/**************************************************************************
 *                                                                        *
 *         Copyright (c) 2012 by iCatch Technology Co., Ltd.             *
 *                                                                        *
 *  This software is copyrighted by and is the property of Sunplus        *
 *  Technology Co., Ltd. All rights are reserved by Sunplus Technology    *
 *  Co., Ltd. This software may only be used in accordance with the       *
 *  corresponding license agreement. Any unauthorized use, duplication,   *
 *  distribution, or disclosure of this software is expressly forbidden.  *
 *                                                                        *
 *  This Copyright notice MUST not be removed or modified without prior   *
 *  written consent of Sunplus Technology Co., Ltd.                       *
 *                                                                        *
 *  Sunplus Technology Co., Ltd. reserves the right to modify this        *
 *  software without notice.                                              *
 *                                                                        *
 *  Sunplus Technology Co., Ltd.                                          *
 *  19, Innovation First Road, Science-Based Industrial Park,             *
 *  Hsin-Chu, Taiwan, R.O.C.                                              *
 *                                                                        *
 **************************************************************************/
#include "CaptureRTSPServer.hh"
#include "H264VideoCaptureSMSS.hh"
#include "JPEGVideoCaptureSMSS.hh"
#include "PCMAudioCaptureSMSS.hh"
#include "CaptureFramedSource.hh"

static const char* description_h264 = "H.264 Video. Streamed by iCatchTek.";
static const char* description_mjpg = "Motion JPEG. Streamed by iCatchTek.";
static const char* description_audio = "Audio. Streamed by iCatchTek.";

CaptureRTSPServer*
CaptureRTSPServer::
createNew(UsageEnvironment& env
	, Port ourPort
	, UserAuthenticationDatabase* authDatabase
	, unsigned reclamationTestSeconds)
{
	int ourSocket = setUpOurSocket(env, ourPort);
	if (ourSocket == -1)
		return NULL;

	return new CaptureRTSPServer(env, ourSocket, ourPort, authDatabase, reclamationTestSeconds);
}

CaptureRTSPServer::CaptureRTSPServer(UsageEnvironment& env
	, int ourSocket
	, Port ourPort
	, UserAuthenticationDatabase* authDatabase
	, unsigned reclamationTestSeconds)
	: RTSPServer(env, ourSocket, ourPort, authDatabase, reclamationTestSeconds)
{
}

// called only by createNew();
CaptureRTSPServer::~CaptureRTSPServer()
{
}

void CaptureRTSPServer::reclaim()
{
	Medium::close(this);
}

ServerMediaSession*
CaptureRTSPServer::lookupServerMediaSession(char const* streamName, char const* fullRequestStr)
{
	NDKStMode st_mode = ndk_st_get_mode();
	UINT32 audio_on = ndk_st_is_audio_on();
	long st_type = NDK_ST_TYPE_DEFAULT;

	ServerMediaSession* sms = RTSPServer::lookupServerMediaSession(streamName, fullRequestStr);
	if (sms)
		return sms;

	if (!ndk_st_send_event(NDK_ST_EVT_RTSP_REQUEST, (UINT32)streamName))
		return NULL;

	ndk_st_get_attribute(NDK_ST_ATTR_STREAMING_TYPE, &st_type);
	if (st_type == NDK_ST_TYPE_DEFAULT)
		st_type = ndk_st_parse_streaming_type(streamName);

	printf("CaptureRTSPServer: type=%ld\n", st_type);
	// Streaming Type and Mode checking.
	switch (st_type) {
	case NDK_ST_TYPE_JPEG:
		switch (st_mode) {
		case NDK_ST_MONO_JPEG:
		case NDK_ST_DUAL_STREAMING:
		case NDK_ST_DUAL_H264JPEG:
		case NDK_ST_MULTI_STREAM:
			break;
		default: ndk_err_return(NULL);
		}
		break;
	case NDK_ST_TYPE_H264S0:
		switch (st_mode) {
		case NDK_ST_MONO_H264:
		case NDK_ST_DUAL_H264JPEG:
		case NDK_ST_MULTI_STREAM:
			break;
		default: ndk_err_return(NULL);
		}
		break;
	case NDK_ST_TYPE_H264S1:
		switch (st_mode) {
#ifdef ICAT_STREAMING_H264H264
		case NDK_ST_MULTI_STREAM: break;
#endif
		default: ndk_err_return(NULL);
		}
		break;
	case NDK_ST_TYPE_AUDIO:
		if (!audio_on) ndk_err_return(NULL);
		break;
	case NDK_ST_TYPE_FILE:
		break;
	default:
		ndk_err_return(NULL);
	}

	if (st_type == NDK_ST_TYPE_JPEG) {
		sms = createCaptureSMS(streamName, "MJPG", description_mjpg
			, (SMSSCreateNewFunc)JPEGVideoCaptureSMSS::createNew
			, NDK_ST_FIFO_JPEG
			, audio_on ? (SMSSCreateNewFunc)AudioCaptureSMSS::createNew : NULL
			, audio_on ? NDK_ST_FIFO_AUDIO : NDK_ST_FIFO_INVALID);
	}
	else if (st_type == NDK_ST_TYPE_H264S0) {
		sms = createCaptureSMS(streamName, "H264S0", description_h264
			, (SMSSCreateNewFunc)H264VideoCaptureSMSS::createNew
			, NDK_ST_FIFO_H264S0
			, audio_on ? (SMSSCreateNewFunc)AudioCaptureSMSS::createNew : NULL
			, audio_on ? NDK_ST_FIFO_AUDIO : NDK_ST_FIFO_INVALID);
	}
	else if (st_type == NDK_ST_TYPE_H264S1) {
#ifdef ICAT_STREAMING_H264H264
		sms = createCaptureSMS(streamName, "H264S1", description_h264
			, (SMSSCreateNewFunc)H264VideoCaptureSMSS::createNew
			, NDK_ST_FIFO_H264S1
			, audio_on ? (SMSSCreateNewFunc)AudioCaptureSMSS::createNew : NULL
			, audio_on ? NDK_ST_FIFO_AUDIO : NDK_ST_FIFO_INVALID);
#else
		ndk_err_return(NULL);
#endif
	}
	else if (st_type == NDK_ST_TYPE_AUDIO) {
		sms = createCaptureSMS(streamName, "AUD", description_audio
			, NULL
			, NDK_ST_FIFO_INVALID
			, (SMSSCreateNewFunc)AudioCaptureSMSS::createNew
			, NDK_ST_FIFO_AUDIO);
	}
	else {
#ifdef ICAT_STREAMING_FILE
		// File, such as DCIM/100MEDIA/SUNP0001.MOV?O=AV
		char _fileName[FILENAME_MAX];
		char const *fileName, *option;

		BOOL fAudio = FALSE, fVideo = FALSE;

		option = strchr(streamName, '?');
		if (option) {
			if ((UINT32)(option - streamName) > (UINT32)(sizeof(_fileName)-1)) {
				ndk_error("URL is too long\n");
				return NULL;
			}

			memcpy(_fileName, streamName, option - streamName);
			_fileName[option - streamName] = 0;
			fileName = _fileName;

			if (!strcasecmp(option, "?O=AV") || !strcasecmp(option, "?O=VA")) {
				fAudio = fVideo = TRUE;
			}
			else if (!strcasecmp(option, "?O=A")) {
				fAudio = TRUE;
			}
			else if (!strcasecmp(option, "?O=V")) {
				fVideo = TRUE;
			}
			else {
				ndk_err_return(NULL);
			}
		}
		else {
			fileName = streamName;
			fAudio = fVideo = TRUE;
		}

		sms = createMediaSrcSMS(streamName, fileName, fAudio, fVideo);
#endif
	}

	if (sms) {
		// Check if it is bICatchMediaClient
		if (strstr(fullRequestStr, NDK_MEDIACLI_APPNAME_V1)) {
			ServerMediaSubsession *smss;
			ServerMediaSubsessionIterator iter(*sms);

			ndk_info("iCatchMediaClient found");

			while ((smss = iter.next()) != NULL) {
				CaptureServerMediaSubsession *csmss = (CaptureServerMediaSubsession *)smss;
				csmss->addSourceFlags(CaptureServerMediaSubsession::FLG_AUD_MARKER_PER_FRAME);
			}
		}
	}
	else
		ndk_error("!sms");

	return sms;
}

ServerMediaSession *CaptureRTSPServer::
createCaptureSMS(char const *streamName, char const* info, char const* description
	, SMSSCreateNewFunc videoCreateNewFunc, NDKFrmFifoType videoFifoType
	, SMSSCreateNewFunc audioCreateNewFunc, NDKFrmFifoType audioFifoType)
{
	NDK_ASSERT(videoCreateNewFunc || audioCreateNewFunc);

	ServerMediaSession *sms = NULL;
	CaptureServerMediaSubsession *v_smss = NULL;
	CaptureServerMediaSubsession *a_smss = NULL;
	NSCapFrmFifoMgr *fifoMgr = NULL;
	bool ret = false;

	do {
		sms = ServerMediaSession::createNew(envir(), streamName, info, description);
		if (!sms)
			return NULL;

		fifoMgr = new NSCapFrmFifoMgr(streamName, audioFifoType, videoFifoType);
		if (!fifoMgr || !fifoMgr->openFIFO())
			break;

		if (videoCreateNewFunc) {
			v_smss = videoCreateNewFunc(envir(), fifoMgr, NULL);
			if (!v_smss)
				break;
		}

		if (audioCreateNewFunc) {
			a_smss = audioCreateNewFunc(envir(), fifoMgr, v_smss);
			if (!a_smss)
				break;
		}

		if (v_smss)
			sms->addSubsession(v_smss);
		if (a_smss)
			sms->addSubsession(a_smss);

		addServerMediaSession(sms);
		ret = true;
	} while (0);

	// The createNewFunc will ref the fifo group. So here must call unref once. Otherwise, the fifo group
	// will never be freed.
	NSFrmFifoMgr::unref(fifoMgr);

	if (ret) {
		return sms;
	}
	else {
		Medium::close(v_smss);
		Medium::close(a_smss);
		Medium::close(sms);
		return NULL;
	}
}

#ifdef ICAT_STREAMING_FILE
ServerMediaSession *CaptureRTSPServer::
createMediaSrcSMS(char const *streamName, char const *fileName, BOOL fAudio, BOOL fVideo)
{
	ServerMediaSession *sms = NULL;
	CaptureServerMediaSubsession *v_smss = NULL;
	CaptureServerMediaSubsession *a_smss = NULL;
	char const* info = NULL;
	char const* description = NULL;
	bool ret = false;

	NSMediaSrcFrmFifoMgr *fifoMgr = new NSMediaSrcFrmFifoMgr(ndk_st_get_rootdir(), fileName, fAudio, fVideo);
	if (!fifoMgr || !fifoMgr->openFIFO())
		return NULL;

	UINT32 v_codec = fifoMgr->getMediaSrc()->getCodecType(false);
	UINT32 a_codec = fifoMgr->getMediaSrc()->getCodecType(true);

	if (fVideo && v_codec != SP5K_MEDIA_VIDEO_UNKNOWN) {
		switch (v_codec) {
		case SP5K_MEDIA_VIDEO_H264:
			v_smss = H264VideoCaptureSMSS::createNew(envir(), fifoMgr, NULL);
			info = "H264";
			description = description_h264;
			break;
		case SP5K_MEDIA_VIDEO_MJPG:
			v_smss = JPEGVideoCaptureSMSS::createNew(envir(), fifoMgr, NULL);
			info = "MJPG";
			description = description_mjpg;
			break;
		default:
			break;
		}

		if (!v_smss)
			goto lExit;
	}

	if (fAudio && a_codec != SP5K_MEDIA_AUDIO_UNKNOWN) {
		switch (a_codec) {
		case SP5K_MEDIA_AUDIO_PCM:
			a_smss = AudioCaptureSMSS::createNew(envir(), fifoMgr, v_smss);
			if (!info) // If no video
				info = "PCM";
			if (!description)
				description = description_audio;
			break;
		default:
			NDK_ASSERT(0);
			break;
		}

		if (!a_smss)
			goto lExit;
	}

	if (!v_smss && !a_smss) {
		ndk_error("no smss");
		goto lExit;
	}

	sms = ServerMediaSession::createNew(envir(), streamName, info, description);
	if (!sms)
		goto lExit;

	if (v_smss)
		sms->addSubsession(v_smss);
	if (a_smss)
		sms->addSubsession(a_smss);

	addServerMediaSession(sms);
	ret = true;

lExit:
	NSFrmFifoMgr::unref(fifoMgr);

	if (ret) {
		return sms;
	}
	else {
		Medium::close(v_smss);
		Medium::close(a_smss);
		Medium::close(sms);
		return NULL;
	}
}
#endif

