/**************************************************************************
 *                                                                        *
 *         Copyright (c) 2012 by iCatch Technology Co., Ltd.             *
 *                                                                        *
 *  This software is copyrighted by and is the property of Sunplus        *
 *  Technology Co., Ltd. All rights are reserved by Sunplus Technology    *
 *  Co., Ltd. This software may only be used in accordance with the       *
 *  corresponding license agreement. Any unauthorized use, duplication,   *
 *  distribution, or disclosure of this software is expressly forbidden.  *
 *                                                                        *
 *  This Copyright notice MUST not be removed or modified without prior   *
 *  written consent of Sunplus Technology Co., Ltd.                       *
 *                                                                        *
 *  Sunplus Technology Co., Ltd. reserves the right to modify this        *
 *  software without notice.                                              *
 *                                                                        *
 *  Sunplus Technology Co., Ltd.                                          *
 *  19, Innovation First Road, Science-Based Industrial Park,             *
 *  Hsin-Chu, Taiwan, R.O.C.                                              *
 *                                                                        *
 **************************************************************************/

#include "JPEGVideoCaptureSource.hh"

///////////////////////////////////////////////////////////////////////////////
struct SOF0 {
	UINT8  precision;
	UINT16 height;
	UINT16 width;
	UINT8  comp_nr;
	struct {
		UINT8 id;
		UINT8 samp_factor;
		UINT8 quant_tbl_no;
	} comps[3];
} __attribute__((__packed__));

///////////////////////////////////////////////////////////////////////////////
#define JPEG_CAPSRC_NUM                 (NDK_ARRAY_SIZE(g_capsrcs))
#ifndef MILLION
#define MILLION 1000000
#endif

///////////////////////////////////////////////////////////////////////////////
// Class JPEGVideoCaptureSource
JPEGVideoCaptureSource *JPEGVideoCaptureSource::createNew(
	UsageEnvironment& env,
	CaptureServerMediaSubsession &capSMSS,
	unsigned sessId)
{
	return new JPEGVideoCaptureSource(env, capSMSS, sessId);
}

JPEGVideoCaptureSource::JPEGVideoCaptureSource(UsageEnvironment& env,
	CaptureServerMediaSubsession &capSMSS,
	unsigned sessId)
: CaptureFramedSource(env, capSMSS, sessId)
{
	//ndk_info("++ %d %s", _objid, name());

	tmrTimeStampGet(&fLastSentTime);

	fImgWidth  = 0;
	fImgHeight = 0;
	fImgType   = 1;
	fQFactor   = 0;
	fDqtLen    = 0;
	fDqtPrec   = 0;

	profLogPrintf(0, "JpgVidCapSrc%d: fifo idx: "NSFRMIDX_FMT", "NSFRMIDX_FMT, fCliId
		, capSMSS.getFIFO().getHeadFrmIdx()
		, capSMSS.getFIFO().getTailFrmIdx());
}

JPEGVideoCaptureSource::~JPEGVideoCaptureSource()
{
	//ndk_info("-- %d %s", _objid, name());
}

UINT8 JPEGVideoCaptureSource::type()
{
	return fImgType;
}

UINT8 JPEGVideoCaptureSource::qFactor()
{
	return 128; //(_qFactor = (UINT8)ndk_st_get_jpeg_q_factor());
}

UINT8 JPEGVideoCaptureSource::width()
{
	return fImgWidth;
}

UINT8 JPEGVideoCaptureSource::height()
{
	return fImgHeight;
}

UINT16 JPEGVideoCaptureSource::restartInterval()
{
	return 0;
}

UINT8 const* JPEGVideoCaptureSource::quantizationTables(UINT8& precision, UINT16& length)
{
	precision = fDqtPrec;
	length    = fDqtLen;
	return fDqtTable;
}

bool JPEGVideoCaptureSource::parseDQT(CaptureFramedSource::FrmReadInf *frinf, int len)
{
	if (fDqtLen > sizeof(fDqtTable))
		ndk_err_return(false);

	NSFrame *frm = frinf->frm;
	UINT8 b;
	int dqt_nr, dqt_id;
	UINT8 *buf = &fDqtTable[0];

	fDqtPrec = 0;

	for (dqt_nr = (len > (64+1)) ? 2 : 1; dqt_nr > 0; --dqt_nr) {
		if (!frm->readByte(frinf->startPos, &b)) // precision and id
			ndk_err_return(false);

		if ((b>>4) != 0)
			ndk_err_return(false);

		dqt_id = b & 0xFF;
		if (dqt_id > 4)
			ndk_err_return(false);

		if (!frm->readBytes(frinf->startPos, buf + dqt_id*64, 64))
			ndk_err_return(false);
		fDqtLen += 64;
	}

	return true;
}

bool JPEGVideoCaptureSource::parseSOF(CaptureFramedSource::FrmReadInf *frinf)
{
	NSFrame *frm = frinf->frm;
	UINT16 len;
	struct SOF0 sof0;

	if (!frm->readWord(frinf->startPos, (UINT8*)&len))
		ndk_err_return(false);

	if (len != sizeof(struct SOF0) + 2)
		ndk_err_return(false);

	if (!frm->readBytes(frinf->startPos, (UINT8*)&sof0, sizeof(sof0)))
		ndk_err_return(false);

	UINT16 w, h;

	w = CONST_SWAP_16(sof0.width);
	h = CONST_SWAP_16(sof0.height);
	if ((w & 0x07) || (h & 0x07))
		ndk_err_return(false);

	fImgWidth  = w >> 3; //divide by 8
	fImgHeight = h >> 3;

	UINT8 horz_samp_factor = sof0.comps[0].samp_factor >> 4;
	UINT8 vert_samp_factor = sof0.comps[0].samp_factor & 0x0f;

	if (horz_samp_factor != 2)
		ndk_err_return(false);

	if (vert_samp_factor == 1)
		fImgType = 0;
	else if (vert_samp_factor == 2)
		fImgType = 1;
	else
		ndk_err_return(false);

	return true;
}

bool JPEGVideoCaptureSource::parseFrame(CaptureFramedSource::FrmReadInf *frinf)
{
	bool parse_result;

#ifdef ICAT_DUALSTREAM
	bool dualstream_jpeg = false;

	if (fFrmFifo.getMediaSrc() && fFrmFifo.getMediaSrc()->isDualStreamJPEG())
		dualstream_jpeg = true;

	if (dualstream_jpeg) {
		parse_result = parseJpegData(frinf);
	}
	else
#endif
	{
		parse_result = parseJfifHeader(frinf);
	}

	return parse_result;
}

bool JPEGVideoCaptureSource::parseJfifHeader(CaptureFramedSource::FrmReadInf *frinf)
{
	NSFrame *frm = frinf->frm;
	UINT16 marker, len;
	char finished = 0;

	fImgWidth  = 0;
	fImgHeight = 0;
	fImgType   = 0;
	fDqtLen    = 0;
	fDqtPrec   = 0;

	while (!finished) {
		if (!frm->readWord(frinf->startPos, (UINT8*)&marker)) {
			ndk_error("pos=%d, len=%d", frinf->startPos, frm->length);
			return false;
		}

		//ndk_info("Marker=%04X pos=%d", marker, _frmInfo.readPos-2);

		switch (marker) {
		case 0xFFC0:
			if (!parseSOF(frinf))
				ndk_err_return(false);
			break;
		case 0xFFDB:
			if (!frm->readWord(frinf->startPos, (UINT8*)&len))
				ndk_err_return(false);
			if (!parseDQT(frinf, len - 2))
				ndk_err_return(false);
			break;
		case 0xFFDD:
			if (!frm->readWord(frinf->startPos, (UINT8*)&len))
				ndk_err_return(false);
			if (!frm->readWord(frinf->startPos, (UINT8*)&len))
				ndk_err_return(false);
			break;
		case 0xFFDA:
			if (!frm->readWord(frinf->startPos, (UINT8*)&len))
				ndk_err_return(false);
			frm->skipBytes(frinf->startPos, len-2);
			finished = 1;
			break;
		case 0xFFFF:
		case 0xFFD8:
			break;
		//case 0xFFD9:
		case 0xFFC1:
		case 0xFFC2:
		case 0xFFC3:
		case 0xFFC5:
		case 0xFFC6:
		case 0xFFC7:
		case 0xFFC8:
		case 0xFFC9:
		case 0xFFCA:
		case 0xFFCB:
		case 0xFFCD:
		case 0xFFCE:
		case 0xFFCF:
		case 0xFFC4:
		case 0xFFCC:
		case 0xFFD0:
		case 0xFFD1:
		case 0xFFD2:
		case 0xFFD3:
		case 0xFFD4:
		case 0xFFD5:
		case 0xFFD6:
		case 0xFFD7:
		case 0xFFDC:
		case 0xFFDE:
		case 0xFFDF:
		case 0xFFE0:
		case 0xFFE1:
		case 0xFFE2:
		case 0xFFE3:
		case 0xFFE4:
		case 0xFFE5:
		case 0xFFE6:
		case 0xFFE7:
		case 0xFFE8:
		case 0xFFE9:
		case 0xFFEA:
		case 0xFFEB:
		case 0xFFEC:
		case 0xFFED:
		case 0xFFEE:
		case 0xFFEF:
		case 0xFFF0:
		case 0xFFFD:
		case 0xFFFE:
		case 0xFF01:
			if (!frm->readWord(frinf->startPos, (UINT8*)&len))
				ndk_err_return(false);
			frm->skipBytes(frinf->startPos, len-2);
			break;
		default:
			printf("Unknow marker 0x%04X\n", marker);
			break;
		}
	}

	return true;
}

#ifdef ICAT_DUALSTREAM
bool JPEGVideoCaptureSource::parseJpegData(CaptureFramedSource::FrmReadInf *frinf)
{
	NSDsjFrame *dsjfrm = (NSDsjFrame *)frinf->frm;
	mediaRecDualStreamJpegAttr_t *jpegAttr = &dsjfrm->jpegAttr;

	//ndk_hexdump("jpeg", jpegAttr, sizeof(*jpegAttr), sizeof(*jpegAttr));

	fQFactor = 128;
	fImgWidth   = (UINT8)(jpegAttr->width >> 3);
	fImgHeight  = (UINT8)(jpegAttr->height >> 3);
	//printf("w=%x, h=%x\n", _width, _height);
	memcpy(fDqtTable, jpegAttr->qtable, sizeof(jpegAttr->qtable));
	fDqtLen  = sizeof(jpegAttr->qtable);
	fDqtPrec = 0;

	if (jpegAttr->dataFmt == 0) /*EXIF_YUV420*/
		fImgType = 1;
	else
		fImgType = 0;

	frinf->startPos = jpegAttr->vlcPos;

	//ndk_hexdump("jpeg data", frmhdr->data + jpegAttr->vlcPos, jpegAttr->vlcSize, 128);
	return true;
}
#endif

Boolean JPEGVideoCaptureSource::prepareFrame()
{
	if (!CaptureFramedSource::prepareFrame())
		return false;

	if (parseFrame(&fFrmReadInf) && (fFrmReadInf.remainder() > 0)/*sanity check*/) {
		return true;
	}
	else {
		ndk_error("parse");
		fFrmReadInf.detach();
		return false;
	}
}

CaptureFramedSource::FrameStatus JPEGVideoCaptureSource::updateFrameStatus()
{
	FrameStatus frmstat = checkFrame();

#ifdef ICAT_DUALSTREAM
	if (ndk_st_is_dualstream_jpeg() && getFifo().isLiveMode() && fFrmReadInf.frm) {
		// Current frame is fall behind the queue head. Clone it.
		if (fFrmFifo.getHeadFrmIdx() > fFrmReadInf.idx + fFrmFifo.getQueueSize()/2) {
			NSFrame *newfrm = fFrmReadInf.frm->clone();
			if (newfrm) {
				NSFrame::unref(fFrmReadInf.frm);
				fFrmReadInf.frm = newfrm;
			}
			else {
				ndk_warning("clone");
			}
		}
	}
#endif

	return frmstat;
}

void JPEGVideoCaptureSource::rtpPayloadIterate(unsigned payloadSizeMax, RTPPayloadVec &rpvec, RTPPayloadInf &rpinf)
{
	NDK_ASSERT(fFrmReadInf.frm);

	u_int8_t *buf = fHeaderBuffer;
	rpinf.index = fFrmReadInf.frm->index;
	rpinf.size = 0;

	// Pack jepg headers.
	*(buf++) = 0; // Type-specific
	*(buf++) = ((UINT8 *)&fFrmReadInf.readPos)[2];
	*(buf++) = ((UINT8 *)&fFrmReadInf.readPos)[1];
	*(buf++) = ((UINT8 *)&fFrmReadInf.readPos)[0];

	*(buf++) = this->type();
	*(buf++) = this->qFactor();
	*(buf++) = this->width();
	*(buf++) = this->height();

	NDK_ASSERT(payloadSizeMax >= 8);
	payloadSizeMax -= 8;

	rpinf.size += 8;
	rpinf.specialHeaderSize = 8;
	rpinf.pts = fFrmReadInf.frm->pts;
	rpinf.timeCreated = fFrmReadInf.frm->timeCreated;
	rpinf.duration = (long)fFrmReadInf.frm->duration;

	if (fFrmReadInf.readPos == 0) {
		if (this->qFactor() >= 128) {
			// There is also a Quantization Header:
			UINT8  precision;
			UINT16 length;
			UINT8 const *qTables = this->quantizationTables(precision, length);
			NDK_ASSERT(length <= 256);

			*(buf++) = 0; // MBZ
			*(buf++) = precision;
			*(buf++) = length >> 8;
			*(buf++) = length & 0xFF;
			memcpy(buf, qTables, length);
			buf += length;

			rpinf.size += (4 + length);
			rpinf.specialHeaderSize += (length + 4);

			NDK_ASSERT(payloadSizeMax >= (unsigned)(length + 4));
			payloadSizeMax -= (4 + length);
		}
	}

	rpvec.append(fHeaderBuffer, rpinf.size);

	UINT32 nRead = fFrmReadInf.remainder();
	if (nRead > payloadSizeMax)
		nRead = payloadSizeMax;

	rpvec.append(fFrmReadInf.advance(nRead), nRead);
	rpinf.size += nRead;

	rpinf.lastPacket = fFrmReadInf.end();
	if (rpinf.lastPacket) {
		tmrTimeStampGet(&fLastSentTime);
	}
}

