/**************************************************************************
 *                                                                        *
 *         Copyright (c) 2014 by iCatch Technology Co., Ltd.             *
 *                                                                        *
 *  This software is copyrighted by and is the property of Sunplus        *
 *  Technology Co., Ltd. All rights are reserved by Sunplus Technology    *
 *  Co., Ltd. This software may only be used in accordance with the       *
 *  corresponding license agreement. Any unauthorized use, duplication,   *
 *  distribution, or disclosure of this software is expressly forbidden.  *
 *                                                                        *
 *  This Copyright notice MUST not be removed or modified without prior   *
 *  written consent of Sunplus Technology Co., Ltd.                       *
 *                                                                        *
 *  Sunplus Technology Co., Ltd. reserves the right to modify this        *
 *  software without notice.                                              *
 *                                                                        *
 *  Sunplus Technology Co., Ltd.                                          *
 *  19, Innovation First Road, Science-Based Industrial Park,             *
 *  Hsin-Chu, Taiwan, R.O.C.                                              *
 *                                                                        *
 **************************************************************************/

#include "VideoCaptureRTPSink.hh"

#ifdef SENDPACKET_UDP_PBLIST
struct RTPPktPBuf
{
	struct pbuf_custom pc;
	char mem[64 + RTP_PAYLOAD_SIZE];
};

static SP5K_QUEUE rppb_que = 0;
static int rppb_refcnt = 0;

void rtp_pktpbuf_global_init()
{
	if (rppb_que) return;
	sp5kOsQueueCreate(&rppb_que, (char*)"RTPPktBuf", sizeof(RTPPktPBuf*)/sizeof(UINT32), NULL, 64*sizeof(RTPPktPBuf*));
	NDK_ASSERT(rppb_que);
}

static void rtp_pktpbuf_free(struct pbuf *p)
{
	RTPPktPBuf *rb = (struct RTPPktPBuf*)p;

	if (rppb_refcnt == 0 || sp5kOsQueueSend(&rppb_que, &rb, TX_NO_WAIT) != SUCCESS) {
		ndk_mem_free(rb);
	}
}

static void rtp_pktpbuf_free_all()
{
	struct RTPPktPBuf *rb = NULL;

	while (sp5kOsQueueReceive(&rppb_que, &rb, TX_NO_WAIT) == SUCCESS) {
		ndk_mem_free(rb);
		rb = NULL;
	}

	profLogPrintf(0, "RTPPktPBuf: free all");
}

static struct pbuf *rtp_pktpbuf_alloc(unsigned paylaodsize)
{
	RTPPktPBuf  *rb = NULL;
	struct pbuf *pb = NULL;

	if (sp5kOsQueueReceive(&rppb_que, &rb, TX_NO_WAIT) != SUCCESS) {
		rb = (RTPPktPBuf*)ndk_mem_malloc(sizeof(RTPPktPBuf));
	}

	if (rb) {
		//memset(&rb->pc, 0, sizeof(rb->pc));
		pb = pbuf_alloced_custom(PBUF_TRANSPORT, (u16_t)paylaodsize, PBUF_RAM, &rb->pc, rb->mem, sizeof(rb->mem));
		NDK_ASSERT(pb);
		rb->pc.custom_free_function = rtp_pktpbuf_free;
	}

	return pb;
}
#else
void rtp_pktpbuf_global_init()
{
}
#endif // SENDPACKET_UDP_PBLIST

///////////////////////////////////////////////////////////////////////////////
CaptureRTPSink::
CaptureRTPSink(UsageEnvironment &env
	, Groupsock *RTPgs
	, unsigned char rtpPayloadFormat /* equals to rtpPayloadType */
	, unsigned rtpTimestampFrequency
	, char const *rtpPayloadFormatName
	, unsigned numChannels
	, Boolean audio)
// Init Parent Class
: RTPSink(env
, RTPgs
, rtpPayloadFormat
, rtpTimestampFrequency
, rtpPayloadFormatName
, numChannels)
{
	fMaxOutPacketNumInOnePlaying = 7;
	fAudio = audio;
	fPrefix = fAudio ? "aud" : "vid";

	long payloadSize = 0;
	ndk_st_get_attribute(NDK_ST_ATTR_RTP_PAYLOAD_SIZE, &payloadSize);
	fRTPPayloadSize = (payloadSize <= 0 || payloadSize > RTP_PAYLOAD_SIZE) ? RTP_PAYLOAD_SIZE : payloadSize;

	//printf("%sCapRTPSnk: payload size = %u\n", fPrefix, fRTPPayloadSize);

#ifdef SENDPACKET_UDP_PBLIST
	++rppb_refcnt;
#endif
}

CaptureRTPSink::~CaptureRTPSink()
{
#ifdef SENDPACKET_UDP_PBLIST
	if (--rppb_refcnt == 0)
		rtp_pktpbuf_free_all();
#endif
}

char const *CaptureRTPSink::sdpMediaType() const
{
	return fAudio ? "audio" : "video";
}

Boolean CaptureRTPSink::onBeforePlaying()
{
	return (ndk_st_get_netload(envir().ifname, NULL) == NDK_ST_NETLOAD_HIGH) ? False : True;
}

Boolean CaptureRTPSink::continuePlaying()
{
	NDK_ASSERT(fSource);
	profLogPrintf(0, "Continue Playing: %c", fAudio?'A':'V');

	fFirstPlaying = True;
	fStartTime = mtime_now_get();
	fStartPTS  = -1;

	envir().taskScheduler().scheduleDelayedTask(0, (TaskFunc *)sendNextPacket, this);
	return True;
}

void CaptureRTPSink::buildRTPHeader(u_int8_t *rtphdr,
	RTPPayloadInf const &rpinf,
	u_int32_t ssrc,
	unsigned bPadding,
	Boolean firstPacket,
	Boolean bExten,
	RTPExtenHeader *&extenHeader)
{
	u_int8_t *wrptr = rtphdr;
	Boolean bMarker;

	if (fFirstPlaying) {
		fFirstPlaying = False;
		fInitialPresentationTime = rpinf.pts;
		//ndk_info("InitialPTS: %s", tv2str(&fInitialPresentationTime));
	}

	if (firstPacket) {
		fMostRecentPresentationTime = rpinf.pts;
		fCurrentTimestamp = convertToRTPTimestamp(rpinf.pts);
		//profLogPrintf(0, "%c => %u", fAudio?'A':'V', fCurrentTimestamp);
	}

	if (fAudio) {
		fCurrentTimestamp = convertToRTPTimestamp(rpinf.pts);
		// Generally, marker is added to every audio packet.
		// If rpinf.markerPerFrame is true, the marker bit is added only to the last packet of the frame.
		bMarker = !rpinf.markerPerFrame || rpinf.lastPacket;
	}
	else {
		bMarker = rpinf.lastPacket;
	}

	// Set up the RTP header:
	// RTP version 2; marker ('M') bit not set (by default; it can be set later)
	// 0 ~ 3
	if (bExten) {
		*(wrptr++) = (0x80 | RTP_EXTEN_HEADER_BIT) | (bPadding ? 0x20 : 0);
	}
	else {
		*(wrptr++) = 0x80 | (bPadding ? 0x20 : 0);
	}

	*(wrptr++) = fRTPPayloadType | (bMarker ? 0x80 : 0);
	*(wrptr++) = ((u_int8_t*)&fSeqNo)[1];
	*(wrptr++) = ((u_int8_t*)&fSeqNo)[0];

	// 4 ~ 7
	*((u_int32_t*)wrptr) = platform_htonl(fCurrentTimestamp);
	wrptr += 4; // Timestamp

	// 8 ~ 11
	*((u_int32_t*)wrptr) = ssrc;
	wrptr += 4;

	// CSRC is ignored.

	if (bExten)
		extenHeader = (RTPExtenHeader *)wrptr;
	else
		extenHeader = NULL;
}

void CaptureRTPSink::buildExtenHeader(RTPExtenHeader *extenHeader, RTPPayloadInf const &rpinf)
{
	int32_t msec = (int32_t)(rpinf.timeCreated.tv_usec/1000);

	extenHeader->ehl1 = 0;
	extenHeader->ehl0 = (RTP_EXTEN_HEADER_SIZE - 4)/4;
	extenHeader->creationTimeSec = rpinf.timeCreated.tv_sec;
	extenHeader->creationTimeMsecHigh = (u_int8_t)((msec >> 8) & 0x03);
	extenHeader->creationTimeMsecLow = (u_int8_t)(msec & 0xFF);
	extenHeader->index = rpinf.index;
}

Boolean CaptureRTPSink::preCheckPlaying(CaptureFramedSource *capsrc, Boolean &continuePlayingResult)
{
	CaptureFramedSource::FrameStatus frmstat = capsrc->checkFrameReady();
	TaskScheduler& sched = envir().taskScheduler();
	Boolean rescheTask = False;

	continuePlayingResult = True;

	switch (frmstat) {
	case CaptureFramedSource::FRAME_READY:
		break;

	case CaptureFramedSource::FRAME_NOT_READY:
		nextTask() = sched.scheduleDelayedTask((int)frmstat * 1000, (TaskFunc*)sendNextPacket, this);
		return False;

	case CaptureFramedSource::SOURCE_ABORTED:
		ndk_info("%s abort", fPrefix);
	default:
		onSourceClosure(this);
		continuePlayingResult = False;
		return False;
	}

	if (!capsrc->getFifo().isLiveMode()) {
		Boolean newFrame = False;
		SINT64  timePTS;

		if (!capsrc->getFrameInfo(newFrame, timePTS))
			NDK_ASSERT(0);

		if (fStartPTS == -1)
			fStartPTS = timePTS;

		if (newFrame) {
			SINT64 timeNow = mtime_now_get();
			int timeDelay = fStartTime + (timePTS - fStartPTS) + 200 - timeNow;
			//profLogPrintf(0, "precheck %c %d %lld %lld %lld", fAudio?'A':'V', timeDelay, fStartTime, timeNow, timePTS);

			if (timeDelay > 10) {
				nextTask() = sched.scheduleDelayedTask((timeDelay-5)*1000, (TaskFunc*)sendNextPacket, this);
				return False;
			}
		}
	}

	if (fRTPInterface.hasTCPStreams() && ndk_st_defsoc_is_used()) {
		rescheTask = !fRTPInterface.pollWrite(10);
	}
	else {
		rescheTask = !onBeforePlaying();
	}

	if (rescheTask) {
#if 0
		if (!fFirstPlaying && fCorrTvLastPlaying.tv_sec == -1 && fCorrTvLastPlaying.tv_usec == -1)
			tmrTimeStampGet(&fCorrTvLastPlaying);
#endif
		nextTask() = sched.scheduleDelayedTask(10 * 1000, (TaskFunc*)sendNextPacket, this);
		return False;
	}

	return True;
}

#ifdef SENDPACKET_UDP_PBLIST
Boolean CaptureRTPSink::sendPacket_UDP_PBList(
	CaptureFramedSource *capsrc,
	u_int32_t ssrc,
	RTPPayloadInf &rpinf)
{
	struct pbuf *pbList = NULL, **ppb = &pbList, *pb;
	unsigned listDepth = 0;
	Boolean firstPacket, bExten;
	RTPExtenHeader *extenHeader = NULL;

	u_int8_t *outbuf, *outbufEnd, *wrptr;

	for (;;) {
		pb = rtp_pktpbuf_alloc(fRTPPayloadSize);
		if (!pb) {
			pbuf_list_free(pbList);
			ndk_error("%s pbuf", fPrefix);
			return False;
		}
		outbuf = (u_int8_t*)pb->payload;
		outbufEnd = outbuf + fRTPPayloadSize;
		wrptr = outbuf;
		firstPacket = capsrc->rtpPayloadIsFirstPacket();

		// RTP header
		wrptr += RTP_HEADER_SIZE;
		if (RTP_EXTEN_HEADER_SIZE && firstPacket) {
			wrptr += RTP_EXTEN_HEADER_SIZE;
			bExten = True;
		}
		else
			bExten = False;

		//ndk_info("fWrPos = %u", wrptr - outbuf);
		capsrc->readFrameData(wrptr, outbufEnd - wrptr, rpinf);
		wrptr += rpinf.size;

		buildRTPHeader(outbuf, rpinf, ssrc, rpinf.size & 0x01, firstPacket, bExten, extenHeader);
		if (rpinf.size & 0x01) {
			*wrptr++ = 1; // Padding length: 1
		}

		if (bExten)
			buildExtenHeader(extenHeader, rpinf);

		NDK_ASSERT(wrptr <= outbufEnd);
		pb->tot_len = pb->len = (u16_t)(wrptr - outbuf);

		// Chain pbuf into the pbuf-list
		*ppb = pb;
		ppb = &pb->list_next;

		++listDepth;
		++fSeqNo;
		++fPacketCount;

		fTotalOctetCount += pb->len;
		fOctetCount += pb->len - RTP_HEADER_SIZE - rpinf.specialHeaderSize;
		if (bExten)
			fOctetCount += RTP_EXTEN_HEADER_SIZE;

		if (rpinf.lastPacket || listDepth == fMaxOutPacketNumInOnePlaying)
			break;
	}

	if (pbList && !fRTPInterface.sendPacket_pblist(pbList)) {
		ndk_error("%s send", fPrefix);
		return False;
	}

	return True;
}
#endif // SENDPACKET_UDP_PBLIST

Boolean CaptureRTPSink::sendPacket(
	CaptureFramedSource *capsrc,
	u_int32_t ssrc,
	RTPPayloadInf &rpinf)
{
	RTPPayloadVec rpvec;
	u_int8_t rtphdr[RTP_HEADER_SIZE_MAX];
	u_int8_t padding[2];
	Boolean  sendSucceed = True;
	Boolean  bTCPStreams = fRTPInterface.hasTCPStreams();
	Boolean  firstPacket, bExten;
	RTPExtenHeader *extenHeader = NULL;
	unsigned payloadSizeMax = bTCPStreams ? RTP_PAYLOAD_SIZE_MAX : fRTPPayloadSize;
	unsigned numPktOut = 0;
	unsigned rtphdr_len, rpvec_len;

	for (;;) {
		rpvec.reset();
		firstPacket = capsrc->rtpPayloadIsFirstPacket();

		// RTP header
		rtphdr_len = RTP_HEADER_SIZE;
		if (RTP_EXTEN_HEADER_SIZE && firstPacket) {
			rtphdr_len += RTP_EXTEN_HEADER_SIZE;
			bExten = True;
		}
		else
			bExten = False;

		NDK_ASSERT(rtphdr_len <= sizeof(rtphdr));

		rpvec.append(rtphdr, rtphdr_len);
		capsrc->rtpPayloadIterate(payloadSizeMax, rpvec, rpinf);
		rpvec_len = rpvec.length();

		NDK_ASSERT_EXPR(rpvec_len == rpinf.size + rtphdr_len,
			printf("%u != %u + %u\n", rpvec_len, rpinf.size, rtphdr_len);
		);

		buildRTPHeader(rtphdr, rpinf, ssrc, rpvec_len & 0x01, firstPacket, bExten, extenHeader);

		if (rpvec_len & 0x01) {
			// Padding. Let the length be even number. Maybe this can minimize the occurrence of tcpip checksum error.
			padding[0] = 1;
			rpvec.append(padding, 1);
			rpvec_len += 1;
		}

		if (bExten)
			buildExtenHeader(extenHeader, rpinf);

		NDK_ASSERT(rpvec.noError());
		if (bTCPStreams) {
			sendSucceed = fRTPInterface.sendPayloadVec(rpvec);
		}
		else {
			NDK_ASSERT_EXPR(rpvec_len <= sizeof(fRtpOutBuf),
				printf("%u > %u\n", rpvec_len, sizeof(fRtpOutBuf));
			);
			rpvec.copyToBuffer(fRtpOutBuf);
			sendSucceed = fRTPInterface.sendPacket(fRtpOutBuf, rpvec_len);
			//profLogPrintf(0, "VidCapRTPSnk Send %d %d", sendSucceed, rpvec_len);
		}

		if (!sendSucceed) {
			ndk_error("%s send", fPrefix);
			break;
		}

		fTotalOctetCount += rpvec_len;
		// FIXME: Extend Header ?
		fOctetCount += rpvec_len - (RTP_HEADER_SIZE + rpinf.specialHeaderSize);
		if (bExten)
			fOctetCount += RTP_EXTEN_HEADER_SIZE;

		++fSeqNo;
		++fPacketCount;
		++numPktOut;

		if (rpinf.lastPacket || numPktOut == fMaxOutPacketNumInOnePlaying || bTCPStreams)
			break;
	}

	if (rpinf.lastPacket)
		capsrc->rtpPayloadIterateEnd();

	return sendSucceed;
}

Boolean CaptureRTPSink::continuePlaying1()
{
	CaptureFramedSource *capsrc = (CaptureFramedSource*)fSource;
	TaskScheduler& sched = envir().taskScheduler();
	u_int32_t ssrc;
	RTPPayloadInf rpinf;
	Boolean continuePlayingResult;

	memset(&rpinf, 0, sizeof(rpinf));

	if (!preCheckPlaying(capsrc, continuePlayingResult))
		return continuePlayingResult;

	ssrc = SSRC();
	ssrc = platform_htonl(ssrc);

#if 0 // May cause audio pts errors.
	if (fCorrTvLastPlaying.tv_sec != -1 && fCorrTvLastPlaying.tv_usec != -1) {
		struct timeval tv_now;
		tmrTimeStampGet(&tv_now);

		fCorrPTS += timeval_diff_ms(&tv_now, &fCorrTvLastPlaying);;
		fCorrTvLastPlaying.tv_sec = fCorrTvLastPlaying.tv_usec = -1;
	}
#endif

#ifdef SENDPACKET_UDP_PBLIST
	if (ndk_st_defsoc_is_used() && !fRTPInterface.hasTCPStreams()) {
		if (!sendPacket_UDP_PBList(capsrc, ssrc, rpinf)) {
			onSourceClosure(this);
			return False;
		}
	}
	else
#endif
	{
		if (!sendPacket(capsrc, ssrc, rpinf)) {
			onSourceClosure(this);
			return False;
		}
	}

	if (!rpinf.lastPacket || capsrc->getReadyFramesInFifo() > 0 || !capsrc->getFifo().isLiveMode())
		nextTask() = sched.scheduleDelayedTask(0, (TaskFunc*)sendNextPacket, this);
	else
		nextTask() = sched.schedulePostponeTask((TaskFunc*)sendNextPacket, this);

#if 0
	long playing_time = timeval_diff_now_ms(&_tvCurrPlayTime);
	if (playing_time > 300) {
		profLogPrintf(0, "vid play takes %ldms", playing_time);
	}
#endif

	return True;
}

void CaptureRTPSink::sendNextPacket(void *firstArg)
{
	((CaptureRTPSink*)firstArg)->continuePlaying1();
}

///////////////////////////////////////////////////////////////////////////////
VideoCaptureRTPSink::VideoCaptureRTPSink(UsageEnvironment &env, Groupsock *RTPgs,
	unsigned char rtpPayloadFormat, char const *rtpPayloadFormatName)
: CaptureRTPSink(env, RTPgs, rtpPayloadFormat, 90000, rtpPayloadFormatName, 1, False)
{
	fDbrcRatio = 100;
	fDbrcRecentTime = 0;
}

Boolean VideoCaptureRTPSink::onBeforePlaying()
{
	int  loading, level;
	long ratio_min  = ndk_st_get_attr(NDK_ST_ATTR_VDBRC_MIN);

	level = ndk_st_get_netload(envir().ifname, &loading);

	if (ratio_min > 0 && ratio_min < 100 && !(level == NDK_ST_NETLOAD_LOW && fDbrcRatio == 100)) {
		CaptureFramedSource *src = (CaptureFramedSource *)fSource;
		SINT64 time_now = mtime_now_get();
		int ratio_new = 0;

		if (level == NDK_ST_NETLOAD_HIGH) {
			ratio_new = ratio_min;
		}
		else if (level == NDK_ST_NETLOAD_MIDDLE) {
			if (loading > (NDK_ST_NETLOAD_LOW_THRESHOLD + ndk_st_get_attr(NDK_ST_ATTR_NET_LOADING_HIGH))/2)
				ratio_new = ratio_min;
			else
				ratio_new = (100 + ratio_min)/2;
		}
		else {
			ratio_new = 100;
		}

		if (ratio_new > fDbrcRatio) {
			long time_restore = ndk_st_get_attr(NDK_ST_ATTR_VDBRC_RESTORE_TIME);

			if (time_restore > 0) {
				if (time_now - fDbrcRecentTime >= time_restore/5) {
					int r = fDbrcRatio + (100 - ratio_min)/5;
					if (r < ratio_new)
						ratio_new = r;
				}
				else
					ratio_new = 0; // Don't update this time
			}
		}

		if (ratio_new > 0 && ratio_new != fDbrcRatio) {
			if (ratio_new > 100)
				ratio_new = 100;

			src->changeBitrateTo(ratio_new/100.0);
			fDbrcRatio = ratio_new;
			fDbrcRecentTime = time_now;
		}
	}

	return (level == NDK_ST_NETLOAD_HIGH) ? False : True;
}
