/**************************************************************************
 *                                                                        *
 *         Copyright (c) 2012 by iCatch Technology Co., Ltd.             *
 *                                                                        *
 *  This software is copyrighted by and is the property of Sunplus        *
 *  Technology Co., Ltd. All rights are reserved by Sunplus Technology    *
 *  Co., Ltd. This software may only be used in accordance with the       *
 *  corresponding license agreement. Any unauthorized use, duplication,   *
 *  distribution, or disclosure of this software is expressly forbidden.  *
 *                                                                        *
 *  This Copyright notice MUST not be removed or modified without prior   *
 *  written consent of Sunplus Technology Co., Ltd.                       *
 *                                                                        *
 *  Sunplus Technology Co., Ltd. reserves the right to modify this        *
 *  software without notice.                                              *
 *                                                                        *
 *  Sunplus Technology Co., Ltd.                                          *
 *  19, Innovation First Road, Science-Based Industrial Park,             *
 *  Hsin-Chu, Taiwan, R.O.C.                                              *
 *                                                                        *
 **************************************************************************/

#include <UsageEnvironment.hh>
#include <ndk_streaming.h>
#include <livePort/streaming_api.h>
#include <livePort/fifo.h>
#include <livePort/mediasrc.hh>

///////////////////////////////////////////////////////////////////////////////
// Global Variables
static NSFrmFifo       *frmfifo_list_head    = NULL;
static NSFrmFifoClient *frmfifocli_list_head = NULL;

///////////////////////////////////////////////////////////////////////////////
// Statistics
UINT32 NSFrame::staFrmAlloced = 0;
UINT32 NSFrame::staFrmFreed = 0;

///////////////////////////////////////////////////////////////////////////////
// Class NSFrame
void NSFrame::unref(NSFrame *frm)
{
	if (frm) {
		unsigned int x;
		int n;

		ndk_global_lock(&x);
		n = --frm->fRefCnt;
		if (n == 0)
			++staFrmFreed;
		ndk_global_unlock(&x);

		if (n == 0)
			delete frm;

		NDK_ASSERT(n >= 0);
	}
}

void NSFrame::ref()
{
	unsigned int x;

	ndk_global_lock(&x);
	++fRefCnt;
	ndk_global_unlock(&x);
}

NSFrame *NSFrame::clone()
{
	unsigned alloc_size = NDK_ALIGN_TO(sizeof(NSFrame), 4) + this->length;
	UINT8 *p_mem = new UINT8[alloc_size];
	if (!p_mem)
		return NULL;

	NSFrame *newfrm = new(p_mem) NSFrame();
	newfrm->data = p_mem + NDK_ALIGN_TO(sizeof(NSFrame), 4);
	newfrm->length = this->length;
	newfrm->duration = this->duration;
	newfrm->pts = this->pts;
	newfrm->timeCreated = this->timeCreated;

	memcpy(newfrm->data, this->data, this->length);
	return newfrm;
}

///////////////////////////////////////////////////////////////////////////////
enum { MEDIA_SRC_FRMPOOL_SIZE = NSFrmFifo::FIFO_QSIZE*2 + 4 };

NDKStExitFunc NSMediaSrcFrame::fFrmPoolOnExit;
void *NSMediaSrcFrame::fFrmPool = NULL;

NSMediaSrcFrame::~NSMediaSrcFrame()
{
	//profLogPrintf(0, "%p %p -", this, _bufobj);
	mediaSrcReleaseBufferObject(fBufObj);
}

void NSMediaSrcFrame::attachBuffer(void *buffer, void *extra_param)
{
	mediaSrcBuffer_t *msbuf = (mediaSrcBuffer_t *)buffer;
	mediaSrcReleaseBufferObject(this->fBufObj);

	if (msbuf) {
		this->fBufObj   = msbuf->bufobj;
		this->fBufFlags = msbuf->bufflags;
		this->data      = msbuf->data;
		this->length    = msbuf->length;
		this->pts       = msbuf->pts;
		this->duration  = msbuf->duration;
		if (msbuf->enctime.tv_sec == -1 && msbuf->enctime.tv_usec == -1)
			tmrTimeStampGet(&this->timeCreated);
		else
			this->timeCreated = msbuf->enctime;

		mediaSrcHoldBufferObject(msbuf->bufobj);
	}
	else {
		this->fBufObj   = NULL;
		this->data      = NULL;
		this->length    = 0;
	}
}

void NSMediaSrcFrame::frmPoolOnExit(NDKStExitFunc *exit_func)
{
	void *pool;

	ndk_st_sys_protect(-1);
	pool = fFrmPool;
	fFrmPool = 0;
	ndk_st_sys_unprotect();

	ndk_blkpool_destroy(pool);
	profLogPrintf(0, "Clean MsrcFrm Pool");
}

void *NSMediaSrcFrame::operator new (size_t size) throw()
{
	NDK_ASSERT(size == sizeof(NSMediaSrcFrame));

	ndk_st_sys_protect(-1);
	if (!fFrmPool) {
		fFrmPool = ndk_blkpool_create("NSMsrcFrm", (int)sizeof(NSMediaSrcFrame), MEDIA_SRC_FRMPOOL_SIZE);
		if (fFrmPool) {
			fFrmPoolOnExit.on_exit = frmPoolOnExit;
			fFrmPoolOnExit.user_data = NULL;
			ndk_st_register_exitfunc(&fFrmPoolOnExit);
		}
		else {
			ndk_st_sys_unprotect();
			return NULL;
		}
	}
	ndk_st_sys_unprotect();

	return ndk_blkpool_alloc_block(fFrmPool);
}

void NSMediaSrcFrame::operator delete (void* ptr)
{
	if (ptr) {
		NDK_ASSERT(fFrmPool);
		ndk_blkpool_free_block(fFrmPool, ptr);
	}
}

///////////////////////////////////////////////////////////////////////////////
NSCustomMediaFrame::NSCustomMediaFrame(void (*fpFreeBufferObject)(NDKStMediaBufferObject))
: NSFrame()
, fFreeBufferObjectFunc(fpFreeBufferObject)
, fBufObj(NULL)
{
	fFlgBufType    = NDK_ST_MEDIABUFFER_UNKNOW;
	fFlgKeyFrame   = 0;
	fFlgParamFrame = 0;
}

NSCustomMediaFrame::~NSCustomMediaFrame()
{
	if (fBufObj)
		fFreeBufferObjectFunc(fBufObj);
}

void NSCustomMediaFrame::attachBuffer(void *buffer, void *extra_param)
{
	NDKStMediaBuffer *mbuf = (NDKStMediaBuffer *)buffer;

	if (this->fBufObj)
		fFreeBufferObjectFunc(this->fBufObj);

	if (mbuf) {
		this->fBufObj      = mbuf->bufobj;
		this->fFlgBufType  = mbuf->f_buftype;
		this->fFlgKeyFrame = mbuf->f_keyframe;
		this->fFlgParamFrame = mbuf->f_paramframe;
		this->data         = mbuf->data;
		this->length       = mbuf->length;
		this->pts          = mbuf->pts;
		this->duration     = mbuf->duration;
		tmrTimeStampGet(&this->timeCreated);
	}
	else {
		this->fBufObj = NULL;
		this->data    = NULL;
		this->length  = 0;
	}
}

///////////////////////////////////////////////////////////////////////////////
#ifdef ICAT_DUALSTREAM_JPEG
NSDsjFrame::~NSDsjFrame()
{
	mediaSrcReleaseBufferObject(fBufObj);
}

void NSDsjFrame::attachBuffer(void *buffer, void *extra_param)
{
	mediaSrcBuffer_t *msbuf = (mediaSrcBuffer_t *)buffer;
	mediaRecDualStreamJpegAttr_t *attr = (mediaRecDualStreamJpegAttr_t *)extra_param;

	mediaSrcReleaseBufferObject(this->fBufObj);

	if (msbuf) {
		this->fBufObj  = msbuf->bufobj;
		this->data     = msbuf->data;
		this->length   = msbuf->length;
		this->pts      = msbuf->pts;
		tmrTimeStampGet(&this->timeCreated);
		this->duration = 0;
		memcpy(&this->jpegAttr, attr, sizeof(this->jpegAttr));
		mediaSrcHoldBufferObject(msbuf->bufobj);
	}
	else {
		this->fBufObj  = NULL;
		this->data     = NULL;
		this->length   = 0;
	}
}
#endif

///////////////////////////////////////////////////////////////////////////////
// NSFrmFifoClient
void NSFrmFifoClient::FrmReadInf::transferTo(NSFrmFifoClient::FrmReadInf& f)
{
	NDK_ASSERT(!f.frm);

	f.frm = frm;
	f.idx = idx;
	f.readPos = readPos;
	f.startPos = startPos;

	frm = NULL;
	idx = 0;
	readPos = 0;
	startPos = 0;
}

NSFrmFifoClient::FrmReadInf::~FrmReadInf()
{
	NSFrame::unref(frm);
}

UINT8 *NSFrmFifoClient::FrmReadInf::advance(unsigned n)
{
	NDK_ASSERT((readPos + startPos) + n <= frm->length);
	UINT8 *p = frm->data + (readPos + startPos);
	readPos += n;
	return p;
}

NSFrmFifoClient::NSFrmFifoClient(NSFrmFifo &fifo, unsigned sessId)
: fFifoNext(NULL)
, fCliNext(NULL)
, fSessId(sessId)
, fCliId(-1)
, fFrmFifo(fifo)
, fFrmReadInf()
, fAbortStreaming(false)
, fAvSyncPeerCli(NULL)
, fAvSyncPTSStart(-1)
, fAvSyncPTSRecent(-1)
, fAvSyncPTSNext(-1)
, fStaFrmNumSent(0)
, fStaFrmNumDrop(0)
, fStaFrmAvgLifetime(0)
, fStaFrmMaxLifetime(0)
, fFrmNextIdx(0)
, fFrmInitialIdx(0)
{
	static int cliId = 0;

	if (sessId != 0) { // Not dummy client
		NSFrmFifoClient **pp;

		ndk_st_sys_protect(-1);
		fCliId = cliId++;

		for (pp = &frmfifocli_list_head; *pp; pp = &(*pp)->fCliNext)
			;
		*pp = this;
		fFrmFifo.fifoClientRegister(this);
		ndk_st_sys_unprotect();
	}
}

NSFrmFifoClient::~NSFrmFifoClient()
{
	if (fCliId >= 0) {
		NSFrmFifoClient **pp;

		ndk_st_sys_protect(-1);
		for (pp = &frmfifocli_list_head; *pp; pp = &(*pp)->fCliNext) {
			if (*pp == this) {
				*pp = this->fCliNext;
				break;
			}
		}
		fFrmFifo.fifoClientUnregister(this);
		ndk_st_sys_unprotect();
	}
}

int NSFrmFifoClient::getReadyFramesInFifo() const
{
	return fFrmFifo.getHeadFrmIdx() - fFrmNextIdx;
}

bool NSFrmFifoClient::fetchNextFrame(NSFrame *&frm, NSFrmIdx &idx)
{
	if (!fFrmFifo.getMediaSrc())
		return false;

	bool gotfrm = false;
	NSFrmIdx curidx;

	ndk_st_sys_protect(-1);
	curidx = fFrmNextIdx;

	if (fFrmFifo.isQueueEmpty()) {
		goto lExit;
	}

	if (curidx < fFrmFifo.getTailFrmIdx()) {
		bool succeed = fFrmFifo.syncIdxFromTail(curidx);
		if (!succeed) {
			//ndk_warning("sync fail: %c "NSFRMIDX_FMT" "NSFRMIDX_FMT, isLiveSource() ? 'L' : '-', curidx, fFrmFifo.getTailFrmIdx());
			goto lExit;
		}
		NDK_ASSERT(curidx >= fFrmFifo.getTailFrmIdx());
	}

	if (curidx < fFrmFifo.getHeadFrmIdx()) {
		frm = fFrmFifo.getFrame(curidx);
		NDK_ASSERT_EXPR(frm, printf("head="NSFRMIDX_FMT" tail="NSFRMIDX_FMT" idx="NSFRMIDX_FMT"\n", fFrmFifo.getHeadFrmIdx(), fFrmFifo.getTailFrmIdx(), curidx));
		frm->ref();
		gotfrm = true;
	}
	else { // Ahead
	}

lExit:
	if (gotfrm) {
		idx = curidx;
		fFrmNextIdx = curidx + 1;
		NDK_ASSERT(fFrmNextIdx <= fFrmFifo.getHeadFrmIdx());

		SINT64 pts = (SINT64)timeval2mtime(&frm->pts);

		if (fAvSyncPTSStart == -1) {
			fAvSyncPTSStart = pts;
			fAvSyncPTSRecent = pts - 100;
		}

		// Estimate next PTS
		if (pts > fAvSyncPTSRecent) {
			int diff = pts - fAvSyncPTSRecent;
			if (diff < 30)
				diff = 30;
			fAvSyncPTSNext = pts + diff;
		}
		else {
			fAvSyncPTSNext = pts + 100;
		}

		fAvSyncPTSRecent = pts;
		//profLogPrintf(0, "fetch %c %d %d", fFrmFifo.isAudio()?'A':'V', (int)fAvSyncPTSRecent, (int)fAvSyncPTSNext);

		// Reclaim for every 8 frames.
		if ((fFrmNextIdx & 0x7) == 0)
			fFrmFifo.reclaimUnusedFrames();
	}

	ndk_st_sys_unprotect();
	return gotfrm;
}

bool NSFrmFifoClient::avSyncInit()
{
	if (!fAvSyncPeerCli) {
		NSFrmFifoClient **pp, *p, *peer = NULL;

		ndk_st_sys_protect(-1);
		for (pp = &frmfifocli_list_head; *pp; pp = &(*pp)->fCliNext) {
			p = *pp;
			if ( (this == p) ||
			     (this->fSessId != p->fSessId) ||
			     (this->fFrmFifo.isAudio() && p->fFrmFifo.isAudio()) ||
			     (this->fFrmFifo.isVideo() && p->fFrmFifo.isVideo())
			   )
			{
				continue;
			}

			if (peer) {
				ndk_error("Multiple a/v clients");
				peer = NULL;
				break;
			}

			peer = *pp;
		}

		fAvSyncPeerCli = peer;
		ndk_st_sys_unprotect();
	}

	return fAvSyncPeerCli != NULL;
}

int NSFrmFifoClient::avSyncCheck(struct timeval &tvPTS)
{
	if (!fFrmReadInf.frm) // error
		return AV_SYNC_OK;

	SINT64 pts = timeval2mtime(&tvPTS);
	int res = AV_SYNC_OK;

	if (fFrmFifo.isAudio()) {
		if (fAvSyncPeerCli->fAvSyncPTSStart == -1)
			res = AV_SYNC_INVALID;
		else if (pts < fAvSyncPeerCli->fAvSyncPTSStart) {
			res = AV_SYNC_BEFORE;
		}
		else if (pts >= fAvSyncPeerCli->fAvSyncPTSNext)
			res = AV_SYNC_WAIT;
	}
	else {
		// Video always return OK.
		//if (pts >= fAvSyncPeerCli->fAvSyncPTSNext)
			//res = AV_SYNC_WAIT;
	}

	//profLogPrintf(0, "avsync %d %c %d %d %d", fFrmFifo.isAudio()?'A':'V', res, (int)pts, (int)fAvSyncPeerCli->fAvSyncPTSRecent, (int)fAvSyncPeerCli->fAvSyncPTSNext);
	return res;
}

bool NSFrmFifoClient::isEndOfSource() const
{
	return fFrmFifo.getMediaSrc()->endOfSource();
}

enum { FIFOCLI_MON_IDX = 0, FIFOCLI_MON_DROP, FIFOCLI_MON_LTAVG, FIFOCLI_MON_LTMAX, FIFOCLI_MON_FR };

int NSFrmFifoClient::monGetVal(ULONG var_id, char *value, int width, unsigned long user_data)
{
	NSFrmFifo *fifo = (NSFrmFifo *)user_data;

	if (fifo) {
		NSFrmFifoClient *cli = (NSFrmFifoClient*)fifo->fifoClientFind(var_id >> 8);
		if (cli) {
			UINT32 n;

			switch (var_id & 0xFF) {
			case FIFOCLI_MON_IDX:
				n = fifo->getHeadFrmIdx() - cli->fFrmNextIdx;
				sprintf(value, "%4u", n);
				break;
			case FIFOCLI_MON_DROP:
				sprintf(value, "%2u", cli->fStaFrmNumDrop);
				cli->fStaFrmNumDrop = 0;
				break;
			case FIFOCLI_MON_LTAVG:
				sprintf(value, "%4u", cli->fStaFrmAvgLifetime>>10);
				break;
			case FIFOCLI_MON_LTMAX:
				sprintf(value, "%4u", cli->fStaFrmMaxLifetime>>10);
				cli->fStaFrmMaxLifetime = 0;
				break;
			case FIFOCLI_MON_FR:
				n = cli->fStaFrmNumSent;
				cli->fStaFrmNumSent = 0;
				sprintf(value, "%3u", n);
				break;
			default:
				return NDK_MON_RES_INVALID;
			}
			return NDK_MON_RES_OK;
		}
	}

	return NDK_MON_RES_INVALID;
}

void NSFrmFifoClient::monAddVars() const
{
	if (fCliId < 0)
		return;

	ULONG id = fCliId;

	char idx_str[8];
	sprintf(idx_str, "<%c%d", *fFrmFifo.getClassName(), id);

	ndk_mon_add_var(idx_str, (id<<8) | FIFOCLI_MON_IDX,   4, monGetVal, (unsigned long)&fFrmFifo); // Index
	ndk_mon_add_var("df",    (id<<8) | FIFOCLI_MON_DROP,  2, monGetVal, (unsigned long)&fFrmFifo); // Drop Frame
	//ndk_mon_add_var("lt-a",  (id<<8) | FIFOCLI_MON_LTAVG, 4, monGetVal, (unsigned long)&fFrmFifo); // Lifetime Avg.
	//ndk_mon_add_var("lt-m",  (id<<8) | FIFOCLI_MON_LTMAX, 4, monGetVal, (unsigned long)&fFrmFifo); // LifeTime Max.
	ndk_mon_add_var("fr>",   (id<<8) | FIFOCLI_MON_FR,    3, monGetVal, (unsigned long)&fFrmFifo); // FPS
}

///////////////////////////////////////////////////////////////////////////////
// Fifo
NSFrmFifo::NSFrmFifo(bool liveMode, bool activated, bool audio)
: fLiveMode(liveMode)
, fActivated(activated)
, fAudio(audio)
, fFrmSeqno(0)
, fFrmIdxResetCnt(0)
, fNext(NULL)
, fClientHead(NULL)
, fMediaSrcAgent(NULL)
, fFrmQ_head(0)
, fFrmQ_tail(0)
, fFrmSizeTot(0)
, fFrmCTJitter(0)
, fFrmCTRecent(0)
{
	onGotFrame = NULL;
	memset(fFrmQ, 0, sizeof(fFrmQ));
	// Insert into the global FIFO list.
	NSFrmFifo **pp;
	ndk_st_sys_protect(-1);
	for (pp = &frmfifo_list_head; *pp; pp = &(*pp)->fNext)
		;
	*pp = this;
	ndk_st_sys_unprotect();
}

NSFrmFifo::~NSFrmFifo()
{
	NDK_ASSERT(!fMediaSrcAgent);
	NDK_ASSERT(!fClientHead);

	NSFrmFifo **pp;

	// Remove from the global FIFO list
	ndk_st_sys_protect(-1);
	for (pp = &frmfifo_list_head; *pp; pp = &(*pp)->fNext) {
		if (*pp == this) {
			*pp = this->fNext;
			break;
		}
	}
	ndk_st_sys_unprotect();

	safeReclaimAllFrames(false);
}

int NSFrmFifo::getQueueTimeLength() const
{
	if (isQueueEmpty())
		return 0;

	NDK_ASSERT(frmIdxSanityCheck());

	NSFrame *head = fFrmQ[idx2pos(fFrmQ_head - 1)];
	NSFrame *tail = fFrmQ[idx2pos(fFrmQ_tail)];

	NDK_ASSERT(head && tail);

	long diff = timeval_diff_ms(&head->pts, &tail->pts);
	if (diff < 0) {
		ndk_error("qTimeLength err: %s %s", tv2str(&head->pts), tv2str(&tail->pts));
		return 0;
	}
	else if (diff == 0) {
		return 10;
	}
	else
		return (int)diff;
}

NSFrame *NSFrmFifo::popFrame()
{
	if (fFrmQ_tail < fFrmQ_head) {
		int n = idx2pos(fFrmQ_tail++);
		NDK_ASSERT_EXPR(frmIdxSanityCheck(), printf("head="NSFRMIDX_FMT" tail="NSFRMIDX_FMT"\n", fFrmQ_head, fFrmQ_tail));
		NSFrame *frm = fFrmQ[n];
		fFrmQ[n] = NULL;
		NDK_ASSERT(frm);
		fFrmSizeTot -= frm->length;
		return frm;
	}

	return NULL;
}

bool NSFrmFifo::syncIdxFromHead(NSFrmIdx &idx) const
{
	UINT32 cnt = 0;
	NSFrmIdx i;

	if (fFrmQ_head == 0)
		return false;

	NDK_ASSERT_EXPR(frmIdxSanityCheck(), printf("head="NSFRMIDX_FMT" tail="NSFRMIDX_FMT"\n", fFrmQ_head, fFrmQ_tail));
	for (i = fFrmQ_head - 1; i >= fFrmQ_tail && i < fFrmQ_head && cnt <= FIFO_QSIZE; --i, ++cnt) {
		if (getFrame(i)->isKeyFrame()) {
			idx = i;
			return true;
		}
	}
	return false;
}

bool NSFrmFifo::syncIdxFromTail(NSFrmIdx &idx) const
{
	NSFrmIdx i;

	NDK_ASSERT_EXPR(frmIdxSanityCheck(), printf("head="NSFRMIDX_FMT" tail="NSFRMIDX_FMT"\n", fFrmQ_head, fFrmQ_tail));
	for (i = fFrmQ_tail; i < fFrmQ_head; ++i) {
		if (getFrame(i)->isKeyFrame()) {
			idx = i;
			return true;
		}
	}
	return false;
}

void NSFrmFifo::reclaimUnusedFrames()
{
	NSFrmIdx minNextIdx = fifoClientGetMinNextFrmIdx();

	while ((minNextIdx > 1) && (fFrmQ_tail < minNextIdx -1) && (fFrmQ_tail < fFrmQ_head)) {
		NSFrame::unref(popFrame());
	}

	//ndk_info(NSFRMIDX_FMT " " NSFRMIDX_FMT " " NSFRMIDX_FMT, minNextIdx, getHeadFrmIdx(), getTailFrmIdx());
}

UINT32 NSFrmFifo::reclaimTailFrames()
{
	UINT32 n = 0;
	NDK_ASSERT(frmIdxSanityCheck());

	while (fFrmQ_tail < fFrmQ_head) {
		if (n && getFrame(fFrmQ_tail)->isKeyFrame()) {
			break;
		}

		NSFrame::unref(popFrame());
		++n;
	}

	return n;
}

void NSFrmFifo::safeReclaimAllFrames(bool resetIndex)
{
	if (fFrmQ) {
		ndk_st_sys_protect(-1);
		NDK_ASSERT(frmIdxSanityCheck());

		while (fFrmQ_tail < fFrmQ_head) {
			NSFrame::unref(popFrame());
		}

		// Set frmQ head and tail idx to 0 to notify the client that the fifo is reset.
		if (resetIndex) {
			fFrmQ_head = fFrmQ_tail = 0;
			++fFrmIdxResetCnt;
		}
		fFrmSizeTot = 0;
		ndk_st_sys_unprotect();

		//profLogPuts(0, "NSFrmFifo: reclaim all");
	}
}

NSMediaSrcAgent* NSFrmFifo::unbindMediaSrc()
{
	NSMediaSrcAgent* src = fMediaSrcAgent;
	fMediaSrcAgent = NULL;

	if (src) {
		src->closeSource();

		ndk_st_sys_protect(-1);
		reclaimSpecialFrames();
		src->unbindFIFOs();
		ndk_st_sys_unprotect();
	}

	safeReclaimAllFrames(true);
	return src;
}

bool NSFrmFifo::bindMediaSrc(NSFrmFifo *fifo, NSMediaSrcAgent *src)
{
	NDK_ASSERT(!fifo->fMediaSrcAgent);

	fifo->fMediaSrcAgent = src;
	src->bindFIFO(fifo);

	return true;
}

bool NSFrmFifo::fifoClientRegister(NSFrmFifoClient *cli)
{
	// non-live mode only support one client.
	NDK_ASSERT(fLiveMode || !fClientHead);
	NSFrmFifoClient **pos;

	ndk_st_sys_protect(-1);
	for (pos = &fClientHead; *pos; pos = &(*pos)->fFifoNext)
		;
	*pos = cli;
	cli->fFifoNext = NULL;
	ndk_st_sys_unprotect();
	return true;
}

void NSFrmFifo::fifoClientUnregister(NSFrmFifoClient *cli)
{
	NSFrmFifoClient **pos;
	ndk_st_sys_protect(-1);
	for (pos = &fClientHead; *pos; pos = &(*pos)->fFifoNext) {
		if (*pos == cli) {
			*pos = cli->fFifoNext;
			break;
		}
	}
	ndk_st_sys_unprotect();
}

NSFrmFifoClient *NSFrmFifo::fifoClientFind(int id)
{
	NSFrmFifoClient *p, *cli = NULL;

	ndk_st_sys_protect(-1);
	for (p = fClientHead; p; p = p->fFifoNext) {
		if (p->fCliId == id) {
			cli = p;
			break;
		}
	}
	ndk_st_sys_unprotect();

	return cli;
}

NSFrmIdx NSFrmFifo::fifoClientGetMinNextFrmIdx() const
{
	NSFrmIdx idx = NSFRMIDX_MAX;

	for (NSFrmFifoClient *cli = fClientHead; cli; cli = cli->fFifoNext) {
		if (idx > cli->fFrmNextIdx)
			idx = cli->fFrmNextIdx;
	}

	if (idx == NSFRMIDX_MAX)
		idx = 0;

	return idx;
}

int NSFrmFifo::fifoClientGetCount() const
{
	NSFrmIdx n = 0;

	ndk_st_sys_protect(-1);
	for (NSFrmFifoClient *cli = fClientHead; cli; cli = cli->fFifoNext) {
		++n;
	}
	ndk_st_sys_unprotect();

	return n;
}

int NSFrmFifo::monGetVal(ULONG var_id, char *value, int width, unsigned long user_data)
{
	NSFrmFifo *p, *f = (NSFrmFifo*)user_data;
	NSFrmIdx head = NSFRMIDX_MAX, tail = 0;
	UINT32   ctj = 0;

	ndk_st_sys_protect(-1);
	for (p = frmfifo_list_head; p; p = p->fNext) {
		if (p == f) {
			head = f->fFrmQ_head;
			tail = f->fFrmQ_tail;
			ctj  = f->fFrmCTJitter/1000;
			break;
		}
	}
	ndk_st_sys_unprotect();

	if (head == NSFRMIDX_MAX)
		return NDK_MON_RES_INVALID;

	if (var_id == 0)
		sprintf(value, "%4d/%3d", (int)(head % 10000), (int)(head - tail));
	else
		sprintf(value, "%3u", ctj);

	return NDK_MON_RES_OK;
}

void NSFrmFifo::monAddVars()
{
	NSFrmFifo *p;
	NSFrmFifoClient *c;
	char name[5];

	ndk_st_sys_protect(-1);
	for (p = frmfifo_list_head; p; p = p->fNext) {
		if (p->fifoClientGetCount() == 0)
			continue;

		name[0] = 'F';
		name[1] = *p->getClassName();
		name[2] = p->fLiveMode ? 'L' : '-';
		name[3] = p->isAudio() ? 'A' : 'V';
		name[4] = 0;

		ndk_mon_add_var(name,  0, 8, monGetVal, (unsigned long)p);
		//if (p->isLiveMode())
			//ndk_mon_add_var("ctj", 1, 3, monGetVal, (unsigned long)p); // Creation time jitter.

		for (c = p->fClientHead; c; c = c->fFifoNext)
			c->monAddVars();
	}
	ndk_st_sys_unprotect();
}

void NSFrmFifo::dumpInfo(char **buf) const
{
	char *p = *buf;

	p += sprintf(p, "<%s> %c this=%p <"NSFRMIDX_FMT", "NSFRMIDX_FMT"> src=%p\n",
		getClassName(), fLiveMode ? 'L' : '-', this,
		fFrmQ_head, fFrmQ_tail,
		fMediaSrcAgent);
	p += sprintf(p, "  depth=%d, timeLen=%d, frmSizeTot=%u\n", getQueueDepth(), getQueueTimeLength(), fFrmSizeTot);

	for (NSFrmFifoClient *cli = fClientHead; cli; cli = cli->fFifoNext) {
		p += sprintf(p, "    cli %d: sessId=%u ptr=0x%p\n", cli->fCliId, cli->fSessId, cli);
	}

	*buf = p;
}

void NSFrmFifo::dumpFifoInfo()
{
	NSFrmFifo *f;
	char *buf0, *buf;

	if (!(buf0 = (char*)ndk_mem_malloc(1024*4)))
		return;

	buf0[0] = 0;
	ndk_st_sys_protect(-1);
	buf = buf0;
	for (f = frmfifo_list_head; f; f = f->fNext)
		f->dumpInfo(&buf);
	ndk_st_sys_unprotect();

	puts(buf0);
	ndk_mem_free(buf0);
}

bool NSFrmFifo::safePushFrame(NSFrame *frm, int timeInMs)
{
	NDK_ASSERT(frm);

	if (!fActivated) {
		NSFrame::unref(frm);
		return true;
	}

	if (!ndk_st_sys_protect(timeInMs)) {
		ndk_error("%s: ndk_st_sys_protect", getClassName());
		return false;
	}

	if (fLiveMode) {
		long fifoSize = ndk_st_get_attr(isAudio() ? NDK_ST_ATTR_AUDIO_FIFO_SIZE : NDK_ST_ATTR_VIDEO_FIFO_SIZE);

		if (isQueueFull() || getQueueDepth() >= fifoSize) {
			if (reclaimTailFrames() == 0) {
				ndk_st_sys_unprotect();
				ndk_err_return(false);
			}
		}

		SINT64 ct = timeval2utime(&frm->timeCreated);
		if (fFrmCTRecent == 0)
			fFrmCTRecent = ct;

		fFrmCTJitter = ((ct - fFrmCTRecent) + fFrmCTJitter*29)/30;
		fFrmCTRecent = ct;
	}
	else {
		if (isQueueFull()) {
			ndk_error("%s: non-live: no space ("NSFRMIDX_FMT" "NSFRMIDX_FMT")"
				, getClassName(), getHeadFrmIdx(), getTailFrmIdx());
			ndk_st_sys_unprotect();
			return false;
		}
	}

	frm->index = fFrmSeqno++;
	fFrmQ[idx2pos(fFrmQ_head++)] = frm;
	fFrmSizeTot += frm->length;

	NDK_ASSERT_EXPR(frmIdxSanityCheck(), printf("head="NSFRMIDX_FMT" tail="NSFRMIDX_FMT"\n", fFrmQ_head, fFrmQ_tail));
	ndk_st_sys_unprotect();

	//profLogPrintf(0, "fifo idx: " NSFRMIDX_FMT " " NSFRMIDX_FMT, getHeadFrmIdx(), getTailFrmIdx());
	if (onGotFrame && isLiveMode())
		onGotFrame(this, frm);

	return true;
}

///////////////////////////////////////////////////////////////////////////////
NSH264FrmFifo::NSH264FrmFifo(bool liveMode, bool activated)
: NSFrmFifo(liveMode, activated, false)
{
	//ndk_info("+");
	fFrmSPS.frm = fFrmPPS.frm = NULL;
	fFrmSPS.idx = fFrmPPS.idx = 1; // Let the source's SPS & PPS index be invalid.
}

NSH264FrmFifo::~NSH264FrmFifo()
{
	//ndk_info("-");
	NSH264FrmFifo::reclaimSpecialFrames();
}

void NSH264FrmFifo::dumpInfo(char **buf) const
{
	NSFrmFifo::dumpInfo(buf);
	char *p = *buf;
	p += sprintf(p, "  sps=%d, pps=%d\n", fFrmSPS.idx, fFrmPPS.idx );
	*buf = p;
}

bool NSH264FrmFifo::safePushFrame(NSFrame *frm, int timeInMs)
{
	if (!frm->data || frm->length < 5)
		ndk_err_return(false);

	UINT8 nal_unit_type = frm->data[4] & 0x1f;

	if (nal_unit_type == 7) {
		if (fFrmSPS.frm) {
			NSFrame::unref(frm);
			return true;
		}
	}
	else if (nal_unit_type == 8) {
		if (fFrmPPS.frm) {
			NSFrame::unref(frm);
			return true;
		}
	}

	if (nal_unit_type == 7 || nal_unit_type == 8) {
		H264ParamFrame *pf = nal_unit_type == 7 ? &fFrmSPS : &fFrmPPS;

		if (!ndk_st_sys_protect(timeInMs))
			return false;

		if (pf->frm
		    && pf->frm->length == frm->length
		    && memcmp(pf->frm->data, frm->data, pf->frm->length) == 0)
		{
			NSFrame::unref(frm);
		}
		else {
			NSFrame::unref(pf->frm);
			pf->idx++;
			pf->frm = frm;
		}
		ndk_st_sys_unprotect();

		puts(nal_unit_type == 7 ? "sps" : "pps");
		memdump(frm->data, frm->length);
		if (onGotFrame)
			onGotFrame(this, frm);

		return true;
	}
	else {
		return NSFrmFifo::safePushFrame(frm, timeInMs);
	}
}

void NSH264FrmFifo::reclaimSpecialFrames()
{
	NSFrame::unref(fFrmSPS.frm);
	NSFrame::unref(fFrmPPS.frm);
	fFrmSPS.frm = fFrmPPS.frm = NULL;
	fFrmSPS.idx = fFrmPPS.idx = 1;
}

///////////////////////////////////////////////////////////////////////////////
// Class NSJpegFrmFifo
NSJpegFrmFifo::NSJpegFrmFifo(bool liveMode, bool activated)
: NSFrmFifo(liveMode, activated, false)
{
	//ndk_info("+");
}

NSJpegFrmFifo::~NSJpegFrmFifo()
{
	//ndk_info("-");
}

void NSJpegFrmFifo::dumpInfo(char **buf) const
{
	NSFrmFifo::dumpInfo(buf);
}

///////////////////////////////////////////////////////////////////////////////
// Class NSPcmFrmFifo
NSAudioFrmFifo::NSAudioFrmFifo(bool liveMode, bool activated)
: NSFrmFifo(liveMode, activated, true)
{
	//ndk_info("+");
}

NSAudioFrmFifo::~NSAudioFrmFifo()
{
	//ndk_info("-");
}

void NSAudioFrmFifo::dumpInfo(char **buf) const
{
	NSFrmFifo::dumpInfo(buf);
}

///////////////////////////////////////////////////////////////////////////////
// NSFrmFifoMgr
void NSFrmFifoMgr::ref()
{
	unsigned int x;
	ndk_global_lock(&x);
	++fRefCnt;
	ndk_global_unlock(&x);
}

void NSFrmFifoMgr::unref(NSFrmFifoMgr *fifoMgr)
{
	if (fifoMgr) {
		unsigned int x;
		int n;

		ndk_global_lock(&x);
		n = --fifoMgr->fRefCnt;
		ndk_global_unlock(&x);

		if (n == 0)
			delete fifoMgr;

		NDK_ASSERT(n >= 0);
	}
}

///////////////////////////////////////////////////////////////////////////////
NSCapFrmFifoMgr::NSCapFrmFifoMgr(const char *streamName, NDKFrmFifoType audFifoType, NDKFrmFifoType vidFifoType)
: NSFrmFifoMgr()
, fAudFifoType(audFifoType)
, fVidFifoType(vidFifoType)
, fOpened(false)
{
	NDK_ASSERT(streamName);
	strncpy(fStreamName, streamName, sizeof(fStreamName)-1);
	fStreamName[sizeof(fStreamName)-1] = 0;
}

NSCapFrmFifoMgr::~NSCapFrmFifoMgr()
{
	if (fOpened)
		ndk_st_capfifo_unref(fStreamName, fAudFifoType, fVidFifoType);
}

bool NSCapFrmFifoMgr::openFIFO()
{
	if (!fOpened)
		fOpened = ndk_st_capfifo_ref(fStreamName, fVidFifoType, &fVidFifo, fAudFifoType, &fAudFifo);
	return fOpened;
}

///////////////////////////////////////////////////////////////////////////////
// NSDmxFrmFifoMgr
#ifdef ICAT_STREAMING_FILE

NSMediaSrcFrmFifoMgr::
NSMediaSrcFrmFifoMgr(const char* rootDir, const char* fileName, BOOL audio, BOOL video)
: NSFrmFifoMgr()
, fAudio(audio)
, fVideo(video)
, fMediaSrcAgent(NULL)
{
	snprintf(fFileName, sizeof(fFileName), "%s/%s", rootDir, fileName);
	ndk_info("filename = %s", fFileName);
	NDK_ASSERT(audio || video);
}

NSMediaSrcFrmFifoMgr::~NSMediaSrcFrmFifoMgr()
{
	if (fAudFifo)
		fAudFifo->unbindMediaSrc();
	if (fVidFifo)
		fVidFifo->unbindMediaSrc();
	if (fMediaSrcAgent)
		fMediaSrcAgent->closeSource();

	delete fMediaSrcAgent;
	delete fAudFifo;
	delete fVidFifo;
}

bool NSMediaSrcFrmFifoMgr::openFIFO()
{
	if (fMediaSrcAgent)
		return true;

	NSMediaSrcAgent *msrc = NULL;
	NSFrmFifo *v_fifo = NULL;
	NSFrmFifo *a_fifo = NULL;
	UINT32 v_codec, a_codec;

	if (ndk_st_has_custom_mediasrc(NDK_ST_MEDIASRC_FILE)) {
		ndk_info("custom msrc is used");
		msrc = new NSCustomMediaSrcAgent(NDK_ST_MEDIASRC_FILE, fFileName);
	}
	else
		msrc = new NSDmxMediaSrcAgent(fFileName);

	if (!msrc || !msrc->openSource()) {
		ndk_error("open");
		goto lFail;
	}

	a_codec = msrc->getCodecType(true);
	v_codec = msrc->getCodecType(false);

	if ((v_codec != SP5K_MEDIA_VIDEO_UNKNOWN) && fVideo) {
		switch (v_codec) {
		case SP5K_MEDIA_VIDEO_H264:
			v_fifo = new NSH264FrmFifo(false, true);
			break;
		case SP5K_MEDIA_VIDEO_MJPG:
			v_fifo = new NSJpegFrmFifo(false, true);
			break;
		default:
			NDK_ASSERT(0);
			break;
		}

		if (!v_fifo)
			goto lFail;
	}

	if ((a_codec != SP5K_MEDIA_AUDIO_UNKNOWN) && fAudio) {
		switch (a_codec) {
		case SP5K_MEDIA_AUDIO_PCM:
		case SP5K_MEDIA_AUDIO_ALAW:
		case SP5K_MEDIA_AUDIO_MULAW:
		case SP5K_MEDIA_AUDIO_IMA_ADPCM:
		case SP5K_MEDIA_AUDIO_AAC:
			a_fifo = new NSAudioFrmFifo(false, true);
			break;
		default:
			NDK_ASSERT(0);
			break;
		}

		if (!a_fifo)
			goto lFail;
	}

	if (!v_fifo && !a_fifo) {
		ndk_error("no fifo");
		goto lFail;
	}

	if (v_fifo) {
		if (!NSFrmFifo::bindMediaSrc(v_fifo, msrc)) {
			ndk_error("bind");
			goto lFail;
		}
	}

	if (a_fifo) {
		if (!NSFrmFifo::bindMediaSrc(a_fifo, msrc)) {
			ndk_error("bind");
			goto lFail;
		}
	}

	fVidFifo = v_fifo;
	fAudFifo = a_fifo;

	fMediaSrcAgent = msrc;
	fMediaSrcAgent->resumeSource();
	return true;

lFail:
	delete msrc;
	delete v_fifo;
	delete a_fifo;
	return false;
}

#endif // ICAT_STREAMING_FILE

