/**************************************************************************
 *                                                                        *
 *         Copyright (c) 2012 by iCatch Technology Co., Ltd.             *
 *                                                                        *
 *  This software is copyrighted by and is the property of Sunplus        *
 *  Technology Co., Ltd. All rights are reserved by Sunplus Technology    *
 *  Co., Ltd. This software may only be used in accordance with the       *
 *  corresponding license agreement. Any unauthorized use, duplication,   *
 *  distribution, or disclosure of this software is expressly forbidden.  *
 *                                                                        *
 *  This Copyright notice MUST not be removed or modified without prior   *
 *  written consent of Sunplus Technology Co., Ltd.                       *
 *                                                                        *
 *  Sunplus Technology Co., Ltd. reserves the right to modify this        *
 *  software without notice.                                              *
 *                                                                        *
 *  Sunplus Technology Co., Ltd.                                          *
 *  19, Innovation First Road, Science-Based Industrial Park,             *
 *  Hsin-Chu, Taiwan, R.O.C.                                              *
 *                                                                        *
 **************************************************************************/

#ifndef __LIVEST_FIFO_H__
#define __LIVEST_FIFO_H__

#include <new>
#include <livePort/livest_internal.h>
#include <livePort/streaming_def.h>

#if 1
typedef UINT32        NSFrmIdx;
#define NSFRMIDX_MAX  0xFFFFFFFFU
#define NSFRMIDX_FMT  "%u"
#else
typedef SINT64        NSFrmIdx;
#define NSFRMIDX_MAX  0x7fffFFFFffffFFFFLL
#define NSFRMIDX_FMT  "%lld"
#endif

// Forward reference
class NSFrmFifo;
class NSMediaSrcAgent;
class NSFrmFifoMgr;
class NSFrmFifo;

///////////////////////////////////////////////////////////////////////////////
// Base Frame Struct
// NS = NDK Streaming
class NSFrame
{
	friend class NSFrmFifo;
public:
	UINT8* data;
	UINT32 length;
	UINT32 duration; // Unit: ms. 0: live a/v frame.
	UINT32 index;
	struct timeval pts;
	struct timeval timeCreated; // for lag time checking

public:
	virtual bool isKeyFrame() const { return true; }
	virtual bool isParamFrame() const { return false; }
	void    ref();
	static  void unref(NSFrame *frm);
	int     getRefcnt() const { return fRefCnt; }

	inline bool readByte(UINT32 &pos, UINT8* buf) const {
		if (pos + 1 > this->length)
			return false;
		*buf = *(this->data + pos++);
		return true;
	}

	inline bool readWord(UINT32 &pos, UINT8* buf) const {
		if (pos + 2 > this->length)
			return false;
		UINT8 *p = this->data + pos;
		buf[1] = *(p++); // swap
		buf[0] = *(p++);
		pos += 2;
		return true;
	}

	inline bool readBytes(UINT32 &pos, UINT8* buf, UINT32 n) const {
		if (pos + n > this->length)
			return false;
		memcpy(buf, this->data + pos, n);
		pos += n;
		return true;
	}

	inline void skipBytes(UINT32 &pos, int n) const {
		pos += n;
		if (pos > this->length)
			pos = this->length;
	}

	NSFrame *clone();
	virtual void attachBuffer(void *buffer, void *extra_param) {}

	static UINT32 staFrmAlloced;
	static UINT32 staFrmFreed;

protected:
	signed short fRefCnt;

	NSFrame() : data(NULL), length(0), fRefCnt(1) { ++staFrmAlloced; }
	virtual ~NSFrame() {}
};

///////////////////////////////////////////////////////////////////////////////
// Frames got from basefw media-sources functions.
class NSMediaSrcFrame : public NSFrame
{
	static NDKStExitFunc fFrmPoolOnExit;
	static void *fFrmPool;

	static void frmPoolOnExit(NDKStExitFunc *exit_func);

protected:
	mediaBufferObject_t fBufObj; // original frame object given by firmware.
	UINT32 fBufFlags;

	virtual ~NSMediaSrcFrame();

public:
	NSMediaSrcFrame() : NSFrame(), fBufObj(NULL), fBufFlags(0) {}
	virtual bool isKeyFrame()   const { return (fBufFlags & MEDIA_BUF_KEY_FRAME) != 0; }
	virtual bool isParamFrame() const { return (fBufFlags & MEDIA_BUF_PARAM_FRAME) != 0; }

	// After assigned, the bufobj in msbuf is hold by this NSMediaSrcFrame object.
	virtual void attachBuffer(void *buffer, void *extra_param);

	void *operator new (size_t size) throw();
	void operator delete (void* ptr);
};

///////////////////////////////////////////////////////////////////////////////
// Frames got from custom media-sources functions.
class NSCustomMediaFrame : public NSFrame
{
protected:
	void (*fFreeBufferObjectFunc)(NDKStMediaBufferObject obj);
	NDKStMediaBufferObject fBufObj; // original frame object given by firmware.

	UINT32 fFlgBufType    : 2,
	       fFlgKeyFrame   : 1,
	       fFlgParamFrame : 1;

	virtual ~NSCustomMediaFrame();

public:
	NSCustomMediaFrame(void (*free_buffer_object)(NDKStMediaBufferObject));
	virtual bool isKeyFrame()   const { return fFlgKeyFrame; }
	virtual bool isParamFrame() const { return fFlgParamFrame; }
	virtual void attachBuffer(void *buffer, void *extra_param);
};

///////////////////////////////////////////////////////////////////////////////
// Dual-Streaming Jpeg frame
#ifdef ICAT_DUALSTREAM_JPEG
class NSDsjFrame : public NSFrame
{
protected:
	mediaBufferObject_t fBufObj;

	virtual ~NSDsjFrame();

public:
	mediaRecDualStreamJpegAttr_t jpegAttr;

	NSDsjFrame() : NSFrame(), fBufObj(NULL) {}
	virtual void attachBuffer(void *buffer, void *extra_param);
};
#endif

///////////////////////////////////////////////////////////////////////////////
// Class NSFrmFifoClient
class NSFrmFifoClient
{
	friend class NSFrmFifo;
public:
	NSFrmFifoClient(NSFrmFifo &fifo, unsigned sessId);
	virtual  ~NSFrmFifoClient();

	NSFrmIdx getNextFrmIdx()    const { return fFrmNextIdx; }
	NSFrmIdx getInitialFrmIdx() const { return fFrmInitialIdx; }
	void     setInitialFrmIdx(NSFrmIdx idx) { fFrmInitialIdx = fFrmNextIdx = idx; }
	NSFrmFifo &getFifo() { return fFrmFifo; }

	bool     isEndOfSource() const;
	int      getReadyFramesInFifo() const;

	// Fetch frame and advance the fFrmNextIdx to the next.
	// idx: the frame-index of current frame.
	bool     fetchNextFrame(NSFrame *&frm, NSFrmIdx &idx);

	bool     avSyncInit();
	enum { // Result of avSyncCheck.
		AV_SYNC_INVALID = 'I',
		AV_SYNC_BEFORE  = 'B', // Before peer start
		AV_SYNC_WAIT    = 'W',
		AV_SYNC_OK      = 'O'
	};
	int  avSyncCheck(struct timeval &tvPTS);

	void monAddVars() const;

protected:
	struct FrmReadInf {
		NSFrame* frm;
		NSFrmIdx idx;
		unsigned readPos, startPos;

		FrmReadInf() : frm(NULL), idx(0), readPos(0), startPos(0) {}
		~FrmReadInf();
		UINT8    *getData() { return frm->data + (readPos + startPos); }
		unsigned remainder() const { NDK_ASSERT(frm->length >= (readPos + startPos)); return frm->length - (readPos + startPos); }
		bool     end() const { return (readPos + startPos) == frm->length; }
		void     attach(NSFrame *f, NSFrmIdx i) { NDK_ASSERT(!frm); frm = f; idx = i; readPos = startPos = 0; }
		void     detach() { NSFrame::unref(frm); frm = NULL; readPos = startPos = 0; }
		void     transferTo(FrmReadInf& f);
		UINT8    *advance(unsigned n);
	};

protected:
	NSFrmFifoClient *fFifoNext; // Next pointer used in FIFO
	NSFrmFifoClient *fCliNext;  // Next pointer used by fCliList.
	unsigned   fSessId; // RTP session ID
	int        fCliId;

	NSFrmFifo  &fFrmFifo;
	FrmReadInf fFrmReadInf;
	bool       fAbortStreaming;

	// AVSync
	NSFrmFifoClient *fAvSyncPeerCli;
	SINT64     fAvSyncPTSStart;
	SINT64     fAvSyncPTSRecent;
	SINT64     fAvSyncPTSNext;

	// Statistics, can be cleared
	UINT32     fStaFrmNumSent;
	UINT32     fStaFrmNumDrop;
	UINT32     fStaFrmAvgLifetime; /* Lifetime: Encoding finished -> sending finished. Unit: 1/1024000 sec. */
	UINT32     fStaFrmMaxLifetime;

	static int monGetVal(ULONG var_id, char *value, int width, unsigned long user_data);

private:
	NSFrmIdx   fFrmNextIdx; // Index of the next frame to be sent.
	NSFrmIdx   fFrmInitialIdx;
};

///////////////////////////////////////////////////////////////////////////////
// Stream Frame FIFO
// FIFO:     tail(oldest)                            head(newest)
// [reclaim] +----+----+----+----+----+----+----+----+  <== [new frame]
//  idx:     0    1    2    3    4    5    6    7    8
// A FIFO only contains one kind of stream.
class NSFrmFifo
{
public:
	enum {
#ifdef SPCA6350
		FIFO_QSIZE     = 64,
#else
		FIFO_QSIZE     = 32,
#endif
		FIFO_QIDX_MASK = FIFO_QSIZE - 1,
	};

	friend   class NSFrmFifoClient;
	friend   class NSFrmFifoMgr;

private:
	bool     frmIdxSanityCheck() const { return (fFrmQ_tail <= fFrmQ_head) && (fFrmQ_tail + FIFO_QSIZE >= fFrmQ_head); }

public:
	virtual  ~NSFrmFifo();
	virtual  const char* getClassName() const { return "FrmFifo"; }
	virtual  bool safePushFrame(NSFrame *frm, int timeInMs/*-1 = wait forever */);

	int      idx2pos(NSFrmIdx idx) const { return (int)(idx & FIFO_QIDX_MASK); }
	bool     isAudio()       const { return fAudio; }
	bool     isVideo()       const { return !isAudio(); }
	bool     isLiveMode()    const { return fLiveMode; }

	int      getQueueSize()  const { return FIFO_QSIZE; }
	int      getQueueTimeLength() const; /* Unit: ms */
	int      getQueueDepth() const { NDK_ASSERT(frmIdxSanityCheck()); return fFrmQ_head - fFrmQ_tail; }
	int      getQueueSpace() const { return FIFO_QSIZE - getQueueDepth(); }
	bool     isQueueEmpty()  const { return fFrmQ_head == fFrmQ_tail; }
	bool     isQueueFull()   const { NDK_ASSERT(frmIdxSanityCheck()); return fFrmQ_head == fFrmQ_tail + FIFO_QSIZE; }
	NSFrmIdx getHeadFrmIdx() const { return fFrmQ_head; }
	NSFrmIdx getTailFrmIdx() const { return fFrmQ_tail; }
	int      getFrmIdxResetCnt() const { return fFrmIdxResetCnt; }

	void     activate(bool activated) { fActivated = activated; }
	bool     syncIdxFromHead(NSFrmIdx &idx) const;
	bool     syncIdxFromTail(NSFrmIdx &idx) const;
	// Return and remove the last frame from FIFO.
	NSFrame  *popFrame();

	// Reclaim frames from tail to idx. Useless in auto-reclaim mode.
	void     reclaimUnusedFrames();
	void     safeReclaimAllFrames(bool resetIndex);

	NSMediaSrcAgent* getMediaSrc() { return fMediaSrcAgent; }
	NSMediaSrcAgent* unbindMediaSrc();
	static   bool bindMediaSrc(NSFrmFifo *fifo, NSMediaSrcAgent *src);

	// Below client functions are thread-safe.
	bool     fifoClientRegister(NSFrmFifoClient *cli);
	void     fifoClientUnregister(NSFrmFifoClient *cli);
	NSFrmFifoClient *fifoClientFind(int id);
	NSFrmIdx fifoClientGetMinNextFrmIdx() const;
	int      fifoClientGetCount() const;

	static   void monAddVars();
	static   void dumpFifoInfo();
	virtual  void dumpInfo(char **buf) const;

	// Event Callback
	void     (*onGotFrame)(NSFrmFifo *fifo, NSFrame *frm);

protected:
	bool     fLiveMode;
	bool     fActivated;
	bool     fAudio;
	UINT32   fFrmSeqno;
	int      fFrmIdxResetCnt;

	NSFrmFifo *fNext;
	NSFrmFifoClient *fClientHead;
	NSMediaSrcAgent *fMediaSrcAgent;

	NSFrmIdx fFrmQ_head; // write position. Normally, _frmQ_head >= _frmQ_tail
	NSFrmIdx fFrmQ_tail; // read position
	NSFrame* fFrmQ[FIFO_QSIZE]; // storage space is provided by sub-classes

	UINT32   fFrmSizeTot;
	UINT32   fFrmCTJitter; // Creation time jitter. Valid only for live mode.
	SINT64   fFrmCTRecent; // us.

protected:
	NSFrmFifo(bool liveMode, bool activated, bool audio);
	NSFrame *getFrame(NSFrmIdx idx) {
		NDK_ASSERT_EXPR((idx >= fFrmQ_tail) && (idx < fFrmQ_head),
			printf(NSFRMIDX_FMT"/"NSFRMIDX_FMT"/"NSFRMIDX_FMT"\n", idx, fFrmQ_head, fFrmQ_tail);
		);
		NSFrame *frm = fFrmQ[idx2pos(idx)];
		NDK_ASSERT(frm);
		return frm;
	}
	NSFrame const *getFrame(NSFrmIdx idx) const {
		NDK_ASSERT_EXPR((idx >= fFrmQ_tail) && (idx < fFrmQ_head),
			printf(NSFRMIDX_FMT"/"NSFRMIDX_FMT"/"NSFRMIDX_FMT"\n", idx, fFrmQ_head, fFrmQ_tail);
		);
		NSFrame *frm = fFrmQ[idx2pos(idx)];
		NDK_ASSERT(frm);
		return frm;
	}

	virtual void reclaimSpecialFrames() {}
	UINT32 reclaimTailFrames();
	static int monGetVal(ULONG var_id, char *value, int width, unsigned long user_data);
};

///////////////////////////////////////////////////////////////////////////////
// NSH264FrmFifo
class NSH264FrmFifo : public NSFrmFifo
{
	struct H264ParamFrame {
		NSFrame* frm;
		UINT32 idx;
	};

public:
	NSH264FrmFifo(bool liveMode, bool activated);
	virtual ~NSH264FrmFifo();
	virtual const char* getClassName() const { return "H264FrmFifo"; }
	virtual void dumpInfo(char **buf) const;
	virtual bool safePushFrame(NSFrame *frm, int timeInMs);

	NSFrmIdx getSPSFrmIdx() const { return fFrmSPS.idx; }
	NSFrmIdx getPPSFrmIdx() const { return fFrmPPS.idx; }

	NSFrame *getSPSFrame() { return fFrmSPS.frm; }
	NSFrame *getPPSFrame() { return fFrmPPS.frm; }

protected:
	virtual void reclaimSpecialFrames();

private:
	H264ParamFrame fFrmSPS, fFrmPPS;
};

///////////////////////////////////////////////////////////////////////////////
// JPEG FIFO
class NSJpegFrmFifo : public NSFrmFifo
{
public:
	NSJpegFrmFifo(bool liveMode, bool activated);
	virtual ~NSJpegFrmFifo();
	virtual const char* getClassName() const { return "JPEGFrmFifo"; }
	virtual void dumpInfo(char **buf) const;
};

///////////////////////////////////////////////////////////////////////////////
// JPEG FIFO
class NSAudioFrmFifo : public NSFrmFifo
{
public:
	NSAudioFrmFifo(bool liveMode, bool activated);
	virtual ~NSAudioFrmFifo();
	virtual const char* getClassName() const { return "AUDFrmFifo"; }
	virtual void dumpInfo(char **buf) const;
};

///////////////////////////////////////////////////////////////////////////////
class NSFrmFifoMgr
{
public:
	NSFrmFifoMgr() : fRefCnt(1), fAudFifo(NULL), fVidFifo(NULL) {}
	void ref();
	static void unref(NSFrmFifoMgr *fifoMgr);

	// Support multi-open
	virtual bool openFIFO() = 0;
	NSFrmFifo* getFIFO(bool bAudio) { return bAudio ? fAudFifo : fVidFifo; }
protected:
	int fRefCnt;
	NSFrmFifo *fAudFifo;
	NSFrmFifo *fVidFifo;

	virtual ~NSFrmFifoMgr() { NDK_ASSERT(fRefCnt == 0); }
};

///////////////////////////////////////////////////////////////////////////////
// Capture Frame FIFO Manager. Only one FIFO in the manager.
class NSCapFrmFifoMgr : public NSFrmFifoMgr
{
private:
	NDKFrmFifoType fAudFifoType;
	NDKFrmFifoType fVidFifoType;
	bool fOpened;
	char fStreamName[128];

	virtual ~NSCapFrmFifoMgr();
public:
	NSCapFrmFifoMgr(const char *streamName, NDKFrmFifoType audFifoType, NDKFrmFifoType vidFifoType);
	virtual bool openFIFO();
};

///////////////////////////////////////////////////////////////////////////////
// Frame FIFO Manager for basefw media-source or custom media-source.
#ifdef ICAT_STREAMING_FILE
class NSMediaSrcFrmFifoMgr : public NSFrmFifoMgr
{
	char fFileName[FILENAME_MAX]; // Full file name
	BOOL fAudio, fVideo;
	NSMediaSrcAgent *fMediaSrcAgent;

	virtual ~NSMediaSrcFrmFifoMgr();
public:
	NSMediaSrcFrmFifoMgr(const char* rootDir, const char* fileName, BOOL audio, BOOL video);
	virtual bool openFIFO();
	NSMediaSrcAgent *getMediaSrc() { return fMediaSrcAgent; }
};
#endif

#endif

