/**************************************************************************
 *                                                                        *
 *         Copyright (c) 2012 by iCatch Technology Co., Ltd.             *
 *                                                                        *
 *  This software is copyrighted by and is the property of Sunplus        *
 *  Technology Co., Ltd. All rights are reserved by Sunplus Technology    *
 *  Co., Ltd. This software may only be used in accordance with the       *
 *  corresponding license agreement. Any unauthorized use, duplication,   *
 *  distribution, or disclosure of this software is expressly forbidden.  *
 *                                                                        *
 *  This Copyright notice MUST not be removed or modified without prior   *
 *  written consent of Sunplus Technology Co., Ltd.                       *
 *                                                                        *
 *  Sunplus Technology Co., Ltd. reserves the right to modify this        *
 *  software without notice.                                              *
 *                                                                        *
 *  Sunplus Technology Co., Ltd.                                          *
 *  19, Innovation First Road, Science-Based Industrial Park,             *
 *  Hsin-Chu, Taiwan, R.O.C.                                              *
 *                                                                        *
 **************************************************************************/
#include <BasicUsageEnvironment.hh>
#include <H264VideoRTPSource.hh>
#include <MediaSession.hh>
#include <RTSPClient.hh>
#include <ndk_streaming.h>
#include <ndk_mediacli.h>
#include <livePort/streaming_api.h>
#include <livePort/mediacli_rtp.h>

/////////////////////////////////////////////////////////////////////////////////
// Macros & Constants
#define RTPCLI_FLG_ISSET(flags, f)  ((flags) & (1 << (f)))
#define RTPCLI_FLG_SET(flags, f)    ((flags) |= (1 << (f)))
#define RTPCLI_FLG_CLR(flags, f)    ((flags) &= ~(1 << (f)))

/////////////////////////////////////////////////////////////////////////////////
// StringAutoDelete
class StringAutoDelete
{
	char *fString;
public:
	StringAutoDelete(char *s) : fString(s) {}
	~StringAutoDelete() { delete [] fString; }
};

/////////////////////////////////////////////////////////////////////////////////
// Types
struct RTPMediaClientScheduler
{
	void *thrid;
	BasicTaskScheduler *scheduler;
	BasicUsageEnvironment *env;
	UINT32 h264_play_err_handle_mode;
	char abort;
};

/////////////////////////////////////////////////////////////////////////////////
// Forward Reference

/////////////////////////////////////////////////////////////////////////////////
UINT32 codecNameToCode(const char *codec)
{
	if (!strcmp(codec, "H264")) {
		return SP5K_MEDIA_VIDEO_H264;
	}
	else if (!strcmp(codec, "JPEG")) {
		return SP5K_MEDIA_VIDEO_MJPG;
	}
	else if (!strcmp(codec, "L16")) {
		return SP5K_MEDIA_AUDIO_PCM;
	}
	else if (!strcmp(codec, "L8")) {
		return SP5K_MEDIA_AUDIO_PCM;
	}
	else if (!strcmp(codec, "DVI4")) {
		return SP5K_MEDIA_AUDIO_IMA_ADPCM;
	}
	else
		return 0;
}

/////////////////////////////////////////////////////////////////////////////////
// Global Variables
static int mediacli_cnt = 0;
static RTPMediaClientScheduler mediacli_sche;
static RTPMediaClient *mediacli_list = NULL;

/////////////////////////////////////////////////////////////////////////////////
static void mediacli_sche_deinit(RTPMediaClientScheduler *sche)
{
	ndk_info("rtpmc sche deinit");
	if (sche->env)
		sche->env->reclaim();

	sp5kMediaPlayCfgSet(SP5K_MEDIA_PLAY_ERROR_HANDLE_MODE, sche->h264_play_err_handle_mode);
	delete sche->scheduler;
	ndk_st_unregister_rtp_client();
	ndk_thread_destroy(sche->thrid);
}

static void mediacli_sche_thread(RTPMediaClientScheduler *sche)
{
	sche->scheduler->doEventLoop(&sche->abort);
	mediacli_sche_deinit(sche);
}

static Boolean mediacli_sche_init(RTPMediaClientScheduler *sche)
{
	memset(sche, 0, sizeof(*sche));

	if ((sche->scheduler = BasicTaskScheduler::createNew()) == NULL) {
		ndk_error("scheduler");
	}
	else if ((sche->env = BasicUsageEnvironment::createNew(*sche->scheduler)) == NULL) {
		ndk_error("env");
	}
	else if ((sche->thrid = ndk_thread_create("rtpcli", ndk_thread_get_priority(),
	                        (void (*)(void*))mediacli_sche_thread, sche, 16*1024 - 256)) == 0) {
		ndk_error("thread");
	}
	else {
		if (ndk_netif_ioctl(NDK_IOCG_IF_INDEXTONAME, 1, (long*)sche->env->ifname) != 0)
			ndk_error("indextoname");

		sp5kMediaPlayCfgGet(SP5K_MEDIA_PLAY_ERROR_HANDLE_MODE, &sche->h264_play_err_handle_mode);
		sp5kMediaPlayCfgSet(SP5K_MEDIA_PLAY_ERROR_HANDLE_MODE, 1);
		return True;
	}

	if (sche->env)
		sche->env->reclaim();

	delete sche->scheduler;
	return False;
}

/////////////////////////////////////////////////////////////////////////////////
struct RTPMediaClientBuffer
{
	//struct timeval createTime;
	unsigned dataSize;
	unsigned bufferSize;
	UINT8    *data;
	UINT8    *buffer;
	UINT32   magicNumber;

	int presize() const { return (int)((long)data - (long)buffer); }
	static RTPMediaClientBuffer *allocBuffer(unsigned dataSize, unsigned presize);
	static void freeBuffer(RTPMediaClientBuffer *buf);
	static void freeBufferSetNull(RTPMediaClientBuffer *&buf) { freeBuffer(buf); buf = NULL; }
	static void freeBufferObject(void *p);
};

RTPMediaClientBuffer *
RTPMediaClientBuffer::allocBuffer(unsigned dataSize, unsigned presize)
{
	RTPMediaClientBuffer *buf;
	unsigned allocSize;

	presize   = NDK_ALIGN_TO(presize, 4);
	allocSize = sizeof(RTPMediaClientBuffer) + dataSize + presize;
	buf       = (RTPMediaClientBuffer*)osMemAlloc(allocSize); /* It must be uncached. */
	if (!buf)
		return NULL;

	buf->magicNumber = 0xBEAFBEAF;
	buf->buffer     = (UINT8*)(buf + 1);
	buf->bufferSize = dataSize + presize;
	buf->data       = &buf->buffer[presize];
	buf->dataSize   = dataSize;
	//tmrTimeStampGet(&buf->createTime);

	return buf;
}

void RTPMediaClientBuffer::freeBuffer(RTPMediaClientBuffer *buf)
{
	if (buf) {
		NDK_ASSERT(buf->magicNumber == 0xBEAFBEAF);
		osMemFree(buf);
	}
}

void RTPMediaClientBuffer::freeBufferObject(void *p)
{
	RTPMediaClientBuffer::freeBuffer((RTPMediaClientBuffer*)p);
}

/////////////////////////////////////////////////////////////////////////////////
class RTPMediaClientAlarm : public AlarmHandler
{
public:
	static TaskToken scheduleTask(UsageEnvironment &e,
		RTPMediaClient *c,
		RTPMediaClient::MsgFunc f,
		RTPMediaClient::MsgStru const &m,
		unsigned timeout,
		TaskToken token)
	{
		DelayInterval delay((long)(timeout/1000), (long)((timeout%1000) * 1000));
		RTPMediaClientAlarm *h = new RTPMediaClientAlarm(c, f, m, delay);

		if (!h)
			return NULL;

		if (token)
			h->token((intptr_t)token);
		return e.taskScheduler().scheduleDelayedTask(h);
	}

	void *operator new (size_t size) throw();
	void operator delete (void* ptr);
private:
	RTPMediaClientAlarm(RTPMediaClient *c,
		RTPMediaClient::MsgFunc f,
		RTPMediaClient::MsgStru const &m,
		DelayInterval const &timeToDelay)
	: AlarmHandler(timeToDelay), fMediaCli(c), fMsgFunc(f) {
		fMsgStru = m;
	}

	virtual void handleTimeout() {
		(fMediaCli->*fMsgFunc)(&fMsgStru);
		DelayQueueEntry::handleTimeout();
	}

	RTPMediaClient *fMediaCli;
	RTPMediaClient::MsgFunc fMsgFunc;
	RTPMediaClient::MsgStru fMsgStru;
};

void *RTPMediaClientAlarm::operator new (size_t size) throw()
{
	return ndk_mem_malloc(size);
}

void RTPMediaClientAlarm::operator delete (void* ptr)
{
	ndk_mem_free(ptr);
}

/////////////////////////////////////////////////////////////////////////////////
RTPMediaClientSession*
RTPMediaClientSession::createNew(
	UsageEnvironment& env,
	char const* sdpDescription,
	RTPMediaClient &mcli)
{
	RTPMediaClientSession* s = new RTPMediaClientSession(env, mcli);

	if (s)
		if (!s->initializeWithSDP(sdpDescription)) {
			delete s;
			return NULL;
		}

	ndk_info("RTPMediaClientSession %f %f", s->playStartTime(), s->playEndTime());
	return s;
}

RTPMediaClientSession::
RTPMediaClientSession(UsageEnvironment& env, RTPMediaClient &mcli)
: MediaSession(env)
, fMediaCli(mcli)
{
}

RTPMediaClientSession::~RTPMediaClientSession()
{
	ndk_info("~RTPMediaClientSession");
}

/////////////////////////////////////////////////////////////////////////////////
class RTPMediaClientSink : public MediaSink
{
public:
	RTPMediaClientSink(UsageEnvironment& env,
		MediaSubsession *subsession,
		NDKMediaCliAVSink *avsnk,
		unsigned codec,
		int audioPTSCorrection,
		RTPMediaClientStats &stats);
	virtual ~RTPMediaClientSink();

private: // redefined virtual functions:
	virtual Boolean continuePlaying();
	int  prependSpsAndPps(UINT8 *data, int presize);
	void processFrame(unsigned frameSize, unsigned numTruncatedBytes, struct timeval &presentationTime);
	void rtpTimestamp2Timeval(struct timeval &tv, UINT32 rtpTimestamp, UINT32 frequency);

	static void afterGettingFrame(RTPMediaClientSink* This,
		unsigned frameSize,
		unsigned numTruncatedBytes,
		struct timeval presentationTime,
		unsigned durationInMicroseconds);

	static Boolean allocFrameBuffer(unsigned frameSize, RTPMediaClientSink* This);

private:
	MediaSubsession *fSubsession;
	NDKMediaCliAVSink *fAVSink;
	UINT32 fCodec;

	RTPMediaClientBuffer *fBuffer;
	int    fAudioPTSCorrection;
	UINT32 fPresize;

	struct timeval fLastPresentationTime;
	SINT64 fPrevRTPTimestampHigh;
	UINT32 fPrevRTPTimestampLow;
	UINT32 fInitialRTPTimestamp;

	// Statistics
	RTPMediaClientStats &fStats;

	// For H264 SPS & PPS
	UINT8  *fSpsPpsBuffer;
	UINT32 fSpsPpsSize;
	UINT8  _SpsPpsBuffer[4 + 64];

	// Flags
	UINT32 // H264
	       fSpsPpsSent : 1,
	       fSyncKeyFrame : 1;
};

RTPMediaClientSink::
RTPMediaClientSink(UsageEnvironment& env,
	MediaSubsession *subsession,
	NDKMediaCliAVSink *avsnk,
	unsigned codec,
	int audioPTSCorrection,
	RTPMediaClientStats &stats)
: MediaSink(env)
, fStats(stats)
{
	fSubsession = subsession;
	fAVSink = avsnk;
	fCodec = codec;
	fBuffer = NULL;
	fAudioPTSCorrection = audioPTSCorrection;
	fSpsPpsBuffer = NULL;

	switch (fCodec) {
	case SP5K_MEDIA_VIDEO_H264:
		fPresize = 4 + 64;
		break;
	case SP5K_MEDIA_VIDEO_MJPG:
		fPresize = 1024*2;
		break;
	default:
		fPresize = 0;
		break;
	}

	fLastPresentationTime.tv_sec = fLastPresentationTime.tv_usec = 0;
	fPrevRTPTimestampHigh = -1;
	fPrevRTPTimestampLow = 0;
	fInitialRTPTimestamp = 0;
	fSpsPpsSent = 0;
	fSyncKeyFrame = 1;

	memset(&fStats, 0, sizeof(fStats));
	fStats.used = 1;
}

RTPMediaClientSink::~RTPMediaClientSink()
{
	profLogPrintf(0, "~RTPMediaClientSink");
	RTPMediaClientBuffer::freeBuffer(fBuffer);
	fStats.used = 0;
}

Boolean RTPMediaClientSink::continuePlaying()
{
	if (fSource == NULL)
		return False;

	fSource->setAllocFrameBufferFunc((FramedSource::allocFrameBufferFunc*)allocFrameBuffer, this);
	fSource->getNextFrame(NULL, 0,
		(FramedSource::afterGettingFunc*)afterGettingFrame, this,
		onSourceClosure, this);

	return True;
}

void RTPMediaClientSink::rtpTimestamp2Timeval(struct timeval &tv, UINT32 rtpTimestamp, UINT32 frequency)
{
	if (fPrevRTPTimestampHigh == -1) {
		fInitialRTPTimestamp  = rtpTimestamp;
		fPrevRTPTimestampHigh = fPrevRTPTimestampLow = 0;
	}

	// Overflow
	if (rtpTimestamp < fPrevRTPTimestampLow) {
		if (fPrevRTPTimestampLow > 0xF0000000U && rtpTimestamp < 0x10000000U) {
			fPrevRTPTimestampHigh += 0xFFFFFFFFU;
			profLogPrintf(0, "RTPCli timestamp overflow %u %u %lld", rtpTimestamp, fPrevRTPTimestampLow, fPrevRTPTimestampHigh);
		}
		else {
			ndk_error("RTPCli Wrong Timestamp: %u %u", rtpTimestamp, fPrevRTPTimestampLow);
		}
	}

	fPrevRTPTimestampLow = rtpTimestamp;

	SINT64 pts = (fPrevRTPTimestampHigh + fPrevRTPTimestampLow) - fInitialRTPTimestamp;
	tv.tv_sec  = (long)(pts/frequency);
	tv.tv_usec = (long)((pts%frequency) * (1000000/frequency));
}

void RTPMediaClientSink::processFrame(unsigned frameSize,
	unsigned numTruncatedBytes,
	struct timeval &presentationTime)
{
	//profLogPrintf(0, "[%s] frmsz %u trunc %u", tv2str(&presentationTime), frameSize, numTruncatedBytes);

	NDKMediaCliAVBuffer mbuf;
	UINT32 err;
	int ptsCorrection = 0;

	NDK_ASSERT(fBuffer && fBuffer->dataSize >= frameSize);
	fBuffer->dataSize = frameSize;
	memset(&mbuf, 0, sizeof(mbuf));

	if (numTruncatedBytes) {
		profLogPrintf(0, "RTPCli: buffer truncked %u bytes. Discard", numTruncatedBytes);
		RTPMediaClientBuffer::freeBufferSetNull(fBuffer);
		fSyncKeyFrame = 1;
		return;
	}

	switch (fCodec) {
	case SP5K_MEDIA_VIDEO_H264:
	{
		mbuf.f_video = 1;

		if ((fBuffer->data[0] & 0x1f) == 5) {
			mbuf.f_keyframe = 1;
			if (fSyncKeyFrame)
				fSyncKeyFrame = 0;
		}
		else {
			if (fSyncKeyFrame) {
				RTPMediaClientBuffer::freeBufferSetNull(fBuffer);
				return;
			}
		}

		fBuffer->data -= 4;
		fBuffer->data[0] = fBuffer->data[1] = fBuffer->data[2] = 0;
		fBuffer->data[3] = 1;
		fBuffer->dataSize += 4;

		if (/*!fSpsPpsSent && */(mbuf.f_keyframe)) {
			NDK_ASSERT(fBuffer->data > fBuffer->buffer);
			int n = prependSpsAndPps(fBuffer->data, fBuffer->presize());
			if (n < 0) {
				ndk_error("prepend %d", n);
				RTPMediaClientBuffer::freeBufferSetNull(fBuffer);
				return;
			}

			mbuf.f_paramframe = 1;
			fBuffer->data -= n;
			fBuffer->dataSize += n;
			//fSpsPpsSent = 1;
		}

		//profLogPrintf(0, "v %s", tv2str(&presentationTime));
		break;
	}
	case SP5K_MEDIA_VIDEO_MJPG:
		mbuf.f_keyframe = 1;
		mbuf.f_video = 1;
		break;
	case SP5K_MEDIA_AUDIO_PCM:
	case SP5K_MEDIA_AUDIO_IMA_ADPCM:
	case SP5K_MEDIA_AUDIO_ALAW:
	case SP5K_MEDIA_AUDIO_MULAW:
		mbuf.f_keyframe = 1;
		mbuf.f_audio = 1;
		ptsCorrection = fAudioPTSCorrection;
		//profLogPrintf(0, "a %s", tv2str(&presentationTime));
		break;
	default:
		RTPMediaClientBuffer::freeBufferSetNull(fBuffer);
		return;
	}

	mbuf.bufobj  = (mediaBufferObject_t)fBuffer;
	mbuf.pf_free = RTPMediaClientBuffer::freeBufferObject;
	mbuf.data    = fBuffer->data;
	mbuf.length  = fBuffer->dataSize;
#if 1
	MultiFramedRTPSource *rtpsrc = (MultiFramedRTPSource *)fSource;

	rtpTimestamp2Timeval(mbuf.pts, rtpsrc->curPacketRTPTimestamp(), rtpsrc->timestampFrequency());
	timeval_add_ms(&mbuf.pts, ptsCorrection);
#else
	mbuf.pts = presentationTime;
#endif
	//printf("rtp %c %s %s\n", mbuf.f_audio ? 'A' : 'V', tv2str(&mbuf.pts), tv2str(&presentationTime));

	//ndk_hexdump("mbuf", mbuf.data, mbuf.length, 48);
	//int delay = (mtime_now_get() + ndk_ntp_get_timediff()) - timeval2mtime(&rtpsrc->frameCreationTime());
	//fStats.avg_rx_delay = ((fStats.avg_rx_delay * 29) + delay)/30;

	err = fAVSink->pf_send_buffer(fAVSink, &mbuf);
	if (err != SUCCESS) {
		RTPMediaClientBuffer::freeBufferSetNull(fBuffer);
	}

	fBuffer = NULL;
}

void RTPMediaClientSink::afterGettingFrame(RTPMediaClientSink* This,
	unsigned frameSize,
	unsigned numTruncatedBytes,
	struct timeval presentationTime,
	unsigned durationInMicroseconds)
{
	if (This->fAVSink) {
		This->processFrame(frameSize, numTruncatedBytes, presentationTime);
	}
	else {
		switch (This->fCodec) {
		case SP5K_MEDIA_VIDEO_H264:
		case SP5K_MEDIA_VIDEO_MJPG:
			printf("vid: %ld\n", timeval_diff_ms(&presentationTime, &This->fLastPresentationTime));
			break;
		case SP5K_MEDIA_AUDIO_PCM:
			printf("aud: %ld\n", timeval_diff_ms(&presentationTime, &This->fLastPresentationTime));
			break;
		default:
			break;
		}

		This->fLastPresentationTime = presentationTime;
		RTPMediaClientBuffer::freeBufferSetNull(This->fBuffer);
	}

	This->fSource->getNextFrame(NULL, 0,
		(FramedSource::afterGettingFunc*)afterGettingFrame, This,
		onSourceClosure, This);
}

Boolean RTPMediaClientSink::allocFrameBuffer(unsigned frameSize, RTPMediaClientSink* This)
{
	NDK_ASSERT(!This->fBuffer);

	// Allocate more bytes for the estimateFrameSize function may get wrong size.
	This->fBuffer = RTPMediaClientBuffer::allocBuffer(frameSize + 1024, This->fPresize);
	if (!This->fBuffer)
		return False;

	This->fSource->updateFrameBuffer(This->fBuffer->data, This->fBuffer->dataSize);
	return True;
}

int RTPMediaClientSink::prependSpsAndPps(UINT8 *data, int presize)
{
	if (!fSpsPpsBuffer) {
		SPropRecord const *recs;
		UINT8 *p;
		int i, nrecs;

		recs = fSubsession->sPropRecs(nrecs);
		p = fSpsPpsBuffer = _SpsPpsBuffer;
		fSpsPpsSize = 0;

		for (i = 0; i < nrecs; ++i) {
			SPropRecord const *r = &recs[i];

			fSpsPpsSize += (int)(r->sPropLength + 4);
			NDK_ASSERT(fSpsPpsSize <= sizeof(_SpsPpsBuffer));

			p[0] = p[1] = p[2] = 0;
			p[3] = 1;
			p += 4;

			memcpy(p, r->sPropBytes, r->sPropLength);
			p += r->sPropLength;
		}

		NDK_ASSERT(fSpsPpsSize > 0);
		NDK_ASSERT(fSpsPpsSize <= (UINT32)presize);
	}

	memcpy(data - fSpsPpsSize, fSpsPpsBuffer, fSpsPpsSize);
	return fSpsPpsSize;
}

/////////////////////////////////////////////////////////////////////////////////
long RTPMediaGetOpt::getl(long id, long defval)
{
	int i;
	for (i = 0; i < fOptNum; ++i) {
		if (fOpts[i].id == id)
			return fOpts[i].u.l;
	}

	return defval;
}

double RTPMediaGetOpt::getd(long id, double defval)
{
	int i;
	for (i = 0; i < fOptNum; ++i) {
		if (fOpts[i].id == id)
			return fOpts[i].u.d;
	}

	return defval;
}

/////////////////////////////////////////////////////////////////////////////////
RTPMediaClient::
RTPMediaClient(UsageEnvironment& env,
	char const* rtspURL,
	int  verbosityLevel,
	char const* applicationName,
	NDKMediaCliAVSink *avsink,
	NDKMediaCliRtpMsgHandler *msgHandler,
	int optnum,
	NDKMediaCliOpt *opts)
: RTSPClient(env, rtspURL, verbosityLevel, applicationName, 0, -1)
, fState(STAT_INITIAL)
, fMsgHandler(msgHandler)
{
	RTPMediaGetOpt mopts(optnum, opts);
	unsigned int x;

	initClient();
	sp5kOsEventFlagsCreate(&fStateEvt, (CHAR*)"rtpcli");

	fFlgTCP = mopts.getl(NDK_MEDIACLI_RTP_OPT_TCP, 0) ? 1 : 0;

	if (mopts.getl(NDK_MEDIACLI_RTP_OPT_AUDIO_ONLY, 0))
		fSingleMedium = "audio";
	else if (mopts.getl(NDK_MEDIACLI_RTP_OPT_VIDEO_ONLY, 0))
		fSingleMedium = "video";
	else
		fSingleMedium = NULL;

	fPktArrivalTimeout = mopts.getl(NDK_MEDIACLI_RTP_OPT_TIMEOUT, 10*1000);
	fPlaybackCacheTime = (unsigned int)mopts.getl(NDK_MEDIACLI_RTP_OPT_CACHETIME, 500);
	fAudioPTSCorrection = (int)mopts.getl(NDK_MEDIACLI_RTP_OPT_AVSYNCTIME, 0);
	fH264GopNo = (UINT32)mopts.getl(NDK_MEDIACLI_RTP_OPT_MEDIA_ATTR + SP5K_MEDIA_ATTR_H264_GOP_NO, 15);
	fAVSink = avsink;

	fMsgToken = &fMsgToken; // Use the address of the fMsgToken to be the value of the msg token.

	ndk_global_lock(&x);
	fNext = mediacli_list;
	mediacli_list = this;
	ndk_global_unlock(&x);
}

RTPMediaClient::~RTPMediaClient()
{
	NDK_ASSERT(fState == STAT_INITIAL || fState == STAT_CLOSED);

	int mcli_cnt;
	unsigned int x;

	ndk_global_lock(&x);
	for (RTPMediaClient **pp = &mediacli_list; *pp; pp = &(*pp)->fNext) {
		if (*pp == this) {
			*pp = fNext;
			break;
		}
	}
	mcli_cnt = --mediacli_cnt;
	ndk_global_unlock(&x);

	if (mcli_cnt == 0) {
		mediacli_sche.abort = 1;
	}

	sp5kOsEventFlagsDelete(&fStateEvt);
	if (fAVSink)
		fAVSink->pf_destroy(fAVSink);
	ndk_info("~RTPMediaClient");
}

Boolean RTPMediaClient::findClient(RTPMediaClient *This)
{
	unsigned int x;
	Boolean found = False;

	ndk_global_lock(&x);
	for (RTPMediaClient *p = mediacli_list; p; p = p->fNext) {
		if (p == This) {
			found = True;
			break;
		}
	}
	ndk_global_unlock(&x);

	return found;
}

void RTPMediaClient::initClient()
{
	memset(&__start, 0, &__end - &__start);

	fDurationStop = -1.0; // extra seconds to play at the end
	fInitialSeekTime = 0.0f;
	fScale = 1.0f;
	fTotNumPacketsReceived = ~0; // used if checking inter-packet gaps
	fWaitForResponseToTEARDOWN = True;
}

void RTPMediaClient::setState(unsigned int state)
{
	Boolean illegal = False;

	switch (state) {
	case STAT_OPTIONS:
		if (fState != STAT_INITIAL)
			illegal = True;
		break;
	case STAT_DESCRIBE:
		if (fState != STAT_OPTIONS)
			illegal = True;
		break;
	case STAT_SETUP:
		if (fState != STAT_DESCRIBE)
			illegal = True;
		break;
	case STAT_PLAYING:
		if (fState != STAT_SETUP && fState != STAT_PAUSED)
			illegal = True;
		break;
	case STAT_PAUSED:
		if (fState != STAT_PLAYING)
			illegal = True;
		break;
	case STAT_TEARDOWN:
		if (fState != STAT_PLAYING && fState != STAT_PAUSED)
			illegal = True;
		break;
	case STAT_CLOSED:
		if (fState == STAT_PLAYING || fState == STAT_PAUSED)
			illegal = True;
		break;
	default:
		illegal = True;
		break;
	}

	if (illegal) {
		ndk_error("Illegal state transition: %d => %d", fState, state);
		NDK_ASSERT(0);
	}
	else {
		//profLogPrintf(0, "rtpcli %u -> %u", fState, state);
		fState = state;

		sp5kOsEventFlagsSet(&fStateEvt, 0, TX_AND);
		sp5kOsEventFlagsSet(&fStateEvt, 1 << fState, TX_OR);
	}
}

Boolean RTPMediaClient::waitState(unsigned int state)
{
	UINT32 cur_flags = 0;

	sp5kOsEventFlagsGet(&fStateEvt, (1 << state) | (1 << STAT_TRANS_ERROR), TX_OR, &cur_flags, TX_WAIT_FOREVER);
	if (cur_flags & (1 << STAT_TRANS_ERROR))
		return False;

	if (cur_flags == (1 << state))
		return True;

	NDK_ASSERT(0);
	return False;
}

void RTPMediaClient::start(double startTime)
{
	NDK_ASSERT(fState == STAT_INITIAL || fState == STAT_CLOSED);
	if (fState == STAT_CLOSED) {
		initClient();
		fState = STAT_INITIAL;
	}

	fInitialSeekTime = startTime;
	sendOptionsCommand(&continueAfterOPTIONS0, NULL);
}

UINT32 RTPMediaClient::resume(long position, Boolean wait)
{
	unsigned int x;
	int in_progress;

	ndk_global_lock(&x);
	in_progress = fPauseOrResumeInProgress;
	ndk_global_unlock(&x);

	if (fState == STAT_PLAYING)
		return SUCCESS;

	if (fState != STAT_PAUSED || in_progress)
		return FAIL;

	MsgStru m;

	m.param0.d = (double)position/1000.0;

	ndk_global_lock(&x);
	fPauseOrResumeInProgress = 1;
	ndk_global_unlock(&x);

	profLogPrintf(0, "RTP Cli resume %ld", position);
	scheduleMsgTask(&RTPMediaClient::msgResume, m);

	if (wait) {
		NDK_ASSERT(ndk_thread_identify() != mediacli_sche.thrid);
		if (!waitState(STAT_PLAYING))
			return FAIL;
	}

	return SUCCESS;
}

void RTPMediaClient::msgResume(RTPMediaClient::MsgStru *m)
{
	double startTime = m->param0.d;
	const char *strEndTime = fSession->absEndTime();
	double endTime = strEndTime ? strtod(strEndTime, NULL) : -1.0;

	ndk_info("resume (%f %f)", startTime, endTime);
	sendPlayCommand(*fSession, &continueAfterPLAY0, startTime, endTime, fScale);
}

UINT32 RTPMediaClient::pause(Boolean wait)
{
	unsigned int x;
	int in_progress;

	ndk_global_lock(&x);
	in_progress = fPauseOrResumeInProgress;
	ndk_global_unlock(&x);

	if (fState == STAT_PAUSED)
		return SUCCESS;

	if (fState != STAT_PLAYING || in_progress)
		return FAIL;

	MsgStru m;

	ndk_global_lock(&x);
	fPauseOrResumeInProgress = 1;
	ndk_global_unlock(&x);
	scheduleMsgTask(&RTPMediaClient::msgPause, m);

	profLogPrintf(0, "RTP Cli pause");

	if (wait) {
		NDK_ASSERT(ndk_thread_identify() != mediacli_sche.thrid);
		if (!waitState(STAT_PAUSED))
			return FAIL;
	}

	return SUCCESS;
}

void RTPMediaClient::msgPause(RTPMediaClient::MsgStru *m)
{
	sendPauseCommand(*fSession, continueAfterPAUSE0, NULL);
}

void RTPMediaClient::shutdown()
{
	ndk_info("shutdown %u %c%c", fState, fDestroyed ? 'd' : '-', fPauseOrResumeInProgress ? 'p' : '-');

	if (fShutdownInProgress)
		return;
	if (fGetParameterTask)
		envir().taskScheduler().unscheduleDelayedTask(fGetParameterTask);
	if (fArrivalCheckTimerTask)
		envir().taskScheduler().unscheduleDelayedTask(fArrivalCheckTimerTask);
	if (fInterPacketGapCheckTimerTask)
		envir().taskScheduler().unscheduleDelayedTask(fInterPacketGapCheckTimerTask);

	fShutdownInProgress = 1;
	if (fAVSink)
		fAVSink->pf_stop(fAVSink);

	if (fState == STAT_CLOSED || fState < STAT_PLAYING) {
		if (fState != STAT_CLOSED)
			shutdown2();
		return;
	}

	// Teardown, then shutdown, any outstanding RTP/RTCP subsessions
	Boolean shutdownImmediately = True; // by default
	if (fSession != NULL) {
		RTSPClient::responseHandler *responseHandlerForTEARDOWN = NULL; // unless:
		if (fWaitForResponseToTEARDOWN) {
			shutdownImmediately = False;
			responseHandlerForTEARDOWN = &continueAfterTEARDOWN0;
			fTeardownTask = scheduleTask(500, checkForTEARDOWN);
		}
		sendTeardownCommand(*fSession, responseHandlerForTEARDOWN, NULL);
	}

	if (shutdownImmediately) {
		setState(STAT_TEARDOWN);
		shutdown2();
	}
	else {
		// The shutdown2 will be called in continueAfterTEARDOWN
	}
}

void RTPMediaClient::msgShutdown(RTPMediaClient::MsgStru *m)
{
	shutdown();
}

void RTPMediaClient::shutdown2()
{
	if (fSession) {
		closeMediaSinks();
		Medium::close(fSession);
		fSession = NULL;
	}

	setState(STAT_CLOSED);
	fShutdownInProgress = 0;

	if (fDestroyed) {
		envir().taskScheduler().unscheduleDelayedTaskAll(fMsgToken);
		delete this;
	}

	ndk_info("shutdown2 %d", fDestroyed);
}

void RTPMediaClient::sendUserMsg(NDKMediaCliRtpMsg msg, long param)
{
	if (fMsgHandler && !fDestroyed)
		(*fMsgHandler)((NDKMediaCliHandle)&this->fMediaCliBase, msg, param);
}

TaskToken RTPMediaClient::scheduleMsgTask(RTPMediaClient::MsgFunc f, RTPMediaClient::MsgStru const &m)
{
	return RTPMediaClientAlarm::scheduleTask(envir(), this, f, m, 0, fMsgToken);
}

TaskToken RTPMediaClient::scheduleTask(int64_t mSecsToDelay, void (*func)(RTPMediaClient *))
{
	NDK_ASSERT(mSecsToDelay < 1000*1000);
	return envir().taskScheduler().scheduleDelayedTask(mSecsToDelay*1000, (TaskFunc *)func, (void *)this);
}

Boolean RTPMediaClient::updateAttributes()
{
	if (!fAVSink)
		return True;

	MediaSubsessionIterator iter(*fSession);
	MediaSubsession *subsession;

	fAVSink->pf_set_attr(fAVSink, SP5K_MEDIA_ATTR_DURATION, fDuration < 0.000001 ? (UINT32)-1 : (UINT32)(fDuration*1000));
	fAVSink->pf_set_attr(fAVSink, MEDIA_SNK_ATTR_CACHETIME, fPlaybackCacheTime);

	iter.reset();
	while ((subsession = iter.next()) != NULL) {
		if (subsession->readSource() == NULL) {
			continue;    // was not initiated
		}

		if (strcmp(subsession->mediumName(), "audio") == 0) {
			const char *codecName = subsession->codecName();
			UINT32 codec = codecNameToCode(codecName);

			switch (codec) {
			case SP5K_MEDIA_AUDIO_PCM:
			case SP5K_MEDIA_AUDIO_IMA_ADPCM:
			case SP5K_MEDIA_AUDIO_ALAW:
			case SP5K_MEDIA_AUDIO_MULAW:
				fAVSink->pf_set_attr(fAVSink, SP5K_MEDIA_ATTR_AUDIO_CODEC, codec);
				break;
			default:
				return False;
			}

			unsigned sampleRate = subsession->rtpTimestampFrequency();
			unsigned numChan = subsession->numChannels();
			unsigned char payloadFmt = subsession->rtpPayloadFormat();
			unsigned sampleBits = 0;

			if (payloadFmt < 96) {
				switch (payloadFmt) {
				case 0: // mulaw
				case 8: // alaw
					sampleBits = 8;
					break;
				case 10: case 11: // pcm
					sampleBits = 16;
					break;
				case 5: case 6: case 16: case 17: // adpcm
					sampleBits = 4;
					break;
				default:
					ndk_error("unsupported payload: %s %d", codecName, payloadFmt);
					return False;
				}
			}
			else {
				if (!strcmp(codecName, "L16")) {
					sampleBits = 16;
				}
				else if (!strcmp(codecName, "L8")) {
					sampleBits = 8;
				}
				else if (!strcmp(codecName, "DVI4")) {
					sampleBits = subsession->sampleBits();
				}
				else {
					ndk_error("unsupported payload: %s %d", codecName, payloadFmt);
					return False;
				}
			}

			fAVSink->pf_set_attr(fAVSink, SP5K_MEDIA_ATTR_AUDIO_SAMPLE_RATE, sampleRate);
			fAVSink->pf_set_attr(fAVSink, SP5K_MEDIA_ATTR_AUDIO_SAMPLE_BITS, sampleBits);
			fAVSink->pf_set_attr(fAVSink, SP5K_MEDIA_ATTR_AUDIO_CHANNELS, numChan);
			fAVSink->pf_set_attr(fAVSink, SP5K_MEDIA_ATTR_AUDIO_AVG_BITRATE, sampleRate * sampleBits * numChan);
		}
		else if (strcmp(subsession->mediumName(), "video") == 0) {
			switch (codecNameToCode(subsession->codecName())) {
			case SP5K_MEDIA_VIDEO_H264:
				fAVSink->pf_set_attr(fAVSink, SP5K_MEDIA_ATTR_VIDEO_CODEC, SP5K_MEDIA_VIDEO_H264);
				fAVSink->pf_set_attr(fAVSink, SP5K_MEDIA_ATTR_H264_GOP_NO, fH264GopNo);
				break;
			case SP5K_MEDIA_VIDEO_MJPG:
				fAVSink->pf_set_attr(fAVSink, SP5K_MEDIA_ATTR_VIDEO_CODEC, SP5K_MEDIA_VIDEO_MJPG);
				break;
			default:
				ndk_error("not supported");
				return False;
			}

			fAVSink->pf_set_attr(fAVSink, SP5K_MEDIA_ATTR_WIDTH, subsession->width());
			fAVSink->pf_set_attr(fAVSink, SP5K_MEDIA_ATTR_HEIGHT, subsession->height());
			fAVSink->pf_set_attr(fAVSink, SP5K_MEDIA_ATTR_VIDEO_FRAME_RATE, subsession->videoFPS());
			fAVSink->pf_set_attr(fAVSink, SP5K_MEDIA_ATTR_VIDEO_BRC_TYPE, MEDIA_CBR);
			//SP5K_MEDIA_ATTR_VIDEO_AVG_BITRATE
		}
		else {
			return False;
		}
	}

	return True;
}

void RTPMediaClient::continueAfterOPTIONS(int resultCode, char *resultString)
{
	StringAutoDelete str_audo_delete(resultString);

	//ndk_trace("%s", __func__);
	NDK_ASSERT(fState == STAT_INITIAL);

	if (resultCode != 0) {
		ndk_error("OPTIONS failed: %s", resultString);
		sendUserMsg(NDK_MEDIACLI_RTP_MSG_ECONN, __LINE__);
		return;
	}

	ndk_info("OPTIONS succeed: %s", resultString);
	setState(STAT_OPTIONS);
	sendDescribeCommand(continueAfterDESCRIBE0, NULL);
}

void RTPMediaClient::continueAfterDESCRIBE(int resultCode, char *resultString)
{
	StringAutoDelete str_audo_delete(resultString);

	//ndk_trace("%s", __func__);
	NDK_ASSERT(fState == STAT_OPTIONS);

	if (resultCode != 0) {
		ndk_error("DESCRIBE failed: '%s'", resultString);
		sendUserMsg(NDK_MEDIACLI_RTP_MSG_ECONN, __LINE__);
		return;
	}

	char *sdpDescription = resultString;
	ndk_info("DESCRIBE succeed: '");
	printf("%s\n", sdpDescription);

	// Create a media session object from this SDP description:
	fSession = RTPMediaClientSession::createNew(envir(), sdpDescription, *this);

	if (fSession == NULL) {
		ndk_error("MediaSession error");
		sendUserMsg(NDK_MEDIACLI_RTP_MSG_ECONN, __LINE__);
		return;
	}
	else if (!fSession->hasSubsessions()) {
		ndk_error("No media subsessions");
		sendUserMsg(NDK_MEDIACLI_RTP_MSG_ECONN, __LINE__);
		return;
	}

	// Then, setup the "RTPSource"s for the session:
	MediaSubsessionIterator iter(*fSession);
	MediaSubsession *subsession;
	int subsession_cnt = 0;
	char const *singleMediumToTest = fSingleMedium;

	while ((subsession = iter.next()) != NULL) {
		// If we've asked to receive only a single medium, then check this now:
		if (singleMediumToTest != NULL) {
			if (strcmp(subsession->mediumName(), singleMediumToTest) != 0) {
				ndk_warning("Ignoring '%s'/%s for single session %s",
					subsession->mediumName(), subsession->codecName(), fSingleMedium);
				continue;
			}
			else {
				// Receive this subsession only
				singleMediumToTest = "xxxxx";
				// this hack ensures that we get only 1 subsession of this type
			}
		}

		if (fDesiredPortNum != 0) {
			subsession->setClientPortNum(fDesiredPortNum);
			fDesiredPortNum += 2;
		}

		if (!subsession->initiate(-1)) {
			ndk_error("Unable to create subsession for '%s'/%s", subsession->mediumName(), subsession->codecName());
		}
		else {
			ndk_info("Created subsession for '%s'/%s, ports %d, %d",
				subsession->mediumName(),
				subsession->codecName(),
				subsession->clientPortNum(),
				subsession->clientPortNum() + 1);
			++subsession_cnt;

			if (subsession->rtpSource() != NULL) {
				// Because we're saving the incoming data, rather than playing
				// it in real time, allow an especially large time threshold
				// (1 second) for reordering misordered incoming packets:
				subsession->rtpSource()->setPacketReorderingThresholdTime(1000000/*1 second*/);
			}
		}
	}

	if (!subsession_cnt) {
		sendUserMsg(NDK_MEDIACLI_RTP_MSG_ECONN, __LINE__);
		return;
	}

	NDK_ASSERT(fAVSink);
	fAVSink->pf_init(fAVSink, avSinkMsgHandler, this,
		(fSession->playEndTime() - fSession->playStartTime() > 0.000001) ? 0 : NDK_MEDIACLI_AVSNK_FLG_LIVE);
	ndk_info("avsink = %s", fAVSink->name);

	setState(STAT_DESCRIBE);
	// Perform additional 'setup' on each subsession, before playing them:
	setupStreams();
}

void RTPMediaClient::continueAfterSETUP(int resultCode, char *resultString)
{
	StringAutoDelete str_audo_delete(resultString);
	//ndk_trace("%s", __func__);

	if (resultCode != 0) {
		ndk_info("SETUP failed: '%s'/%s",
			fSetupSubsession->mediumName(),
			fSetupSubsession->codecName());
		sendUserMsg(NDK_MEDIACLI_RTP_MSG_ECONN, __LINE__);
		return;
	}

	ndk_info("Setup '%s'/%s, ports %d,%d",
		fSetupSubsession->mediumName(),
		fSetupSubsession->codecName(),
		fSetupSubsession->clientPortNum(),
		fSetupSubsession->clientPortNum());

	fSetupInProgress = 1;
	setupStreams();
}

void RTPMediaClient::continueAfterTEARDOWN(int resultCode, char *resultString)
{
	StringAutoDelete str_audo_delete(resultString);
	//ndk_trace("%s", __func__);
	if (fTeardownTask)
		envir().taskScheduler().unscheduleDelayedTask(fTeardownTask);

	setState(STAT_TEARDOWN);
	shutdown2();
}

void RTPMediaClient::continueAfterPLAY(int resultCode, char *resultString)
{
	StringAutoDelete str_audo_delete(resultString);

	//ndk_trace("%s", __func__);
	NDK_ASSERT(fState == STAT_SETUP || fState == STAT_PAUSED);

	if (resultCode != 0) {
		sp5kOsEventFlagsSet(&fStateEvt, 1 << STAT_TRANS_ERROR, TX_OR);
		ndk_error("PLAY failed: %s", resultString);
		sendUserMsg(NDK_MEDIACLI_RTP_MSG_ECONN, __LINE__);
		return;
	}

	ndk_info("PLAY (%f, %f)", fSession->playStartTime(), fSession->playEndTime());

	fPauseOrResumeInProgress = 0;
	setState(STAT_PLAYING);

	if (!fGetParameterTask)
		fGetParameterTask = scheduleTask(10*1000, checkForGET_PARAMETER);

	if (fPktArrivalTimeout > 0 && !fArrivalCheckTimerTask) {
		fArrivalCheckTimerTask = scheduleTask(fPktArrivalTimeout, checkForPacketArrival);
	}

	//checkInterPacketGaps();
}

void RTPMediaClient::continueAfterPAUSE(int resultCode, char *resultString)
{
	StringAutoDelete str_audo_delete(resultString);

	//ndk_trace("%s", __func__);
	NDK_ASSERT(fState == STAT_PLAYING);

	if (resultCode != 0) {
		sp5kOsEventFlagsSet(&fStateEvt, 1 << STAT_TRANS_ERROR, TX_OR);

		ndk_error("PUASE failed: %s", resultString);
		sendUserMsg(NDK_MEDIACLI_RTP_MSG_EGENERIC, __LINE__);
		return;
	}

	fPauseOrResumeInProgress = 0;
	setState(STAT_PAUSED);
	//sendUserMsg(NDK_MEDIACLI_MSG_PAUSED, 0);
}

void RTPMediaClient::setupStreams()
{
	//ndk_trace("%s", __func__);
	NDK_ASSERT(fState == STAT_DESCRIBE);
	NDK_ASSERT(fAVSink);

	if (!fSetupIter) {
		fSetupIter = new MediaSubsessionIterator(*fSession);
		fSetupInProgress = 0;
	}

	while ((fSetupSubsession = fSetupIter->next()) != NULL) {
		// We have another subsession left to set up:
		if (fSetupSubsession->clientPortNum() == 0) {
			continue;    // port # was not set
		}

		sendSetupCommand(*fSetupSubsession, &continueAfterSETUP0, False/*streamOutgoing*/,
			fFlgTCP, False/*forceMulticastOnUnspecified*/, NULL);
		return;
	}

	// We're done setting up subsessions.
	delete fSetupIter;
	fSetupIter = NULL;
	fSetupInProgress = 0;

	// Create and start "FileSink"s for each subsession:
	MediaSubsessionIterator iter(*fSession);
	MediaSubsession *subsession;
	RTPMediaClientSink *sink;
	int subsession_cnt = 0;
	Boolean subsession_error = False;

	while ((subsession = iter.next()) != NULL) {
		if (subsession->readSource() == NULL) {
			continue;    // was not initiated
		}

		unsigned int codec = codecNameToCode(subsession->codecName());
		if (!codec) {
			ndk_error("codec %s", subsession->codecName());
			subsession_error = True;
			break;
		}

		if (strcmp(subsession->mediumName(), "audio") == 0) {
			sink = new RTPMediaClientSink(envir(), subsession, fAVSink, codec,
			                              fAudioPTSCorrection, fStatsAudio);
		}
		else if (strcmp(subsession->mediumName(), "video") == 0) {
			sink = new RTPMediaClientSink(envir(), subsession, fAVSink, codec,
			                              0, fStatsVideo);
		}
		else {
			ndk_error("mediumName err: %s", subsession->mediumName());
			continue;
		}

		if (!sink) {
			subsession_error = True;
			ndk_info("createNew");
			break;
		}

		if (!sink->startPlaying(*(subsession->readSource()), subsessionAfterPlaying, subsession)) {
			Medium::close(sink);
			subsession_error = True;
			ndk_info("startPlaying");
			break;
		}

		// Also set a handle to be called if a RTCP "BYE" arrives for this subsession:
		if (subsession->rtcpInstance() != NULL) {
			subsession->rtcpInstance()->setByeHandler(subsessionByeHandler, subsession);
		}

		subsession->sink = sink;
		++subsession_cnt;
	}

	if (subsession_cnt == 0 || subsession_error) {
		sendUserMsg(NDK_MEDIACLI_RTP_MSG_ECONN, __LINE__);
		return;
	}

	// Finally, start playing each subsession, to start the data flow:
	if (fDuration < 0.0001f) {
		if (fScale > 0.0001f) {
			fDuration = fSession->playEndTime() - fInitialSeekTime;    // use SDP end time
		}
		else if (fScale < 0) {
			fDuration = fInitialSeekTime;
		}
	}
	if (fDuration < 0.0001f) {
		fDuration = 0.0f;
	}

	fEndTime = fInitialSeekTime;
	if (fScale > 0) {
		if (fDuration <= 0) {
			fEndTime = -1.0f;
		}
		else {
			fEndTime = fInitialSeekTime + fDuration;
		}
	}
	else {
		fEndTime = fInitialSeekTime - fDuration;
		if (fEndTime < 0.0001f) {
			fEndTime = 0.0f;
		}
	}
	ndk_info("duration=%s", f2str(fDuration, 2));

	if (!updateAttributes()) {
		sendUserMsg(NDK_MEDIACLI_RTP_MSG_EGENERIC, __LINE__);
		return;
	}

	// Start Media Sink
	UINT32 err = fAVSink->pf_start(fAVSink);
	if (err != SUCCESS) {
		ndk_error("MEDIA_SNK_CTRL_START err %d", err);
		sendUserMsg(NDK_MEDIACLI_RTP_MSG_EGENERIC, __LINE__);
		return;
	}

	setState(STAT_SETUP);
	sendUserMsg(NDK_MEDIACLI_RTP_MSG_STARTED, 0);

	char const *absStartTime = fInitialAbsoluteSeekTime != NULL ? fInitialAbsoluteSeekTime : fSession->absStartTime();
	if (absStartTime != NULL) {
		// Either we or the server have specified that seeking should be done by 'absolute' time:
		ndk_info("play %s %s %f", absStartTime, fSession->absEndTime(), fScale);
		sendPlayCommand(*fSession, &continueAfterPLAY0, absStartTime, fSession->absEndTime(), fScale);
	}
	else {
		// Normal case: Seek by relative time (NPT):
		ndk_info("play %f %f %f", fInitialSeekTime, fEndTime, fScale);
		sendPlayCommand(*fSession, &continueAfterPLAY0, fInitialSeekTime, fEndTime, fScale);
	}
}

void RTPMediaClient::closeMediaSinks()
{
	MediaSubsessionIterator iter(*fSession);
	MediaSubsession *subsession;
	while ((subsession = iter.next()) != NULL) {
		Medium::close(subsession->sink);
		subsession->sink = NULL;
	}
}

void RTPMediaClient::subsessionByeHandler(void *ss)
{
	struct timeval timeNow;
	MediaSubsession *subsession = (MediaSubsession *)ss;
	RTPMediaClientSession *session = (RTPMediaClientSession *)&subsession->parentSession();
	RTPMediaClient &This = session->mediaClient();

	gettimeofday(&timeNow, NULL);
	unsigned secsDiff = timeNow.tv_sec - This.fStartTime.tv_sec;

	ndk_info("Received RTCP \"BYE\" on '%s'/%s subsession (after %u seconds)",
		subsession->mediumName(),
		subsession->codecName(),
		secsDiff);

	// Act now as if the subsession had closed:
	subsessionAfterPlaying(subsession);
}

void RTPMediaClient::subsessionAfterPlaying(void *ss)
{
	// Begin by closing this media subsession's stream:
	MediaSubsession *subsession = (MediaSubsession *)ss;
	RTPMediaClientSession *session = (RTPMediaClientSession *)&subsession->parentSession();

	Medium::close(subsession->sink);
	subsession->sink = NULL;

	// Next, check whether *all* subsessions' streams have now been closed:
	MediaSubsessionIterator iter(*session);
	while ((subsession = iter.next()) != NULL) {
		if (subsession->sink != NULL) {
			return;    // this subsession is still active
		}
	}

	// All subsessions' streams have now been closed
	session->mediaClient().sessionAfterPlaying(True);
}

void RTPMediaClient::sessionAfterPlaying(Boolean bye)
{
	shutdown();
	sendUserMsg(NDK_MEDIACLI_RTP_MSG_STOPPED, 0);
}

void RTPMediaClient::checkForGET_PARAMETER(RTPMediaClient *This)
{
	if (This->fState != STAT_PLAYING && This->fState != STAT_PAUSED)
		return;

	This->sendGetParameterCommand(*This->fSession, &continueAfterGET_PARAMETER0, NULL, NULL);
	This->fGetParameterTask = This->scheduleTask(10*1000, checkForGET_PARAMETER);
}

void RTPMediaClient::checkForPacketArrival(RTPMediaClient *This)
{
	if (This->fState != STAT_PLAYING && This->fState != STAT_PAUSED)
		return;

	// Check each subsession, to see whether it has received data packets:
	unsigned numSubsessionsChecked = 0;
	unsigned numSubsessionsWithReceivedData = 0;
	unsigned numSubsessionsThatHaveBeenSynced = 0;

	MediaSubsessionIterator iter(*This->fSession);
	MediaSubsession *subsession;
	while ((subsession = iter.next()) != NULL) {
		RTPSource *src = subsession->rtpSource();
		if (src == NULL) {
			continue;
		}
		++numSubsessionsChecked;

		if (src->receptionStatsDB().numActiveSourcesSinceLastReset() > 0) {
			// At least one data packet has arrived
			++numSubsessionsWithReceivedData;
		}
		if (src->hasBeenSynchronizedUsingRTCP()) {
			++numSubsessionsThatHaveBeenSynced;
		}

		src->receptionStatsDB().reset();
	}

	//ndk_info("%u %u", numSubsessionsChecked, numSubsessionsWithReceivedData);
	if (numSubsessionsWithReceivedData == 0 && (This->fState == STAT_PLAYING)) {
		ndk_error("recv timeout");
		This->shutdown();
		This->sendUserMsg(NDK_MEDIACLI_RTP_MSG_TIMEOUT, 0);
	}

	if (This->fPktArrivalTimeout > 0)
		This->fArrivalCheckTimerTask = This->scheduleTask(This->fPktArrivalTimeout, checkForPacketArrival);
}

void RTPMediaClient::checkInterPacketGaps(RTPMediaClient *This)
{
	if (This->fState != STAT_PLAYING && This->fState != STAT_PAUSED)
		return;

	if (This->fInterPacketGapMaxTime == 0) {
		return;    // we're not checking
	}

	// Check each subsession, counting up how many packets have been received:
	unsigned newTotNumPacketsReceived = 0;
	MediaSubsessionIterator iter(*This->fSession);
	MediaSubsession *subsession;
	while ((subsession = iter.next()) != NULL) {
		RTPSource *src = subsession->rtpSource();
		if (src == NULL) {
			continue;
		}
		newTotNumPacketsReceived += src->receptionStatsDB().totNumPacketsReceived();
	}

	if (newTotNumPacketsReceived == This->fTotNumPacketsReceived) {
		// No additional packets have been received since the last time we
		// checked, so end this stream:
		ndk_info("Closing session, because we stopped receiving packets");
		This->fInterPacketGapCheckTimerTask = NULL;
		This->sessionAfterPlaying(False);
	}
	else {
		This->fTotNumPacketsReceived = newTotNumPacketsReceived;
		// Check again, after the specified delay:
		This->fInterPacketGapCheckTimerTask = This->scheduleTask(This->fInterPacketGapMaxTime * 1000, This->checkInterPacketGaps);
	}
}

void RTPMediaClient::checkForTEARDOWN(RTPMediaClient *This)
{
	ndk_info("TEARDOWN timeout");
	This->continueAfterTEARDOWN(0, NULL);
}

UINT32 RTPMediaClient::avSinkMsgHandler(UINT32 msg, long param, void *user_data)
{
	RTPMediaClient *This = (RTPMediaClient *)user_data;
	RTPMediaClient::MsgStru m;
	BOOL bWait = TRUE;

	// In the media client scheduler thread. Cannot wait.
	if (mediacli_sche.thrid == ndk_thread_identify())
		bWait = FALSE;

	switch (msg) {
	case NDK_MEDIACLI_AVSNK_MSG_PAUSE : case MEDIA_SNK_MSG_PAUSE_WAIT:
		return This->pause(bWait);
	case NDK_MEDIACLI_AVSNK_MSG_RESUME: case MEDIA_SNK_MSG_RESUME_WAIT:
		return This->resume(param, bWait);
	default:
		return FAIL;
	}
}

int RTPMediaClient::mediaCliInit(
	NDKMediaCliHandle *handle,
	const char *source_url,
	NDKMediaCliAVSink *avsink,
	NDKMediaCliRtpMsgHandler *msgHandler,
	int optnum,
	NDKMediaCliOpt *opts)
{
	RTPMediaClient *This = NULL;
	unsigned int x;
	int mcli_cnt;
	RTPMediaGetOpt mopts(optnum, opts);

	if (!avsink || !source_url || !msgHandler)
		ndk_err_return(-1);

	ndk_st_global_init();
	streaming_server_global_init();

	ndk_global_lock(&x);
	mcli_cnt = mediacli_cnt++;
	ndk_global_unlock(&x);

	if (mcli_cnt == 0) {
		if (ndk_st_register_rtp_client(FALSE) != 0)
			return -1;

		if (!mediacli_sche_init(&mediacli_sche)) {
			ndk_error("init");
			goto lFail;
		}
	}

	This = new RTPMediaClient(*mediacli_sche.env,
		source_url,
		mopts.getl(NDK_MEDIACLI_RTP_OPT_DEBUG, 0),
		NDK_MEDIACLI_APPNAME_V1,
		avsink,
		msgHandler,
		optnum,
		opts);

	if (This == NULL) {
		ndk_error("new");
		goto lFail;
	}

	This->fMediaCliBase.priv_data = This;
	This->fMediaCliBase.pf_start = mediaCliStart;
	This->fMediaCliBase.pf_stop = mediaCliStop;
	This->fMediaCliBase.pf_destroy = mediaCliDestroy;
	*handle = (NDKMediaCliHandle)&This->fMediaCliBase;

	ndk_info("Create RTPClient url='%s'", source_url);
	return 0;

lFail:
	ndk_global_lock(&x);
	mcli_cnt = --mediacli_cnt;
	ndk_global_unlock(&x);

	delete This;

	if (mcli_cnt == 0) {
		mediacli_sche_deinit(&mediacli_sche);
		ndk_st_unregister_rtp_client();
	}
	return -1;
}

int RTPMediaClient::mediaCliStart(struct NDKMediaCli *cli, int optnum, NDKMediaCliOpt *opts)
{
	RTPMediaClient *This = (RTPMediaClient *)cli->priv_data;

	if (This->fState != STAT_INITIAL && This->fState != STAT_CLOSED)
		return -1;

	double startTime;
	RTPMediaGetOpt mopts(optnum, opts);
	startTime = mopts.getd(NDK_MEDIACLI_OPT_STARTTIME, 0.0);
	This->start(startTime);

	return 0;
}

int RTPMediaClient::mediaCliStop(struct NDKMediaCli *cli)
{
	RTPMediaClient *This = (RTPMediaClient *)cli->priv_data;

	RTPMediaClient::MsgStru m;
	This->scheduleMsgTask(&RTPMediaClient::msgShutdown, m);
	return 0;
}

void RTPMediaClient::mediaCliDestroy(struct NDKMediaCli *cli)
{
	RTPMediaClient *This = (RTPMediaClient *)cli->priv_data;

	if (!This->fDestroyed) {
		This->fDestroyed = 1;

		RTPMediaClient::MsgStru m;
		This->scheduleMsgTask(&RTPMediaClient::msgShutdown, m);
	}
}

int ndk_mediacli_rtp_init(
	NDKMediaCliHandle *handle,
	const char *source_url,
	NDKMediaCliAVSink *avsink,
	NDKMediaCliRtpMsgHandler *msgHandler,
	int optnum,
	NDKMediaCliOpt *opts)
{
	return RTPMediaClient::mediaCliInit(handle, source_url, avsink, msgHandler, optnum, opts);
}

