/**************************************************************************
 *                                                                        *
 *         Copyright (c) 2012 by iCatch Technology Co., Ltd.             *
 *                                                                        *
 *  This software is copyrighted by and is the property of Sunplus        *
 *  Technology Co., Ltd. All rights are reserved by Sunplus Technology    *
 *  Co., Ltd. This software may only be used in accordance with the       *
 *  corresponding license agreement. Any unauthorized use, duplication,   *
 *  distribution, or disclosure of this software is expressly forbidden.  *
 *                                                                        *
 *  This Copyright notice MUST not be removed or modified without prior   *
 *  written consent of Sunplus Technology Co., Ltd.                       *
 *                                                                        *
 *  Sunplus Technology Co., Ltd. reserves the right to modify this        *
 *  software without notice.                                              *
 *                                                                        *
 *  Sunplus Technology Co., Ltd.                                          *
 *  19, Innovation First Road, Science-Based Industrial Park,             *
 *  Hsin-Chu, Taiwan, R.O.C.                                              *
 *                                                                        *
 **************************************************************************/

#include <ctype.h>
#include <string.h>

#include <livePort/streaming_api.h>
#include <livePort/mediasrc.hh>
#include <livePort/fifo.h>

// Copied from SPCA6350 vlc.h
#ifdef SPCA6350
#define IS_NONINTEGER_FRAMERATE(fps)    ((fps) & 0x80000000)
#define NONINTEGER_NUMERATOR_GET(fps)   ((fps) & 0x0000ffff)
#define NONINTEGER_DENOMINATOR_GET(fps) (((fps) & 0x7fff0000)>>16)
#endif

#define ES2CHAR(es)   (((es) == MEDIA_BUF_ES_AUDIO) ? 'A' : 'V')

///////////////////////////////////////////////////////////////////////////////
struct FrcMaskBits
{
	UINT32 frmrate;
	UINT32 maskbits;
};

static struct FrcMaskBits frcMaskBits[] = {
	/* 3         2         1         0*/
	/* 0987654321098765432109876543210*/
	{6,  /*0b000010000100001000010000100001*/0x02108421},
	{10, /*0b001001001001001001001001001001*/0x09249249},
	{15, /*0b010101010101010101010101010101*/0x15555555},
	{20, /*0b011011011011011011011011011011*/0x1b6db6db},
	{24, /*0b011110111101111011110111101111*/0x1ef7bdef},
};

///////////////////////////////////////////////////////////////////////////////
static UINT32 codec_to_upperase(UINT32 codec)
{
	return toupper(codec & 0xFF) |
		(toupper((codec >> 8)  & 0xFF) << 8) |
		(toupper((codec >> 16) & 0xFF) << 16) |
		(toupper((codec >> 24) & 0xFF) << 24);
}

static int jfif_remove_tail_FFD9(NSFrame *frm)
{
	// Remove FF D9 .. .. at the end of the frame.
	UINT8 *p_tail = (UINT8*)frm->data + (frm->length - 1);
	UINT8 prev_char = 0, cnt = 0;

	while ((p_tail > (UINT8*)frm->data) && (cnt < 16)) {
		++cnt;

		if (*p_tail == 0xFF && prev_char == 0xD9) {
			frm->length -= cnt;
			return cnt;
		}

		prev_char = *(p_tail--);
	}

	return 0;
}

///////////////////////////////////////////////////////////////////////////////
void NSMediaSrcAgent::MediaInf::pushFrame(NSFrame *frm, int timeout)
{
	if (bFirstBuffer && !frm->isParamFrame()) {
		setStartTime(frm->pts);
		bFirstBuffer = false;
		//profLogPrintf(0, "MedSrcRmx: %s %s", tv2str(&startTime), tv2str(&startPTS));
	}

	if (codec == SP5K_MEDIA_VIDEO_MJPG)
		jfif_remove_tail_FFD9(frm);

	if (fifo) {
		if (fifo->safePushFrame(frm, timeout))
			return;

		profLogPrintf(0, "safePushFrame err");
	}

	NSFrame::unref(frm);
}

void NSMediaSrcAgent::MediaInf::setStartTime(struct timeval const &startPTS)
{
	tmrTimeStampGet(&this->startTime);
	this->startPTS = startPTS;
}

///////////////////////////////////////////////////////////////////////////////
// Class NSMediaSrcAgent
NSMediaSrcAgent::NSMediaSrcAgent(bool fDualSource)
: fMediaSrcHandle(NULL)
, fIsDualSrc(fDualSource)
, fMediaSrcStat(MSRC_STAT_UNKNOW)
{
	for (int i = 0; i < MINF_NUM; ++i)
		fMediaInf[i].init(0);
}

NSMediaSrcAgent::~NSMediaSrcAgent()
{
	// Make sure the source is closed.
	NDK_ASSERT(fMediaSrcHandle == NULL);
}

float NSMediaSrcAgent::getSourceFrameRate() const
{
	UINT32 fr;
	getSourceAttribute(SP5K_MEDIA_ATTR_VIDEO_FRAME_RATE, &fr);

#ifdef SPCA6350
	if (IS_NONINTEGER_FRAMERATE(fr)) {
		return ((float)NONINTEGER_NUMERATOR_GET(fr)) / ((float)NONINTEGER_DENOMINATOR_GET(fr));
	}
#endif

	return (float)fr;
}

bool NSMediaSrcAgent::getSourceAttribute(UINT32 attr, UINT32 *val) const
{
	return mediaSrcGetAttribute(fMediaSrcHandle, attr, val) == SUCCESS;
}

void NSMediaSrcAgent::onGotBufferStatic(mediaSrcBuffer_t *buf, unsigned long user_data)
{
	((NSMediaSrcAgent*)user_data)->onGotBuffer(buf);
}

void NSMediaSrcAgent::closeSource()
{
	fMediaSrcStat = MSRC_STAT_CLOSED;
	if (fMediaSrcHandle) {
		mediaSrcClose((mediaSrcHandle_t)fMediaSrcHandle);
		sp5kTimeDelay(SP5K_TIME_DELAY_1MS, 10); // Remove it ?
		fMediaSrcHandle = NULL;
	}
}

void NSMediaSrcAgent::bindFIFO(NSFrmFifo *fifo)
{
	MediaInf &minf = getMediaInf(fifo->isAudio());

	NDK_ASSERT(!minf.fifo);
	minf.fifo = fifo;
}

void NSMediaSrcAgent::unbindFIFOs()
{
	for (int i = 0; i < MINF_NUM; ++i)
		fMediaInf[i].fifo = NULL;
}

UINT32 NSMediaSrcAgent::getCodecType(bool audio) const
{
	return getMediaInf(audio).codec;
}

bool NSMediaSrcAgent::seekSource(double npt, double &cur_npt)
{
	if (!seekable())
		return false;

	struct timeval tv_npt, tv_cur_npt;
	tv_npt.tv_sec =  (long)npt;
	tv_npt.tv_usec = (long)((npt - tv_npt.tv_sec) * 1000000);

	if (doSeek(tv_npt, tv_cur_npt)) {
		cur_npt  = tv_cur_npt.tv_sec;
		cur_npt += (double)tv_cur_npt.tv_usec/1000000.0;
		return true;
	}

	return false;
}

bool NSMediaSrcAgent::doSeek(struct timeval const &tv_npt, struct timeval &tv_cur_npt)
{
	if (fIsDualSrc) {
		bool seeking_ok;
		MediaInf *minf_main = NULL;

		seeking_ok = doMediaSrcSeek(tv_npt);
		if (seeking_ok) {
			MediaInf &mainf = getMediaInf(true);
			MediaInf &mvinf = getMediaInf(false);

			if (mvinf.fifo) {
				mvinf.bFirstBuffer = true;
				minf_main = &mvinf;
			}

			if (mainf.fifo) {
				mainf.bFirstBuffer = true;
				if (!minf_main)
					minf_main = &mainf;
			}

			NDK_ASSERT(minf_main);

			// Wait for the coming of the first frame.
			while (minf_main->bFirstBuffer)
				sp5kTimeDelay(SP5K_TIME_DELAY_1MS, 10);

			tv_cur_npt = minf_main->startPTS;
			printf("MSA: seek ok: %s, %s\n", tv2str(&tv_npt), tv2str(&tv_cur_npt));
			return true;
		}
		else {
			tv_cur_npt.tv_sec = tv_cur_npt.tv_usec = 0;
			printf("MSA: seek fail: %s\n", tv2str(&tv_npt));
			return false;
		}
	}
	else {
		// TODO: not implemented
		return false;
	}
}

///////////////////////////////////////////////////////////////////////////////
NSSingleMediaSrcAgent::NSSingleMediaSrcAgent(UINT32 esType, UINT32 codec, UINT32 sidBits)
: NSMediaSrcAgent(false)
, fESType(esType)
, fFrmCnt(0)
{
#ifdef SPCA6350
	fSIDBits = sidBits & (SP5K_MEDIA_REC_STREAM_1ST | SP5K_MEDIA_REC_STREAM_2ND);
	fCodec   = codec;
#endif

	fFrcFrmRate   = 0;
	fFrcMaskBits  = 0;
	fFrcShiftBits = 0;

	fStartTime   = -1;
	fStartPTS    = -1;
	fPrevTime    = 0;
	fPrevPTS     = 0;
	fDuration    = 0;
	fPTSCorrStep = 0;
	fOrigBitrate = 0;
}

bool NSSingleMediaSrcAgent::frameRateControl()
{
	UINT32 fr = 0;
	BOOL skipped;

	ndk_st_get_attribute(NDK_ST_ATTR_FRAME_RATE, (long*)&fr);
	if (fr == 0 || fr >= 30)
		return true;

	if (fr != fFrcFrmRate) { // Update frmrate mask bits.
		for (unsigned i = 0; i < NDK_ARRAY_SIZE(frcMaskBits); ++i) {
			if (frcMaskBits[i].frmrate == fr) {
				fFrcFrmRate   = fr;
				fFrcMaskBits  = frcMaskBits[i].maskbits;
				fFrcShiftBits = 0;
				goto lNext;
			}
		}

		// Invalid frmrate. Ignore it.
		return true;
	}

lNext:
	skipped = ((0x01 << fFrcShiftBits) & fFrcMaskBits) == 0;

	if (++fFrcShiftBits > 29)
		fFrcShiftBits = 0;

	return !skipped;
}

void NSSingleMediaSrcAgent::calculatePTS(mediaSrcBuffer_t *buf)
{
	SINT64 utime_now;
	int    utime_gap;
	long   mtime_pts_reset = ndk_st_get_attr(NDK_ST_ATTR_PTS_RESET_TIME);

	if (buf->enctime.tv_sec > 0)
		utime_now = timeval2utime(&buf->enctime);
	else
		utime_now = utime_now_get();

	utime_gap = (int)(utime_now - fPrevTime);

	if ( (fStartTime == -1) || (fDuration && mtime_pts_reset && (utime_gap > mtime_pts_reset * 1000)) ) {
		if (fStartTime == -1) {
			fStartPTS = 0;
		}
		else {
			fStartPTS += utime_now - fStartTime;
		}

		fStartTime = utime_now;
		fDuration  = 0;
		printf("LSMSA-%c: start=(%lld %lld) time-gap=%d\n", ES2CHAR(fESType), fStartTime, fStartPTS, utime_gap);
	}

	SINT64 upts_correct, upts_systime;
	upts_systime = (utime_now - fStartTime) + fStartPTS;

	if (fDuration) {
		upts_correct = fPrevPTS + fDuration;

		if (upts_systime > upts_correct)
			upts_correct += fPTSCorrStep;
		else if (upts_systime < upts_correct)
			upts_correct -= fPTSCorrStep;
	}
	else {
		fDuration = buf->duration;
		if (fDuration < 1000) { fDuration *= 1000; } // ms => us.
		fPTSCorrStep = fDuration/200; // maximum 5 ms for every second.
		upts_correct = upts_systime;

		if (fDuration)
			printf("LSMSA-%c: dura=%u step=%u\n", ES2CHAR(fESType), fDuration, fPTSCorrStep);
	}

	fPrevTime = utime_now;
	fPrevPTS  = upts_correct;
	utime2timeval(&buf->pts, upts_correct);
}

void NSSingleMediaSrcAgent::onGotBuffer(mediaSrcBuffer_t *buf)
{
	MediaInf &minf = getMediaInf();
	//profLogPrintf(0, "SMSA-%c: %u %u %s %x", ES2CHAR(fESType), buf->duration, buf->length, tv2str(&buf->pts), buf->bufflags);

	calculatePTS(buf);

	if (fFrmCnt == 0) {
		UINT32 attr_codec = fESType == MEDIA_BUF_ES_AUDIO ? SP5K_MEDIA_ATTR_AUDIO_CODEC : SP5K_MEDIA_ATTR_VIDEO_CODEC;

		if (getSourceAttribute(attr_codec, &minf.codec)) {
			minf.codec = (UINT32)codec_to_upperase(minf.codec);
			printf("SMSA-%c: codec=%x\n", ES2CHAR(fESType), getMediaInf().codec);
		}
		else
			ndk_error("getSourceAttribute %c", ES2CHAR(fESType));
	}

	++fFrmCnt;

	if (minf.codec == SP5K_MEDIA_VIDEO_MJPG && !frameRateControl())
		return;

	NSMediaSrcFrame *frm = new NSMediaSrcFrame();
	if (frm) {
		frm->attachBuffer(buf, NULL);
		minf.pushFrame(frm, 30);
	}
}

void NSSingleMediaSrcAgent::dynamicBitRateSet(float ratio)
{
#ifdef SPCA6350
	if (fOrigBitrate == 0) {
		getSourceAttribute(SP5K_MEDIA_ATTR_VIDEO_AVG_BITRATE, &fOrigBitrate);
		if (fOrigBitrate == 0)
			ndk_err_return();
	}

	if (ratio < 0.2f)
		ratio = 0.2f;
	else if (ratio > 1.0f)
		ratio = 1.0f;

	UINT32 newBitrate = (UINT32)(fOrigBitrate * ratio);

	if (newBitrate > fOrigBitrate)
		newBitrate = fOrigBitrate;

	//ndk_info("dbr %u => %u", fOrigBitrate, newBitrate);
	mediaSrcControl(fMediaSrcHandle, MEDIA_SRC_CTRL_VDBR_SET, newBitrate);
#endif
}

///////////////////////////////////////////////////////////////////////////////
bool NSAudioStreamMediaSrcAgent::openSource()
{
	if (fMediaSrcHandle) {
		NDK_ASSERT(0);
		return false;
	}

	UINT32 flags = 0;

	if (ndk_st_is_audio_on() & NDK_ST_AUDIO_ON_NOSWAP)
		flags |= MEDIA_SRC_NO_AUDIO_SWAP;

#ifdef SPCA6350
	fMediaSrcHandle = mediaSrcOpenAudioStream(onGotBufferStatic, (unsigned long)this, flags);
#else
	fMediaSrcHandle = mediaSrcOpenLiveAudio(onGotBufferStatic, (unsigned long)this, flags);
#endif

	return fMediaSrcHandle != NULL;
}

///////////////////////////////////////////////////////////////////////////////
bool NSVideoStreamMediaSrcAgent::openSource()
{
	if (fMediaSrcHandle) {
		NDK_ASSERT(0);
		return false;
	}

#ifdef SPCA6350
	fMediaSrcHandle = mediaSrcOpenVideoStream(onGotBufferStatic, (unsigned long)this, 0, fSIDBits, fCodec);
#else
	fMediaSrcHandle = mediaSrcOpenRmx(MEDIA_BUF_ES_VIDEO, onGotBufferStatic, (unsigned long)this, 0);
#endif

	return fMediaSrcHandle != NULL;
}

///////////////////////////////////////////////////////////////////////////////
#ifdef ICAT_DUALSTREAM_JPEG
NSDsjMediaSrcAgent::NSDsjMediaSrcAgent()
: NSMediaSrcAgent(false)
{
	fOrigVlcSize = 0;

	getMediaInf().init(SP5K_MEDIA_VIDEO_MJPG);
}

void NSDsjMediaSrcAgent::onRecvData(mediaSrcBuffer_t *buf, mediaRecDualStreamJpegAttr_t const *attr)
{
	NSDsjFrame *dsjfrm = NULL;
	MediaInf  &minf = getMediaInf(false);
	NSFrmFifo *fifo = minf.fifo;

	NDK_ASSERT_EXPR(attr->vlcPos + attr->vlcSize == buf->length, {
		printf("vlcPos = %u, vlcSize = %u, length = %u\n"
			, attr->vlcPos, attr->vlcSize, buf->length);
	});

	if (minf.bFirstBuffer) {
		minf.setStartTime(buf->pts);
		minf.bFirstBuffer = false;
		//profLogPrintf(0, "MedSrcDsj: %s %s", tv2str(&minf.startTime), tv2str(&minf.startPTS));
	}

	if (!ndk_st_sys_protect(15)) {
		profLogPrintf(0, "MedSrcDsj: sys protect err");
		return;
	}

	if (fifo->isQueueFull()) {
		NSFrame *frm = NULL;

		if (NULL != (frm = fifo->popFrame())) {
			if (frm->getRefcnt() > 1) {
				// Hold by other user or have not enough space
				NSFrame::unref(frm);
			}
			else {
				// Reuse
				dsjfrm = (NSDsjFrame*)frm;
			}
		}
	}
	ndk_st_sys_unprotect();

	if (!dsjfrm) {
		dsjfrm = new NSDsjFrame();
		if (!dsjfrm) {
			ndk_error("memory");
			return;
		}
	}

	//println("dsj: pts: %ld.%ld", buf->pts.tv_sec, buf->pts.tv_usec);

	// The buffer object is hold by dsjfrm now.
	dsjfrm->attachBuffer(buf, (void*)attr);
	jfif_remove_tail_FFD9(dsjfrm);

	if (!fifo->safePushFrame(dsjfrm, 30)) {
		NSFrame::unref(dsjfrm);
		profLogPrintf(0, "MedSrcDsj: pushFrame err");
	}
}

void NSDsjMediaSrcAgent::onRecvData_static(mediaSrcBuffer_t *buf
	, mediaRecDualStreamJpegAttr_t const *attr
	, void* user_data)
{
	((NSDsjMediaSrcAgent*)user_data)->onRecvData(buf, attr);
}

bool NSDsjMediaSrcAgent::openSource()
{
	ndk_dualstream_config(onRecvData_static, (void*)this);
	return true;
}

void NSDsjMediaSrcAgent::closeSource()
{
	ndk_dualstream_config(NULL, NULL);
	sp5kTimeDelay(SP5K_TIME_DELAY_1MS, 10);
}

void NSDsjMediaSrcAgent::dynamicBitRateSet(float ratio)
{
	if (fOrigVlcSize == 0) {
		sp5kMediaRecAttrGet(SP5K_MEDIA_ATTR_DUALSTREAM_TARGETVLCSIZE, &fOrigVlcSize);
		if (fOrigVlcSize == 0)
			ndk_err_return();
	}

	if (ratio < 0.2f) ratio = 0.2f;
	else if (ratio > 1.0f) ratio = 1.0f;

	UINT32 newVlcSize = (UINT32)(fOrigVlcSize * ratio);

	if (newVlcSize > fOrigVlcSize)
		newVlcSize = fOrigVlcSize;

	//ndk_info("DSJ vlc-size %u => %u", fOrigVlcSize, newVlcSize);
	mediaRecDualStreamControl(MEDIA_REC_DUALSTREAM_TGTVLCSZ_SET, newVlcSize);
}
#endif

///////////////////////////////////////////////////////////////////////////////
#ifdef ICAT_STREAMING_FILE
NSDmxMediaSrcAgent::NSDmxMediaSrcAgent(const char *filename)
: NSMediaSrcAgent(true)
{
	fFileName  = filename;
}

NSDmxMediaSrcAgent::~NSDmxMediaSrcAgent()
{
}

bool NSDmxMediaSrcAgent::openSource()
{
	if (fMediaSrcHandle) {
		ndk_error("reopen");
		return false;
	}

	UINT32 flags = 0;

	if (ndk_st_is_audio_on() & NDK_ST_AUDIO_ON_NOSWAP)
		flags |= MEDIA_SRC_NO_AUDIO_SWAP;

	fMediaSrcHandle = mediaSrcOpenFile(fFileName, onGotBufferStatic, (unsigned long)this, flags);
	if (!fMediaSrcHandle)
		return false;

	UINT32 v_codec, a_codec;

	if (!getSourceAttribute(SP5K_MEDIA_ATTR_VIDEO_CODEC, &v_codec) ||
	    !getSourceAttribute(SP5K_MEDIA_ATTR_AUDIO_CODEC, &a_codec))
	{
		mediaSrcClose(fMediaSrcHandle);
		fMediaSrcHandle = NULL;
		ndk_err_return(false);
	}

	fMediaInf[MINF_AUDIO].codec = (UINT32)codec_to_upperase(a_codec);
	fMediaInf[MINF_VIDEO].codec = (UINT32)codec_to_upperase(v_codec);
	ndk_info("a_codec=%x, v_codec=%x", fMediaInf[MINF_AUDIO].codec, fMediaInf[MINF_VIDEO].codec);

	return true;
}

void NSDmxMediaSrcAgent::closeSource()
{
	NSMediaSrcAgent::closeSource();
}

void NSDmxMediaSrcAgent::pauseSource()
{
	fMediaSrcStat = MSRC_STAT_PAUSED;
}

void NSDmxMediaSrcAgent::resumeSource()
{
	mediaSrcControl((mediaSrcHandle_t)fMediaSrcHandle, MEDIA_SRC_CTRL_RESUME);
	fMediaSrcStat = MSRC_STAT_RESUMED;
}

bool NSDmxMediaSrcAgent::endOfSource() const
{
	return mediaSrcEndOfSource((mediaSrcHandle_t)fMediaSrcHandle) ? true : false;
}

bool NSDmxMediaSrcAgent::doMediaSrcSeek(struct timeval const &tv_npt)
{
	UINT32 ret;
	fMediaSrcStat = MSRC_STAT_SEEKING;

	mediaSrcControl((mediaSrcHandle_t)fMediaSrcHandle, MEDIA_SRC_CTRL_PAUSE);
	ret = mediaSrcControl((mediaSrcHandle_t)fMediaSrcHandle, MEDIA_SRC_CTRL_SEEK, tv_npt.tv_sec * 1000 + tv_npt.tv_usec / 1000);
	if (ret == SUCCESS) {
		if (getMediaInf(true).fifo) getMediaInf(true).fifo->safeReclaimAllFrames(true);
		if (getMediaInf(false).fifo) getMediaInf(false).fifo->safeReclaimAllFrames(true);
	}

	mediaSrcControl((mediaSrcHandle_t)fMediaSrcHandle, MEDIA_SRC_CTRL_RESUME);
	fMediaSrcStat = MSRC_STAT_RESUMED;
	return ret == SUCCESS;
}

float NSDmxMediaSrcAgent::getSourceDuration() const
{
	UINT32 d; // milli-seconds
	if (!getSourceAttribute(SP5K_MEDIA_ATTR_DURATION, &d))
		return 0.0;

	return (float)d/1000;
}

bool NSDmxMediaSrcAgent::getSourceAttribute(UINT32 attr, UINT32 *val) const
{
	return mediaSrcGetAttribute((mediaSrcHandle_t)fMediaSrcHandle, attr, val) == SUCCESS;
}

void NSDmxMediaSrcAgent::onGotBuffer(mediaSrcBuffer_t *buf)
{
	MediaInf *minf = NULL;

	if (MEDIA_BUF_ES_TYPE(buf->bufflags) == MEDIA_BUF_ES_AUDIO) {
		minf = &getMediaInf(true);
		//profLogPrintf(0, "dmxmsa: a %d.%d", buf->pts.tv_sec, buf->pts.tv_usec);
	}
	else if (MEDIA_BUF_ES_TYPE(buf->bufflags) == MEDIA_BUF_ES_VIDEO) {
		minf = &getMediaInf(false);
		//profLogPrintf(0, "dmxmsa: v %d.%d", buf->pts.tv_sec, buf->pts.tv_usec);
	}
	else
		NDK_ASSERT(0);

	// Th buf will be freed by media-source if mediaSrcHoldBufferObject is not called.
	if (!minf->fifo)
		return;

	for (;;) {
		if (fMediaSrcStat == MSRC_STAT_SEEKING || fMediaSrcStat == MSRC_STAT_CLOSED) {
			return;
		}

		if (minf->fifo->getQueueSpace() > 8)
			break;

		ndk_msleep(20);
	}

	NSFrame *frm = new NSMediaSrcFrame();
	if (frm) {
		frm->attachBuffer(buf, NULL); // Transfer the ownership to the NSMediaSrcFrame object.
		minf->pushFrame(frm, -1);
	}
}
#endif // ICAT_STREAM_LIVEAUDIO_DEMUX

/////////////////////////////////////////////////////////////////////////////////
NSCustomMediaSrcAgent::NSCustomMediaSrcAgent(NDKStMediaSrcType type, const char *filename)
: NSMediaSrcAgent(true)
, fCustSrcType(type)
{
	strncpy(fFileName, filename, sizeof(fFileName) - 1);
	fFileName[sizeof(fFileName) - 1] = 0;
	fInPushing = 0;
}

NSCustomMediaSrcAgent::~NSCustomMediaSrcAgent()
{
}

bool NSCustomMediaSrcAgent::openSource()
{
	if (fMediaSrcHandle) {
		ndk_error("reopen");
		return false;
	}

	if (!ndk_st_has_custom_mediasrc(fCustSrcType))
		return false;

	ndk_st_get_custom_mediasrc(fCustSrcType, &fCustSrcCb);
	fMediaSrcHandle = fCustSrcCb.open(pushBuffer0, this, (void*)fFileName);
	if (!fMediaSrcHandle)
		ndk_err_return(false);

	UINT32 v_codec, a_codec;

	if (!getSourceAttribute(SP5K_MEDIA_ATTR_VIDEO_CODEC, &v_codec) ||
	    !getSourceAttribute(SP5K_MEDIA_ATTR_AUDIO_CODEC, &a_codec))
	{
		goto lFail;
	}

	fMediaInf[MINF_AUDIO].codec = (UINT32)codec_to_upperase(a_codec);
	fMediaInf[MINF_VIDEO].codec = (UINT32)codec_to_upperase(v_codec);
	ndk_info("a_codec=%x, v_codec=%x", fMediaInf[MINF_AUDIO].codec, fMediaInf[MINF_VIDEO].codec);

	return true;

lFail:
	if (fMediaSrcHandle) {
		fCustSrcCb.close(fMediaSrcHandle);
		fMediaSrcHandle = NULL;
	}

	freeResources();
	return false;
}

void NSCustomMediaSrcAgent::closeSource()
{
	fMediaSrcStat = MSRC_STAT_CLOSED;

	if (fMediaSrcHandle) {
		fCustSrcCb.close(fMediaSrcHandle);
		fMediaSrcHandle = NULL;
	}

	freeResources();
}

void NSCustomMediaSrcAgent::freeResources()
{
}

bool NSCustomMediaSrcAgent::doMediaSrcSeek(struct timeval const &tv_npt)
{
	UINT32 pos = tv_npt.tv_sec*1000 + tv_npt.tv_usec/1000;
	fMediaSrcStat = MSRC_STAT_SEEKING;

	if (fCustSrcCb.pause(fMediaSrcHandle) != SUCCESS)
		ndk_err_return(false);

	if (fCustSrcCb.seek_to(fMediaSrcHandle, pos) != SUCCESS)
		ndk_err_return(false);

	if (getMediaInf(true).fifo) getMediaInf(true).fifo->safeReclaimAllFrames(true);
	if (getMediaInf(false).fifo) getMediaInf(false).fifo->safeReclaimAllFrames(true);

	if (fCustSrcCb.resume(fMediaSrcHandle) != SUCCESS)
		ndk_err_return(false);

	fMediaSrcStat = MSRC_STAT_RESUMED;
	return true;
}

void NSCustomMediaSrcAgent::pauseSource()
{
	fMediaSrcStat = MSRC_STAT_PAUSED;
}

void NSCustomMediaSrcAgent::resumeSource()
{
	fMediaSrcStat = MSRC_STAT_RESUMED;
}

bool NSCustomMediaSrcAgent::endOfSource() const
{
	return fCustSrcCb.end_of_source(fMediaSrcHandle) ? true : false;
}

float NSCustomMediaSrcAgent::getSourceDuration() const
{
	UINT32 d; // milli-seconds

	if (!getSourceAttribute(SP5K_MEDIA_ATTR_DURATION, &d)) {
		NDK_ASSERT(0);
		return 0.0f;
	}

	return d/1000.0f;
}

bool NSCustomMediaSrcAgent::getSourceAttribute(UINT32 attr, UINT32 *val) const
{
	return fCustSrcCb.get_attribute(fMediaSrcHandle, attr, val) == SUCCESS ? true : false;
}

void NSCustomMediaSrcAgent::pushBuffer(NDKStMediaBuffer *buf, bool bAudio)
{
	MediaInf &minf = getMediaInf(bAudio);
	int  maxDepth = bAudio ? 32 : 16;
	bool gotPushing = false;
	NSCustomMediaFrame *frm;
	unsigned int x;

	//profLogPrintf(0, "custms: %c %d.%d", bAudio?'A':'C', buf->pts.tv_sec, buf->pts.tv_usec);

	if (!minf.fifo) {
		fCustSrcCb.free_buffer_object(buf->bufobj);
		return;
	}

lAgain:
	if (fMediaSrcStat == MSRC_STAT_SEEKING || fMediaSrcStat == MSRC_STAT_CLOSED) {
		fCustSrcCb.free_buffer_object(buf->bufobj);
		if (gotPushing) {
			fInPushing = 0;
		}
		return;
	}

	if (!gotPushing) {
		ndk_global_lock(&x);
		if (!fInPushing) {
			fInPushing = 1;
			gotPushing = true;
		}
		ndk_global_unlock(&x);
	}

	if (!gotPushing || minf.fifo->getQueueDepth() >= maxDepth) {
		ndk_msleep(20);
		goto lAgain;
	}

	frm = new NSCustomMediaFrame(fCustSrcCb.free_buffer_object);

	if (frm) {
		frm->attachBuffer(buf, NULL); // Transfer the ownership to the NSCustomMediaFrame object.
		minf.pushFrame(frm, -1);
	}
	else {
		fCustSrcCb.free_buffer_object(buf->bufobj);
	}

	fInPushing = 0;
}

void NSCustomMediaSrcAgent::pushBuffer0(NDKStMediaBuffer *buf, void *arg)
{
	((NSCustomMediaSrcAgent*)arg)->pushBuffer(buf, buf->f_buftype == NDK_ST_MEDIABUFFER_AUDIO);
}

