/**************************************************************************
 *
 *       Copyright (c) 2011 by iCatch Technology, Inc.
 *
 *  This software is copyrighted by and is the property of iCatch Technology,
 *  Inc.. All rights are reserved by iCatch Technology, Inc..
 *  This software may only be used in accordance with the corresponding
 *  license agreement. Any unauthorized use, duplication, distribution,
 *  or disclosure of this software is expressly forbidden.
 *
 *  This Copyright notice MUST not be removed or modified without prior
 *  written consent of iCatch Technology, Inc..
 *
 *  iCatch Technology, Inc. reserves the right to modify this software
 *  without notice.
 *
 *  iCatch Technology, Inc.
 *  19-1, Innovation First Road, Science-Based Industrial Park,
 *  Hsin-Chu, Taiwan, R.O.C.
 *
 **************************************************************************/
#ifndef __LIVEST_MEDIASRC_H__
#define __LIVEST_MEDIASRC_H__

#include <livePort/streaming_def.h>
#include <livePort/livest_internal.h>
#include <livePort/taskqueue.h>

class NSFrmFifo;
class NSFrame;
class NSMediaSrcFrame;
class NSCustomMediaFrame;

///////////////////////////////////////////////////////////////////////////////
class NSMediaSrcAgent
{
public:
	struct MediaInf {
		NSFrmFifo *fifo;
		UINT32 codec;
		bool   bFirstBuffer;
		struct timeval startTime;
		struct timeval startPTS;

		void init(UINT32 c) { fifo = NULL; codec = c; bFirstBuffer = true; }
		void setStartTime(struct timeval const &startPTS);
		void pushFrame(NSFrame *frm, int timeout);
	};

public:
	virtual ~NSMediaSrcAgent();

	// Can open more than one time.
	virtual bool  openSource() = 0;
	virtual void  closeSource();

	// true: file streaming; false: live streaming.
	virtual bool  seekable() const { return false; }
	virtual void  pauseSource() { ; }
	virtual void  resumeSource() { ; }
	virtual bool  endOfSource() const { return false; } // End of source
	virtual bool  isDualStreamJPEG() const { return false; }

	virtual float getSourceDuration() const { return 0.0; }
	virtual float getSourceFrameRate() const;
	virtual bool  getSourceAttribute(UINT32 attr, UINT32 *val) const;

	virtual void  bindFIFO(NSFrmFifo *fifo);
	virtual void  unbindFIFOs();

	virtual UINT32 getCodecType(bool audio) const;
	virtual void  dynamicBitRateSet(float ratio/* 0.1f ~ 1.0f */) {}

	bool seekSource(double npt, double &cur_npt);

protected:
	enum { MINF_AUDIO = 0, MINF_VIDEO = 1, MINF_NUM = 2 }; // For Dual-Source

	void     *fMediaSrcHandle;
	MediaInf fMediaInf[MINF_NUM]; /* Single Source: only 0; Dual-Source, 0 for audio, 1 for video.*/
	bool     fIsDualSrc; /* true: has audio and video bitstream. false: only audio or video bitstream. */
	enum { MSRC_STAT_UNKNOW = 0, MSRC_STAT_PAUSED, MSRC_STAT_RESUMED, MSRC_STAT_SEEKING, MSRC_STAT_CLOSED } fMediaSrcStat;

protected:
	NSMediaSrcAgent(bool fDualSource);

	MediaInf &getMediaInf(bool aud) { return fIsDualSrc ? (aud ? fMediaInf[MINF_AUDIO] : fMediaInf[MINF_VIDEO]) : fMediaInf[0]; }
	MediaInf const &getMediaInf(bool aud) const { return fIsDualSrc ? (aud ? fMediaInf[MINF_AUDIO] : fMediaInf[MINF_VIDEO]) : fMediaInf[0]; }
	MediaInf &getMediaInf() { return fMediaInf[0]; }

	bool doSeek(struct timeval const &npt, struct timeval &cur_npt);
	virtual bool doMediaSrcSeek(struct timeval const &npt) { return false; }
	virtual void onGotBuffer(mediaSrcBuffer_t *buf) {}
	static  void onGotBufferStatic(mediaSrcBuffer_t *buf, unsigned long user_data);
};

///////////////////////////////////////////////////////////////////////////////
// Single Live MediaSrcAgent based on mediaSrcXXX APIs.
class NSSingleMediaSrcAgent : public NSMediaSrcAgent
{
public:
	virtual void dynamicBitRateSet(float ratio/* 0.1f ~ 1.0f */);

protected:
	UINT32 fESType;
	UINT32 fFrmCnt;
#ifdef SPCA6350
	UINT32 fSIDBits;
	UINT32 fCodec;
#endif

protected:
	NSSingleMediaSrcAgent(UINT32 esType, UINT32 codec, UINT32 sidBits /*stream id. Not used on V33 */);
	bool frameRateControl();
	void calculatePTS(mediaSrcBuffer_t *buf);
	virtual void onGotBuffer(mediaSrcBuffer_t *buf);

private:
	// Frame rate control
	UINT32 fFrcFrmRate;   // 0: no frame-rate control
	UINT32 fFrcShiftBits;
	UINT32 fFrcMaskBits;

	// PTS calculation
	SINT64 fStartTime;
	SINT64 fStartPTS;
	SINT64 fPrevTime;
	SINT64 fPrevPTS;
	UINT32 fDuration;
	UINT32 fPTSCorrStep;
	UINT32 fOrigBitrate;
};

///////////////////////////////////////////////////////////////////////////////
class NSAudioStreamMediaSrcAgent : public NSSingleMediaSrcAgent
{
public:
	NSAudioStreamMediaSrcAgent() : NSSingleMediaSrcAgent(MEDIA_BUF_ES_AUDIO, 0, 0) {}
	virtual bool openSource();
};

///////////////////////////////////////////////////////////////////////////////
class NSVideoStreamMediaSrcAgent : public NSSingleMediaSrcAgent
{
public:
	NSVideoStreamMediaSrcAgent(UINT32 codec, UINT32 sidBits=0xFF/*Not used on V33*/) : NSSingleMediaSrcAgent(MEDIA_BUF_ES_VIDEO, codec, sidBits) {}
	virtual bool openSource();
};

///////////////////////////////////////////////////////////////////////////////
// Single Source: DualStreamJpeg
#ifdef ICAT_DUALSTREAM_JPEG
class NSDsjMediaSrcAgent : public NSMediaSrcAgent
{
public:
	NSDsjMediaSrcAgent();
	virtual bool openSource();
	virtual void closeSource();
	virtual bool isDualStreamJPEG() const { return true; }
	virtual void dynamicBitRateSet(float ratio/* 0.1f ~ 1.0f */);

private:
	UINT32 fOrigVlcSize;

	void onRecvData(mediaSrcBuffer_t *buf, mediaRecDualStreamJpegAttr_t const *attr);
	static void onRecvData_static(mediaSrcBuffer_t *buf
		, mediaRecDualStreamJpegAttr_t const *attr
		, void* user_data);
};
#endif

///////////////////////////////////////////////////////////////////////////////
// Dual Source (Audio & Video): Demux
#ifdef ICAT_STREAMING_FILE
class NSDmxMediaSrcAgent : public NSMediaSrcAgent
{
public:
	NSDmxMediaSrcAgent(const char *filename);
	virtual ~NSDmxMediaSrcAgent();

	virtual bool  openSource();
	virtual void  closeSource();
	virtual bool  seekable() const { return true; }
	virtual void  pauseSource();
	virtual void  resumeSource();
	virtual bool  endOfSource() const;

	virtual float getSourceDuration() const;
	virtual bool  getSourceAttribute(UINT32 attr, UINT32 *val) const;

private:
	const char* fFileName;

	virtual bool doMediaSrcSeek(struct timeval const &npt);
	virtual void onGotBuffer(mediaSrcBuffer_t *buf);
};
#endif // ICAT_STREAM_LIVEAUDIO_DEMUX

///////////////////////////////////////////////////////////////////////////////
// Dual Source (Audio & Video)
class NSCustomMediaSrcAgent : public NSMediaSrcAgent
{
public:
	NSCustomMediaSrcAgent(NDKStMediaSrcType type, const char *filename);
	virtual ~NSCustomMediaSrcAgent();

	virtual bool  openSource();
	virtual void  closeSource();
	virtual bool  seekable() const { return true; }
	virtual void  pauseSource();
	virtual void  resumeSource();
	virtual bool  endOfSource() const;

	virtual float getSourceDuration() const;
	virtual bool  getSourceAttribute(UINT32 attr, UINT32 *val) const;

private:
	char fFileName[256];
	NDKStMediaSrcType fCustSrcType;
	NDKStMediaSrcCallback fCustSrcCb;
	UINT32 fSeekPosition;
	UINT32 fInPushing : 1;

	void freeResources();
	virtual bool doMediaSrcSeek(struct timeval const &npt);

	void pushBuffer(NDKStMediaBuffer *buf, bool bAudio);
	static void pushBuffer0(NDKStMediaBuffer *buf, void *arg);
};

#endif
