#include <stdio.h>
#include <stdlib.h>

#include <livePort/nal_parser.h>

static UINT32 read_bits(struct NALBufferReader *buf, int len);
static UINT32 read_exp_golomb(struct NALBufferReader *buf);
static SINT32 read_exp_golomb_s(struct NALBufferReader *buf);
static void skip_scaling_list(struct NALBufferReader *buf, int size);

static inline UINT32 read_bits (struct NALBufferReader *buf, int len)
{
	UINT32 i_mask[33] = {
		0x00,
		0x01,      0x03,      0x07,      0x0f,
		0x1f,      0x3f,      0x7f,      0xff,
		0x1ff,     0x3ff,     0x7ff,     0xfff,
		0x1fff,    0x3fff,    0x7fff,    0xffff,
		0x1ffff,   0x3ffff,   0x7ffff,   0xfffff,
		0x1fffff,  0x3fffff,  0x7fffff,  0xffffff,
		0x1ffffff, 0x3ffffff, 0x7ffffff, 0xfffffff,
		0x1fffffff, 0x3fffffff, 0x7fffffff, 0xffffffff
	};

	int i_shr;
	UINT32 bits = 0;

	while (len > 0 && (buf->cur_pos - buf->buf) < buf->len) {
		if ((i_shr = buf->cur_offset - len) >= 0) {
			bits |= (*buf->cur_pos >> i_shr)&i_mask[len];
			buf->cur_offset -= len;
			if (buf->cur_offset == 0) {
				buf->cur_pos++;
				buf->cur_offset = 8;
			}
			return bits;
		}
		else {
			bits |= (*buf->cur_pos & i_mask[buf->cur_offset]) << -i_shr;
			len -= buf->cur_offset;
			buf->cur_pos++;
			buf->cur_offset = 8;
		}
	}
	return bits;
}

UINT32 read_exp_golomb(struct NALBufferReader *buf)
{
	int leading_zero_bits = 0;

	while (read_bits(buf, 1) == 0 && leading_zero_bits < 32) {
		leading_zero_bits++;
	}

	UINT32 code = (1 << leading_zero_bits) - 1 + read_bits(buf, leading_zero_bits);
	return code;
}

SINT32 read_exp_golomb_s(struct NALBufferReader *buf)
{
	UINT32 ue = read_exp_golomb(buf);
	SINT32 code = ue & 0x01 ? (ue + 1) / 2 : -(ue / 2);
	return code;
}

void skip_scaling_list(struct NALBufferReader *buf, int size)
{
	int i;
	for (i = 0; i < size; i++) {
		read_exp_golomb_s(buf);
	}
}

int ndk_parse_sps(struct NALBufferReader *buf, struct SeqParameterSetRbsp *sps)
{
	sps->profile_idc = buf->buf[0];
	sps->constraint_setN_flag = (buf->buf[1] >> 4) & 0x0f;
	sps->level_idc = buf->buf[2];

	buf->cur_pos = buf->buf + 3;

	sps->seq_parameter_set_id = read_exp_golomb(buf);
	if (sps->profile_idc == 100 || sps->profile_idc == 110 ||
		sps->profile_idc == 122 || sps->profile_idc == 144) {
		sps->chroma_format_idc = read_exp_golomb(buf);
		if (sps->chroma_format_idc == 3) {
			sps->residual_colour_transform_flag = read_bits(buf, 1);
		}

		sps->bit_depth_luma_minus8 = read_exp_golomb(buf);
		sps->bit_depth_chroma_minus8 = read_exp_golomb(buf);
		sps->qpprime_y_zero_transform_bypass_flag = read_bits(buf, 1);
		sps->seq_scaling_matrix_present_flag = read_bits(buf, 1);
		if (sps->seq_scaling_matrix_present_flag) {
			sps->seq_scaling_lists_present_flag = read_bits(buf, 8);
			int i;
			for (i = 0; i < 8; i++) {
				if ((sps->seq_scaling_lists_present_flag >> (7 - i)) & 0x01) {
					// NOTE: just skip the scaling lists, as we do not
					// need their data for parsing
					if (i < 6) {
						skip_scaling_list(buf, 16);
					}
					else {
						skip_scaling_list(buf, 64);
					}
				}
			}
		}
	}

	sps->log2_max_frame_num_minus4 = read_exp_golomb(buf);

	sps->pic_order_cnt_type = read_exp_golomb(buf);
	if (!sps->pic_order_cnt_type) {
		sps->log2_max_pic_order_cnt_lsb_minus4 = read_exp_golomb(buf);
	}
	else {
		sps->delta_pic_order_always_zero_flag = read_bits(buf, 1);
		sps->offset_for_non_ref_pic = read_exp_golomb_s(buf);
		sps->offset_for_top_to_bottom_field = read_exp_golomb_s(buf);
		sps->num_ref_frames_in_pic_order_cnt_cycle = read_exp_golomb(buf);
		int i;
		for (i = 0; i < sps->num_ref_frames_in_pic_order_cnt_cycle; i++) {
			sps->offset_for_ref_frame[i] = read_exp_golomb_s(buf);
		}
	}
	sps->num_ref_frames = read_exp_golomb(buf);
	sps->gaps_in_frame_num_value_allowed_flag = read_bits(buf, 1);

	sps->pic_width = 16 * (read_exp_golomb(buf) + 1);
	sps->pic_height = 16 * (read_exp_golomb(buf) + 1);

	sps->frame_mbs_only_flag = read_bits(buf, 1);

	/* compute the height correctly even for interlaced material */
	sps->pic_height = (2 - sps->frame_mbs_only_flag) * sps->pic_height;

	if (!sps->frame_mbs_only_flag) {
		sps->mb_adaptive_frame_field_flag = read_bits(buf, 1);
	}

	sps->direct_8x8_inference_flag = read_bits(buf, 1);
	sps->frame_cropping_flag = read_bits(buf, 1);
	if (sps->frame_cropping_flag) {
		sps->frame_crop_left_offset = read_exp_golomb(buf);
		sps->frame_crop_right_offset = read_exp_golomb(buf);
		sps->frame_crop_top_offset = read_exp_golomb(buf);
		sps->frame_crop_bottom_offset = read_exp_golomb(buf);
	}
	sps->vui_parameters_present_flag = read_bits(buf, 1);

	return 0;
}

int ndk_parse_pps(struct NALBufferReader *buf, struct PicParameterSetRbsp *pps)
{
	pps->pic_parameter_set_id = read_exp_golomb(buf);
	pps->seq_parameter_set_id = read_exp_golomb(buf);
	pps->entropy_coding_mode_flag = read_bits(buf, 1);
	pps->pic_order_present_flag = read_bits(buf, 1);
	return 0;
}

