/**************************************************************************
 *                                                                        *
 *         Copyright (c) 2012 by iCatch Technology Co., Ltd.             *
 *                                                                        *
 *  This software is copyrighted by and is the property of Sunplus        *
 *  Technology Co., Ltd. All rights are reserved by Sunplus Technology    *
 *  Co., Ltd. This software may only be used in accordance with the       *
 *  corresponding license agreement. Any unauthorized use, duplication,   *
 *  distribution, or disclosure of this software is expressly forbidden.  *
 *                                                                        *
 *  This Copyright notice MUST not be removed or modified without prior   *
 *  written consent of Sunplus Technology Co., Ltd.                       *
 *                                                                        *
 *  Sunplus Technology Co., Ltd. reserves the right to modify this        *
 *  software without notice.                                              *
 *                                                                        *
 *  Sunplus Technology Co., Ltd.                                          *
 *  19, Innovation First Road, Science-Based Industrial Park,             *
 *  Hsin-Chu, Taiwan, R.O.C.                                              *
 *                                                                        *
 **************************************************************************/
#include <new>

#include <UsageEnvironment.hh>
#include <RTSPServer.hh>

#include <ndk_global_api.h>
#include <livePort/livest_internal.h>
#include <livePort/streaming_api.h>
#include <livePort/fifo.h>
#include <livePort/mediasrc.hh>

extern "C" {
#include <sp5k_modesw_api.h>
}

///////////////////////////////////////////////////////////////////////////////
static char   g_st_root_dir[64];
static UINT32 g_st_mem_alloced = 0;
static UINT32 g_st_mem_freed = 0;
static NDKStExitFunc *g_st_exitfunc_list = NULL;

#ifdef LIVEST_SUPPORT_TUTK
NDK_SOC_OPS_IMPLEMENT(livest);

static BOOL g_st_rtp_server_registered = FALSE;
static BOOL g_st_rtp_client_registered = FALSE;
static BOOL g_st_use_tutk = FALSE;
#endif

///////////////////////////////////////////////////////////////////////////////
extern "C" {

int closeSocket(int s)
{
	//ndk_info("close socket %d", s);
	return CLOSESOCKET(s);
}

///////////////////////////////////////////////////////////////////////////////
// Streaming API implementions
void streaming_server_thread(void *arg);

static NDKStState g_stctx_state = NDK_ST_STAT_INVALID;
NDKStCtx g_stctx;

#define STATTR(attr)   g_stctx.attrs[attr]

#ifdef ICAT_DUALSTREAM_JPEG
#define BITRATE2TGTVLCSZ(br, fr)   ((br)/(fr)/8)

struct DualStreamJpegState {
	mediaRecDualStreamClient_t client;
	NDKDualStreamCbFunc cbfunc;
	UINT32 frmrate;
	// Dynamic BitRate Control
	UINT32 dbrc_vlcsz_orig;
	UINT32 dbrc_vlcsz_cur;
	struct timeval dbrc_recent_time;
	void *udata;
};

static BOOL g_dualstream_started = FALSE;
static struct DualStreamJpegState g_dualstream_stat;
#endif

int ndk_st_get_conn_num()
{
	unsigned int x;
	int i, cnt = 0;

	ndk_global_lock(&x);
	for (i = 0; i < NDK_ST_FIFO_NUM; ++i) {
		if (g_stctx.capfifos[i].fifo)
			cnt += g_stctx.capfifos[i].refcnt;
	}
	ndk_global_unlock(&x);

	return cnt;
}

BOOL _ndk_st_is_streaming_started()
{
	return g_stctx_state == NDK_ST_STAT_STARTED;
}

BOOL ndk_st_defsoc_is_used()
{
#ifdef LIVEST_SUPPORT_TUTK
	return NDK_SOC_OPS_GET_ID(livest) == NDK_SOC_ID_DEFAULT;
#else
	return TRUE;
#endif
}

void ndk_st_register_exitfunc(NDKStExitFunc *exit_func)
{
	NDKStExitFunc **pp;
	ndk_st_sys_protect(-1);
	for (pp = &g_st_exitfunc_list; *pp; pp = &(*pp)->next) {
		if (*pp == exit_func) {
			ndk_st_sys_unprotect();
			NDK_ASSERT(0);
			return;
		}
	}
	*pp = exit_func;
	exit_func->next = NULL;
	ndk_st_sys_unprotect();
}

BOOL ndk_st_send_event(UINT32 event, UINT32 data)
{
	BOOL r = FALSE;
	if (g_stctx.cfg.evt_handler) {
		r = g_stctx.cfg.evt_handler(event, data);
	}
	return r;
}

BOOL ndk_st_in_mediasrv_thread()
{
	if (g_stctx_state == NDK_ST_STAT_INVALID)
		return FALSE;

	return ndk_thread_identify() == g_stctx.thrid ? TRUE : FALSE;
}

static int ndk_st_get_max_rtp_payload_size()
{
	int tcp_mss;

#ifdef LIVEST_SUPPORT_TUTK
	tcp_mss = ndk_soc_get_tcp_mss(NDK_SOC_OPS_GET_ID(livest));
#else
	tcp_mss = TCP_MSS;
#endif

	return tcp_mss - 12 /* RTP header size */ - RTP_EXTEN_HEADER_SIZE;
}

static void fifo_unbind_source(NDKStCtx *stctx, NDKFrmFifoType fft)
{
	NSFrmFifo *fifo = stctx->capfifos[fft].fifo;

	if (fifo) {
		NDK_ASSERT_EXPR(stctx->capfifos[fft].refcnt > 0,
			printf("fft=%d refcnt=%d\n", fft, stctx->capfifos[fft].refcnt);
		);
		if (--stctx->capfifos[fft].refcnt == 0) {
			fifo->activate(false);
			delete fifo->unbindMediaSrc();
		}
	}
}

// Stage one binding.
/*
 svn revision:18709:
 NSFrmFifoW[activatekGV33]basefw DualStream-JPEG]Abstage 1q}audioA
 ڦ쪺Audio dataqu@bkAҥHnbstage 2q~}AudioCOJPEGwg
 ͤ@frameAo|yAV UnsyncCQActivatekAbStage 2FIFO~}lAV frame.
 */
static int fifo_bind_source_1(
	NDKStCtx       *stctx,
	NDKFrmFifoType fft,
	NSFrmFifo      **ppFifo)
{
	NDK_ASSERT((UINT32)fft < NDK_ST_FIFO_NUM);

	NSFrmFifo *fifo = stctx->capfifos[fft].fifo;
	if (!fifo)
		ndk_err_return(-1);

	*ppFifo  = fifo;

	if (fifo->getMediaSrc()) {
		NDK_ASSERT(stctx->capfifos[fft].refcnt > 0);
		++stctx->capfifos[fft].refcnt;
		return 1;
	}

	// H264
	if (fft == NDK_ST_FIFO_H264S0) {
		UINT32 sidBits = 0xFF;

#ifdef ICAT_STREAMING_H264H264
		if (stctx->cfg.st_mode == NDK_ST_MULTI_STREAM)
			sidBits = SP5K_MEDIA_REC_STREAM_1ST;
#endif

		NSVideoStreamMediaSrcAgent *src = new NSVideoStreamMediaSrcAgent(MEDIA_BUF_CODEC_H264, sidBits);
		if (!src || !src->openSource() || !NSFrmFifo::bindMediaSrc(fifo, src)) {
			delete src;
			ndk_err_return(-1);
		}
	}
#ifdef ICAT_STREAMING_H264H264
	else if (fft == NDK_ST_FIFO_H264S1) {
		NSVideoStreamMediaSrcAgent *src = new NSVideoStreamMediaSrcAgent(MEDIA_BUF_CODEC_H264, SP5K_MEDIA_REC_STREAM_2ND);
		if (!src || !src->openSource() || !NSFrmFifo::bindMediaSrc(fifo, src)) {
			delete src;
			ndk_err_return(-1);
		}
	}
#endif
	else if (fft == NDK_ST_FIFO_JPEG) {
		NSMediaSrcAgent *src;

#ifdef ICAT_DUALSTREAM_JPEG
		if (ndk_st_is_dualstream_jpeg())
			src = new NSDsjMediaSrcAgent();
		else
#endif
		{
			src = new NSVideoStreamMediaSrcAgent(MEDIA_VIDEO_MJPG);
		}

		if (!src || !src->openSource() || !NSFrmFifo::bindMediaSrc(fifo, src)) {
			delete src;
			ndk_err_return(-1);
		}
	}
	else if (fft == NDK_ST_FIFO_AUDIO) {
#ifdef ICAT_DUALSTREAM_JPEG
		if (ndk_st_is_dualstream_jpeg()) {
			// Postpone to stage two.
			fifo = NULL;
		}
		else
#endif
		{
			NSAudioStreamMediaSrcAgent *src = new NSAudioStreamMediaSrcAgent();
			if (!src || !src->openSource() || !NSFrmFifo::bindMediaSrc(fifo, src)) {
				delete src;
				ndk_err_return(-1);
			}
		}
	}
	else
		ndk_err_return(-1);

	if (fifo) {
		++stctx->capfifos[fft].refcnt;
#ifndef ICAT_DUALSTREAM_JPEG
		fifo->activate(true);
#endif
		return 1;
	}

	return 0;
}

// Stage two binding.
static int fifo_bind_source_2(NDKStCtx *stctx, NDKFrmFifoType fft)
{
	int ret = 0;
#if defined(ICAT_DUALSTREAM_JPEG)
	NSFrmFifo *fifo = stctx->capfifos[fft].fifo;

	if (stctx->cfg.audio_on) {
		if (ndk_st_is_dualstream_jpeg()) {
			NSMediaSrcAgent *src = new NSAudioStreamMediaSrcAgent();

			if (!src || !src->openSource() || !NSFrmFifo::bindMediaSrc(fifo, src)) {
				delete src;
				ndk_err_return(-1);
			}

			++stctx->capfifos[fft].refcnt;
			++ret;
		}
	}
	// Note, for V33 H264, this may cause error.
	fifo->activate(true);
#endif

	return ret;
}

// streamName: H264 H264S0 H264S1 MJPG AUD
bool ndk_st_capfifo_ref(const char *streamName,
	NDKFrmFifoType vfft, NSFrmFifo **ppvff,
	NDKFrmFifoType afft, NSFrmFifo **ppaff)
{
	NDK_ASSERT(g_stctx_state != NDK_ST_STAT_INVALID);

	NDKStCtx *stctx = &g_stctx;
	bool reqSent = false;
	int vbound = 0, abound = 0, bound;

	*ppaff = NULL;
	*ppvff = NULL;

	if (vfft != NDK_ST_FIFO_INVALID) {
		bound = fifo_bind_source_1(stctx, vfft, ppvff);
		if (bound < 0) {
			ndk_error("bind vfifo 1");
			goto lError;
		}
		vbound += bound;
	}

	if (afft != NDK_ST_FIFO_INVALID) {
		bound = fifo_bind_source_1(stctx, afft, ppaff);
		if (bound < 0) {
			ndk_error("bind afifo 1");
			goto lError;
		}
		abound += bound;
	}

	if ( (vfft != NDK_ST_FIFO_INVALID && stctx->capfifos[vfft].refcnt == 1) ||
	     (afft != NDK_ST_FIFO_INVALID && stctx->capfifos[afft].refcnt == 1) )
	{
		// Notify the host program to start basefw
		if (!ndk_st_send_event(NDK_ST_EVT_ON_STARTED, (UINT32)streamName)) {
			ndk_error("request");
			goto lError;
		}
		reqSent = true;
	}

	if (vfft != NDK_ST_FIFO_INVALID) {
		bound = fifo_bind_source_2(stctx, vfft);
		if (bound < 0) {
			ndk_error("bind vfifo 2");
			goto lError;
		}
		vbound += bound;
	}

	if (afft != NDK_ST_FIFO_INVALID) {
		bound = fifo_bind_source_2(stctx, afft);
		if (bound < 0) {
			ndk_error("bind afifo 2");
			goto lError;
		}
		abound += bound;
	}

	if (!ndk_st_is_started()) {
		stctx->abort_streaming = 0;
		tmrTimeStampGet(&stctx->tv_start);
		g_stctx_state = NDK_ST_STAT_STARTED;
	}

	return true;

lError:
	if (reqSent)
		ndk_st_send_event(NDK_ST_EVT_ON_STOPPED, (UINT32)streamName);

	if (vbound)
		fifo_unbind_source(stctx, vfft);

	if (abound)
		fifo_unbind_source(stctx, afft);

	return false;
}

void ndk_st_capfifo_unref(
	const char *streamName,
	NDKFrmFifoType afft,
	NDKFrmFifoType vfft)
{
	NDKStCtx *stctx = &g_stctx;

	NDK_ASSERT(g_stctx_state != NDK_ST_STAT_INVALID);

	ndk_info("cap-fifo unref a:%d v:%d", afft, vfft);
	if ((vfft != NDK_ST_FIFO_INVALID && stctx->capfifos[vfft].refcnt == 1) ||
	    (afft != NDK_ST_FIFO_INVALID && stctx->capfifos[afft].refcnt == 1)  )
	{
		ndk_st_send_event(NDK_ST_EVT_ON_STOPPED, (UINT32)streamName);
	}

	if (vfft != NDK_ST_FIFO_INVALID) {
		fifo_unbind_source(stctx, vfft);
	}

	if (afft != NDK_ST_FIFO_INVALID) {
		fifo_unbind_source(stctx, afft);
	}

	if (ndk_st_get_conn_num() == 0 && ndk_st_is_started()) {
		g_stctx_state = NDK_ST_STAT_STOPPED;
	}
}

#ifdef ICAT_DUALSTREAM_JPEG
static void dualstream_handle_data(mediaSrcBuffer_t *buf
	, mediaRecDualStreamJpegAttr_t const *jpegAttr
	, UINT32 udata)
{
	NDKDualStreamCbFunc ds_cbfunc = g_dualstream_stat.cbfunc;
	void *ds_udata = g_dualstream_stat.udata;

	if (ndk_st_is_started() && ndk_st_is_dualstream_jpeg() && ds_cbfunc) {
		ds_cbfunc(buf, jpegAttr, ds_udata);
	}
}

void ndk_dualstream_config(NDKDualStreamCbFunc cbfunc, void *user_data)
{
	if (cbfunc) {
		g_dualstream_stat.udata = user_data;
		g_dualstream_stat.cbfunc = cbfunc;
	}
	else {
		g_dualstream_stat.cbfunc = NULL;
	}
}

void ndk_dualstream_set_framerate(UINT32 frmrate)
{
	if (g_dualstream_started)
		mediaRecDualStreamSetFrameRate(&g_dualstream_stat.client, frmrate);
}

void ndk_st_dualstream_setup(UINT32 width, UINT32 height
	, UINT32 qInitial, UINT32 qMax /*0: disable brc*/
	, UINT32 frmrate, UINT32 bitrate)
{
	UINT32 bufsz;

#if 0
	UINT32 q = qMax ? qMax : qInitial;
	// Compression ratio assumption: Q=100, 2.0; Q=50, 4.0; min. 6.0.
	// Function: y = 6 - x*0.04
	bufsz  = (UINT32)((float)(width*height) / (6.0f - (float)q*2/50));
	bufsz *= 4;
	bufsz  = NDK_ALIGN_TO(bufsz, 256);
#else
	bufsz = NDK_ALIGN_TO(width*height, 256);
#endif

	sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_DUALSTREAM_WIDTH, width);
	sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_DUALSTREAM_HEIGHT, height);
	sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_DUALSTREAM_Q_FACTOR, qInitial);
	sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_DUALSTREAM_MAXQ_FACTOR, qMax);
	sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_DUALSTREAM_TARGETVLCSIZE, BITRATE2TGTVLCSZ(bitrate, frmrate));
	sp5kMediaRecCfgSet (SP5K_MEDIA_REC_DUALSTREAM_BUF_SIZE, bufsz);
	sp5kMediaRecCfgSet (SP5K_MEDIA_REC_DUALSTREAM_BUF_CNT, 16 + 2);
}

int _ndk_st_dualstream_start(UINT32 width, UINT32 height
	, UINT32 qInitial, UINT32 qMax /*0: disable brc*/
	, UINT32 frmrate, UINT32 bitrate)
{
	UINT32 mode;

	if (g_dualstream_started)
		ndk_err_return(-1);

	sp5kModeGet(&mode);
	if (mode != SP5K_MODE_VIDEO_PREVIEW && mode != SP5K_MODE_VIDEO_RECORD)
		ndk_err_return(-1);

	if (frmrate > 30) frmrate = 30;
	if (qMax > 99) qMax = 99;

	if (qInitial > 99) qInitial = 99;
	else if (qInitial < 10) qInitial = 10;

	if (qMax && (qInitial > qMax)) qInitial = qMax;

	if (width > 0 && height > 0)
		ndk_st_dualstream_setup(width, height, qInitial, qMax, frmrate, bitrate);

	sp5kMediaRecCfgSet(MEDIA_REC_SOUT_RDWR_SIZE_MAX, 64*1024);

	memset(&g_dualstream_stat.client, 0, sizeof(g_dualstream_stat.client));
	g_dualstream_stat.client.cbfunc = dualstream_handle_data;
	g_dualstream_stat.frmrate = frmrate;
	sp5kMediaRecAttrGet(SP5K_MEDIA_ATTR_DUALSTREAM_TARGETVLCSIZE, &g_dualstream_stat.dbrc_vlcsz_orig);
	g_dualstream_stat.dbrc_vlcsz_cur = g_dualstream_stat.dbrc_vlcsz_orig;

	if (mediaRecDualStreamOpen(&g_dualstream_stat.client) != SUCCESS)
		ndk_err_return(-1);

	mediaRecDualStreamSetFrameRate(&g_dualstream_stat.client, frmrate);
	g_dualstream_started = TRUE;

	ndk_info("dualstream start: w=%u h=%u br=%u qmax=%u", width, height, bitrate, qMax);
	return 0;
}

int _ndk_st_dualstream_control(NDKDualStreamCtrlCode code, long param)
{
	if (!g_dualstream_started)
		return -1;

	switch (code) {
	case NDK_DUALSTREAM_PAUSE:
		mediaRecDualStreamControl(MEDIA_REC_DUALSTREAM_PAUSE);
		break;

	case NDK_DUALSTREAM_RESUME:
		mediaRecDualStreamControl(MEDIA_REC_DUALSTREAM_RESUME);
		break;

	case NDK_DUALSTREAM_MAXQ_SET:
		if (param < 0) param = 1;
		else if (param > 99) param = 99;

		if (mediaRecDualStreamControl(MEDIA_REC_DUALSTREAM_QVAL_SET, (UINT32)param, 0) != SUCCESS)
			return -1;
		break;

	case NDK_DUALSTREAM_MINQ_SET:
		if (param < 0) param = 1;
		else if (param > 99) param = 99;

		if (mediaRecDualStreamControl(MEDIA_REC_DUALSTREAM_QVAL_SET, 0, (UINT32)param) != SUCCESS)
			return -1;
		break;

	case NDK_DUALSTREAM_BITRATE_SET:
		if (param < 0) return -1;

		param = BITRATE2TGTVLCSZ(param, g_dualstream_stat.frmrate);
		mediaRecDualStreamControl(MEDIA_REC_DUALSTREAM_TGTVLCSZ_SET, (UINT32)param);
		break;

	default:
		return -1;
	}

	return 0;
}

int _ndk_st_dualstream_stop()
{
	if (g_dualstream_started) {
		sp5kMediaRecCfgSet(MEDIA_REC_SOUT_RDWR_SIZE_MAX, 0);
		mediaRecDualStreamClose(&g_dualstream_stat.client);
		g_dualstream_started = FALSE;
	}

	return 0;
}
#endif

int _ndk_st_register_mediasrc(NDKStMediaSrcType type, NDKStMediaSrcCallback *cb)
{
	if (g_stctx_state == NDK_ST_STAT_INVALID)
		return -1;

	NDK_ASSERT(type >= 0 && type < NDK_ST_MEDIASRC_NUM);

	ndk_st_sys_protect(-1);
	memcpy(&g_stctx.custom_mediasrc_cb[type], cb, sizeof(NDKStMediaSrcCallback));
	ndk_st_sys_unprotect();

	return 0;
}

BOOL ndk_st_has_custom_mediasrc(NDKStMediaSrcType type)
{
	NDK_ASSERT(type >= 0 && type < NDK_ST_MEDIASRC_NUM);

	return g_stctx.custom_mediasrc_cb[type].open != NULL;
}

void ndk_st_get_custom_mediasrc(NDKStMediaSrcType type, NDKStMediaSrcCallback *cb)
{
	NDK_ASSERT(type >= 0 && type < NDK_ST_MEDIASRC_NUM);

	memcpy(cb, &g_stctx.custom_mediasrc_cb[type], sizeof(NDKStMediaSrcCallback));
}

static void st_destroy_fifos(NDKStCtx *stctx)
{
	for (int i = 0; i < NDK_ST_FIFO_NUM; ++i) {
		if (stctx->capfifos[i].fifo) {
			NDK_ASSERT(stctx->capfifos[i].refcnt == 0);
			delete stctx->capfifos[i].fifo;
			stctx->capfifos[i].fifo = NULL;
		}
	}
}

static void st_on_got_frame(NSFrmFifo *fifo, NSFrame *frm)
{
	g_stctx.rtspServ->envir().taskScheduler().wakeup();
}

static bool st_create_fifos(NDKStCtx *stctx)
{
	int i;
	NDKStMode mode = stctx->cfg.st_mode;

	if (mode == NDK_ST_DUAL_H264JPEG || mode == NDK_ST_MONO_H264 || mode == NDK_ST_MULTI_STREAM) {
		stctx->capfifos[NDK_ST_FIFO_H264S0].fifo = new NSH264FrmFifo(true, false);
		if (!stctx->capfifos[NDK_ST_FIFO_H264S0].fifo)
			goto lFail;
	}

#ifdef ICAT_STREAMING_H264H264
	if (mode == NDK_ST_MULTI_STREAM) {
		stctx->capfifos[NDK_ST_FIFO_H264S1].fifo = new NSH264FrmFifo(true, false);
		if (!stctx->capfifos[NDK_ST_FIFO_H264S1].fifo)
			goto lFail;
	}
#endif

	if (mode == NDK_ST_DUAL_H264JPEG || mode == NDK_ST_MONO_JPEG || mode == NDK_ST_MULTI_STREAM || mode == NDK_ST_DUAL_STREAMING) {
		stctx->capfifos[NDK_ST_FIFO_JPEG].fifo = new NSJpegFrmFifo(true, false);
		if (!stctx->capfifos[NDK_ST_FIFO_JPEG].fifo)
			goto lFail;
	}

	if (stctx->cfg.audio_on) {
		stctx->capfifos[NDK_ST_FIFO_AUDIO].fifo = new NSAudioFrmFifo(true, false);
		if (!stctx->capfifos[NDK_ST_FIFO_AUDIO].fifo)
			goto lFail;
	}

	for (i = 0; i < NDK_ST_FIFO_NUM; ++i) {
		if (stctx->capfifos[i].fifo)
			stctx->capfifos[i].fifo->onGotFrame = st_on_got_frame;
	}

	return true;

lFail:
	st_destroy_fifos(stctx);
	return false;
}

int _ndk_st_start_server(NDKStCfg *cfg)
{
	NDKStCtx *stctx = &g_stctx;
	void *thrid = NULL;

#ifndef ICAT_DUALSTREAM_JPEG
	NDK_ASSERT(cfg->st_mode != NDK_ST_DUAL_STREAMING);
#endif
#ifndef ICAT_STREAMING_H264H264
	NDK_ASSERT(cfg->st_mode != NDK_ST_MULTI_STREAM);
#endif

	if (g_stctx_state != NDK_ST_STAT_INVALID)
		ndk_err_return(-1);

	ndk_st_global_init();
	streaming_server_global_init();
	if (ndk_st_register_rtp_server(cfg->flags & NDK_ST_FLG_TUTK ? TRUE : FALSE) != 0)
		return -1;

	strncpy(g_st_root_dir, cfg->root_dir, sizeof(g_st_root_dir) - 1);
	g_st_root_dir[sizeof(g_st_root_dir) - 1] = 0;

	memset(stctx, 0, sizeof(*stctx));
	stctx->cfg = *cfg;
	stctx->cfg.root_dir = g_st_root_dir;

	printf("start server: root=%s, port=%d, mode=%d, audio=%d, ka=%d, tutk=%d\n",
		stctx->cfg.root_dir, stctx->cfg.port, stctx->cfg.st_mode,
		stctx->cfg.audio_on, stctx->cfg.keepalive_secs,
		(stctx->cfg.flags & NDK_ST_FLG_TUTK) ? 1 : 0);

	if (sp5kOsMutexCreate(&stctx->mutex, (CHAR*)"streaming", 0) != 0)
		goto lFail;

	if (!st_create_fifos(stctx)) {
		ndk_error("fifo");
		goto lFail;
	}

	thrid = ndk_thread_create("mediasrv", ndk_thread_get_priority(), streaming_server_thread, stctx, 16 * 1024 - 256);
	if (!thrid) {
		ndk_error("thread");
		goto lFail;
	}

	// Default attribute values.
	STATTR(NDK_ST_ATTR_TIME_TO_DROP_FRAME) = 250;
	STATTR(NDK_ST_ATTR_NET_LOADING_HIGH) = 75;
	STATTR(NDK_ST_ATTR_TIME_CLR_TXBUFQ) = 0;
	STATTR(NDK_ST_ATTR_VDBRC_MIN) = 50;
	STATTR(NDK_ST_ATTR_VDBRC_RESTORE_TIME) = 1000;
	STATTR(NDK_ST_ATTR_AUDIO_FIFO_SIZE) = 24;
	STATTR(NDK_ST_ATTR_VIDEO_FIFO_SIZE) = 32;
	STATTR(NDK_ST_ATTR_PTS_RESET_TIME) = 250;
	STATTR(NDK_ST_ATTR_STREAMING_TYPE) = NDK_ST_TYPE_DEFAULT;
	STATTR(NDK_ST_EXT_ATTR_RANDOM_LOADING) = 0;
	STATTR(NDK_ST_EXT_ATTR_PSEUDO_LOADING) = 0;
	STATTR(NDK_ST_ATTR_RTP_PAYLOAD_SIZE) = ndk_st_get_max_rtp_payload_size();

	ndk_cmd_dbgswitch_add("dropfrm", &STATTR(NDK_ST_ATTR_TIME_TO_DROP_FRAME));
	ndk_cmd_dbgswitch_add("vdbrmin", &STATTR(NDK_ST_ATTR_VDBRC_MIN));
	ndk_cmd_dbgswitch_add("vdbrest", &STATTR(NDK_ST_ATTR_VDBRC_RESTORE_TIME));
	ndk_cmd_dbgswitch_add("afifosz", &STATTR(NDK_ST_ATTR_AUDIO_FIFO_SIZE));
	ndk_cmd_dbgswitch_add("vfifosz", &STATTR(NDK_ST_ATTR_VIDEO_FIFO_SIZE));
	ndk_cmd_dbgswitch_add("pseload", &STATTR(NDK_ST_EXT_ATTR_PSEUDO_LOADING));

	g_stctx_state = NDK_ST_STAT_STOPPED;

	return 0;

lFail:
	st_destroy_fifos(stctx);
	sp5kOsMutexDelete(&stctx->mutex);
	ndk_st_unregister_rtp_server();
	return -1;
}

void _ndk_st_stop_server()
{
	NDKStCtx *stctx = &g_stctx;

	if (g_stctx_state != NDK_ST_STAT_INVALID) {
		ndk_st_close_streaming();

		stctx->abort_server = 1;
		stctx->rtspServ->envir().taskScheduler().wakeup();
		while (g_stctx_state != NDK_ST_STAT_INVALID)
			sp5kTimeDelay(SP5K_TIME_DELAY_1MS, 10);
	}
}

BOOL _ndk_st_is_server_started()
{
	return g_stctx_state != NDK_ST_STAT_INVALID;
}

int _ndk_st_set_attribute(NDKStAttr attr, long value)
{
	if (g_stctx_state == NDK_ST_STAT_INVALID)
		ndk_err_return(-1);

	NDK_ASSERT((UINT32)attr < NDK_ST_EXT_ATTR_MAX);

	if (value < 0)
		value = 0;

	switch (attr) {
	default:
		break;

	case NDK_ST_ATTR_TIME_TO_RESEND_FRAME:
		if (value > 0 && value < 30) value = 30;
		break;

	case NDK_ST_ATTR_TIME_TO_DROP_FRAME:
		if (value > 0 && value < 100) value = 100;
		break;

	case NDK_ST_ATTR_NET_LOADING_HIGH:
		if (value > 0 && value > 100) value = 100;
		break;

	case NDK_ST_ATTR_FRAME_RATE:
		switch (value) {
		case 6: case 10: case 15: case 20: case 24: case 30: break;
		default: return -1;
		}
		break;

	case NDK_ST_ATTR_VDBRC_MIN:
		if (value < 0) value = 0;
		else if (value > 100) value = 100;
		break;

	case NDK_ST_ATTR_VDBRC_RESTORE_TIME:
		if (value < 0) value = 0;
		break;

	case NDK_ST_ATTR_AUDIO_FIFO_SIZE:
	case NDK_ST_ATTR_VIDEO_FIFO_SIZE:
		if (value < 0) return -1;
		break;

	case NDK_ST_ATTR_STREAMING_TYPE:
		if (value <= NDK_ST_TYPE_INVALID || value >= NDK_ST_TYPE_NUM)
			ndk_err_return(-1);
		break;

	case NDK_ST_ATTR_RTP_PAYLOAD_SIZE:
		if (value <= 0 || value > ndk_st_get_max_rtp_payload_size())
			value = ndk_st_get_max_rtp_payload_size();
		break;

	case NDK_ST_EXT_ATTR_RANDOM_LOADING:
		srand(sp5kOsTimeGet());
		break;

	case NDK_ST_EXT_ATTR_PSEUDO_LOADING:
		if (value < 0) value = 0;
		else if (value > 100) value = 100;
		break;
	}

	STATTR(attr) = value;
	return 0;
}

long ndk_st_get_attr(NDKStAttr attr)
{
	return STATTR(attr);
}

int _ndk_st_get_attribute(NDKStAttr attr, long *value)
{
	if (g_stctx_state == NDK_ST_STAT_INVALID)
		return -1;

	NDK_ASSERT(attr >= 0 && attr < (NDKStAttr)NDK_ST_EXT_ATTR_MAX);
	*value = ndk_st_get_attr(attr);
	return 0;
}

void ndk_st_stop_server_done(NDKStCtx *stctx)
{
	ndk_info("done, conn nr=%d", ndk_st_get_conn_num());

	st_destroy_fifos(stctx);
	sp5kOsMutexDelete(&stctx->mutex);
	memset(&g_stctx, 0, sizeof(g_stctx));
	g_stctx_state = NDK_ST_STAT_INVALID;
	ndk_st_unregister_rtp_server();
}

void _ndk_st_close_streaming()
{
	NDKStCtx *stctx = &g_stctx;

	NDK_ASSERT(g_stctx_state != NDK_ST_STAT_INVALID);
	NDK_ASSERT(stctx->thrid);
	// Don't run this function in the streaming event handler.
	NDK_ASSERT(stctx->thrid != ndk_thread_identify());

	TaskScheduler &sche = stctx->rtspServ->envir().taskScheduler();

	sche.lockScheduler();

	// Notify the streaming server to close all sessions. The ndk_st_close_capfifo functions
	// is called when SMSS class is deleted. Streaming is closed if all SMSSs are deleted.
	if (ServerMediaSession::getServerMediaSessionCount() > 0) {
		int n = 100;

		sche.signal(TaskScheduler::SIGNAL_ABORT_STREAMING);
		sche.unlockScheduler();
		sche.wakeup();

		while (ServerMediaSession::getServerMediaSessionCount() > 0 && --n > 0) {
			sp5kTimeDelay(SP5K_TIME_DELAY_1MS, 50);
		}

		sche.lockScheduler();
		stctx->abort_streaming = 0;
	}

	sche.unlockScheduler();
	ndk_info("streaming closed");
}

NDKStMode _ndk_st_get_mode()
{
	if (g_stctx_state == NDK_ST_STAT_INVALID)
		ndk_warning("unititialized");

	return g_stctx.cfg.st_mode;
}

BOOL _ndk_st_is_dualstream_jpeg()
{
#ifdef ICAT_DUALSTREAM_JPEG
	return g_stctx.cfg.st_mode == NDK_ST_DUAL_H264JPEG || g_stctx.cfg.st_mode == NDK_ST_DUAL_STREAMING ? TRUE : FALSE;
#else
	return FALSE;
#endif
}

UINT32 _ndk_st_is_audio_on()
{
	if (g_stctx_state == NDK_ST_STAT_INVALID)
		ndk_warning("unititialized");

	return g_stctx.cfg.audio_on;
}

int ndk_st_parse_streaming_type(const char *name)
{
	UINT32 st_type = NDK_ST_TYPE_INVALID;
	char rootname[32], *p = rootname, *p_end = &rootname[sizeof(rootname) - 1];

	for (p = rootname; *name && *name != '/' && *name != '?' && p < p_end; ++name, ++p)
		*p = *name;
	*p = 0;

	if (!strcasecmp("MJPG", rootname)) {
		st_type = NDK_ST_TYPE_JPEG;
	}
	else if (!strcasecmp("H264", rootname) || !strcasecmp("H264S0", rootname)) {
		st_type = NDK_ST_TYPE_H264S0;
	}
	else if (!strcasecmp("H264S1", rootname)) {
		st_type = NDK_ST_TYPE_H264S1;
	}
	else if (!strcasecmp("AUD", rootname)) {
		st_type = NDK_ST_TYPE_AUDIO;
	}
	else {
		st_type = NDK_ST_TYPE_FILE;
	}

	return st_type;
}

int ndk_st_get_netload(const char* ifname, int *loading_ret)
{
	long loading = 0, loading_high = STATTR(NDK_ST_ATTR_NET_LOADING_HIGH);
	int  level;

	if (STATTR(NDK_ST_EXT_ATTR_PSEUDO_LOADING) > 0) {
		loading = STATTR(NDK_ST_EXT_ATTR_PSEUDO_LOADING);
	}
	else if (STATTR(NDK_ST_EXT_ATTR_RANDOM_LOADING) > 0) {
		loading = rand() % STATTR(NDK_ST_EXT_ATTR_RANDOM_LOADING);
	}
	else if (loading_high == 0) {
		if (loading_ret)
			*loading_ret = 0;
		return NDK_ST_NETLOAD_LOW;
	}
	else {
		//ndk_netif_ioctl(NDK_IOCG_IF_BUF_LOADING, (long)ifname, &loading);
		ndk_netif_ioctl(NDK_IOCG_IF_LOADING, (long)ifname, &loading);
	}

	if (loading < NDK_ST_NETLOAD_LOW_THRESHOLD)
		level = NDK_ST_NETLOAD_LOW;
	else if (loading < loading_high)
		level = NDK_ST_NETLOAD_MIDDLE;
	else
		level = NDK_ST_NETLOAD_HIGH;

	if (loading_ret)
		*loading_ret = loading;

	return level;
}

///////////////////////////////////////////////////////////////////////////////
// RTP Server and Client manangement
static int livest_register(BOOL bServer, BOOL bTUTK)
{
#ifdef LIVEST_SUPPORT_TUTK
	BOOL bFirst = FALSE;

	if (g_st_rtp_server_registered || g_st_rtp_client_registered) {
		if (g_st_use_tutk != bTUTK)
			ndk_err_return(-1);
	}

	if (!g_st_rtp_server_registered && !g_st_rtp_client_registered) {
		bFirst = TRUE;
	}

	if (bServer) {
		NDK_ASSERT(!g_st_rtp_server_registered);
		g_st_rtp_server_registered = TRUE;
		g_st_use_tutk = bTUTK;
	}
	else {
		NDK_ASSERT(!g_st_rtp_client_registered);
		g_st_rtp_client_registered = TRUE;
		g_st_use_tutk = bTUTK;
	}

	if (bFirst) {
		if (NDK_SOC_OPS_INITIALIZE(livest, bTUTK ? NDK_SOC_ID_TUTK : NDK_SOC_ID_DEFAULT) != 0)
			ndk_err_return(-1);
	}
#endif

	return 0;
}

static void livest_unregister(BOOL bServer)
{
#ifdef LIVEST_SUPPORT_TUTK
	if (bServer) {
		NDK_ASSERT(g_st_rtp_server_registered);
		g_st_rtp_server_registered = FALSE;
	}
	else {
		NDK_ASSERT(g_st_rtp_client_registered);
		g_st_rtp_client_registered = FALSE;
	}

	if (!g_st_rtp_server_registered && !g_st_rtp_client_registered) {
		NDKStExitFunc *p;
		for (p = g_st_exitfunc_list; p; p = p->next) {
			p->on_exit(p);
		}

		g_st_exitfunc_list = NULL;
	}
#endif
}

int ndk_st_register_rtp_server(BOOL bTUTK)
{
	return livest_register(TRUE, bTUTK);
}

int ndk_st_register_rtp_client(BOOL bTUTK)
{
	return livest_register(FALSE, bTUTK);
}

void ndk_st_unregister_rtp_server()
{
	livest_unregister(TRUE);
}

void ndk_st_unregister_rtp_client()
{
	livest_unregister(FALSE);
}

///////////////////////////////////////////////////////////////////////////////
// Monitor & dump-info
enum {
	MON_VAR_FRM_MEM,
	MON_VAR_CPU_IDLE,
	MON_VAR_IF_LOAD,
	MON_VAR_JPG_QVAL,
};

#ifdef TCPIP_LWIP
extern UINT32 lwip_sta_mem_alloced;
extern UINT32 lwip_sta_mem_freed;
extern UINT32 lwip_tcpip_msg_api_cnt;
extern UINT32 lwip_tcpip_msg_inpkt_cnt;
#endif

extern UINT32 stStaScheduledTaskCnt;

static int    mon_rtp_conn_num = 0;

static void _init_monitor();

static int st_mon_getval(ULONG var_id, char *value, int width, unsigned long user_data)
{
	const char *fmtstr;
	switch (width) {
	case 2: fmtstr = "%2u"; break;
	case 3: fmtstr = "%3u"; break;
	case 4: fmtstr = "%4u"; break;
	case 6: fmtstr = "%6u"; break;
	case 9: fmtstr = "%9u"; break;
	default: fmtstr = "%7u"; break;
	}

	int val;

	switch (var_id) {
	case MON_VAR_CPU_IDLE: {
		int n = ndk_st_get_conn_num();
		if (mon_rtp_conn_num != n) {
			_init_monitor();
			mon_rtp_conn_num = n;
			return NDK_MON_RES_BREAK;
		}

		sprintf(value, fmtstr, ndk_cpuidle_get());
		break; }

	case MON_VAR_FRM_MEM:
		sprintf(value, "%4u", NSFrame::staFrmAlloced - NSFrame::staFrmFreed);
		break;

	case MON_VAR_IF_LOAD: {
		int loading;
		ndk_st_get_netload("wlan0", &loading);
		sprintf(value, "%2d", loading);
		break; }

#ifdef ICAT_DUALSTREAM_JPEG
	case MON_VAR_JPG_QVAL: {
		UINT32 maxQ, minQ, avgQ = 0;
		mediaRecDualStreamControl(MEDIA_REC_DUALSTREAM_QVAL_GET, &maxQ, &minQ, &avgQ);
		sprintf(value, fmtstr, avgQ);
		break; }
#endif

	default: {
		val = *((UINT32*)var_id);
		*((UINT32*)var_id) = 0;
		sprintf(value, fmtstr, val);
		break; }
	}

	return NDK_MON_RES_OK;
}

static void _init_monitor()
{
	ndk_mon_reset();

	ndk_mon_add_var("cpu",  MON_VAR_CPU_IDLE, 4, st_mon_getval, 0);
	//ndk_mon_add_var("FRM/U", MON_VAR_FRM_MEM, 9, st_mon_getval);
	//ndk_mon_add_var("sche", (ULONG)&stStaScheduledTaskCnt, 4, st_mon_getval, 0);
	ndk_mon_add_var("ld", MON_VAR_IF_LOAD, 2, st_mon_getval, 0);
#ifdef ICAT_DUALSTREAM_JPEG
	ndk_mon_add_var("q", MON_VAR_JPG_QVAL, 3, st_mon_getval, 0);
#endif

	NSFrmFifo::monAddVars();
}

void ndk_st_start_monitor()
{
	int c;

	printf ("<<< Press Q/q to exit >>>\n");
	ndk_cpuidle_begin(31);

	mon_rtp_conn_num = ndk_st_get_conn_num();
	_init_monitor();

	ndk_mon_start(1000);
	do {
		c = getch();
	} while (c != 'q' && c != 'Q');

	ndk_mon_stop();
	ndk_cpuidle_end();
}

#define ST_ATTR_DESCS                      \
ST_ATTR_DES_DEF(TIME_TO_RESEND_FRAME,"\t") \
ST_ATTR_DES_DEF(TIME_TO_DROP_FRAME,"\t")   \
ST_ATTR_DES_DEF(NET_LOADING_HIGH,"\t")     \
ST_ATTR_DES_DEF(TIME_CLR_TXBUFQ,"\t")      \
ST_ATTR_DES_DEF(FRAME_RATE,"\t\t")         \
ST_ATTR_DES_DEF(VDBRC_MIN,"\t\t")          \
ST_ATTR_DES_DEF(VDBRC_RESTORE_TIME,"\t")   \
ST_ATTR_DES_DEF(AUDIO_FIFO_SIZE,"\t")      \
ST_ATTR_DES_DEF(VIDEO_FIFO_SIZE,"\t")      \
ST_ATTR_DES_DEF(PTS_RESET_TIME,"\t")       \
ST_ATTR_DES_DEF(STREAMING_TYPE,"\t")       \
ST_ATTR_DES_DEF(RTP_PAYLOAD_SIZE,"\t")

void ndk_st_dump_info()
{
	const char *mode_str[] = { "DHJ", "DJ", "MH", "MJ", "DHHJ" };
	printf("[Streaming Info]\n");
	printf("  Mode = %s, State = %d, Audio = %d, RTP Conn Num=%d\n",
		mode_str[g_stctx.cfg.st_mode],
		g_stctx_state,
		g_stctx.cfg.audio_on,
		ndk_st_get_conn_num());

	printf("<Attrs>\n");
#define ST_ATTR_DES_DEF(a, b)  printf("  " #a "%s= %ld\n", b, g_stctx.attrs[NDK_ST_ATTR_##a]);
	ST_ATTR_DESCS;
#undef  ST_ATTR_DES_DEF

	printf("<Memory>\n");
	printf("  Memory: Alloced=%u, Freed=%u, Unfreed=%u\n",
		g_st_mem_alloced, g_st_mem_freed, g_st_mem_alloced - g_st_mem_freed);
	printf("  Frames: Alloced=%d, Freed=%d, Unfreed=%d\n",
		NSFrame::staFrmAlloced, NSFrame::staFrmFreed,
		NSFrame::staFrmAlloced - NSFrame::staFrmFreed);

	if (g_stctx_state == NDK_ST_STAT_INVALID)
		return;

	printf("[FrmFIFO]\n");
	printf("  refcnt: H264=%d/%d, JPEG = %d, PCM = %d\n",
		g_stctx.capfifos[NDK_ST_FIFO_H264S0].refcnt,
#ifdef ICAT_STREAMING_H264H264
		g_stctx.capfifos[NDK_ST_FIFO_H264S1].refcnt,
#else
		0,
#endif
		g_stctx.capfifos[NDK_ST_FIFO_JPEG].refcnt,
		g_stctx.capfifos[NDK_ST_FIFO_AUDIO].refcnt);

	NSFrmFifo::dumpFifoInfo();
	g_stctx.rtspServ->envir().taskScheduler().printSchedulerInfo(False);
	g_stctx.rtspServ->dumpClientSessions();
}

} // extern "C"

void* operator new(std::size_t size) throw (std::bad_alloc)
{
	void* p = ndk_mem_malloc(size);
	if (!p)
		NDK_ASSERT(0);

	++g_st_mem_alloced;
	return p;
}

void* operator new [] (std::size_t size) throw (std::bad_alloc)
{
	void* p = ndk_mem_malloc(size);
	if (!p)
		NDK_ASSERT(0);

	++g_st_mem_alloced;
	return p;
}

void operator delete(void *p) throw ()
{
	if (p) {
		++g_st_mem_freed;
		ndk_mem_free(p);
	}
}

void operator delete[] (void* p) throw ()
{
	if (p) {
		++g_st_mem_freed;
		ndk_mem_free(p);
	}
}

extern "C" void __cxa_pure_virtual()
{
	NDK_ASSERT(0);
}

