/**************************************************************************
 *
 *       Copyright (c) 2014 by iCatch Technology, Inc.
 *
 *  This software is copyrighted by and is the property of iCatch Technology,
 *  Inc.. All rights are reserved by iCatch Technology, Inc..
 *  This software may only be used in accordance with the corresponding
 *  license agreement. Any unauthorized use, duplication, distribution,
 *  or disclosure of this software is expressly forbidden.
 *
 *  This Copyright notice MUST not be removed or modified without prior
 *  written consent of iCatch Technology, Inc..
 *
 *  iCatch Technology, Inc. reserves the right to modify this software
 *  without notice.
 *
 *  iCatch Technology, Inc.
 *  19-1, Innovation First Road, Science-Based Industrial Park,
 *  Hsin-Chu, Taiwan, R.O.C.
 *
 **************************************************************************/
extern "C" {
#include <api/sp5k_modesw_api.h>
#include <api/sp5k_aud_api.h>
#include <api/sp5k_sensor_api.h>
#include <api/sp5k_capture_api.h>
#include <api/sp5k_dcf_api.h>

unsigned _dbg_proc_ps_en(unsigned en);
void     _dbg_proc_ps_dump(unsigned n);
UINT32   osTaskInfoDisplay(UINT32 mode);
void     wdgOperSet(UINT32 mode, UINT32 interval);

UINT32 sp5kDcfFsSubFreeCharSet(UINT32 uiFreeCharNum,UINT8 ** uiFreeChar);
UINT32 dcfFsNextSubFileNameGet(UINT32 fileType,UINT32 subFileType,UINT8 *pfilename);
UINT32 dcfFsNextFileNameGet(UINT32 fileType,UINT8 *pfilename);
}

/**************************************************************************
 *                           C O N S T A N T S                            *
 **************************************************************************/
static BOOL cmdPsDumpEnable = FALSE;

/**************************************************************************
 *                              M A C R O S                               *
 **************************************************************************/
#define ALIGN_TO(x, n)      ( ((x)+(n-1)) & ~(n-1) )

#define VIDEO_RESOL_NR		NDK_ARRAY_SIZE(h264ResolDefs)

/**************************************************************************
 *                          D A T A    T Y P E S                          *
 **************************************************************************/
typedef struct {
	BOOL   fSaveFile;
	BOOL   fDualSave;
	BOOL   fCustomMediaSrc;

	UINT32 file_type;
	UINT32 frmrate;
	UINT32 seamless_secs;

	UINT32 h264_width;
	UINT32 h264_height;
	UINT32 h264_bitrate;

	UINT32 jpeg_width;
	UINT32 jpeg_height;
	UINT32 jpeg_q_factor;
	UINT32 jpeg_frmrate;
	UINT32 jpeg_bitrate;
} CmdRtpStreamParams;

static CmdRtpStreamParams st_params;

// URL parsing
struct H264ResolDef {
	const char *resol_str;
	UINT32 width, height;
	UINT32 bitrates[3]; /* bit-rate maps */
};

struct UrlAttrDef
{
	const char *name;
	int id;
};

enum {
	URL_H264_BITRATE,
	URL_H264_FRMRATE,

	URL_MJPG_WIDTH,
	URL_MJPG_HEIGHT,
	URL_MJPG_Q_FACTOR,
	URL_MJPG_BITRATE
};

enum {
	RTP_MJPG_Q_FINE = 80,
	RTP_MJPG_Q_NORMAL = 60,
	RTP_MJPG_Q_ECONOMY = 40,

	RTP_MJPG_BR_FINE = 8000000,
	RTP_MJPG_BR_NORMAL = 6000000,
	RTP_MJPG_BR_ECONOMY = 4000000
};

/**************************************************************************
 *                 E X T E R N A L    R E F E R E N C E S                 *
 **************************************************************************/
extern void dosSimDel(UINT8 *);
/* test hapd start/stop */
// PS-DUMP
extern UINT32 osTaskInfoDisplay(UINT32 mode);
extern void wdgOperSet(UINT32 mode, UINT32 interval);

/**************************************************************************
 *               F U N C T I O N    D E C L A R A T I O N S               *
 **************************************************************************/
static BOOL urlStrNoCaseCmp(const char *src, const char *dst, size_t len);
static void wcmd_custom_mediasrc_test();

/**************************************************************************
 *                         G L O B A L    D A T A                         *
 **************************************************************************/

static int			h264FpsDefs[3] = {30, 30, 15};
static struct UrlAttrDef	h264UrlAttrs[] =
{
	{"BITRATE", URL_H264_BITRATE},
	{"FRMRATE", URL_H264_FRMRATE},
	{NULL, 0}
};

static struct UrlAttrDef	mjpgUrlAttrs[] =
{
	{"W",  URL_MJPG_WIDTH},
	{"H",  URL_MJPG_HEIGHT},
	{"Q",  URL_MJPG_Q_FACTOR},
	{"BR", URL_MJPG_BITRATE},
	{NULL, 0}
};

static struct H264ResolDef      h264ResolDefs[] = {
	{ "HD1080", 1920, 1080, {8000000, 6000000, 4000000}},
	{ "HD720",  1280, 720,  {4000000, 3000000, 2000000}},
	{ "XGA",    1024, 768,  {3000000, 2250000, 1500000}},
	{ "SVGA",   800,  600,  {1800000, 1350000, 9000000}},
	{ "VGA",    640,  480,  {1200000, 900000,  600000}},
	{ "QVGA",   320,  240,  {300000,  225000,  150000}}
};

static inline void cmdSetAndWaitMode(UINT32 mode)
{
	sp5kModeSet(mode);
	sp5kModeWait(mode);
}

static void cmdRtpUpdateMediaAttrs(CmdRtpStreamParams *sp)
{
	sp->jpeg_width = ALIGN_TO(sp->jpeg_width, 8);
	sp->jpeg_height = ALIGN_TO(sp->jpeg_height, 8);

	#if 0
	if (sp->jpeg_q_factor > RTP_MJPG_Q_FINE)
		sp->jpeg_q_factor = RTP_MJPG_Q_FINE;
	else if (sp->jpeg_q_factor < RTP_MJPG_Q_ECONOMY)
		sp->jpeg_q_factor = RTP_MJPG_Q_ECONOMY;

	if (sp->jpeg_bitrate > RTP_MJPG_BR_FINE)
		sp->jpeg_bitrate = RTP_MJPG_BR_FINE;
	else if (sp->jpeg_bitrate < RTP_MJPG_BR_ECONOMY)
		sp->jpeg_bitrate = RTP_MJPG_BR_ECONOMY;
	#endif

	sp5kMediaRecCfgSet(SP5K_MEDIA_REC_RTP_STREAMING_EN, 0);
	sp5kMediaRecCfgSet(SP5K_MEDIA_REC_DISABLE_STORAGE, 0);

	sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_STAMP_OPTION, 0);
	sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_VIDEO_BRC_TYPE, SP5K_MEDIA_CBR);
	sp5kMediaRecCfgSet(SP5K_MEDIA_REC_MUTE_PERIOD, 100);
	sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_FILE_TYPE, sp->file_type);
	sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_AUDIO_CODEC, SP5K_MEDIA_AUDIO_PCM);

	sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_AUDIO_SAMPLE_RATE, 44100);
	sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_AUDIO_SAMPLE_BITS, 16);
	sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_AUDIO_CHANNELS, 2);

	sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_AUDIO_ALC_MAX_VOL, 31);
	sp5kMediaRecCfgSet(SP5K_MEDIA_REC_ALC_MUTE, 0);
	sp5kMediaRecCfgSet(SP5K_MEDIA_REC_ALC, SP5K_MEDIA_REC_OFF);
	sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_AUDIO_ALC_HB, 500);
	sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_AUDIO_ALC_LB, 100);
	sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_SEAMLESS_TIME_SLOT, 0);

	//sp5kMediaRecCfgSet(SP5K_MEDIA_REC_ALC, SP5K_MEDIA_REC_OFF);
	//sp5kMediaRecCfgSet(SP5K_MEDIA_REC_ALC_MODE, SP5K_MEDIA_REC_ALC_DRC_MODE);
	//sp5kMediaRecCfgSet(SP5K_MEDIA_REC_ALC, SP5K_MEDIA_REC_ON);

	printf("Mode = %d\n", ndk_st_get_mode());
	if (ndk_st_get_mode() == NDK_ST_MONO_JPEG) {
		sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_WIDTH, sp->jpeg_width);
		sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_HEIGHT, sp->jpeg_height);
		sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_MJPG_INITIAL_Q_FACTOR, sp->jpeg_q_factor);
		sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_VIDEO_AVG_BITRATE, sp->jpeg_bitrate);
		sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_VIDEO_FRAME_RATE, sp->frmrate);/*30*/
		sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_VIDEO_CODEC, SP5K_MEDIA_VIDEO_MJPG);

		sp5kSensorModeCfgSet(SP5K_MODE_VIDEO_PREVIEW, 0x30);
	}
	else {
		sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_WIDTH, sp->h264_width);
		sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_HEIGHT, sp->h264_height);
		sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_VIDEO_AVG_BITRATE, sp->h264_bitrate);
		sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_VIDEO_FRAME_RATE, sp->frmrate);
		sp5kMediaRecAttrSet(MEDIA_ATTR_H264_GOP_STRUCTURE, 0x10);
		sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_VIDEO_CODEC, SP5K_MEDIA_VIDEO_H264);

		/*if (sp->h264_width *sp->h264_height == 1920 * 1080) {
			sp5kSensorModeCfgSet(SP5K_MODE_VIDEO_PREVIEW, SP5K_SENSOR_MODE_PREVIEW + 1);
		}
		else */{
			sp5kSensorModeCfgSet(SP5K_MODE_VIDEO_PREVIEW, 0x30);
		}
	}

	if (0) {
		UINT32 w, h;
		sp5kMediaRecAttrGet(SP5K_MEDIA_ATTR_WIDTH, &w);
		sp5kMediaRecAttrGet(SP5K_MEDIA_ATTR_HEIGHT, &h);
		printf("width = %d, height = %d\n", w, h);
	}

	sp5kMediaRecCfgSet(SP5K_MEDIA_REC_DISABLE_STORAGE, sp->fSaveFile ? 0 : 1);
#if 0
	if (sp->fSaveFile && sp->fDualSave) {
		if (!ndk_st_is_started()) {
			ndk_st_dualstream_setup(sp->jpeg_width
				, sp->jpeg_height
				, sp->jpeg_q_factor - 20
				, sp->jpeg_q_factor
				, sp->jpeg_frmrate
				, sp->jpeg_bitrate);
		}

		sp5kMediaRecCfgSet (SP5K_MEDIA_REC_2ND_RECORD_EN, 1);
		sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_2ND_VIDEO_CODEC, SP5K_MEDIA_VIDEO_MJPG);
		sp5kMediaRecAttrSet(SP5K_MEDIA_ATTR_2ND_FILE_TYPE, sp->file_type);
	}
	else
#endif
	{
		sp5kMediaRecCfgSet (SP5K_MEDIA_REC_2ND_RECORD_EN, 0);
	}

	if (sp->seamless_secs)
		sp5kMediaRecAttrSet (SP5K_MEDIA_ATTR_SEAMLESS_TIME_SLOT, sp->seamless_secs);
	else
		sp5kMediaRecAttrSet (SP5K_MEDIA_ATTR_SEAMLESS_TIME_SLOT, 0);

	/* Only after the RTP Muxer is enabled, the liveStreaming can get stream data from encoder. */
	sp5kMediaRecCfgSet(SP5K_MEDIA_REC_RTP_STREAMING_EN, ndk_st_is_dualstream_jpeg() ? 0 : 1);
}

static void cmdRtpStartStreaming()
{
	CmdRtpStreamParams *sp = &st_params;

	UINT32 curMode;
	sp5kModeGet(&curMode);

	if (ndk_st_is_dualstream_jpeg()) {
		cmdRtpUpdateMediaAttrs(sp);
		if (curMode != SP5K_MODE_VIDEO_RECORD && curMode != SP5K_MODE_VIDEO_PREVIEW) {
			cmdSetAndWaitMode(SP5K_MODE_VIDEO_PREVIEW);
		}

		if (ndk_st_dualstream_start_brc(sp->jpeg_width
			, sp->jpeg_height
			, sp->jpeg_q_factor
			, sp->jpeg_frmrate
			, sp->jpeg_bitrate) != 0)
			printf("error dualstream");
	}
	else {
		cmdRtpUpdateMediaAttrs(sp);
		if (curMode != SP5K_MODE_VIDEO_PREVIEW) {
			cmdSetAndWaitMode(SP5K_MODE_VIDEO_PREVIEW);
			cmdRtpUpdateMediaAttrs(sp);
		}

		cmdSetAndWaitMode(SP5K_MODE_VIDEO_RECORD);
	}
	printf("start record\n");

	if (cmdPsDumpEnable) {
		printf("\nPS DUMP begin\n");
		//wdgOperSet(0, 0);
		_dbg_proc_ps_en(1);
	}
}

static void cmdRtpStopStreaming()
{
	if (cmdPsDumpEnable) {
		osTaskInfoDisplay(1);
	}

	if (ndk_st_is_dualstream_jpeg()) {
		if (ndk_st_dualstream_stop())
			printf("error dualstream");
	}
	else {
		UINT32 curMode;
		sp5kModeGet(&curMode);
		if (curMode == SP5K_MODE_VIDEO_RECORD)
			cmdSetAndWaitMode(SP5K_MODE_VIDEO_PREVIEW);
	}

	if (cmdPsDumpEnable) {
		_dbg_proc_ps_en(0);
		_dbg_proc_ps_dump(0);
	}
	printf("stop record\n");
}

static void cmdRtpChangeMode(NDKStMode newmode)
{
	if (ndk_st_get_mode() == newmode)
		return;

	UINT32 t = tmrTimeClockGet();

	if (ndk_st_is_started())
		cmdRtpStopStreaming();

	if (ndk_st_change_mode(newmode, NULL) < 0) {
		printf("Change mode failed\n");
		return;
	}

	if (ndk_st_is_started())
		cmdRtpStartStreaming();

	t = tmrTimeClockGet() - t;
	printf("time spent: %ums\n", t/1000);
}

static void cmdRtpSwitchMode(NDKStMode newmode, int cnt)
{
	NDKStMode mode0 = newmode;
	NDKStMode mode1 = ndk_st_get_mode();

	if (mode0 == mode1 || cnt == 0)
		return;

	int i;
	UINT32 t = tmrTimeClockGet();
	BOOL st_strted = ndk_st_is_started();

	for (i = 0; i < cnt; ++i) {
		if (st_strted)
			cmdRtpStopStreaming();

		if (ndk_st_change_mode(i & 0x01 ? mode1 : mode0, NULL) < 0) {
			printf("Change mode failed\n");
			return;
		}

		if (st_strted)
			cmdRtpStartStreaming();
	}
	t = tmrTimeClockGet() - t;

	printf("switch mode: cnt = %d, total time spent = %ums\n", cnt, t/1000);
}

static struct H264ResolDef *urlGetResolDef(const char *resol_str, int len) {
	unsigned i;
	for (i = 0; i < NDK_ARRAY_SIZE(h264ResolDefs); ++i) {
		if (urlStrNoCaseCmp(h264ResolDefs[i].resol_str, resol_str, len)) {
			return &h264ResolDefs[i];
		}
	}
	return NULL;
}

static BOOL urlStrNoCaseCmp(const char *src, const char *dst, size_t dst_len)
{
	while (*src && *dst && dst_len > 0) {
		if (toupper(*src) != toupper(*dst)) {
			return FALSE;
		}

		++src;
		++dst;
		--dst_len;
	}

	return dst_len == 0 ? TRUE : FALSE;
}

/* H264: /VGA?bitrate=high&frmrate=high
   H264: /VGA?bitrate=high&frmrate=high
   MJPG: ?W=320&H=160&Q=85 */
static BOOL urlGetNextElem(char **url_, char leading_char, char **name, int *len)
{
	char *url = *url_;
	if (*url == 0) {
		return FALSE;
	}

	if ((leading_char && *url != leading_char) || !strchr("/?=&", *url)) {
		return FALSE;
	}

	char *p = url + 1;
	while (*p) {
		if (strchr("/?=&", *p)) {
			break;
		}
		++p;
	}
	*name = url + 1;
	*len = (int)(p - url - 1);
	*url_ = p;

	return TRUE;
}

static BOOL urlParseAttrs(struct UrlAttrDef *urlAttrs, char *url
	, BOOL (*urlAttrHandler)(int id, char* val))
{
	if (!url || *url == 0)
		return TRUE;

	if (*url != '?') // must start with ?
		return FALSE;

	struct UrlAttrDef *pAttr;
	char *str;
	int  len;

	while (1) {
		if (!urlGetNextElem(&url, 0, &str, &len))
			return FALSE;

		for (pAttr = urlAttrs; pAttr->name != NULL; ++pAttr) {
			if (urlStrNoCaseCmp(pAttr->name, str, len)) {
				if (!urlGetNextElem(&url, '=', &str, &len))
					return FALSE;
				char c = str[len];
				str[len] = 0;
				BOOL r = urlAttrHandler(pAttr->id, str);
				str[len] = c;
				if (!r)
					return FALSE;
				break;
			}
		}

		if (pAttr->name == NULL) // Unknow attributes
			return FALSE;

		if (*url == 0) // finished
			break;

		if (*url != '&') // name=value pair must start with character '&' except the first one.
			return FALSE;
	}

	return TRUE;
}

static BOOL urlH264AttrHandler(int id, char* val)
{
	CmdRtpStreamParams *sp = &st_params;

	switch (id) {
	case URL_H264_BITRATE:
		if (!strcasecmp(val, "HIGH"))
			sp->h264_bitrate = 0;
		else if (!strcasecmp(val, "NORMAL"))
			sp->h264_bitrate = 1;
		else if (!strcasecmp(val, "LOW"))
			sp->h264_bitrate = 2;
		else
			return FALSE;
		break;
	case URL_H264_FRMRATE:
		if (!strcasecmp(val, "HIGH"))
			sp->frmrate = 0;
		else if (!strcasecmp(val, "NORMAL"))
			sp->frmrate = 1;
		else if (!strcasecmp(val, "LOW"))
			sp->frmrate = 2;
		else
			return FALSE;
		break;
	default:
		return FALSE;
	}

	return TRUE;
}

/* HD1080?BITRATE=HIGH&FRMRATE=HIGH
   Return 0 to abort streaming prcess */
static BOOL urlParseH264Attrs(char *url)
{
	CmdRtpStreamParams params_bak;
	memcpy(&params_bak, &st_params, sizeof(params_bak));

	struct H264ResolDef *resoldef;
	char *name;
	int  len;

	if (*url == 0) {
		return TRUE;
	}

	if (!urlGetNextElem(&url, '/', &name, &len)) {
		return FALSE;
	}

	resoldef = urlGetResolDef(name, len);
	if (!resoldef) {
		return FALSE;
	}

	if (!urlParseAttrs(h264UrlAttrs, url, urlH264AttrHandler))
		return FALSE;

	st_params.h264_width = resoldef->width;
	st_params.h264_height = resoldef->height;

	NDK_ASSERT(st_params.h264_bitrate >= 0 && st_params.h264_bitrate <= 2);
	st_params.h264_bitrate = resoldef->bitrates[st_params.h264_bitrate];

	NDK_ASSERT(st_params.frmrate >= 0 && st_params.frmrate <= 2);
	st_params.frmrate = h264FpsDefs[st_params.frmrate];

	if (memcmp(&st_params, &params_bak, sizeof(params_bak)) != 0) {
#if 0
		cmdRtpStopStreaming();
		cmdRtpStartStreaming();
#endif
	}

	return TRUE;
}

static BOOL urlMjpgAttrHandler(int id, char* val)
{
	CmdRtpStreamParams *sp = &st_params;

	switch (id) {
	case URL_MJPG_WIDTH:
		sp->jpeg_width = strtoul(val, NULL, 10);
		break;

	case URL_MJPG_HEIGHT:
		sp->jpeg_height = strtoul(val, NULL, 10);
		break;

	case URL_MJPG_Q_FACTOR:
		if (!strcasecmp(val, "FINE"))
			sp->jpeg_q_factor = RTP_MJPG_Q_FINE;
		else if (!strcasecmp(val, "NORMAL"))
			sp->jpeg_q_factor = RTP_MJPG_Q_NORMAL;
		else if (!strcasecmp(val, "ECONOMY"))
			sp->jpeg_q_factor = RTP_MJPG_Q_ECONOMY;
		else
			sp->jpeg_q_factor = strtoul(val, NULL, 10);
		break;
	case URL_MJPG_BITRATE:
		if (!strcasecmp(val, "FINE"))
			sp->jpeg_bitrate = RTP_MJPG_BR_FINE;
		else if (!strcasecmp(val, "NORMAL"))
			sp->jpeg_bitrate = RTP_MJPG_BR_NORMAL;
		else if (!strcasecmp(val, "ECONOMY"))
			sp->jpeg_bitrate = RTP_MJPG_BR_ECONOMY;
		else
			sp->jpeg_bitrate = strtoul(val, NULL, 10);
		break;
	default:
		return FALSE;
	}

	return TRUE;
}

static BOOL urlParseJpegAttrs(char *url)
{
	NDK_ASSERT(url);

	CmdRtpStreamParams params_bak;
	memcpy(&params_bak, &st_params, sizeof(params_bak));

	if (!urlParseAttrs(mjpgUrlAttrs, url, urlMjpgAttrHandler))
		return FALSE;

	if (memcmp(&st_params, &params_bak, sizeof(params_bak)) != 0) {
		// changed
	}

	return TRUE;
}

static BOOL cmdRtpEventHandler(UINT32 event, UINT32 data)
{
	switch (event) {
	case NDK_ST_EVT_RTSP_REQUEST: {
		char *url = (char *)data;
		NDKStMode st_mode = ndk_st_get_mode();
		printf("URL = '%s'\n", url);

		if (urlStrNoCaseCmp(url, "H264", 4)) {
			if (st_mode != NDK_ST_DUAL_H264JPEG && st_mode != NDK_ST_MONO_H264)
				return FALSE;

			return urlParseH264Attrs(url + 4);
		}
		else if (urlStrNoCaseCmp(url, "MJPG", 4)) {
			if (st_mode == NDK_ST_MONO_H264)
				return FALSE;

			return urlParseJpegAttrs(url + 4);
		}
		else if (urlStrNoCaseCmp(url, "PCM", 3)) {
			return TRUE;
		}
		else {
			return TRUE;
		}
		break; }

	case NDK_ST_EVT_ON_STARTED: {
		printf("NDK_ST_EVT_ON_STARTED\n");
		cmdRtpStartStreaming();
		return TRUE; }

	case NDK_ST_EVT_ON_STOPPED: {
		printf("NDK_ST_EVT_ON_STOPPED\n");
		cmdRtpStopStreaming();
		return TRUE; }

	case NDK_ST_EVT_FRAME_DROPPED: {
		return TRUE; }
	}

	return FALSE;
}

static void cmd_mediasrv(int argc, char **argv)
{
	CmdRtpStreamParams *sp = &st_params;
	NDKStCfg cfg;
	BOOL bTUTK = FALSE;
	int i;

	cfg.root_dir      = "D:";
	cfg.port          = 554;
	cfg.st_mode       = NDK_ST_DUAL_STREAMING;
	cfg.audio_on      = 0;
	cfg.keepalive_secs = 30;
	cfg.evt_handler   = cmdRtpEventHandler;
	cfg.flags         = 0;

	memset(sp, 0, sizeof(*sp));
	sp->fSaveFile     = TRUE;
	sp->fDualSave     = FALSE;
	sp->file_type     = SP5K_MEDIA_MOV;

	sp->h264_width    = 1024;//1920;
	sp->h264_height   = 768;//1080;
	sp->h264_bitrate  = 6000000;
	sp->frmrate       = 30;

	sp->jpeg_width    = 640;
	sp->jpeg_height   = 360;
	sp->jpeg_q_factor = 80;
	sp->jpeg_frmrate  = 30;
	sp->jpeg_bitrate  = 4000000;

	for (i = 0; i < argc; ++i) {
		if (!strcmp(argv[i], "-h")) {
			goto lUSAGE;
		}
		else if (!strcmp(argv[i], "-x")) {
			ndk_st_stop_server();
			return;
		}
		else if (!strcmp(argv[i], "-dhj") || !strcmp(argv[i], "-djh")) {
			cfg.st_mode = NDK_ST_DUAL_H264JPEG;
			sp->jpeg_height = 360;
		}
		else if (!strcmp(argv[i], "-dj")) {
			cfg.st_mode = NDK_ST_DUAL_STREAMING;
			sp->jpeg_height = 360;
		}
		else if (!strcmp(argv[i], "-mh")) {
			cfg.st_mode = NDK_ST_MONO_H264;
		}
		else if (!strcmp(argv[i], "-mj")) {
			cfg.st_mode = NDK_ST_MONO_JPEG;
		}
		else if (!strcmp(argv[i], "-nsave")) {
			sp->fSaveFile = FALSE;
		}
		else if (!strncmp(argv[i], "-A", 2)) {
			const char *c;
			cfg.audio_on = NDK_ST_AUDIO_ON_NORMAL | NDK_ST_AUDIO_ON_NOSWAP;
			for (c = &argv[i][2]; *c; ++c) {
				if (*c == 's')
					cfg.audio_on &= ~NDK_ST_AUDIO_ON_NOSWAP;
				else
					goto lUSAGE;
			}
		}
		else if (!strcmp(argv[i], "-avi")) {
			sp->file_type = SP5K_MEDIA_AVI;
		}
		else if (!strcmp(argv[i], "-mov")) {
			sp->file_type = SP5K_MEDIA_MOV;
		}
		else if (!strcmp(argv[i], "-psdump")) {
			cmdPsDumpEnable = TRUE;
		}
		else if (!strcmp(argv[i], "-dsave")) {
			sp->fDualSave = TRUE;
		}
		else if (!strcmp(argv[i], "-custom")) {
			sp->fCustomMediaSrc = TRUE;
		}
		else if (!strcmp(argv[i], "-tutk")) {
			bTUTK = TRUE;
		}
		// The following options need argument
		else if ( argc-i < 2) {
			goto lUSAGE;
		}
		else if (!strcmp(argv[i], "-root")) {
			cfg.root_dir = argv[++i];
		}
		else if (!strcmp(argv[i], "-fr")) {
			sp->frmrate = atoi(argv[++i]);
		}
		else if (!strcmp(argv[i], "-hres")) {
			const char* resol_str = argv[++i];

			if (!strcmp(resol_str, "fhd")) {
				sp->h264_width = 1920;
				sp->h264_height = 1080;
			}
			else if (!strcmp(resol_str, "hd")) {
				sp->h264_width = 1280;
				sp->h264_height = 720;
			}
			else if (!strcmp(resol_str, "xga")) {
				sp->h264_width = 1024;
				sp->h264_height = 768;
			}
			else if (!strcmp(resol_str, "svga")) {
				sp->h264_width = 800;
				sp->h264_height = 600;
			}
			else if (!strcmp(resol_str, "vga")) {
				sp->h264_width = 640;
				sp->h264_height = 480;
			}
			else if (!strcmp(resol_str, "qvga")) {
				sp->h264_width = 320;
				sp->h264_height = 240;
			}
			else {
				goto lUSAGE;
			}
		}
		else if (!strcmp(argv[i], "-hw")) {
			sp->h264_width = atoi(argv[++i]);
		}
		else if (!strcmp(argv[i], "-hh")) {
			sp->h264_height = atoi(argv[++i]);
		}
		else if (!strcmp(argv[i], "-hbr")) {
			sp->h264_bitrate = kmgstr_to_long(argv[++i]);
		}
		else if (!strcmp(argv[i], "-jw")) {
			sp->jpeg_width = atoi(argv[++i]);
		}
		else if (!strcmp(argv[i], "-jh")) {
			sp->jpeg_height = atoi(argv[++i]);
		}
		else if (!strcmp(argv[i], "-jq")) {
			sp->jpeg_q_factor = atoi(argv[++i]);
		}
		else if (!strcmp(argv[i], "-jfr")) {
			sp->jpeg_frmrate = atoi(argv[++i]);
		}
		else if (!strcmp(argv[i], "-jbr")) {
			sp->jpeg_bitrate = kmgstr_to_long(argv[++i]);
		}
		else if (!strcmp(argv[i], "-ka")) {
			cfg.keepalive_secs = atoi(argv[++i]);
		}
		else if (!strcmp(argv[i], "-seam")) {
			sp->seamless_secs = atoi(argv[++i]);
		}
		else {
			goto lUSAGE;
		}
	}

	if (cfg.st_mode == NDK_ST_DUAL_STREAMING && !sp->fSaveFile) {
		printf("Error: NDK_ST_DUAL_STREAMING is set but file storage is disabled.\n");
		return;
	}

	if (bTUTK) {
		cfg.flags |= NDK_ST_FLG_TUTK;
		cfg.keepalive_secs = 0;

		sp->jpeg_width = 320;
		sp->jpeg_height = 240;
		sp->jpeg_bitrate = 1000*1000*2/5;
		sp->jpeg_frmrate = 6;
	}

	if (ndk_st_start_server(&cfg) == 0) {
		if (sp->fCustomMediaSrc)
			wcmd_custom_mediasrc_test();
	}
	else {
		printf("start server failed");
	}
	return;

lUSAGE:
	printf("mediasrv [options]\n");
	printf("  -h     : Show help\n");
	printf("  -x     : Stop media server\n");
	printf("  -root dir : Set root direcotyr for media server\n");
	printf("  -dhj   : Dual Streaming.\n");
	printf("  -dj    : Dual Streaming without H264.\n");
	printf("  -mh    : Mono Streaming with H264 encoding.\n");
	printf("  -mj    : Mono Streaming with JPEG encoding.\n");
	printf("  -nsave : Disable file storage. Cannot coexist with -dj\n");
	printf("  -A[s]   : Enable audio streaming. s: swap audio bytes.\n");
	printf("  -fr n  : frame-rate.\n");
	printf("  -hw n  : H264 width.\n");
	printf("  -hh n  : H264 height.\n");
	printf("  -hbr n : H264 bit-rate(K,M).\n");
	printf("  -jw n  : JPEG width.\n");
	printf("  -jh n  : JPEG height.\n");
	printf("  -jq n  : JPEG Q factor.\n");
	printf("  -jbr n : JPEG bit-rate(K).\n");
	printf("  -jsf n : JPEG skip frame factor.\n");
}

static void cmd_mediactl(int argc, char **argv)
{
	CmdRtpStreamParams *sp = &st_params;
	char *subcmd;
	/* jump crosses initialization is not allowed since g++ 2.7.0
	 * or compiled with "-fpermissive" option */
	if (argc == 0)
		goto lUSAGE;

	subcmd = argv[0];
	--argc;
	++argv;

	if (!strcmp(subcmd, "-h")) {
		goto lUSAGE;
	}
	else if (!strcmp(subcmd, "rec")) {
		UINT32 mode;
		sp5kModeGet(&mode);

		if (argc == 1 && !strcmp(argv[0], "+")) {
			if (ndk_st_is_started()) {
				if (mode == SP5K_MODE_VIDEO_PREVIEW)
					cmdSetAndWaitMode(SP5K_MODE_VIDEO_RECORD);
			}
			else {
				if (mode != SP5K_MODE_VIDEO_RECORD) {
					cmdRtpUpdateMediaAttrs(sp);
					cmdSetAndWaitMode(SP5K_MODE_VIDEO_PREVIEW);
					cmdSetAndWaitMode(SP5K_MODE_VIDEO_RECORD);
				}
			}
		}
		else if (argc == 1 && !strcmp(argv[0], "-")) {
			if (mode == SP5K_MODE_VIDEO_RECORD) {
				//ndk_st_dualstream_stop();
				cmdSetAndWaitMode(SP5K_MODE_VIDEO_PREVIEW);
				//sp5kTimeDelay(SP5K_TIME_DELAY_1MS, 100);
				//ndk_st_dualstream_start(sp->jpeg_width, sp->jpeg_height, sp->jpeg_q_factor);
			}
		}
		else if (argc >= 1 && !strcmp(argv[0], "sw")) {
			int i, sleep = 0, n = 1;
			if (argc >= 2)
				n = atoi(argv[1]);
			if (argc >= 3)
				sleep = atoi(argv[2]);

			// preview <-> record switch
			cmdRtpUpdateMediaAttrs(sp);
			if (mode == SP5K_MODE_VIDEO_PREVIEW) {
				for (i = 0; i < n; ++i) {
					cmdSetAndWaitMode(SP5K_MODE_VIDEO_RECORD);
					if (sleep > 0)
						sp5kTimeDelay(SP5K_TIME_DELAY_1MS, sleep);
					cmdSetAndWaitMode(SP5K_MODE_VIDEO_PREVIEW);
				}
			}
			// record <-> preview switch
			else if (mode == SP5K_MODE_VIDEO_RECORD) {
				for (i = 0; i < n; ++i) {
					cmdSetAndWaitMode(SP5K_MODE_VIDEO_PREVIEW);
					if (sleep > 0)
						sp5kTimeDelay(SP5K_TIME_DELAY_1MS, sleep);
					cmdSetAndWaitMode(SP5K_MODE_VIDEO_RECORD);
				}
			}
			else {
				printf("mode error\n");
			}
		}
		else
			goto lUSAGE;
	}
	else if (!strcmp(subcmd, "snap")) {
		UINT32 t0, t1, t;
		int i, cnt = 1;

		if (argc >= 1)
			cnt = atoi(argv[0]);

		t0 = t1 = 0;
		for (i = 0; i < cnt; ++i) {
			t = tmrTimeClockGet()/1000;
			cmdRtpStopStreaming();
			t0 += tmrTimeClockGet()/1000 - t;

			t = tmrTimeClockGet()/1000;
			cmdRtpStartStreaming();
			t1 += tmrTimeClockGet()/1000 - t;
		}

		printf("snap test: cnt=%d, stop-time=(%u, %u)ms, start-time=(%u, %u)ms\n"
			, cnt, t0, t0/cnt, t1, t1/cnt);
		return;
	}
	else if (!strcmp(subcmd, "j")) {
		if (argc != 4)
			goto lUSAGE;

		sp->jpeg_width = atoi(argv[0]);
		sp->jpeg_height = atoi(argv[1]);
		sp->jpeg_bitrate = atoi(argv[2]) * 1000;
		sp->jpeg_q_factor = atoi(argv[3]);

		UINT32 t = tmrTimeClockGet()/1000;

		cmdRtpStopStreaming();
		cmdRtpStartStreaming();

		t = tmrTimeClockGet()/1000 - t;
		printf("time spent = %ums\n", t);
	}
#ifdef SPCA6330
	else if (argc == 1 && !strcmp(subcmd, "djfr")) {
		ndk_dualstream_set_framerate(atoi(argv[0]));
	}
#endif
	else if (!strcmp(subcmd, "h")) {
		if (argc != 3)
			goto lUSAGE;

		sp->h264_width = atoi(argv[0]);
		sp->h264_height = atoi(argv[1]);
		sp->h264_bitrate = atoi(argv[2]) * 1000;

		UINT32 t = tmrTimeClockGet()/1000;

		cmdRtpStopStreaming();
		cmdRtpStartStreaming();

		t = tmrTimeClockGet()/1000 - t;
		printf("time spent = %ums\n", t);
	}
	else if (!strcmp(subcmd, "mon")) {
		ndk_st_start_monitor();
	}
	else if (!strcmp(subcmd, "close")) {
		ndk_st_close_streaming();
	}
#if 0
	else if (!strcmp(subcmd, "sm")) {
		int cnt = 1;

		if (argc >= 1)
			cnt = atoi(argv[0]);

		if (ndk_st_is_dualstream_jpeg())
			cmdRtpSwitchMode(NDK_ST_MONO_JPEG, cnt);
		else
			cmdRtpSwitchMode(NDK_ST_DUAL_H264JPEG, cnt);
	}
	else if (!strcmp(subcmd, "cm")) {
		if (argc != 1)
			goto lUSAGE;

		NDKStMode mode;

		if (!strcmp(argv[0], "dj")) {
			mode = NDK_ST_DUAL_STREAMING;
		}
		else if (!strcmp(argv[0], "dhj") || !strcmp(argv[0], "djh")) {
			mode = NDK_ST_DUAL_H264JPEG;
		}
		else if (!strcmp(argv[0], "mh")) {
			mode = NDK_ST_MONO_H264;
		}
		else if (!strcmp(argv[0], "mj")) {
			mode = NDK_ST_MONO_JPEG;
		}
		else {
			printf("Wrong mode\n");
			return;
		}

		cmdRtpChangeMode(mode);
	}
#endif
	else if (!strcmp(subcmd, "dsj")) {
		if (argc < 1)
			goto lUSAGE;

		int r = 0;

		if (!strcmp(argv[0], "p"))
			r = ndk_st_dualstream_control(NDK_DUALSTREAM_PAUSE, 0);
		else if (!strcmp(argv[0], "r"))
			r = ndk_st_dualstream_control(NDK_DUALSTREAM_RESUME, 0);
		else if (argc > 1 && !strcmp(argv[0], "maxq"))
			r = ndk_st_dualstream_control(NDK_DUALSTREAM_MAXQ_SET, atoi(argv[1]));
		else if (argc > 1 && !strcmp(argv[0], "minq"))
			r = ndk_st_dualstream_control(NDK_DUALSTREAM_MINQ_SET, atoi(argv[1]));
		else if (argc > 1 && !strcmp(argv[0], "br"))
			r = ndk_st_dualstream_control(NDK_DUALSTREAM_BITRATE_SET, kmgstr_to_long(argv[1]));
		else
			goto lUSAGE;

		if (r != 0)
			printf("dsj control failed\n");
	}
	else {
		goto lUSAGE;
	}

	return;

lUSAGE:
	printf("mc h w h br      : change h264 settings\n");
	printf("mc j w h br q    : change jpeg settings\n");
	printf("mc close         : close streaming\n");
	printf("mc mon           : monitor streaming\n");
	printf("mc cm [dj|mh|mj] : change mode\n");
	printf("mc sm            : dual <==> mono\n");
	printf("mc rec +/-       : start/stop record in dual-streaming mode\n");
	printf("mc rec sw [nr] [sleep-time] : switch test\n");
	printf("mc dst p/r       : pause/resume dual-streaming\n");
}

static void cmd_dualstream_test(int argc, char *argv[])
{
	if (argc >= 1 && !strcmp(argv[0], "+"))
		ndk_st_dualstream_start_brc(640, 360, 80, 30, 4500000);
	else if (argc >= 1 && !strcmp(argv[0], "-"))
		ndk_st_dualstream_stop();
	else
		printf("wrong arguments\n");
}

static void wcmd_dcf_test(int argc, char** argv)
{
	const char *freechar[5] = {"TUM","TOM"};
	sp5kDcfFsSubFreeCharSet(2, (UINT8**)freechar);

	UINT32 type;
	sp5kMediaRecAttrGet(MEDIA_ATTR_FILE_TYPE, &type);

	char filename[256];
	filename[0] = 0;

	dcfFsNextFileNameGet(type, (UINT8*)filename);
	printf("%s\n", filename);

	dcfFsNextSubFileNameGet(type, type, (UINT8*)filename);
	printf("%s\n", filename);

	dcfFsNextSubFileNameGet(type, type, (UINT8*)filename);
	printf("%s\n", filename);

}

static NDKStMediaSrcCallback g_custom_msrc_cb;
static NDKStMediaSrcPushBufferFunc g_push_func = NULL;
static void *g_push_param;

static void customMediaSrcOnGotBuffer(mediaSrcBuffer_t *buf, unsigned long user_data)
{
	NDKStMediaBuffer mb;

	if (MEDIA_BUF_ES_TYPE(buf->bufflags) == MEDIA_BUF_ES_AUDIO)
		mb.f_buftype = NDK_ST_MEDIABUFFER_AUDIO;
	else if (MEDIA_BUF_ES_TYPE(buf->bufflags) == MEDIA_BUF_ES_VIDEO)
		mb.f_buftype = NDK_ST_MEDIABUFFER_VIDEO;
	else
		return;

	mb.bufobj       = (NDKStMediaBufferObject)buf->bufobj;
	mb.f_keyframe   = buf->bufflags & MEDIA_BUF_KEY_FRAME ?   1 : 0;
	mb.f_paramframe = buf->bufflags & MEDIA_BUF_PARAM_FRAME ? 1 : 0;
	mb.data         = buf->data;
	mb.length       = buf->length;
	mb.duration     = buf->duration;
	mb.pts          = buf->pts;

	if (g_push_func) {
		mediaSrcHoldBufferObject(buf->bufobj);
		g_push_func(&mb, g_push_param);
	}
}

static NDKStMediaSrcHandle
customMediaSrcOpen(
	NDKStMediaSrcPushBufferFunc push_func,
	void *push_param,
	void *src_arg,
	...)
{
	g_push_func = push_func;
	g_push_param = push_param;
	return (NDKStMediaSrcHandle)mediaSrcOpenFile((const char *)src_arg, customMediaSrcOnGotBuffer, 0, 0);
}

static void customMediaSrcClose(NDKStMediaSrcHandle h)
{
	mediaSrcClose((mediaSrcHandle_t)h);
}

static UINT32 customMediaSrcPause(NDKStMediaSrcHandle h)
{
	return mediaSrcControl((mediaSrcHandle_t)h, MEDIA_SRC_CTRL_PAUSE);
}

static UINT32 customMediaSrcResume(NDKStMediaSrcHandle h)
{
	return mediaSrcControl((mediaSrcHandle_t)h, MEDIA_SRC_CTRL_RESUME);
}

static BOOL customMediaSrcEndOfSource(NDKStMediaSrcHandle h)
{
	return mediaSrcEndOfSource((mediaSrcHandle_t)h);
}

static UINT32 customMediaSrcSeekTo(NDKStMediaSrcHandle h, UINT32 position)
{
	return mediaSrcControl((mediaSrcHandle_t)h, MEDIA_SRC_CTRL_SEEK, position);
}

static UINT32 customMediaSrcGetAttribute(NDKStMediaSrcHandle h, UINT32 attr, UINT32 *val)
{
	return mediaSrcGetAttribute((mediaSrcHandle_t)h, attr, val);
}

static void customMediaSrcFreeBufferObject(NDKStMediaBufferObject obj)
{
	mediaSrcReleaseBufferObject((mediaBufferObject_t)obj);
}

static void wcmd_custom_mediasrc_test()
{
	g_custom_msrc_cb.open = customMediaSrcOpen;
	g_custom_msrc_cb.close = customMediaSrcClose;
	g_custom_msrc_cb.pause = customMediaSrcPause;
	g_custom_msrc_cb.resume = customMediaSrcResume;
	g_custom_msrc_cb.end_of_source = customMediaSrcEndOfSource;
	g_custom_msrc_cb.seek_to = customMediaSrcSeekTo;
	g_custom_msrc_cb.get_attribute = customMediaSrcGetAttribute;
	g_custom_msrc_cb.free_buffer_object = customMediaSrcFreeBufferObject;

	ndk_st_register_mediasrc(NDK_ST_MEDIASRC_FILE, &g_custom_msrc_cb);
}

