/**************************************************************************
 *
 *       Copyright (c) 2014 by iCatch Technology, Inc.
 *
 *  This software is copyrighted by and is the property of iCatch Technology,
 *  Inc.. All rights are reserved by iCatch Technology, Inc..
 *  This software may only be used in accordance with the corresponding
 *  license agreement. Any unauthorized use, duplication, distribution,
 *  or disclosure of this software is expressly forbidden.
 *
 *  This Copyright notice MUST not be removed or modified without prior
 *  written consent of iCatch Technology, Inc..
 *
 *  iCatch Technology, Inc. reserves the right to modify this software
 *  without notice.
 *
 *  iCatch Technology, Inc.
 *  19-1, Innovation First Road, Science-Based Industrial Park,
 *  Hsin-Chu, Taiwan, R.O.C.
 *
 **************************************************************************/
extern "C" {
#include <api/sp5k_modesw_api.h>
#include <api/sp5k_aud_api.h>
#include <api/sp5k_sensor_api.h>
#include <api/sp5k_dcf_api.h>

BOOL recutil_stmgr_is_started(void* hmgr);
int  recutil_stmgr_get_info(void* hmgr, UINT32 *stream_bits_file, UINT32 *stream_bits_rtp);
void recutil_stmgr_dump(void* hmgr);

int recutil_token_index();
const char *recutil_token_value();
void recutil_token_skip(const char **pp_tokstr);
BOOL recutil_token_compare(const char **pp_tokstr, const char *tok);
BOOL recutil_token_vcompare(const char **pp_tokstr, ...);

void *recutil_create();
void recutil_destroy(void *h);
void *recutil_get_stream_manager(void *h);
void recutil_config_media(void *h, int stream_id, char klass, const char *mstr, BOOL update_def_codec);
void recutil_get_info(void *h, int stream_id, UINT32 *codec0, UINT32 *codec1);
int  recutil_start_stream(void *h, UINT32 stream_bits, const char *stream_chars);
int  recutil_stop_stream(void *h, UINT32 stream_bits);

UINT32 mediaAttrIdConv(UINT32 attrId, UINT32 codecId);
} // "C"

#define URL_ROOT_NAME_SIZE     16
#define RTP_ERR_GOTO(label)    do { ndk_error("err @ %d", __LINE__); goto label; }while(0)

///////////////////////////////////////////////////////////////////////////////
// Media Attributes

typedef struct {
	NDKStCfg cfg;
	void     *recutil;

	BOOL     auto_start_all_streams;

	UINT32   a_flags;
	int      rtp_payload_size;
	int      dropfrm_time;
	char     root_dir[32];
} RTPStCtx;

static RTPStCtx  *rtp_stctx = NULL;

static RTPStCtx *rtp_stctx_create(BOOL (*evt_handler)(UINT32, UINT32))
{
	RTPStCtx *rtpctx = (RTPStCtx*)ndk_mem_malloc(sizeof(RTPStCtx));

	memset(rtpctx, 0, sizeof(*rtpctx));
	rtpctx->cfg.root_dir       = "D:";
	rtpctx->cfg.port           = 554;
	rtpctx->cfg.audio_on       = NDK_ST_AUDIO_OFF;
	rtpctx->cfg.keepalive_secs = 30;
	rtpctx->cfg.evt_handler    = evt_handler;

	// Default values.
	rtpctx->dropfrm_time       = -1;
	rtpctx->recutil            = recutil_create();

	return rtpctx;
}

static void rtp_stctx_destroy(RTPStCtx *rtpctx)
{
	if (rtpctx) {
		recutil_destroy(rtpctx->recutil);
		ndk_mem_free(rtpctx);
	}
}

///////////////////////////////////////////////////////////////////////////////
// StreamManager Callback.

static void rtp_get_url_root_name(const char *url, char *rootname, int len)
{
	char *p = strchr(url, '?');
	int n = p ? p - url : strlen(url);

	if (n >= len) {
		rootname[0] = 0;
	}
	else {
		strncpy(rootname, url, len);
		rootname[len - 1] = 0;
	}
}

static BOOL rtp_get_stream_info(RTPStCtx *rtpctx, const char *rootname, int *sid, UINT32 *codec)
{
	long st_type;

	ndk_st_get_attribute(NDK_ST_ATTR_STREAMING_TYPE, (long*)&st_type);
	if (st_type == NDK_ST_TYPE_DEFAULT)
		st_type = ndk_st_parse_streaming_type(rootname);

	if (ndk_st_get_mode() == NDK_ST_MULTI_STREAM) {
		if (st_type == NDK_ST_TYPE_H264S0) {
			*sid = 0;
			*codec = SP5K_MEDIA_VIDEO_H264;
			return TRUE;
		}
		else if (st_type == NDK_ST_TYPE_H264S1) {
			*sid = 1;
			*codec = SP5K_MEDIA_VIDEO_H264;
			return TRUE;
		}
	}

	int i;

	for (i = 0; i < SP5K_MEDIA_REC_STREAM_NUM; ++i) {
		UINT32 codec0, codec1;
		recutil_get_info(rtpctx->recutil, i, &codec0, &codec1);

		if ((codec0 == SP5K_MEDIA_VIDEO_H264 || codec1 == SP5K_MEDIA_VIDEO_H264) && st_type == NDK_ST_TYPE_H264S0) {
			*sid = i;
			*codec = SP5K_MEDIA_VIDEO_H264;
			return TRUE;
		}
		else if ((codec0 == SP5K_MEDIA_VIDEO_MJPG || codec1 == SP5K_MEDIA_VIDEO_MJPG) && st_type == NDK_ST_TYPE_JPEG) {
			*sid = i;
			*codec = SP5K_MEDIA_VIDEO_MJPG;
			return TRUE;
		}
	}

	return FALSE;
}

static BOOL rtp_handle_events(UINT32 event, UINT32 data)
{
	switch (event) {
	case NDK_ST_EVT_RTSP_REQUEST:
	{
		int sid = -1;
		UINT32 codec0;
		printf("URL='%s'\n", (const char *)data);

		if (ndk_sscanf((const char *)data, "live%d", &sid) == 1 ||ndk_sscanf((const char *)data, "LIVE%d", &sid) == 1) {
			if (sid < 0 || sid >= SP5K_MEDIA_REC_STREAM_NUM)
				ndk_err_return(FALSE);

			recutil_get_info(rtp_stctx->recutil, sid, &codec0, NULL);
			if (codec0 == SP5K_MEDIA_VIDEO_H264) {
				ndk_st_set_attribute(NDK_ST_ATTR_STREAMING_TYPE, (sid == 0) ? NDK_ST_TYPE_H264S0 : NDK_ST_TYPE_H264S1);
			}
			else if (codec0 == SP5K_MEDIA_VIDEO_MJPG) {
				ndk_st_set_attribute(NDK_ST_ATTR_STREAMING_TYPE, (sid == 0) ? NDK_ST_TYPE_JPEGS0 : NDK_ST_TYPE_JPEGS1);
			}
			else
				ndk_err_return(FALSE);
		}
		else {
			ndk_st_set_attribute(NDK_ST_ATTR_STREAMING_TYPE, NDK_ST_TYPE_DEFAULT);
		}

		return TRUE;
	}

	case NDK_ST_EVT_ON_STARTED:
	case NDK_ST_EVT_ON_STOPPED:
	{
		const  char *estr = event == NDK_ST_EVT_ON_STARTED ? "STARTED" : "STOPPED";
		char   rootname[URL_ROOT_NAME_SIZE] = {0};
		char   stream_chars[SP5K_MEDIA_REC_STREAM_NUM + 1];
		UINT32 rtp_codec, sbits = 0;
		int    i, rtp_sid, err;
		void   *stmgr = recutil_get_stream_manager(rtp_stctx->recutil);

		rtp_get_url_root_name((const char *)data, rootname, sizeof(rootname));
		if (!rootname[0])
			return FALSE;

		if (!rtp_get_stream_info(rtp_stctx, rootname, &rtp_sid, &rtp_codec))
			return FALSE;

		if (recutil_stmgr_is_started(stmgr)) {
			UINT32 v_codec = 0;
			sp5kMediaRecAttrGet(mediaAttrIdConv(SP5K_MEDIA_ATTR_VIDEO_CODEC, rtp_sid), &v_codec);
			if (rtp_codec != v_codec) {
				printf("RTP-EVENT-%s %d err: %x != %x\n", estr, rtp_sid, rtp_codec, v_codec);
				return FALSE;
			}
		}

		for (i = 0; i < SP5K_MEDIA_REC_STREAM_NUM; ++i) {
			if (rtp_sid == i) {
				stream_chars[i] = rtp_codec == SP5K_MEDIA_VIDEO_H264 ? 'H' : 'J';
				sbits |= 1 << i;
			}
			else if (rtp_stctx->auto_start_all_streams) {
				stream_chars[i] = '*';
				sbits |= 1 << i;
			}
			else
				stream_chars[i] = '-';
		}
		stream_chars[i] = 0;

		printf("RTP-EVENT-%s url=%s s-chars=%s rtp_sid=%x\n", estr, (char *)data, stream_chars, rtp_sid);

		if (event == NDK_ST_EVT_ON_STARTED)
			err = recutil_start_stream(rtp_stctx->recutil, sbits, stream_chars);
		else
			err = recutil_stop_stream(rtp_stctx->recutil, sbits);

		if (err != 0) {
			ndk_error("%s %d\n", estr, err);
			return FALSE;
		}

		return TRUE;
	}

	case NDK_ST_EVT_FRAME_DROPPED:
		return TRUE;

	default:
		break;
	}

	return FALSE;
}

///////////////////////////////////////////////////////////////////////////////
// Media Server Command
static void mediasrv_usage()
{
	printf("Stream Options:\n");
	printf("  -As,nswap,pcm,aac,mulaw,alaw,adpcm8,adpcm16,adpcm11,adpcm22,adpcm48,sr= : Audio\n");
	printf("  -Hp60,g30,g60,g120,b,i,w=,h=,fr=,br=,fhd,hd,xga,svga,vga,qvga : H264\n");
	printf("  -Jq=,w=,h=,fr=,br=,fhd,hd,xga,svga,vga,qvga : MJPG\n");
	printf("  -Rdf=,root=,ka=,rplsz= -rtp: RTP StreamCtrl\n");
	printf("  -Favi,mov,seamless= -file: File StreamCtrl\n");
	printf("  -Oasas,tutk,sen= : Miscellaneous Options. asas=auto-start-all-streams.\n");
	printf("  -n       : Define the next stream.\n");
	printf("  -help    : Help.\n");
	printf("  -x       : Stop media server.\n");
}

static BOOL mediasrv_precheck(int argc, char **argv)
{
	int i;

	for (i = 0; i < argc; ++i) {
		if (!strcasecmp(argv[i], "-help")) {
			mediasrv_usage();
			return FALSE;
		}
		else if (!strcasecmp(argv[i], "-x")) {
			ndk_st_stop_server();

			if (rtp_stctx) {
				rtp_stctx_destroy(rtp_stctx);
				rtp_stctx = NULL;
			}
			return FALSE;
		}
	}

	return TRUE;
}

#define MEDIASRV_ERROR(l)  do { printf("err @ %d\n", l); goto lFail; } while (0)

static void cmd_mediasrv(int argc, char **argv)
{
	if (!mediasrv_precheck(argc, argv))
		return;

	if (rtp_stctx) {
		printf("Error: MediaServer Already Started\n");
		return;
	}

	UINT32 sbits_rtp = 0;
	void *stmgr = NULL;
	const char *mstr, *s;
	int i, sid = 0, h264_num = 0, jpeg_num = 0;
	char klass;
	BOOL bTUTK = FALSE;

	RTPStCtx *rtpctx = rtp_stctx_create(rtp_handle_events);

	for (i = 0; i < argc; ++i) {
		s = argv[i];
		if (s[0] != '-' || s[1] == 0) {
			printf("invalid option: %s\n", s);
			rtp_stctx_destroy(rtpctx);
			return;
		}

		klass = s[1];
		mstr = s + 2;

		if (!strcmp(s, "-n")) {
			if (sid == SP5K_MEDIA_REC_STREAM_NUM-1) {
				printf("too many '-n'\n");
				rtp_stctx_destroy(rtpctx);
				return;
			}
			++sid;
		}
		else if (klass == 'A') {
			rtpctx->cfg.audio_on = NDK_ST_AUDIO_ON_NORMAL;
			recutil_config_media(rtpctx->recutil, sid, klass, mstr, FALSE);

			while (*mstr) {
				if (recutil_token_compare(&mstr, "nswap"))
					rtpctx->a_flags |= NDK_ST_AUDIO_ON_NOSWAP;
				else
					recutil_token_skip(&mstr);
			}
		}
		else if (klass == 'H' || klass == 'J' || klass == 'F') {
			recutil_config_media(rtpctx->recutil, sid, klass, mstr, FALSE);
		}
		else if (klass == 'R') {
			recutil_config_media(rtpctx->recutil, sid, klass, mstr, FALSE);

			while (*mstr) {
				if (recutil_token_compare(&mstr, "df=")) {
					rtpctx->dropfrm_time = atoi(recutil_token_value());
				}
				else if (recutil_token_compare(&mstr, "root=")) {
					strcpy(rtpctx->root_dir, recutil_token_value());
					rtpctx->cfg.root_dir = rtpctx->root_dir;
				}
				else if (recutil_token_compare(&mstr, "ka=")) {
					rtpctx->cfg.keepalive_secs = atoi(recutil_token_value());
				}
				else if (recutil_token_compare(&mstr, "rplsz=")) {
					rtpctx->rtp_payload_size = atoi(recutil_token_value());
				}
				else
					recutil_token_skip(&mstr);
			}
		}
		else if (klass == 'O') {
			recutil_config_media(rtpctx->recutil, sid, klass, mstr, FALSE);

			while (*mstr) {
				if (recutil_token_compare(&mstr, "asas")) {
					rtpctx->auto_start_all_streams = 1;
				}
				else if (recutil_token_compare(&mstr, "tutk")) {
					bTUTK = TRUE;
				}
				else
					recutil_token_skip(&mstr);
			}
		}
	}

	for (i = 0; i < SP5K_MEDIA_REC_STREAM_NUM; ++i) {
		UINT32 codec0, codec1;
		recutil_get_info(rtpctx->recutil, i, &codec0, &codec1);

		if (codec0 == SP5K_MEDIA_VIDEO_H264 || codec1 == SP5K_MEDIA_VIDEO_H264)
			++h264_num;
		else if (codec0 == SP5K_MEDIA_VIDEO_H264 || codec1 == SP5K_MEDIA_VIDEO_H264)
			++jpeg_num;
	}

	if (h264_num == 1 && jpeg_num == 1)
		rtpctx->cfg.st_mode = NDK_ST_DUAL_H264JPEG;
	else if (h264_num == 1 && jpeg_num == 0)
		rtpctx->cfg.st_mode = NDK_ST_MONO_H264;
	else if (jpeg_num == 1 && h264_num == 0)
		rtpctx->cfg.st_mode = NDK_ST_MONO_JPEG;
	else
		rtpctx->cfg.st_mode = NDK_ST_MULTI_STREAM;

	stmgr = recutil_get_stream_manager(rtpctx->recutil);
	if (!stmgr)
		MEDIASRV_ERROR(-__LINE__);

	recutil_stmgr_dump(stmgr);
	recutil_stmgr_get_info(stmgr, NULL, &sbits_rtp);
	if (!sbits_rtp)
		MEDIASRV_ERROR(-__LINE__);

	rtp_stctx = rtpctx;

	if (bTUTK) {
		rtpctx->cfg.keepalive_secs = 0;
		rtpctx->cfg.flags |= NDK_ST_FLG_TUTK;
	}

	if (ndk_st_start_server(&rtpctx->cfg) == 0) {
		if (rtpctx->dropfrm_time >= 0)
			ndk_st_set_attribute(NDK_ST_ATTR_TIME_TO_DROP_FRAME, rtpctx->dropfrm_time);
		if (rtpctx->rtp_payload_size > 0)
			ndk_st_set_attribute(NDK_ST_ATTR_RTP_PAYLOAD_SIZE, rtpctx->rtp_payload_size);

		return;
	}

lFail:
	printf("start mediasrv failed\n");
	mediasrv_usage();
	rtp_stctx_destroy(rtpctx);
	rtp_stctx = NULL;
}

static void cmd_mediactl(int argc, char **argv)
{
	int    err = 0;
	char   *subcmd;
	BOOL   multi_start_stop_test = FALSE;
	UINT32 stream_bits = 0;
	void   *stmgr = recutil_get_stream_manager(rtp_stctx->recutil);

	/* jump crosses initialization is not allowed since g++ 2.7.0
	 * or compiled with "-fpermissive" option */
	if (argc == 0) goto lUsage;

	subcmd = argv[0];
	--argc;
	++argv;

	if (!strcmp(subcmd, "help")) {
		goto lUsage;
	}
	else if (!strcmp(subcmd, "close")) {
		ndk_st_close_streaming();
		return;
	}
	else if (!strcmp(subcmd, "mon")) {
		ndk_st_start_monitor();
	}
	else if (!stmgr) {
		printf("no stream mgr\n");
		return;
	}
	else if (!strcmp(subcmd, "dump")) {
		recutil_stmgr_dump(stmgr);
	}
	else if (!strcmp(subcmd, "1+")) {
		err = recutil_start_stream(rtp_stctx->recutil, SP5K_MEDIA_REC_STREAM_1ST, NULL);
	}
	else if (!strcmp(subcmd, "1-")) {
		err = recutil_stop_stream(rtp_stctx->recutil, SP5K_MEDIA_REC_STREAM_1ST);
	}
	else if (!strcmp(subcmd, "2+")) {
		err = recutil_start_stream(rtp_stctx->recutil, SP5K_MEDIA_REC_STREAM_2ND, NULL);
	}
	else if (!strcmp(subcmd, "2-")) {
		err = recutil_stop_stream(rtp_stctx->recutil, SP5K_MEDIA_REC_STREAM_2ND);
	}
	else if (!strcmp(subcmd, "all+")) {
		err = recutil_start_stream(rtp_stctx->recutil, SP5K_MEDIA_REC_STREAMS_ALL, NULL);
	}
	else if (!strcmp(subcmd, "all-")) {
		err = recutil_stop_stream(rtp_stctx->recutil, SP5K_MEDIA_REC_STREAMS_ALL);
	}
	else if (argc == 2 && !strcmp(subcmd, "1*")) {
		multi_start_stop_test = TRUE;
		stream_bits = SP5K_MEDIA_REC_STREAM_1ST;
	}
	else if (argc == 2 && !strcmp(subcmd, "2*")) {
		multi_start_stop_test = TRUE;
		stream_bits = SP5K_MEDIA_REC_STREAM_2ND;
	}
	else if (argc == 2 && !strcmp(subcmd, "all*")) {
		multi_start_stop_test = TRUE;
		stream_bits = 0xFF;
	}
	else if (argc == 2 && !strcmp(subcmd, "hbr")) {
		sp5kSystemCfgSet(SP5K_MEDIA_SYS_CFG, SP5K_MEDIA_SYS_CFG_H264_DYNAMIC_BITRATE,
			(UINT32)atoi(argv[0]), (UINT32)kmgstr_to_long(argv[1]));
	}
	else {
		printf("Unknow Command\n");
		goto lUsage;
	}

	if (multi_start_stop_test) {
		int secs = atoi(argv[0]);
		int cnt = atoi(argv[1]);
		int i;

		for (i = 0; i < cnt; ++i) {
			recutil_start_stream(rtp_stctx->recutil, stream_bits, NULL);
			sp5kTimeDelay(SP5K_TIME_DELAY_1MS, secs * 1000);
			recutil_stop_stream(rtp_stctx->recutil, stream_bits);
		}
	}

	if (err < 0)
		printf("err = %d\n", err);

	return;

lUsage:
	printf("sub-commands:\n");
	printf("  help\n");
	printf("  mon\n");
	printf("  close\n");
	printf("  1+ : start stream 1\n");
	printf("  1- : stop stream 1\n");
	printf("  2+ : start stream 2\n");
	printf("  2- : stop stream 2\n");
	printf("  all+ : start all streams\n");
	printf("  all- : stop all streams\n");
	printf("  1* secs cnt: start/stop stream 1 test\n");
	printf("  2* secs cnt: start/stop stream 1 test\n");
	printf("  all* secs cnt: start/stop all streams test\n");
}

///////////////////////////////////////////////////////////////////////////////
// RTP Playback
static NDKMediaCliHandle rtpcli_handle = NULL;
static NDKMediaCliAVSink *rtpcli_avsink = NULL;
static BOOL rtpcli_pcap = FALSE;

const char *rtpcli_msg2str(NDKMediaCliRtpMsg msg)
{
	switch (msg) {
	case NDK_MEDIACLI_RTP_MSG_STARTED: return "started";
	case NDK_MEDIACLI_RTP_MSG_STOPPED: return "stopped";

	case NDK_MEDIACLI_RTP_MSG_EGENERIC: return "e-generic";
	case NDK_MEDIACLI_RTP_MSG_ECONN: return "e-conn";
	case NDK_MEDIACLI_RTP_MSG_TIMEOUT: return "e-timeout";
	default: return "unknow";
	}
}

static void rtpcli_msg_handler(NDKMediaCliHandle h, NDKMediaCliRtpMsg msg, long param)
{
	printf("RTPCLi msg: %s %ld\n", rtpcli_msg2str(msg), param);

	switch (msg) {
	case NDK_MEDIACLI_RTP_MSG_STARTED:
	{
		unsigned int width = 0xFFFF, height = 0xFFFF;
		rtpcli_avsink->pf_get_attr(rtpcli_avsink, SP5K_MEDIA_ATTR_WIDTH, &width);
		rtpcli_avsink->pf_get_attr(rtpcli_avsink, SP5K_MEDIA_ATTR_HEIGHT, &height);
		printf("START RTP Play: %u %u\n", width, height);

		//sp5kDispAttrSet(SP5K_DISP_CHNL_1, SP5K_DISP_IMG_WINDOW, 0, 0, width, height);
		//sp5kDispAttrSet(SP5K_DISP_CHNL_1, SP5K_DISP_OSD_WINDOW,0, 0, width, height);
		break;
	}
	case NDK_MEDIACLI_RTP_MSG_STOPPED:
	case NDK_MEDIACLI_RTP_MSG_EGENERIC:
	case NDK_MEDIACLI_RTP_MSG_TIMEOUT:
	case NDK_MEDIACLI_RTP_MSG_ECONN:
		if (h) {
			printf("Destroy MediaCli\n");
			ndk_mediacli_destroy(h);
			rtpcli_handle = NULL;
		}
		break;
	default:
		break;
	}
}

static void rtpcli_control()
{
	int c, position = 0, resumed = 0;

	printf("space: resume/pause\n");
	printf("f, F: forward 10.0/30.0 seconds\n");
	printf("b, B: backward 10.0/30.0 seconds\n");
	printf("q : quit\n\n");

	for (;;) {
		c = getch();

		switch (c) {
		case 'q':
			return;
		case ' ':
			resumed = 1 - resumed;
			sp5kMediaPlayControl(resumed ? SP5K_MEDIA_PLAY_RESUME : SP5K_MEDIA_PLAY_PAUSE, 0, 0);
			break;
		case 'f': position = 10 * 1000; break;
		case 'F': position = 30 * 1000; break;
		case 'b': position = -10 * 1000; break;
		case 'B': position = -30 * 1000; break;
		default:
			break;
		}

		switch (c) {
		case 'f': case 'F': case 'b': case 'B': {
			int elapsed;
			sp5kMediaPlayAttrGet(SP5K_MEDIA_ATTR_ELAPSED_TIME, (UINT32*)&elapsed);

			position = elapsed + position;
			if (position < 0)
				position = 0;

			printf("SEEK: %d => %d\n", elapsed, position);
			sp5kMediaPlayControl(SP5K_MEDIA_PLAY_SEEK, (UINT32)position, 0);
			sp5kMediaPlayControl(SP5K_MEDIA_PLAY_RESUME, 0, 0);
			break; }
		default:
			break;
		}
	}
}

#define STATS_(v, n)  v[NDK_MEDIACLI_AVSNK_STAT_##n]

void rtpcli_mon_thread(ULONG arg)
{
	BOOL *abort = (BOOL*)arg;
	UINT32 bak[NDK_MEDIACLI_AVSNK_STAT_NUM], stats[NDK_MEDIACLI_AVSNK_STAT_NUM];
	UINT32 cnt;

	memset(bak, 0, sizeof(bak));

	printf("ct=cache time, df=drop frames, fr=frame rate, la=late, Dl=delay, Pb=playback\n");
	for (cnt = 0; !*abort; ++cnt) {
		if (!rtpcli_handle) {
			sp5kTimeDelay(SP5K_TIME_DELAY_1MS, 100);
			continue;
		}

		if ((cnt % 20) == 0) {
			/* Title */
			printf("\n No. cach A<BR  FR FD La TDly> V<BR  FR FD La TDly>\n");
			printf(  "---- ---- ---- --- -- -- ---- ----- --- -- -- ----\n");
		}

		rtpcli_avsink->pf_control(rtpcli_avsink, NDK_MEDIACLI_AVSNK_CTRL_GET_STATS, NDK_MEDIACLI_AVSNK_STAT_NUM, stats);

		printf("%4u %4d "
		       "%4d %3u %2u %2u %4d "
		       "%5d %3u %2u %2u %4d\n",
			cnt, STATS_(stats, CACHE_TIME),

			(STATS_(stats, AUD_BYTES) - STATS_(bak, AUD_BYTES)) >> 10,
			STATS_(stats, AUD_FRAME_TOTAL) - STATS_(bak, AUD_FRAME_TOTAL),
			STATS_(stats, AUD_FRAME_DROPPED) - STATS_(bak, AUD_FRAME_DROPPED),
			STATS_(stats, AUD_FRAME_LATE) - STATS_(bak, AUD_FRAME_LATE),
			STATS_(stats, AUD_AVG_DELAY_TIME),

			(STATS_(stats, VID_BYTES) - STATS_(bak, VID_BYTES)) >> 10,
			STATS_(stats, VID_FRAME_TOTAL) - STATS_(bak, VID_FRAME_TOTAL),
			STATS_(stats, VID_FRAME_DROPPED) - STATS_(bak, VID_FRAME_DROPPED),
			STATS_(stats, VID_FRAME_LATE) - STATS_(bak, VID_FRAME_LATE),
			STATS_(stats, VID_AVG_DELAY_TIME));

		memcpy(bak, stats, sizeof(bak));
		sp5kTimeDelay(SP5K_TIME_DELAY_1MS, 1000);
	}

	*abort = 0;
}

static void rtpcli_monitor()
{
	if (rtpcli_handle == NULL)
		return;

	int abort = 0;

	SP5K_THREAD* tid = sp5kOsThreadCreate((char*)"rtpmon", rtpcli_mon_thread, (ULONG)&abort, 7, 0, 2048, TX_AUTO_START);
	getch();
	abort = 1;

	while (abort)
		sp5kTimeDelay(SP5K_TIME_DELAY_1MS, 50);

	sp5kOsThreadDelete(tid);
}

static inline void rtpcli_opt_set(NDKMediaCliOpt *opt, long id, long value) { opt->id = id; opt->u.l = value; }

static void cmd_rtpcli_main(int argc, char *argv[])
{
	if (argc < 1)
		return;

	NDKMediaCliOpt cliopts[12];
	NDKMediaCliAVSink *avsink = NULL;
	double starttime = 0.0;
	const char *url = NULL;
	int i, err, optnum = 0;
	BOOL pcap = FALSE;
	const char *mstr, *sarg;

	for (i = 0; i < argc; ++i) {
		sarg = argv[i];
		if (!strcmp(sarg, "-h")) {
			goto lUsage;
		}
		else if (!strcmp(sarg, "-C")) {
			if (rtpcli_handle)
				rtpcli_control();
			return;
		}
		else if (!strcmp(sarg, "-t")) {
			ndk_mediacli_stop(rtpcli_handle);
			return;
		}
		else if (!strcmp(sarg, "-x")) {
			ndk_mediacli_destroy(rtpcli_handle);
			if (rtpcli_pcap) ndk_netif_ioctl(NDK_IOCS_IF_PCAP_STOP, (long)ifname_get(), 0);
			rtpcli_handle = NULL;
			rtpcli_avsink = NULL;
			rtpcli_pcap = FALSE;
			return;
		}
		else if (!strcmp(sarg, "-m")) {
			rtpcli_monitor();
			return;
		}
		else if (!strcmp(sarg, "-nsink")) {
			avsink = ndk_mediacli_nullsink_create();
		}
		else if (!strcmp(sarg, "-pbsink")) {
			avsink = ndk_mediacli_playback_sink_create();
		}
		else if (!strncmp(sarg, "-O", 2)) {
			mstr = &sarg[2];

			if (recutil_token_vcompare(&mstr, "tcp", "nto", "na", "nv", "pcap", NULL)) {
				switch (recutil_token_index()) {
				case 0: rtpcli_opt_set(&cliopts[optnum++], NDK_MEDIACLI_RTP_OPT_TCP, 1); break;
				case 1: rtpcli_opt_set(&cliopts[optnum++], NDK_MEDIACLI_RTP_OPT_TIMEOUT, 0); break;
				case 2: rtpcli_opt_set(&cliopts[optnum++], NDK_MEDIACLI_RTP_OPT_VIDEO_ONLY, 1); break;
				case 3: rtpcli_opt_set(&cliopts[optnum++], NDK_MEDIACLI_RTP_OPT_AUDIO_ONLY, 1); break;
				case 4: pcap = TRUE; break;
				default: goto lUsage;
				}
			}
			else if (recutil_token_compare(&mstr, "ct=")) {
				rtpcli_opt_set(&cliopts[optnum++], NDK_MEDIACLI_RTP_OPT_CACHETIME, atoi(recutil_token_value()));
			}
			else if (recutil_token_compare(&mstr, "avsync=")) {
				rtpcli_opt_set(&cliopts[optnum++], NDK_MEDIACLI_RTP_OPT_AVSYNCTIME, atoi(recutil_token_value()));
			}
			else if (!recutil_token_compare(&mstr, "gop=")) {
				rtpcli_opt_set(&cliopts[optnum++],
					NDK_MEDIACLI_RTP_OPT_MEDIA_ATTR + SP5K_MEDIA_ATTR_H264_GOP_NO,
					atoi(recutil_token_value()));
			}
			else if (recutil_token_compare(&mstr, "starttime=")) {
				starttime = strtod(recutil_token_value(), NULL);
			}
			else
				goto lUsage;
		}
		else if (!strncmp(sarg, "-u", 2)) {
			const char *s = &sarg[2];

			if (!strcmp(s, "h264"))
				url = "rtsp://192.168.1.1/h264";
			else if (!strcmp(s, "mjpg"))
				url = "rtsp://192.168.1.1/mjpg";
			else
				url = s;
		}
		else
			goto lUsage;
	}

	if (rtpcli_handle) {
		printf("rtpcli is already started\n");
		return;
	}

	if (!url || !avsink) {
		printf("No URL or AVSink\n");
		return;
	}

	rtpcli_avsink = avsink;
	rtpcli_pcap = pcap;

	if (rtpcli_pcap) {
		long *caplen = (long*)0xFFFF;
		ndk_netif_ioctl(NDK_IOCS_IF_PCAP_START, (long)ifname_get(), caplen);
	}

	printf("URL=%s\n", url);
	err = ndk_mediacli_rtp_init(&rtpcli_handle, url, avsink, rtpcli_msg_handler, optnum, cliopts);

	if (err == 0) {
		NDKMediaCliOpt snkopts[1];
		memset(snkopts, 0, sizeof(snkopts));
		snkopts[0].id  = NDK_MEDIACLI_OPT_STARTTIME;
		snkopts[0].u.d = starttime;
		ndk_mediacli_start(rtpcli_handle, NDK_MEDIACLI_OPT_NUM, snkopts);
	}
	else {
		ndk_mediacli_destroy(rtpcli_handle);
		if (rtpcli_pcap) ndk_netif_ioctl(NDK_IOCS_IF_PCAP_STOP, (long)ifname_get(), 0);
		rtpcli_handle = NULL;
		rtpcli_avsink = NULL;
		rtpcli_pcap = FALSE;
		ndk_error("init error %d", err);
	}

	return;

lUsage:
	printf("rtp [options or controls] -u RTSP-URL\n");
	printf("Controls:\n");
	printf("  -h : help\n");
	printf("  -C : enter control mode\n");
	printf("  -t : stop\n");
	printf("  -d : destroy\n");
	printf("Options for RTP Client:\n");
	printf("  -nsnk : NULL SINK\n");
	printf("  -pbsnk : Playback SINK\n");
	printf("  -Hgop=###: H24 gop size.\n");
	printf("  -Hgop=###: H24 gop size.\n");
	printf("  -Otcp: Use TCP protocol.\n");
	printf("  -sm.n: start play at time m.n seconds\n");
	printf("  -idemux|null: sink type\n");
	printf("  -Acorr=###: audio pts correction time in ms\n");
}

///////////////////////////////////////////////////////////////////////////////
// Streaming CGI
// do/mattset?sid=n[&location=uri]&vw=nnn&vh=nnn&v=p30&a=16bit&a=2ch&asr=44100

#ifdef NDK_CONFIG_HTTPD
#define FCGI_ST_TERMINATE(inst, errstr)  do { ndk_cgi_terminate(inst, 400, errstr); return; } while(0)

struct FCgiStAttSetPriv {
	char   buf[1024];
};

// set: sid=0&vcodec=H264&w=640&h=360&br=4000000&fr=30
static void fcgi_statt_set(NDKCgiInstance inst, struct FCgiStAttSetPriv *priv)
{
	UINT idx;
	char *val, *loc = NULL;
	const char *name, *name1, *v_codec;
	char klass, v_class, buf[64];
	int  err, sid = 0;

	if (!rtp_stctx)
		FCGI_ST_TERMINATE(inst, "Media not defined.");

	if (ndk_st_is_streaming_started())
		FCGI_ST_TERMINATE(inst, "Media server has started.");

	if (ndk_cgi_arg_get_string(inst, "sid"))
		sid = strtol(ndk_cgi_arg_get_string(inst, "sid"), NULL, 0);
	if (sid >= SP5K_MEDIA_REC_STREAM_NUM)
		FCGI_ST_TERMINATE(inst, "Bad or No StreamId");

	loc = ndk_cgi_arg_get_string(inst, "location");

	v_codec = ndk_cgi_arg_get_string(inst, "vcodec");
	if (!v_codec)
		FCGI_ST_TERMINATE(inst, "No Video Codec");

	if (!strcasecmp(v_codec, "H264"))
		v_class = 'H';
	else if (!strcasecmp(v_codec, "MJPG"))
		v_class = 'J';
	else
		FCGI_ST_TERMINATE(inst, "Invalid Video Codec");

	for (idx = 0; ndk_cgi_arg_get(inst, idx, &name, &val); ++idx) {
		if (!strcasecmp(name, "sid") || !strcasecmp(name, "location") || !strcasecmp(name, "vcodec"))
			continue;

		name1 = name + 1;
		err   = 0;

		switch (*name) {
		case 'v': klass = v_class; break;
		case 'a': klass = 'A'; break;
		default: continue;
		}

		if (*name1 == 0) {
			recutil_config_media(rtp_stctx->recutil, sid, klass, val, TRUE);
		}
		else {
			sprintf(buf, "%s=%s", name1, val);
			recutil_config_media(rtp_stctx->recutil, sid, klass, buf, TRUE);
		}

		if (err != 0) {
			ndk_error("err=%d", err);
			FCGI_ST_TERMINATE(inst, "Attribute Error");
		}
	}

	ndk_cgi_put_status(inst, 200, TRUE, "text/plain", 0, FALSE);
	if (loc)
		ndk_cgi_printf(inst, "Location: %s\r\n", loc);
	ndk_cgi_puts(inst, "\r\n");
	ndk_cgi_flush(inst);
	ndk_cgi_set_finished(inst);
}

static void fcgi_statt_set_main(NDKCgiInstance inst, void *udata, UINT flags, void *priv)
{
	fcgi_statt_set(inst, (struct FCgiStAttSetPriv*)priv);
}

int fcgi_streaming_server_start()
{
	if (rtp_stctx)
		ndk_err_return(-1);

	RTPStCtx *rtpctx = rtp_stctx_create(rtp_handle_events);

	// default media attributes
	cmdProcess((char*)"net recstart -Hfr=30,br=4000000,hd -R -n -Hfr=30,br=4000000,hd -R");
	rtpctx->cfg.st_mode = NDK_ST_MULTI_STREAM;

	if (ndk_st_start_server(&rtpctx->cfg) == 0) {
		rtp_stctx = rtpctx;
		return 0;
	}

	printf("start mediasrv failed\n");
	cmdProcess((char*)"net recstop");
	ndk_mem_free(rtpctx);
	return -1;
}

void cmd_stcgi_main(int argc, char *argv[])
{
	static BOOL run = FALSE;
	if (!run) {
		ndk_cgi_register_fast("stattset", sizeof(struct FCgiStAttSetPriv), fcgi_statt_set_main, NULL);
		fcgi_streaming_server_start();
		run = TRUE;
	}
}
#endif

