/**************************************************************************
 *                                                                        *
 *         Copyright (c) 2014 by iCatch Technology Co., Ltd.              *
 *                                                                        *
 *  This software is copyrighted by and is the property of Sunplus        *
 *  Technology Co., Ltd. All rights are reserved by Sunplus Technology    *
 *  Co., Ltd. This software may only be used in accordance with the       *
 *  corresponding license agreement. Any unauthorized use, duplication,   *
 *  distribution, or disclosure of this software is expressly forbidden.  *
 *                                                                        *
 *  This Copyright notice MUST not be removed or modified without prior   *
 *  written consent of Sunplus Technology Co., Ltd.                       *
 *                                                                        *
 *  Sunplus Technology Co., Ltd. reserves the right to modify this        *
 *  software without notice.                                              *
 *                                                                        *
 *  Sunplus Technology Co., Ltd.                                          *
 *  19, Innovation First Road, Science-Based Industrial Park,             *
 *  Hsin-Chu, Taiwan, R.O.C.                                              *
 *                                                                        *
 **************************************************************************/

#include <livePort/taskqueue.h>
#include <livePort/livest_internal.h>

NSTaskQueue::NSTaskQueue(int dataSize, int queSize)
: fInited(false)
, fFrozen(false)
, fDataSize(dataSize)
, fMaxEntryNum(queSize)
, fEntryNum(0)
, fEntryHead(NULL)
, fBlkPool(0)
, fTotalTime(0)
, fTimeLastSync(-1)
, fEvtFlags(0)
, fMutex(0)
, fThrId(0)
{
	if (fMaxEntryNum <= 0)
		fMaxEntryNum = 32;
	if (fDataSize < 0)
		fDataSize = 0;
}

NSTaskQueue::~NSTaskQueue()
{
	deinitialize();
}

bool NSTaskQueue::initialize()
{
	if (fInited)
		return false;

	if (sp5kOsMutexCreate(&fMutex, (char*)"NSTaskQ", 0) != SUCCESS)
		return false;

	if (sp5kOsEventFlagsCreate(&fEvtFlags, (char*)"NSTaskQ") != SUCCESS)
		return false;

	unsigned entry_size = sizeof(TaskEntry) + fDataSize;
	entry_size = NDK_ALIGN_TO(entry_size, 4);

	fBlkPool = ndk_blkpool_create("NSTaskQ", sizeof(TaskEntry) + fDataSize, (unsigned)fMaxEntryNum);
	if (!fBlkPool)
		return false;

	fThrId = ndk_thread_create("NSTaskQ", ndk_thread_get_priority(), scheTaskFunc0, (UINT32)this, 32*1024 - 256);
	if (!fThrId)
		return false;

	return fInited = true;
}

void NSTaskQueue::deinitialize()
{
	if (fThrId) {
		if (fEvtFlags) {
			ULONG cur_flags;
			sp5kOsEventFlagsSet(&fEvtFlags, SCHE_ABORT, TX_OR);
			sp5kOsEventFlagsGet(&fEvtFlags, SCHE_ABORTED, TX_OR, &cur_flags, TX_WAIT_FOREVER);
		}
		ndk_thread_destroy(fThrId);
		fThrId = 0;
	}

	if (fMutex)
		sp5kOsMutexGet(&fMutex, TX_WAIT_FOREVER);

	if (fEvtFlags) {
		sp5kOsEventFlagsDelete(&fEvtFlags);
		fEvtFlags = 0;
	}

	if (fBlkPool) {
		ndk_blkpool_destroy(fBlkPool);
		fBlkPool = 0;
	}

	if (fMutex) {
		sp5kOsMutexPut(&fMutex);
		sp5kOsMutexDelete(&fMutex);
		fMutex = 0;
	}

	fInited = false;
}

bool NSTaskQueue::showHeadTask(void **data, int *timeLeft)
{
	bool r = false;
	sp5kOsMutexGet(&fMutex, TX_WAIT_FOREVER);
	if (fEntryHead) {
		*data = (void*)(fEntryHead + 1);
		*timeLeft = fEntryHead->timeLeft;
		r = true;
	}
	sp5kOsMutexPut(&fMutex);

	return r;
}

void *NSTaskQueue::pushTask(NSTaskQueue::TaskHandler *handler, void *data, int delayTime, bool backIfEqual)
{
	TaskEntry **pp_cur, *p_new;
	bool wakeup = false;

	if (!(p_new = (TaskEntry*)ndk_blkpool_alloc_block(fBlkPool)))
		ndk_err_return(NULL);

	if (delayTime < 0)
		delayTime = 0;

	p_new->magicNumber = 0xbeafbeaf;
	p_new->next = NULL;
	p_new->handler = handler;
	if (fDataSize)
		memcpy(p_new + 1, data, fDataSize);
	p_new->timeLeft = delayTime;
	//profLogPrintf(0, "push %p %d", p_new, delayTime);

	synchronize();
	sp5kOsMutexGet(&fMutex, TX_WAIT_FOREVER);

	if (fTotalTime < delayTime)
		fTotalTime = delayTime;

	pp_cur = &fEntryHead;
	while (1) {
		if (*pp_cur) {
			TaskEntry *p_cur = *pp_cur;

			if ((p_new->timeLeft > p_cur->timeLeft) ||
			    ((p_new->timeLeft == p_cur->timeLeft) && backIfEqual))
			{
				p_new->timeLeft -= p_cur->timeLeft;
			}
			else {
				p_cur->timeLeft -= p_new->timeLeft;

				p_new->next = p_cur;
				*pp_cur = p_new;
				break;
			}

			pp_cur = &p_cur->next;
		}
		else {
			*pp_cur = p_new;
			break;
		}
	}

	++fEntryNum;
	if (p_new == fEntryHead)
		wakeup = true;
	sp5kOsMutexPut(&fMutex);

	if (wakeup)
		sp5kOsEventFlagsSet(&fEvtFlags, SCHE_WAKEUP, TX_OR);

	return p_new;
}

void NSTaskQueue::clearTasks()
{
	TaskEntry *p_entry;

	while ((p_entry = popTaskEntry(true)) != NULL) {
		p_entry->handler(p_entry + 1, TASK_REMOVED);
		ndk_blkpool_free_block(fBlkPool, p_entry);
	}
}

void NSTaskQueue::freeze(bool froze)
{
	sp5kOsMutexGet(&fMutex, TX_WAIT_FOREVER);
	if (!froze)
		fTimeLastSync = -1;
	fFrozen = froze;
	sp5kOsMutexPut(&fMutex);
}

NSTaskQueue::TaskEntry *NSTaskQueue::popTaskEntry(bool force)
{
	TaskEntry *p_entry = NULL;

	sp5kOsMutexGet(&fMutex, TX_WAIT_FOREVER);
	if (fEntryHead) {
		NDK_ASSERT(fEntryHead->timeLeft >= 0);
		NDK_ASSERT(fEntryNum > 0);

		if (fEntryHead->timeLeft == 0 || force) {
			p_entry = fEntryHead;
			fEntryHead = p_entry->next;
			--fEntryNum;
		}
	}
	else {
		NDK_ASSERT(fEntryNum == 0);
	}
	sp5kOsMutexPut(&fMutex);

	if (p_entry) {
		NDK_ASSERT(p_entry->magicNumber == 0xbeafbeaf);
		//profLogPrintf(0, "pop  %p", p_entry);
	}

	return p_entry;
}

int NSTaskQueue::getTotalDelayTime() const
{
	return fTotalTime;
}

void NSTaskQueue::synchronize()
{
	sp5kOsMutexGet(&fMutex, TX_WAIT_FOREVER);

	SINT64 timeNow = mtime_now_get();

	if (fTimeLastSync == -1)
		fTimeLastSync = timeNow;

	int timeElapsed = timeNow - fTimeLastSync;
	if (fEntryHead) {
		//profLogPrintf(0, "sync %d %d", fEntryHead->timeLeft, timeElapsed);
	}

	fTotalTime -= timeElapsed;
	if (fTotalTime < 0)
		fTotalTime = 0;

	for (TaskEntry *p_cur = fEntryHead; p_cur && timeElapsed > 0; p_cur = p_cur->next) {
		if (p_cur->timeLeft >= timeElapsed) {
			p_cur->timeLeft -= timeElapsed;
			timeElapsed = 0;
		}
		else {
			timeElapsed -= p_cur->timeLeft;
			p_cur->timeLeft = 0;
		}
	}

	fTimeLastSync = timeNow;
	sp5kOsMutexPut(&fMutex);
}

void NSTaskQueue::scheTaskFunc()
{
	ULONG cur_flags = 0;
	TaskEntry *p_entry;
	int delay_time;

	while (!(cur_flags & SCHE_ABORT)) {
		if (fFrozen) {
			ndk_msleep(10);
			continue;
		}

		synchronize();

		sp5kOsMutexGet((SP5K_MUTEX*)&fMutex, TX_WAIT_FOREVER);
		delay_time = fEntryHead ? fEntryHead->timeLeft : 1000;
		sp5kOsMutexPut((SP5K_MUTEX*)&fMutex);

		if (delay_time == 0) {
			while (!fFrozen && (p_entry = popTaskEntry(false)) != NULL) {
				p_entry->handler(p_entry + 1, TASK_TIMEOUT);
				ndk_blkpool_free_block(fBlkPool, p_entry);
			}
			continue;
		}

		cur_flags = 0;
		sp5kOsEventFlagsClear(&fEvtFlags, SCHE_WAKEUP);
		sp5kOsEventFlagsGet(&fEvtFlags, SCHE_WAKEUP | SCHE_ABORT, TX_OR, &cur_flags, (ULONG)delay_time);
	}

	clearTasks();
	sp5kOsEventFlagsSet(&fEvtFlags, SCHE_ABORTED, TX_OR);
}

