/**************************************************************************
 *
 *        Copyright (c) 2002-2009 by Sunplus mMedia Inc., Ltd.
 *
 *  This software is copyrighted by and is the property of Sunplus
 *  mMedia Inc., Ltd. All rights are reserved by Sunplus mMedia
 *  Inc., Ltd. This software may only be used in accordance with the
 *  corresponding license agreement. Any unauthorized use, duplication,
 *  distribution, or disclosure of this software is expressly forbidden.
 *
 *  This Copyright notice MUST not be removed or modified without prior
 *  written consent of Sunplus mMedia Inc., Ltd.
 *
 *  Sunplus mMedia Inc., Ltd. reserves the right to modify this
 *  software without notice.
 *
 *  Sunplus mMedia Inc., Ltd.
 *  19-1, Innovation First Road, Science-Based Industrial Park,
 *  Hsin-Chu, Taiwan.
 *
 **************************************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <general.h>
#include <cmd.h>
#include <debug_mod/dbg_api.h>

#include <mmc_sdio_func.h>
#include <mmc_card.h>
#include <mmc_host.h>
#include <hal/hal_peripheral_sdio.h>
#include <hal/hal_gpio.h>
#include <regmap/reg_flash.h>
#include <sp5k_os_api.h>
#include <middleware/peripherals.h>

/**************************************************************************
 *                           C O N S T A N T S                            *
 **************************************************************************/
#define HELP_SDIO_INIT \
	"Initialize sdio.\n" \
	"  sdioinit < freq(hz) >\n"

/**************************************************************************
 *                              M A C R O S                               *
 **************************************************************************/
#define SDIO_VENDOR_ID_REALTEK         0x024c
#define SDIO_VENDOR_ID_BROADCOM        0x02d0

/**************************************************************************
 *                          D A T A    T Y P E S                          *
 **************************************************************************/
typedef struct SDIO_CCCR
{
    UINT8    SDIO_CCCR_Revision;
    UINT8    SD_Spec_Revision;
    UINT8    IO_Enable;
    UINT8    IO_Ready;
    UINT8    INT_Enable;
    UINT8    INT_Pending;
    UINT8    IO_Abort;
    UINT8    Bus_If_Control;
    UINT8    Card_Capability;
    UINT8    Common_CIS_PointerL;
    UINT8    Common_CIS_PointerM;
    UINT8    Common_CIS_PointerH;
    UINT8    Bus_Suspend;
    UINT8    Function_Select;
    UINT8    Exec_Flags;
    UINT8    Ready_Flags;
    UINT16   FN0_Block_size;
    UINT8    Power_Control;
    UINT8    Bus_Speed_Select;
    UINT8    UHSI_Support;
    UINT8    Driver_Strength;
    UINT8    Interrupt_Extension;
    UINT8    RFU[220];
    UINT8    Vendor_Info[16];
} SDIO_CCCR;

/**************************************************************************
 *                         G L O B A L    D A T A                         *
 **************************************************************************/

/**************************************************************************
 *                 E X T E R N A L    R E F E R E N C E S                 *
 **************************************************************************/
extern int sdhc_initialize(char second_sdio, UINT32 f_max);
extern int halSdDummyTx(void);

/**************************************************************************
 *               F U N C T I O N    D E C L A R A T I O N S               *
 **************************************************************************/
static int sdio_probe(struct sdio_func *func, const struct sdio_device_id *id);

static void cmd_sdio_init(int argc, char *argv[]);
static void cmd_sdio_deinit(int argc, char *argv[]);
static void cmd_sdio_cccr_dump(int argc, char *argv[]);
static void cmd_sdio_io(int argc, char *argv[]);
static void cmd_sdio_detect(int argc, char *argv[]);
static void cmd_sdio_pinmux(int argc, char *argv[]);
static void cmd_sdio_phase_sel(int argc, char *argv[]);
static void cmd_sdio_wcheck(int argc, char *argv[]);
static void cmdSdioPerf(int argc, char *argv[]);

/**************************************************************************
 *                          L O C A L    D A T A                          *
 **************************************************************************/
static cmd_t cmdSdioArray[] = {
	{"sdioinit",   cmd_sdio_init,       HELP_SDIO_INIT, 0x0},
	{"sdioexit",   cmd_sdio_deinit,     "", 0x0},
	{"sdiocccr",   cmd_sdio_cccr_dump,  "", 0x0},
	{"sdiocmd",    cmd_sdio_io,         "", 0x0},
	{"sdiodet",    cmd_sdio_detect,     "", 0x0},
	{"sdiopin",    cmd_sdio_pinmux,     "", 0x0},
	{"sdiowck",    cmd_sdio_wcheck,     "", 0x0},
	{"sdiops",     cmd_sdio_phase_sel,  "", 0x0},
	{"sdioperf",   cmdSdioPerf,         "",  0x0},
	{NULL,    NULL,    NULL,    NULL}
};

static struct sdio_device_id sdio_ids[] = {
	{ SDIO_DEVICE(0, 0) },
};

static struct sdio_driver sdio_driver = {
	.name       = "sdio_test",
	.probe      = sdio_probe,
};

static int sdio_probe(struct sdio_func *func, const struct sdio_device_id *id)
{
	sdio_set_block_size(func, 32);
	return 0;
}

static void sdio_ioreset(struct sdio_func *func)
{
	/* IO reset */
	int i, ret;
	for ( i = 100; i > 0; i-- ) { /* Tx 80 clock for power on sequence */
		ret = halSdDummyTx();
		if ( ret != SUCCESS ) {
			printf("[%s] ERROR\n", __FUNCTION__);
		}
	}

	sdio_claim_host(func);
	sdio_writeb(func, (1 << 3), 0x6, &ret);
	sdio_release_host(func);
}

static void sdio_cccr_dump(void)
{
	char *value;
	unsigned int addr;
	int result = 0;
	SDIO_CCCR   cccr;
	struct sdio_func *func = sdioFuncGet(0);

	if (!func) {
		printf("func0 is NULL\n");
		return;
	}

	for (addr=0; addr < sizeof(cccr); addr++)
	{
		sdio_claim_host(func);
		value = ((char *)&cccr)+addr;
		*value = sdio_f0_readb(func, addr, &result);
		sdio_release_host(func);

		if (result != 0)
		{
			printf("read error: %d\n", result);
			return;
		}
	}

	printf("====== Dump CCCR Register ======\n");
	printf("SDIO_CCCR_Revision     = %02X\n", cccr.SDIO_CCCR_Revision);
	printf("SD_Spec_Revision       = %02X\n", cccr.SD_Spec_Revision);
	printf("IO_Enable              = %02X\n", cccr.IO_Enable);
	printf("IO_Ready               = %02X\n", cccr.IO_Ready);
	printf("INT_Enable             = %02X\n", cccr.INT_Enable);
	printf("INT_Pending            = %02X\n", cccr.INT_Pending);
	printf("IO_Abort               = %02X\n", cccr.IO_Abort);
	printf("Bus_If_Control         = %02X\n", cccr.Bus_If_Control);
	printf("Card_Capability        = %02X\n", cccr.Card_Capability);
	printf("Common_CIS_PointerL    = %02X\n", cccr.Common_CIS_PointerL);
	printf("Common_CIS_PointerM    = %02X\n", cccr.Common_CIS_PointerM);
	printf("Common_CIS_PointerH    = %02X\n", cccr.Common_CIS_PointerH);
	printf("Bus_Suspend            = %02X\n", cccr.Bus_Suspend);
	printf("Function_Select        = %02X\n", cccr.Function_Select);
	printf("Exec_Flags             = %02X\n", cccr.Exec_Flags);
	printf("Ready_Flags            = %02X\n", cccr.Ready_Flags);
	printf("FN0_Block_size         = %04X\n", cccr.FN0_Block_size);
	printf("Power_Control          = %02X\n", cccr.Power_Control);
	printf("Bus_Speed_Select       = %02X\n", cccr.Bus_Speed_Select);
	printf("UHSI_Support           = %02X\n", cccr.UHSI_Support);
	printf("Driver_Strength        = %02X\n", cccr.Driver_Strength);
	printf("Interrupt_Extension    = %02X\n", cccr.Interrupt_Extension);
	printf("RFU[220]               = \n"); memdump((void*) &cccr.RFU, sizeof(cccr.RFU));
	printf("Vendor_Info[16]        = \n"); memdump((void*) &cccr.Vendor_Info, sizeof(cccr.Vendor_Info));
}

static int sdio_init(unsigned int clkFreq)
{
	sdio_ids[0].class = SDIO_ANY_ID;
	sdio_ids[0].vendor = SDIO_ANY_ID;
	sdio_ids[0].device = SDIO_ANY_ID;

	sdio_driver.id_table = sdio_ids;

	if (sdhc_initialize(1, clkFreq) != 0) {
		printf("sdhc init fail\n");
		return -1;
	}

	if (sdio_register_driver(&sdio_driver) != 0) {
		printf("sdio driver register error\n");
		return -1;
	}

	return 0;
}

static int sdio_deinit(void)
{
	struct mmc_card *card;
	struct sdio_func *func = sdioFuncGet(0);

	if (!func) {
		return -1;
	}

	sdio_ioreset(func);
	sdio_unregister_driver(&sdio_driver);

	card = func->card;
	mmc_remove_host(card->host);

	return 0;
}

static void sdio_cmd52w_word(struct sdio_func *func, unsigned int w, unsigned int addr, int *err_ret)
{
	UINT32 value;
	UINT32 i;

	if (!func) {
		return;
	}

	for (i = 0; i < sizeof(UINT32); i++) {
		value = (w >> (i*8)) & 0xFF;
		sdio_writeb(func, value, addr + i, err_ret);
		printf("CMD52w addr:0x%08x = 0x%08x res:0x%x\n", addr, value, *err_ret);
	}
}

static int sdio_cmd52r_word(struct sdio_func *func, unsigned int addr, int *err_ret)
{
	unsigned int value = 0;
	unsigned int i;

	if (!func) {
		return -1;
	}

	for (i = 0; i < sizeof(UINT32); i++) {
		value |= ((UINT32)(sdio_readb(func, addr + i, err_ret)) << (i*8));
	}

	return value;
}

static void sdio_puts_null(const char *str)
{ /* do nothing */ }

static void sdio_putc_null(UINT8 ch)
{ /* do nothing */ }

static void cmd_sdio_cccr_dump(int argc, char *argv[])
{
	sdio_cccr_dump();
}

static void cmd_sdio_init(int argc, char *argv[])
{
	unsigned int sdioClk = 24*1000*1000;

	if (argc == 1) {
		sdioClk = (atoi(argv[0]));
	}

	sdio_init(sdioClk);
}

static void cmd_sdio_deinit(int argc, char *argv[])
{
	sdio_deinit();
}

static void cmd_sdio_wcheck(int argc, char *argv[])
{
	struct sdio_func *func;
	UINT32 addr_start, addr_end;
	UINT8 rval, wval = 0xa;
	UINT32 i;
	UINT32 cnt = 0;
	int result = 0;

	if (argc != 2) {
		printf("Usage:\n");
		printf("  sdiowck <addr> ~ <addr>\n");
		printf("  Check an address range has read write permission by CMD52 func0\n");
		return;
	}

	addr_start = strtoul(argv[0], NULL, 0);
	addr_end = strtoul(argv[1], NULL, 0);

	func = sdioFuncGet(0);
	sdio_claim_host(func);

	printf("The following address coulde be write:\n");
	for (i = addr_start; i <= addr_end; i++) {
		rval = sdio_readb(func, i, &result);
		if (rval == wval) {
			wval = ~wval;
		}

		sdio_writeb(func, wval, i, &result);
		rval = sdio_readb(func, i, &result);
		if (rval == wval && result == 0) {
			printf("0x%08x  ", i);

			if(!(++cnt % 5)) {
				printf("\n");
			}
		}
	}
	printf("\n");

	sdio_release_host(func);
}

static void cmd_sdio_phase_sel(int argc, char *argv[])
{
	struct sdio_func *func;
	flashReg_t *pflashReg = (flashReg_t *)(LOGI_ADDR_REG);

	UINT32 mmc_mode;
	UINT32 clk_delay, clk_reverse, edge_latch;
	UINT32 data_delay;
	UINT32 phase_default;

	int result = 0;
	UINT32 addr = 0x60;
	UINT8 value_default;
	UINT8 rval, wval = 0xa;
	int runcnt = 0;
	int ok_beg, ok_end;
	UINT8 best_chs = 0;
	char buf1[64];
	char buf2[64];

	if (argc == 2) {
		addr = strtoul(argv[0], NULL, 0);
		wval = strtoul(argv[1], NULL, 0);
	}

	func = sdioFuncGet(0);
	sdio_claim_host(func);

	phase_default = pflashReg->sdiosddiphsel;
	value_default = sdio_readb(func, addr, &result);

	if (value_default == wval) {
		wval = ~wval;
		printf("The value on this address is 0x%x. Auto change write pattern to 0x%x\n", value_default, wval);
	}

	for (clk_reverse = 0; clk_reverse <= 1; clk_reverse++) {
		for (edge_latch = 0; edge_latch <= 1; edge_latch++) {
			for (mmc_mode = 0; mmc_mode <= 1; mmc_mode++) {

				for (clk_delay = 0; clk_delay <= 7; clk_delay++) {

					ok_beg = ok_end = -1;
					for (data_delay = 0; data_delay <= 7; data_delay++) {
						/* phase settting */
						halSdioModeSet(mmc_mode);
						halSdioPhaseClkDelaySet(clk_delay);
						halSdioPhaseDataDelaySet(data_delay);
						halSdioPhaseClkReverseSet(clk_reverse);
						halSdioPhasePosLatchSet(edge_latch);

						/* run sdio cmd */
						sioPutsRedirect(sdio_puts_null);
						sioPutcRedirect(sdio_putc_null);

						do {
							sdio_writeb(func, wval, addr, &result);
							rval = sdio_readb(func, addr, &result);
							if (rval != wval || result != 0) {
								/* NG */
								if (!best_chs) {
									if (ok_end == -1) {
										ok_end = (data_delay == 0) ? 0 : (data_delay - 1);
									}
								}

								result = 1;
								break;
							}
							sdio_writeb(func, value_default, addr, &result);
						} while (++runcnt < 3);
						/* PASS */

						if (!best_chs) {
							if (ok_beg == -1) {
								ok_beg = data_delay;
							}

							if (ok_beg != -1 && ok_end != -1)
							{
								best_chs = 1;
								snprintf(buf1, 64, "mmc mode: %u, clock reverse: %u, edge latch: %u", mmc_mode, clk_reverse, edge_latch);
								snprintf(buf2, 64, "clock phase dealy: %u, data phase delay: %d", clk_delay, (ok_beg + ok_end)/2);
							}
						}

						sioPutsRedirect(NULL);
						sioPutcRedirect(NULL);

						/* check result */
						/* printf("write: 0x%x read: 0x%x result: 0x%x\n", wval, rval, result); */
						printf("mmc mode: %d, phase setting: 0x%02x - %s\n", mmc_mode, pflashReg->sdiosddiphsel, result ? "NG" : "PASS");
					}
				}
			}
		}
	}

	pflashReg->sdiosddiphsel = phase_default;
	sdio_release_host(func);

	printf("\nThe suggest settings is:\n");
	printf("\t%s\n", buf1);
	printf("\t%s\n", buf2);
}

static void cmd_sdio_io(int argc, char *argv[])
{
	int result = 0;
	struct sdio_func *func;
	void (*funcSel) (UINT32 func, UINT32 msk);
	void (*dirSet) (UINT32 dir, UINT32 msk);
	void (*outSet) (UINT32 val, UINT32 msk);
	UINT32 cmd;
	UINT32 addr;
	UINT32 fmgpio;
	UINT32 value;

	if (argc != 4 && argc != 5) {
		printf("Usage:\n");
		printf("  sdiocmd [52|53] <fmgpio_num> <func_num> <addr> [value]\n");
		printf("  This sdio command will also raise FMGPIO before running\n");
		printf("  The limitation size of value is 4 bytes\n");
		return;
	}

	cmd = atoi(argv[0]);

	func = sdioFuncGet(atoi(argv[1]));
	if (!func) {
		printf("Unknow function number\n");
		return;
	}

	fmgpio = (atoi(argv[2]));
	if (fmgpio < 32) {
		funcSel = halFmGpioAFuncSel;
		dirSet = halFmGpioADirSet;
		outSet = halFmGpioAOutSet;
	}
	else {
		fmgpio -= 32;
		funcSel = halFmGpioBFuncSel;
		dirSet = halFmGpioBDirSet;
		outSet = halFmGpioBOutSet;
	}

	funcSel(1, 1 << fmgpio);
	dirSet(1 << fmgpio, 1 << fmgpio);
	outSet(0, 1 << fmgpio);

	sdio_claim_host(func);

	outSet(1 << fmgpio, 1 << fmgpio);
	addr = strtoul(argv[3], NULL, 0);

	if (argc == 4) {
		if (cmd == 53) {
			value = sdio_readl(func, addr, &result);
		}
		else {
			value = sdio_cmd52r_word(func, addr, &result);
		}

		printf("CMD%d read addr:0x%08x = 0x%08x res:0x%x\n", cmd, addr, value, result);
	}
	else {
		value = strtoul(argv[4], NULL, 0);

		if (cmd == 53) {
			sdio_writel(func, value, addr, &result);
		}
		else {
			sdio_cmd52w_word(func, value, addr, &result);
		}
		printf("CMD%d write addr:0x%08x = 0x%08x res:0x%x\n", cmd, addr, value, result);
	}

	outSet(0, 1 << fmgpio);
	sdio_release_host(func);
}

static void cmd_sdio_pinmux(int argc, char *argv[])
{
	UINT32 pin_grp;

	if (argc == 1) {
		pin_grp = atoi(argv[0]);

		if (pin_grp <= 2) {
			halSdioModeType(pin_grp);
			return;
		}
	}

	printf("Usage:\n");
	printf("  sdiopin <grp>\n");
	printf("  grp: [0] PIN 15~20, [1] PIN 24~29, [2] PIN 41~46\n");
	printf("  Set sdio pin mux group setting\n");
}

static void cmd_sdio_detect(int argc, char *argv[])
{
	if (sdio_init(24*1000*1000) < 0) {
		return;
	}

	sdio_cccr_dump();
	sdio_deinit();

}

static void sdioPerfIO(struct sdio_func *pfunc, UINT32 wflag, UINT32 addr, UINT32 len, UINT32 chunkbytes)
{
	void *pbuf;
	int res;
	UINT32 remainder = len;
	UINT32 wbyte;
	UINT64 tmr, tspt = 0;

	pbuf = sp5kMalloc(chunkbytes);
	if (pbuf == NULL) {
		printf("malloc %d fail\n", chunkbytes);
		return;
	}

	while(remainder) {
		tmr = tmrTimeClockGet();
		if (wflag) {
			res = sdio_memcpy_toio(pfunc, addr, pbuf, chunkbytes);
		}
		else {
			res = sdio_memcpy_fromio(pfunc, pbuf, addr, chunkbytes);
		}

		if (res != SUCCESS) {
			printf("SDIO write fail: res:0x%x\n", res);
			tspt = tmrTimeClockGet() - tmr;
			break;
		}
		remainder -= chunkbytes;
		tspt += tmrTimeClockGet() - tmr;
	}

	wbyte = len - remainder;
	printf("SDIO %s %d Bytes, Total transfer = %.2f K, Time = %.2f Msec, Bps=%.2f M\n"
	       ,wflag ? "OUT":"IN", chunkbytes, (double)(wbyte/1000.0), (double)(tspt/1000.0), (double)(wbyte/tspt));

	sp5kFree(pbuf);
}

static void cmdSdioPerf(int argc, char *argv[])
{
	struct sdio_func *pfunc;
	UINT32 waddr, raddr;
	UINT32 datalen;

	if (strncmp(argv[0], "rtk", strlen(argv[0]))) {
		printf("Not support: %s\n", argv[0]);
		return;
	}

	waddr = 3; /* BK_QUEUE_INX */
	raddr = (7 << 13); /* WLAN_RX0FF_DEVICE_ID */
	pfunc = sdioFuncGet(1); /* WIFI function is 1 */
	datalen = pfunc->cur_blksize * 100;

	/* Block Write */
	sdioPerfIO(pfunc, 1, waddr, datalen, pfunc->cur_blksize);

	/* Block Read */
	sdioPerfIO(pfunc, 0, raddr, datalen, pfunc->cur_blksize);
}

void cmdSdioInit(void)
{
	printf("[%s]\n", __FUNCTION__);
	cmdBatchRegister(&cmdSdioArray[0]);
}

