/*
 *  linux/drivers/mmc/core/core.c
 *
 *  Copyright (C) 2003-2004 Russell King, All Rights Reserved.
 *  SD support Copyright (C) 2004 Ian Molton, All Rights Reserved.
 *  Copyright (C) 2005-2008 Pierre Ossman, All Rights Reserved.
 *  MMCv4 support Copyright (C) 2006 Philip Langdale, All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <stdarg.h>
#include <os/tx_api.h>
#include <os/ros_api.h>
#include "bus.h"

static void
mmc_start_request(struct mmc_host *host, struct mmc_request *mrq)
{
	/*pr_debug("%s: starting CMD%u arg %08x flags %08x\n",
		 mmc_hostname(host), mrq->cmd->opcode,
		 mrq->cmd->arg, mrq->cmd->flags);*/

	/*if (mrq->data) {
		pr_debug("%s:     blksz %d blocks %d flags %08x\n",
			mmc_hostname(host), mrq->data->blksz,
			mrq->data->blocks, mrq->data->flags);
	}*/

	mrq->cmd->error = 0;
	if (mrq->data) {
		mrq->data->error = 0;
	}
	sdhc_request(host, mrq);
}

/**
 *	mmc_wait_for_req - start a request and wait for completion
 *	@host: MMC host to start command
 *	@mrq: MMC request to start
 *
 *	Start a new MMC custom command request for a host, and wait
 *	for the command to complete. Does not attempt to parse the
 *	response.
 */
void mmc_wait_for_req(struct mmc_host *host, struct mmc_request *mrq)
{
	mmc_start_request(host, mrq);
}

/**
 *	mmc_wait_for_cmd - start a command and wait for completion
 *	@host: MMC host to start command
 *	@cmd: MMC command to start
 *	@retries: maximum number of retries
 *
 *	Start a new MMC command for a host, and wait for the command
 *	to complete.  Return any error that occurred while the command
 *	was executing.  Do not attempt to parse the response.
 */
int mmc_wait_for_cmd(struct mmc_host *host, struct mmc_command *cmd, int retries)
{
	struct mmc_request mrq = {cmd, NULL};

	/*memset(cmd->resp, 0, sizeof(cmd->resp));*/
	cmd->retries = retries;

	/*mmc_wait_for_req(host, &mrq);*/
	mmc_start_request(host, &mrq);

	return cmd->error;
}

/**
 *	mmc_align_data_size - pads a transfer size to a more optimal value
 *	@card: the MMC card associated with the data transfer
 *	@sz: original transfer size
 *
 *	Pads the original data size with a number of extra bytes in
 *	order to avoid controller bugs and/or performance hits
 *	(e.g. some controllers revert to PIO for certain sizes).
 *
 *	Returns the improved size, which might be unmodified.
 *
 *	Note that this function is only relevant when issuing a
 *	single scatter gather entry.
 */
unsigned int mmc_align_data_size(struct mmc_card *card, unsigned int sz)
{
	/*
	 * FIXME: We don't have a system for the controller to tell
	 * the core about its problems yet, so for now we just 32-bit
	 * align the size.
	 */
	sz = ((sz + 3) / 4) * 4;

	return sz;
}

/**
 *	mmc_host_enable - enable a host.
 *	@host: mmc host to enable
 *
 *	Hosts that support power saving can use the 'enable' and 'disable'
 *	methods to exit and enter power saving states. For more information
 *	see comments for struct mmc_host_ops.
 */
int mmc_host_enable(struct mmc_host *host)
{
	if (!(host->caps & MMC_CAP_DISABLE))
		return 0;

	int err = sdhc_enable(host);
	if (err) {
		pr_debug("%s: enable error %d\n", mmc_hostname(host), err);
		return err;
	}

	host->enabled = 1;
	return 0;
}

static int mmc_host_do_disable(struct mmc_host *host, int lazy)
{
	int err = sdhc_disable(host, lazy);
	if (err != 0) {
		pr_debug("%s: disable error %d\n", mmc_hostname(host), err);
		return err;
	}
	host->enabled = 0;
	return 0;
}

/**
 *	mmc_host_disable - disable a host.
 *	@host: mmc host to disable
 *
 *	Hosts that support power saving can use the 'enable' and 'disable'
 *	methods to exit and enter power saving states. For more information
 *	see comments for struct mmc_host_ops.
 */
int mmc_host_disable(struct mmc_host *host)
{
	if (!(host->caps & MMC_CAP_DISABLE))
		return 0;

	if (!host->enabled)
		return 0;

	return mmc_host_do_disable(host, 0);
}

void mmc_claim_host(struct mmc_host *host)
{
	ros_mutex_get(host->lock, TX_WAIT_FOREVER);
}

/**
 *	mmc_release_host - release a host
 *	@host: mmc host to release
 *
 *	Release a MMC host, allowing others to claim the host
 *	for their operations.
 */
void mmc_release_host(struct mmc_host *host)
{
	ros_mutex_put(host->lock);
}

/*
 * Internal function that does the actual ios call to the host driver,
 * optionally printing some debug output.
 */
static void mmc_set_ios(struct mmc_host *host)
{
	struct mmc_ios *ios = &host->ios;

	/*pr_debug("%s: clock %uHz powermode %u width %u\n",
		 mmc_hostname(host), ios->clock, ios->power_mode, ios->bus_width);*/
	sdhc_set_ios(host, ios);
}

/*
 * Sets the host clock to the highest possible frequency that
 * is below "hz".
 */
void mmc_set_clock(struct mmc_host *host, unsigned int hz)
{
	WARN_ON(hz < host->f_min);
	if (hz > host->f_max)
		hz = host->f_max;

	host->ios.clock = hz;
	mmc_set_ios(host);
}

/*
 * Change data bus width of a host.
 */
void mmc_set_bus_width(struct mmc_host *host, unsigned int width)
{
	host->ios.bus_width = width;
	mmc_set_ios(host);
}

/*
 * Apply power to the MMC stack.  This is a two-stage process.
 * First, we enable power to the card without the clock running.
 * We then wait a bit for the power to stabilise.  Finally,
 * enable the bus drivers and clock to the card.
 *
 * We must _NOT_ enable the clock prior to power stablising.
 *
 * If a host does all the power sequencing itself, ignore the
 * initial MMC_POWER_UP stage.
 */
static void mmc_power_up(struct mmc_host *host)
{
}

static void mmc_power_off(struct mmc_host *host)
{
}

/*
 * Assign a mmc bus handler to a host. Only one bus handler may control a
 * host at any given time.
 */
void mmc_attach_bus(struct mmc_host *host, const struct mmc_bus_ops *ops)
{
	BUG_ON(!host);
	BUG_ON(!ops);
	BUG_ON(host->bus_ops);

	mmc_claim_host(host);
	host->bus_ops = ops;
	host->bus_dead = 0;
	mmc_release_host(host);
}

/*
 * Remove the current bus handler from a host. Assumes that there are
 * no interesting cards left, so the bus is powered down.
 */
void mmc_detach_bus(struct mmc_host *host)
{
	BUG_ON(!host);
	WARN_ON(!host->bus_ops);

	mmc_claim_host(host);
	host->bus_ops = NULL;
	host->bus_dead = 1;
	mmc_release_host(host);

	mmc_power_off(host);
}

void mmc_detect_change(struct mmc_host *host)
{
	if (host && host->bus_ops->detect)
		host->bus_ops->detect(host);
}

int mmc_rescan(struct mmc_host *host)
{
	int ret = 0;

	/* if there is a card registered, check whether it is still present */
	if ((host->bus_ops != NULL) && host->bus_ops->detect && !host->bus_dead)
		host->bus_ops->detect(host);

	/* if there still is a card present, stop here */
	if (host->bus_ops != NULL)
		return -1;

	if (sdhc_get_cd(host) == 0)
		return -1;

	mmc_claim_host(host);
	mmc_power_up(host);

	if (sdhc_probe_sdio(host) || mmc_attach_sdio(host)) {
		mmc_power_off(host);
		ret = -1;
	}

	mmc_release_host(host);
	return ret;
}

void mmc_power_save_host(struct mmc_host *host)
{
}

void mmc_power_restore_host(struct mmc_host *host)
{
}

extern int vsnprintf(char *dest, size_t size, const char *format, va_list ap);

void SDBUSDBG(const char *fmt, ...)
{
	static UINT32 repeat = 0;
	static char str_last[120];

	char str[120], *pstr = str;
	char *str_end = pstr + sizeof(str);

	pstr += sprintf(pstr, "[SDBUS] ");

	va_list ap;
	va_start(ap, fmt);
	pstr += vsnprintf(pstr, str_end-pstr-1, fmt, ap);
	va_end(ap);

	*pstr = 0;

	if (!strcmp(str_last, str)) {
		++repeat;
	}
	else {
		if (repeat) {
			printf("  <repeat %d times>\n", repeat);
			repeat = 0;
		}

		puts(str);
		strcpy(str_last, str);
	}
}

