/*
 *  linux/drivers/mmc/core/sdio_bus.c
 *
 *  Copyright 2007 Pierre Ossman
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * SDIO function driver model
 */

#include "bus.h"

extern struct mmc_bus_type g_mmc_bus;

static const struct sdio_device_id *sdio_match_one(struct sdio_func *func,
	const struct sdio_device_id *id)
{
	if (id->class != (UINT8)SDIO_ANY_ID && id->class != func->class)
		return NULL;
	if (id->vendor != (UINT16)SDIO_ANY_ID && id->vendor != func->vendor)
		return NULL;
	if (id->device != (UINT16)SDIO_ANY_ID && id->device != func->device)
		return NULL;
	return id;
}

static const struct sdio_device_id *sdio_match_device(struct sdio_func *func,
	struct sdio_driver *sdrv)
{
	const struct sdio_device_id *ids;

	ids = sdrv->id_table;

	if (ids) {
		while (ids->class || ids->vendor || ids->device) {
			if (sdio_match_one(func, ids))
				return ids;
			ids++;
		}
	}

	return NULL;
}

static int sdio_bus_probe(struct sdio_func *func, struct sdio_driver *sdrv)
{
	const struct sdio_device_id *id;
	int ret = 0;

	id = sdio_match_device(func, sdrv);
	if (!id)
		return -ENODEV;

	/* Set the default block size so the driver is sure it's something
	 * sensible. */
	sdio_claim_host(func);
	ret = sdio_set_block_size(func, 0);
	sdio_release_host(func);
	if (ret)
		return ret;

	if (sdrv && sdrv->probe) {
	ret = sdrv->probe(func, id);
	if (ret)
		return ret;
	}

	func->drv = sdrv;
	func->dev.func = func;
	return ret;
}

static int sdio_bus_remove(struct sdio_func *func, struct sdio_driver *sdrv)
{
	if (sdrv && sdrv->remove) {
	sdrv->remove(func);
	}

	if (func->irq_handler) {
		printf("WARNING: driver %s did not remove "
			"its interrupt handler!\n", sdrv->name);
		sdio_claim_host(func);
		sdio_release_irq(func);
		sdio_release_host(func);
	}

	return 0;
}

int sdio_probe_device()
{
	struct mmc_bus_type* bus = &g_mmc_bus;
	struct sdio_driver* sdrv = bus->sdio_func_driver;
	int n, m;
	int ret = -1;

	if (!sdrv)
		return -1;

	/* FIXME: assuem only one host */
	for (n = 0; n < CONFIG_MMC_HOST_NR; ++n) {
		struct mmc_host* h = bus->hosts[n];
		if (!h)
			continue;

		if (!h->card.initialized) {
			printf("SDIO Card isn't initialized\n");
			continue;
		}

		for (m = 0; m < h->card.sdio_funcs; ++m) {
			ret = sdio_bus_probe(&h->card.sdio_func[m], sdrv);
		}
	}
	return ret;
}

/**
 *	sdio_register_driver - register a function driver
 *	@drv: SDIO function driver
 */
int sdio_register_driver(struct sdio_driver *drv)
{
	if (!drv || g_mmc_bus.sdio_func_driver)
		return -1;

	g_mmc_bus.sdio_func_driver = drv;
	return sdio_probe_device();
}

/**
 *	sdio_unregister_driver - unregister a function driver
 *	@drv: SDIO function driver
 */
void sdio_unregister_driver(struct sdio_driver *drv)
{
	struct mmc_bus_type* bus = &g_mmc_bus;
	struct sdio_driver* sdrv = bus->sdio_func_driver;
	int n, m;

	if (!sdrv)
		return;

	for (n = 0; n < CONFIG_MMC_HOST_NR; ++n) {
		struct mmc_host* h = bus->hosts[n];
		if (!h || !h->card.initialized)
			continue;

		for (m = 0; m < h->card.sdio_funcs; ++m) {
			if (h->card.sdio_func[m].drv == sdrv) {
				sdio_bus_remove(&h->card.sdio_func[m], drv);
			}
		}
	}
	g_mmc_bus.sdio_func_driver = NULL;
}

