/*
 * linux/drivers/mmc/core/sdio_irq.c
 *
 * Author:      Nicolas Pitre
 * Created:     June 18, 2007
 * Copyright:   MontaVista Software Inc.
 *
 * Copyright 2008 Pierre Ossman
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 */

#include <os/tx_api.h>
#include <os/ros_api.h>
#include "bus.h"
#include "debug_mod/dbg_api.h"
#include "middleware/sdhc_cfg.h" /* sdioFuncNumGet() */

static int process_sdio_pending_irqs(struct mmc_card *card)
{
	int ret = 0, count = 0;

	if (sdioFuncNumGet() == 1) {
		struct sdio_func *func = &card->sdio_func[0];
		if (!func) {
			printk("pending IRQ for non-existant function\n");
			ret = -EINVAL;
		} else if (func->irq_handler) {
			func->irq_handler(func);
			count++;
		} else {
			printk("%s: pending IRQ with no handler\n", sdio_func_id(func));
			ret = -EINVAL;
		}
	}
	else {
		int i;
		unsigned char pending;

		ret = mmc_io_rw_direct(card, 0, 0, SDIO_CCCR_INTx, 0, &pending);
		if (ret) {
			printk("error reading SDIO_CCCR_INTx %d\n", ret);
			return ret;
		}

		for (i = 1; i <= CONFIG_SDIO_FUNC_NR; i++) {
			if (pending & (1 << i)) {
				struct sdio_func *func = &card->sdio_func[i - 1];
				if (!func) {
					printk("pending IRQ for non-existant function\n");
					ret = -EINVAL;
				} else if (func->irq_handler) {
					func->irq_handler(func);
					count++;
				} else {
					printk("%s: pending IRQ with no handler\n", sdio_func_id(func));
					ret = -EINVAL;
				}
			}
		}
	}
	
	if (count)
		return count;

	return ret;
}

static UINT32 irq_safety = 1;

int sdio_irq_safety(void)
{
	return irq_safety;
}

static void sdio_irq_thread(UINT32 _host)
{
	pr_debug("sdio_irq_thread +\n");

	struct mmc_host *host = (struct mmc_host *)_host;
	UINT32 flags;
	UINT32 sr;

	if (host->caps & MMC_CAP_SDIO_IRQ) {
		sdhc_exit_irq_cs();
	}

	do {
		ros_event_flag_get(host->sdio_irq_thread_evt, 0x01, ROS_EVT_OR_CLR, &flags, ROS_WAIT_FOREVER);

		/*
		 * We claim the host here on drivers behalf for a couple
		 * reasons:
		 *
		 * 1) it is already needed to retrieve the CCCR_INTx;
		 * 2) we want the driver(s) to clear the IRQ condition ASAP;
		 * 3) we need to control the abort condition locally.
		 *
		 * Just like traditional hard IRQ handlers, we expect SDIO
		 * IRQ handlers to be quick and to the point, so that the
		 * holding of the host lock does not cover too much work
		 * that doesn't require that lock to be held.
		 */
		ENTER_CRITICAL(sr);
		irq_safety = 0;
		EXIT_CRITICAL(sr);

		mmc_claim_host(host);
		process_sdio_pending_irqs(&host->card);
		mmc_release_host(host);

		ENTER_CRITICAL(sr);
		irq_safety = 1;
		EXIT_CRITICAL(sr);

		if (host->caps & MMC_CAP_SDIO_IRQ) {
			sdhc_exit_irq_cs();
		}
	} while (1);

	if (host->caps & MMC_CAP_SDIO_IRQ) {
		sdhc_enter_irq_cs();
	}

	pr_debug("%s: IRQ thread exiting ...\n", mmc_hostname(host));
}

static int sdio_card_irq_get(struct mmc_card *card)
{
	struct mmc_host *host = card->host;

	if (!host->sdio_irqs++) {
		host->sdio_irq_thread_evt = ros_event_flag_create("sdio irq");
		ASSERT(host->sdio_irq_thread_evt);

		host->sdio_irq_thread = ros_thread_create_ext("sdioirq", sdio_irq_priority_get()
			, sdio_irq_thread, (UINT32)host
			, 0, 4096-256, 1);
		ASSERT(host->sdio_irq_thread);
	}

	return 0;
}

static int sdio_card_irq_put(struct mmc_card *card)
{
	struct mmc_host *host = card->host;

	BUG_ON(host->sdio_irqs < 1);

	if (!--host->sdio_irqs) {
		ros_thread_destroy(host->sdio_irq_thread);
		ros_event_flag_destroy(host->sdio_irq_thread_evt);
	}

	return 0;
}

/**
 *	sdio_claim_irq - claim the IRQ for a SDIO function
 *	@func: SDIO function
 *	@handler: IRQ handler callback
 *
 *	Claim and activate the IRQ for the given SDIO function. The provided
 *	handler will be called when that IRQ is asserted.  The host is always
 *	claimed already when the handler is called so the handler must not
 *	call sdio_claim_host() nor sdio_release_host().
 */
int sdio_claim_irq(struct sdio_func *func, sdio_irq_handler_t *handler)
{
	int ret;
	unsigned char reg;

	BUG_ON(!func);
	BUG_ON(!func->card);

	pr_debug("SDIO: Enabling IRQ for %s...\n", sdio_func_id(func));

	if (func->irq_handler) {
		pr_debug("SDIO: IRQ for %s already in use.\n", sdio_func_id(func));
		return -EBUSY;
	}

	ret = mmc_io_rw_direct(func->card, 0, 0, SDIO_CCCR_IENx, 0, &reg);
	if (ret)
		return ret;

	reg |= 1 << func->num;

	reg |= 1; /* Master interrupt enable */

	ret = mmc_io_rw_direct(func->card, 1, 0, SDIO_CCCR_IENx, reg, NULL);
	if (ret)
		return ret;

	func->irq_handler = handler;
	ret = sdio_card_irq_get(func->card);
	if (ret)
		func->irq_handler = NULL;

	return ret;
}

/**
 *	sdio_release_irq - release the IRQ for a SDIO function
 *	@func: SDIO function
 *
 *	Disable and release the IRQ for the given SDIO function.
 */
int sdio_release_irq(struct sdio_func *func)
{
	int ret;
	unsigned char reg;

	BUG_ON(!func);
	BUG_ON(!func->card);

	pr_debug("SDIO: Disabling IRQ for %s...\n", sdio_func_id(func));

	if (func->irq_handler) {
		func->irq_handler = NULL;
		sdio_card_irq_put(func->card);
	}

	ret = mmc_io_rw_direct(func->card, 0, 0, SDIO_CCCR_IENx, 0, &reg);
	if (ret)
		return ret;

	reg &= ~(1 << func->num);

	/* Disable master interrupt with the last function interrupt */
	if (!(reg & 0xFE))
		reg = 0;

	ret = mmc_io_rw_direct(func->card, 1, 0, SDIO_CCCR_IENx, reg, NULL);
	if (ret)
		return ret;

	return 0;
}

