/**************************************************************************
 *                                                                        *
 *         Copyright (c) 2012 by iCatch Technology Co., Ltd.             *
 *                                                                        *
 *  This software is copyrighted by and is the property of Sunplus        *
 *  Technology Co., Ltd. All rights are reserved by Sunplus Technology    *
 *  Co., Ltd. This software may only be used in accordance with the       *
 *  corresponding license agreement. Any unauthorized use, duplication,   *
 *  distribution, or disclosure of this software is expressly forbidden.  *
 *                                                                        *
 *  This Copyright notice MUST not be removed or modified without prior   *
 *  written consent of Sunplus Technology Co., Ltd.                       *
 *                                                                        *
 *  Sunplus Technology Co., Ltd. reserves the right to modify this        *
 *  software without notice.                                              *
 *                                                                        *
 *  Sunplus Technology Co., Ltd.                                          *
 *  19, Innovation First Road, Science-Based Industrial Park,             *
 *  Hsin-Chu, Taiwan, R.O.C.                                              *
 *                                                                        *
 *  Author: Andy.Li                                                       *
 *                                                                        *
 **************************************************************************/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "general.h"
#include <os/tx_api.h>
#include <os/ros_api.h>
#include "os/vfs_api.h"
#include "hal/hal_flash.h"
#include "hal/hal_peripheral_sdio.h"
#include "debug_mod/err_code_tbl.h"
#include "debug_mod/dbg_api.h"
#include "core/bus.h"
#include "middleware/sdhc_cfg.h" /* sdioReadBlkNoRspGet() */
#include "hal/hal_pwr.h"

/* Becareful. Copied from sd_private.h */
typedef enum {
	SD_1BIT_MODE = (0 << 0),
	SD_4BIT_MODE = (1 << 0),
	SD_8BIT_MODE = (2 << 0)
} sdBitMode_t;

/*
 * Macro & Constants
 */
#define SDHC_DEBUG		0

#define FLASH_REG_DIRECT(a)	(*(UINT8*)(LOGI_ADDR_REG + (a)))
void sdio2SockSet(UINT32 on);
#define SDIO_RWFLAG_GET(x) ((x->arg >> 31) & 0x01)
#define SDIO_IS_READCMD(x) SDIO_RWFLAG_GET(x) ? 0 : 1

/*
 * Global variables
 */
static struct mmc_host g_mmc_host;

#define CMD53_DISABLE_IRQ	1

#define SDHC_INTERRUPT_MASK 0
#define SDHC_INTERRUPT_UNMASK 1

char sdhc_dbg_cmd52_on		= 1;
char sdhc_dbg_cmd53_on		= 1;
char sdhc_dbg_cmd53_rx_on	= 1;
char sdhc_dbg_cmd53_tx_on	= 1;

/* Statistics */
UINT32 sdhc_sta_cmd52_count	= 0;
UINT32 sdhc_sta_cmd53_count	= 0;
UINT32 sdhc_sta_cmd53_tx_bytes  = 0;
UINT32 sdhc_sta_cmd53_rx_bytes  = 0;
UINT32 sdhc_sta_irq_count	= 0;

static int irq_ref = 0;
static UINT32 irq_status = 0;
static int irq_released = 0;
/*
 * Forward declarations
 */
extern UINT32 sdioDetect(void);
extern void halSdioData1Int(UINT32 enable, pfFlashIsr_t pisr);
extern void mmc_initialize();
extern UINT32 halFmStsGet(void);
extern UINT32 halFmIntGet(void);
extern UINT32 SdioDma(UINT32 isR, UINT8 *p,UINT32 Cnt);
extern UINT8 * sdioBufGet(void);
/*
 * Functions
 */
void sdhc_enable_sdio_irq(struct mmc_host *host, int enable);
static void sdhc_enter_irq_cs_cmd53(void);
static void sdhc_exit_irq_cs_cmd53(void);

/* FIXME: missing funciton, that's shoulde be in sd_variable.c */
static UINT32 sdioIfModeGet(void)
{
	return 1;
}

static void sdioIfModeSet(UINT32 val)
{
	return;
}

#if SDHC_DEBUG
static struct mmc_command sdhc_last_cmd;
static void sdhc_dump_regs(const char* prompt)
{
#if 0
	printf("$ %s status= %08x enable= %08x\n"
		, prompt , halFmStsGet(), halFmIntGet());
#endif
}
#else
#define sdhc_dump_regs(x)
#endif

int sdhc_enable(struct mmc_host *host)
{
	return 0;
}

int sdhc_disable(struct mmc_host *host, int lazy)
{
	return 0;
}

static void sdio_cmd52(struct mmc_command* cmd)
{
	UINT8 rspbuf[17];
	/*printf("CMD52 arg:0x%X\n",cmd->arg);*/
	halSdioDebugFlagSet(0);
	cmd->error = halSdioCmd(52, cmd->arg, SD_MMC_RSP_R5, rspbuf);
	if (cmd->error == SUCCESS) {
		cmd->resp[0] = rspbuf[4];
#if SDHC_DEBUG
		if (sdhc_dbg_cmd52_on) {
			UINT32 arg = cmd->arg;
			SDBUSDBG("CMD52 arg=%08x %c fn=%d addr=%x resp=%x"
			       , arg
			       , (arg >> 31) & 0x01 ? 'w' : 'r'
			       , (arg >> 28) & 0x07
			       , (arg >> 9) & 0x1FFFF
			       , cmd->resp[0]);
		}
#endif
	} else {
		printf("cmd52 failed(0x%08x)\n", cmd->error);
	}
	#if 0
	halemmcDummy();
	#endif
}

static void sdio_cmd53(struct mmc_host *host, struct mmc_command* cmd, struct mmc_data* data)
{
	UINT8 *pbuf    = data->buf, rspbuf[17];
	UINT32 rsptype = SD_MMC_RSP_R5;
	UINT32 datalen = data->blksz * data->blocks;
	UINT32 blksz   = data->blksz;
	char   r_flag = SDIO_IS_READCMD(cmd);
	static UINT32 r=0, w=0;

	halSdioDebugFlagSet(0);

	sdhc_enter_irq_cs_cmd53();

#if SDHC_DEBUG
	if (sdhc_dbg_cmd53_on) {
		UINT32 arg = cmd->arg;
		SDBUSDBG("CMD53 arg=%08x %c fn=%d addr=%x blksz=%d blks=%d"
		         , arg
		         , (arg >> 31) & 0x01 ? 'w' : 'r'
		         , (arg >> 28) & 0x07
		         , (arg >> 9) & 0x1FFFF
		         , data->blksz
		         , data->blocks);
	}
#endif

	/* Transfer Cmd */
	/* TODO: receive response of read command */
	if (r_flag) {
		rsptype = SD_MMC_RSP_NONE;
	}

	cmd->error = halSdioCmd(53, cmd->arg, rsptype, rspbuf);
	if ( cmd->error != SUCCESS ) {
		sdhc_exit_irq_cs_cmd53();
		SDBUSDBG("cmd53 failed(%x): arg=%x", cmd->error, cmd->arg);
		return;
	}

	/* Transfer Data */
	if (r_flag) {
		sdhc_sta_cmd53_rx_bytes += datalen;
		++r;
	} else {
		sdhc_sta_cmd53_tx_bytes += datalen;
		++w;
	}

	cmd->resp[0] = rspbuf[4];

	halSdioCrcLenSet(blksz);
	data->error = SdioDma(r_flag, pbuf, datalen);

	if(data->error) {
		printf("CMD53 arg:0x%X r_flag:%d r%d w:%d\n\n"
		       , cmd->arg, r_flag, r, w);
	}

	halSdioInfReset();
	halSdioDummyTx();
	halSdioDummyTx();

	if (r_flag && sdioReadBlkNoRspGet()) {
		halSdioDummyTx();
		halSdioDummyTx();
	}
	sdhc_exit_irq_cs_cmd53();
}


void sdhc_request(struct mmc_host *host, struct mmc_request *req)
{
	struct mmc_command* cmd;
	cmd = req->cmd;

	switch (cmd->opcode) {
	case SD_IO_RW_DIRECT:
		sdio_cmd52(cmd);
		++sdhc_sta_cmd52_count;
		break;

	case SD_IO_RW_EXTENDED:
		sdio_cmd53(host, cmd, req->data);
		++sdhc_sta_cmd53_count;
		break;

	default:
		cmd->error = 1;
		break;
	}
#if SDHC_DEBUG
	memcpy(&sdhc_last_cmd, cmd, sizeof(sdhc_last_cmd));
#endif
}

#define MILLION(n)	((n)*1000000)

void sdhc_set_ios(struct mmc_host *host, struct mmc_ios *ios)
{
	if (ios->bus_width >= 0) {
		UINT32 busWidth = (ios->bus_width == MMC_BUS_WIDTH_1) ? SD_1BIT_MODE : SD_4BIT_MODE;
		halSdioBusWidthSet((UINT8)busWidth);
		SDBUSDBG("SDHC: set bus width to %d %d", busWidth, ios->bus_width);
	}

	int clock = ios->clock;
	if (clock > 0) {
		if (clock > host->f_max)
			clock = host->f_max;

		UINT32 clkFreq = HAL_PERI_SDIO_24MHz;

		if (clock >= MILLION(48)) {
			clkFreq = HAL_PERI_SDIO_48MHz;
		}
		else if (clock >= MILLION(24)) {
			clkFreq = HAL_PERI_SDIO_24MHz;
		}
		else if (clock >= MILLION(16)) {
			clkFreq = HAL_PERI_SDIO_16MHz;
		}
		else if (clock >= MILLION(12)) {
			clkFreq = HAL_PERI_SDIO_12MHz;
		}
		else if (clock >= MILLION(6)) {
			clkFreq = HAL_PERI_SDIO_6MHz;
		}
		else {
			clkFreq = HAL_PERI_SDIO_375KHz;
		}

		halSdioClk(clkFreq);
		SDBUSDBG("SDHC: set clock to %d %d", clkFreq, ios->clock);
	}
}

/* card detect is always 1 */
int sdhc_get_cd(struct mmc_host *host)
{
	return 1;
}

static void sdhc_isr();
void sdhc_enable_sdio_irq(struct mmc_host *host, int enable)
{
	halSdioData1Int(enable, sdhc_isr);
}

void sdhc_enter_irq_cs(void)
{
	static struct mmc_host *host = &g_mmc_host;
	UINT32 sr;

	ENTER_CRITICAL(sr);
	sdhc_enable_sdio_irq(host, 0);
	irq_status = SDHC_INTERRUPT_MASK;
	EXIT_CRITICAL(sr);

}

void sdhc_exit_irq_cs(void)
{
	static struct mmc_host *host = &g_mmc_host;
	UINT32 sr;

	ENTER_CRITICAL(sr);
	if ( irq_ref == 0 ) {
		sdhc_enable_sdio_irq(host, 1);
	}
	irq_status = SDHC_INTERRUPT_UNMASK;
	EXIT_CRITICAL(sr);

}

static void sdhc_enter_irq_cs_cmd53(void)
{
#if CMD53_DISABLE_IRQ
	static struct mmc_host *host = &g_mmc_host;
	UINT32 sr;

	ENTER_CRITICAL(sr);
	sdhc_enable_sdio_irq(host, 0);
	++irq_ref;
	EXIT_CRITICAL(sr);
#endif
}

static void sdhc_exit_irq_cs_cmd53(void)
{
#if CMD53_DISABLE_IRQ
	static struct mmc_host *host = &g_mmc_host;
	UINT32 sr;

	ENTER_CRITICAL(sr);
	if ( irq_status == SDHC_INTERRUPT_UNMASK && irq_ref == 1 ) {
		sdhc_enable_sdio_irq(host, 1);
	}
	--irq_ref;
	DBG_ASSERT(irq_ref >= 0);

	EXIT_CRITICAL(sr);
#endif
}

static void sdhc_isr(UINT32 id)
{
	static struct mmc_host *host = &g_mmc_host;

	if (!sdio_irq_safety()) {
		printf("IRQ not safety\n");
	}

	++sdhc_sta_irq_count;

	sdhc_dump_regs("sdhc_isr 0");
	sdhc_enter_irq_cs();
	sdhc_dump_regs("sdhc_isr 1");
	if (host->sdio_irqs > 0)
		ros_event_flag_set(host->sdio_irq_thread_evt, 0x01, ROS_EVT_OR);
}

extern UINT32 sdio_ocr_value;

int sdhc_probe_sdio(struct mmc_host* host)
{
	UINT32 err = -1;
	sdhc_enable_sdio_irq(host, 0);
	err = sdioDetect();
	printf("sdioDetect() Finish\n");
	if (err == SUCCESS ) {
		host->ocr = sdio_ocr_value;
		printf("Probe SDIO is succeed. OCR = 0x%08x\n", host->ocr);
	}
	else {
		printf("Probe SDIO is failed.\n");
	}

	return err;
}

int sdhc_initialize(char second_sdio, UINT32 f_max)
{
	struct mmc_host* host = &g_mmc_host;
	mmc_initialize();
	if (host->card.initialized) {
		return -1;
	}

	if (mmc_init_host(host) != 0) {
		return -1;
	}

	if (f_max > MILLION(48))
		f_max = MILLION(48);
	else if (f_max < 375000)
		f_max = 375000;

	host->index = 0;
	host->f_min = 375000;
	host->f_max = f_max;
	host->max_req_size = 65536;
	host->max_blk_size = 512;
	host->max_blk_count = 511;
	host->caps = MMC_CAP_SDIO_IRQ | MMC_CAP_4_BIT_DATA;

	if (f_max >= MILLION(48)) {
		host->caps |= MMC_CAP_SD_HIGHSPEED;
	}

	printf("SDIO = %d, Max Freq. = %d cap = 0x%x\n", second_sdio, f_max, (UINT16)host->caps);
	halPhaseClkBEnableSet(HAL_PHASE_CLK_PERCLK, HAL_PHASE_CLK_PERCLK);
	sdio2SockSet(1);

	return mmc_add_host(host);
}

void sdhc_deinitialize(void)
{
	struct mmc_host* host = &g_mmc_host;

	halPhaseClkBEnableSet(HAL_PHASE_CLK_PERCLK, 0);
	mmc_remove_host(host);
	mmc_deinit_host(host);
}

#define SDHC_TEST
#ifdef  SDHC_TEST

extern struct mmc_bus_type g_mmc_bus;

static void sdhc_dump_info()
{
	struct mmc_bus_type* bus = &g_mmc_bus;
	struct mmc_host* host = &g_mmc_host;

	printf("SDBUS Function Driver information:\n");

	if (bus->sdio_func_driver) {
		printf("  name   = %s\n", bus->sdio_func_driver->name);
		printf("  class  = %x\n", bus->sdio_func_driver->id_table->class);
		printf("  vendor = %x\n", bus->sdio_func_driver->id_table->vendor);
		printf("  device = %x\n", bus->sdio_func_driver->id_table->device);
	} else {
		printf("  no driver found\n");
	}

	printf("\nSDHC%d information:\n", host->index);
	printf("  card inited     = %d\n", host->card.initialized);
	printf("  card vendor     = %x\n", host->card.cis.vendor);
	printf("  card device     = %x\n", host->card.cis.device);
	printf("  card sdio_funcs = %d\n", host->card.sdio_funcs);
	printf("  card function 1:\n");
	printf("    func inited   = %d\n", host->card.sdio_func[0].initialized);

	printf("\nStatistics:\n");
	printf("  CMD52 count = %d\n", sdhc_sta_cmd52_count);
	printf("  CMD53 count = %d\n", sdhc_sta_cmd53_count);
}
#endif

void cmdSDHCTest(int argc, char* argv[])
{
#ifdef SDHC_TEST
	if (argc != 1) {
		return;
	}

	char* cmd = argv[0];

	if (strcmp(cmd, "init") == 0) {
		if (sdhc_initialize(0, MILLION(24)) != 0) {
			printf("sdhc_initialize failed");
		}
	} else if (strcmp(cmd, "scan") == 0) {
		sdio_probe_device();
	} else if (strcmp(cmd, "dump") == 0) {
		sdhc_dump_info();
	} else if (strcmp(cmd, "info") == 0) {
		printf("DataLen = %x\n", halSdioCrcLenGet());
		printf("BusWidth = %x\n", halSdioBusWidthGet());
	} else {
		printf("sdhc init\n");
		printf("sdhc scan\n");
		printf("sdhc dump\n");
		printf("sdhc info\n");
	}
#endif
}

void sdio2SockSet(UINT32 on)
{
	if(on)
			sdioIfModeSet(1);
	else 	sdioIfModeSet(0);
}

UINT32 sdio2SockGet(void)
{
	return sdioIfModeGet();
}

struct sdio_func* sdioFuncGet(int num)
{
	struct mmc_card *card = &g_mmc_host.card;
	int fn = num;

	if (num >= card->sdio_funcs) {
		fn = card->sdio_funcs - 1;
	}

	return &card->sdio_func[fn];
}

void sdhc_release_irq(void)
{
	if (irq_released) {
		return;
	}

	irq_released = 1;
	sdhc_enter_irq_cs_cmd53();
}

void sdhc_restore_irq(void)
{
	if (!irq_released) {
		return;
	}

	irq_released = 0;
	sdhc_exit_irq_cs_cmd53();
}
