/**************************************************************************
 *
 *       Copyright (c) 2012 by iCatch Technology, Inc.
 *
 *  This software is copyrighted by and is the property of iCatch Technology,
 *  Inc.. All rights are reserved by iCatch Technology, Inc..
 *  This software may only be used in accordance with the corresponding
 *  license agreement. Any unauthorized use, duplication, distribution,
 *  or disclosure of this software is expressly forbidden.
 *
 *  This Copyright notice MUST not be removed or modified without prior
 *  written consent of iCatch Technology, Inc..
 *
 *  iCatch Technology, Inc. reserves the right to modify this software
 *  without notice.
 *
 *  iCatch Technology, Inc.
 *  19-1, Innovation First Road, Science-Based Industrial Park,
 *  Hsin-Chu, Taiwan, R.O.C.
 *
 **************************************************************************/
#include <string.h>
#include <stdarg.h>
#include <ndk/sockios.h>
#include <ndk/common.h>
#include <ndk/linux_compat.h>
#include <ndk/cmd.h>
#include <lwip/sockets.h>
#include <lwip/pbuf.h>
#include <lwip/netif.h>
#include <lwip/tcpip.h>
#include <lwip/mem.h>
#include <netif/etharp.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>

UINT32 netif_sta_skb_alloced = 0;
UINT32 netif_sta_skb_freeed  = 0;
static long   netif_dbg_tx = 0;
static long   netif_dbg_rx = 0;
static long   netif_dbg_cutoff = 0;
static UINT32 netif_static_ipaddr = 0xC0A8FF01;

extern struct netif *netif_list;

static NDKPcapHandle pcap_handler = NULL;

/******************************************************************************
 * sk_buff functions
 ******************************************************************************/
#define NET_SKB_PAD	32
#define SIZEOF_STRUCT_PBUF	LWIP_MEM_ALIGN_SIZE(sizeof(struct pbuf))

static void __free_pbuf_skb(pskbc_t skbc)
{
	pskb_t skb = (pskb_t)skbc;

#if NETCFG_SKBC_SUPPORT_CLONE
	if (skb->cloned) {
		SKBC_FREE(skb->cloned_skb);
		skbc_delete_directly(skbc);
	}
	else
#endif
	{
		struct pbuf* pb = container_of((pbuf_skb_room_t*)skb, struct pbuf, skb_room);
		pbuf_free(pb);
		++netif_sta_skb_freeed;
	}
}

static pskb_t __alloc_pbuf_skb(unsigned int payload_size)
{
	pskb_t skb = NULL;

	struct pbuf* pb = pbuf_alloc(PBUF_RAW, payload_size, PBUF_RAM);

	if (pb) {
		NDK_ASSERT(pb->len == payload_size);
		NDK_ASSERT((UINT8*)pb->payload >= (UINT8*)pb + SIZEOF_STRUCT_PBUF);

		skb = (pskb_t)&pb->skb_room;
		skbc_init_header((pskbc_t)skb, sizeof(struct sk_buff)
			, (UINT8*)pb->payload, pb->len
			, __free_pbuf_skb);

		++netif_sta_skb_alloced;
	}

	return skb;
}

struct sk_buff*	netdev_alloc_skb(struct net_device *dev, UINT32 payload_size)
{
	pskb_t skb = __alloc_pbuf_skb(payload_size + NET_SKB_PAD);
	if (skb) {
#if NET_SKB_PAD > 0
		skb_reserve(skb, NET_SKB_PAD);
#endif
		skb->dev = dev;
	}
	else {
		ndk_error("alloc skb failed. len=%d", payload_size);
	}
	return skb;
}

#if !NETCFG_SKBC_SUPPORT_CLONE
struct sk_buff*	netdev_clone_skb(struct sk_buff *skb)
{
	pskb_t nskb = netdev_alloc_skb(skb->dev, skb->end - skb->head);
	if (!nskb)
		ndk_err_return(NULL);

	SKB_SANITY_CHECK(skb);

	nskb->data = nskb->head + (skb->data - skb->head);
	nskb->tail = nskb->head + (skb->tail - skb->head);
	nskb->len  = skb->len;
	nskb->skbc_srcaddr = skb->skbc_srcaddr;
	nskb->protocol = skb->protocol;

	memcpy(nskb->data, skb->data, skb->len);
	return nskb;
}
#endif

UINT32 netdev_ipaddr_get(struct net_device *netdev)
{
	struct netif *ni = netdev ? (struct netif*)netdev->netif : netif_find((char*)netdev->name);

	if (!ni)
		return 0;

	return (unsigned int)ni->ip_addr.addr;
}

/******************************************************************************
 * netif ethernet device functions
 ******************************************************************************/
/* Define those to better describe your network interface. */
#define IFNAME0 'e'
#define IFNAME1 'n'

int netif_global_init()
{
	NDK_ASSERT_EXPR(sizeof(((struct pbuf*)0)->skb_room) >= sizeof(struct sk_buff), {
		printf("skb_room too small: %d < %d\n", sizeof(((struct pbuf*)0)->skb_room), sizeof(struct sk_buff));
	});

	tcpip_init(NULL, NULL);
	ndk_cmd_dbgswitch_add("ifrx", &netif_dbg_rx);
	ndk_cmd_dbgswitch_add("iftx", &netif_dbg_tx);
	ndk_cmd_dbgswitch_add("ifoff", &netif_dbg_cutoff);
	return 0;
}

/**
 * In this function, the hardware should be initialized.
 * Called from ethernetif_init().
 *
 * @param netif the already initialized lwip network interface structure
 *        for this ethernetif
 */
static void low_level_init(struct netif *netif)
{
	struct net_device * netdev = (struct net_device *)netif->state;

	/* set MAC hardware address length */
	netif->hwaddr_len = NETIF_MAX_HWADDR_LEN;

	/* set MAC hardware address */
	unsigned char invalid_macaddr[6] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
	if (memcmp(netdev->dev_addr, invalid_macaddr, 6) == 0) {
		NDK_ASSERT(0);
	}

	memcpy(netif->hwaddr, netdev->dev_addr, NETIF_MAX_HWADDR_LEN);

	/* maximum transfer unit */
	netif->mtu = 1500;

	/* device capabilities */
	/* don't set NETIF_FLAG_ETHARP if this device is not an ethernet one */
	netif->flags = NETIF_FLAG_BROADCAST | NETIF_FLAG_ETHARP | NETIF_FLAG_IGMP;// | NETIF_FLAG_LINK_UP;

	/* Do whatever else is needed to initialize interface. */
}

/**
 * This function should do the actual transmission of the packet. The packet is
 * contained in the pbuf that is passed to the function. This pbuf
 * might be chained.
 *
 * @param netif the lwip network interface structure for this ethernetif
 * @param p the MAC packet to send (e.g. IP packet including MAC addresses and type)
 * @return ERR_OK if the packet could be sent
 *         an err_t value if the packet couldn't be sent
 *
 * @note Returning ERR_MEM here if a DMA queue of your MAC is full can lead to
 *       strange results. You might consider waiting for space in the DMA queue
 *       to become availale since the stack doesn't retry to send a packet
 *       dropped because of memory failure (except for the TCP timers).
 */
static err_t low_level_output(struct netif *netif, struct pbuf *p)
{
	struct net_device * netdev;
	pskb_t skb;
	err_t err = ERR_OK;

	netdev = (struct net_device *)netif->state;
	//NDK_ASSERT(netdev);
	//NDK_ASSERT(netif == netdev->netif);

	if (!(netif->flags & NETIF_FLAG_LINK_UP)) {
		profLogPuts(0, "netif link off.");
		return ERR_IF;
	}

	if (netif_dbg_cutoff) {
		return ERR_OK; // Cut off Tx.
	}

#if 0
	if (netdev->f_queue_stopped) {
		profLogPuts(0, "netif queue stopped");
		return ERR_IF;
	}
#endif

#if ETH_PAD_SIZE
	pbuf_header(p, -ETH_PAD_SIZE); /* drop the padding word */
#endif

	if (p->next) { // chained pbuf
		struct pbuf *q;
		UINT32 total_sz = 0;

		/*for (q = p; q; q = q->next) {
			total_sz += q->len;
		}*/
		total_sz = p->tot_len;

		skb = dev_alloc_skb(total_sz);
		if (!skb) {
			LWIPDBG("Out of memory");
			err = ERR_IF;
			goto lEXIT;
		}

		for (q = p; q; q = q->next) {
			memcpy(skb_put(skb, q->len), q->payload, q->len);
		}
		++netif_sta_tx_copy_count;
	}
	else {
		pbuf_ref(p); // Prevent the lwIP from freeing it

		skb = (pskb_t)&p->skb_room;
		skbc_init_header((pskbc_t)skb, sizeof(struct sk_buff)
			, p->payload, p->len
			, __free_pbuf_skb);
		skb_put(skb, p->len);

		++netif_sta_skb_alloced;
	}

	//skb->priority = 0;
	skb->protocol = 0;
	skb->dev  = netdev;

	/*if (netif_may_be_blocked(netdev)) {
		ndk_msleep(20);
	}*/

	/* Dump TCPIP packet for debug */
#if 0
	if (skb->data[0x0c] == 0x08 && skb->data[0x0d] == 0x00) { // IP
		if (skb->data[0x17] == 6) { // TCP
			u32_t seqnum;
			u16_t tcp_chksum;

			memcpy(&seqnum, skb->data + 0x26, 4);
			memcpy(&tcp_chksum, skb->data + 0x32, 2);
			seqnum = platform_ntohl(seqnum);
			tcp_chksum = platform_ntohs(tcp_chksum);

			profLogPrintf(0, "tcp l=%u sq=%u cs=%04x", skb->len, seqnum, tcp_chksum);
		}
	}
#endif
	netif_sta_tx_bytes += skb->len;
	++netif_sta_tx_count;

	if (netif_dbg_tx > 0) {
		ndk_hexdump("lwip tx:", skb->data, skb->len, netif_dbg_tx);
	}

	NDK_TCHART_S(IFTX);
	if (netdev->netdev_ops->ndo_start_xmit(skb, netdev) == 0) {
		if (pcap_handler) {
			int err = ndk_pcap_write(&pcap_handler, skb->data, skb->len);
			if (err) {
				ndk_error("netif pcap fail %d", err);
				ndk_pcap_destroy(&pcap_handler);
			}
		}
	}
	else {
		dev_kfree_skb(skb);
		profLogPuts(0, "ndo_start_xmit failed");
		//printf("ndo_start_xmit failed\n");
		err = ERR_IF;
	}
	NDK_TCHART_E(IFTX);

lEXIT:
#if ETH_PAD_SIZE
	pbuf_header(p, ETH_PAD_SIZE); /* reclaim the padding word */
#endif

	return err;
}

#if 0
static void pbuf_dump(struct pbuf* pb)
{
    int i, col = 0;
    unsigned char *pdata = (unsigned char *)pb->payload;

    for (i = 0; i < pb->len; i++) {
        if (col == 0) {
            printk("0x%02X: ", i);
        }

        printk(" %02X", pdata[i]);

        if (++col == 16) {
            printk("\n");
            col = 0;
        }
    }
    if (col) {
        printk("\n");
    }
}
#endif

/**
 * This function should be called when a packet is ready to be read
 * from the interface. It uses the function low_level_input() that
 * should handle the actual reception of bytes from the network
 * interface. Then the type of the received packet is determined and
 * the appropriate input function is called.
 *
 * @param netif the lwip network interface structure for this ethernetif
 */
static int ether_netif_input(struct netif *netif, struct pbuf* pb)
{
	struct net_device * netdev;
	struct eth_hdr *ethhdr;
	netdev = (struct net_device *)netif->state;
	int ret = 0 ;
	/* points to packet payload, which starts with an Ethernet header */
#if ETH_PAD_SIZE
	pbuf_header(pb, ETH_PAD_SIZE);
#endif
	ethhdr = (struct eth_hdr*)pb->payload;

	switch (ethhdr->type) {
		/* IP or ARP packet? */
	case OSA_HTONS(ETH_P_IP): //ETHTYPE_IP:
	case OSA_HTONS(ETH_P_ARP): //ETHTYPE_ARP:
#if PPPOE_SUPPORT
		/* PPPoE packet? */
	case OSA_HTONS(ETH_P_PPP_DISC): //ETHTYPE_PPPOEDISC:
	case OSA_HTONS(ETH_P_PPP_SES): //ETHTYPE_PPPOE:
#endif /* PPPOE_SUPPORT */
		/* full packet send to tcpip_thread to process */
		if (netif->input(pb, netif) != ERR_OK) {
			LWIPDBG("IP input error: %d", pb->len);
			ret = ERR_BUF ;
			goto FREE_PBUF;
		}
		break;

	case OSA_HTONS(ETH_P_PAE): {
		pskb_t skb = (pskb_t)&pb->skb_room;
		skb->protocol = OSA_NTOHS(ETH_P_PAE);
		netif_l2_packet_rx(netdev, skb);
		goto FREE_PBUF; }

	default:
		//ndk_error("Undesirable packet received.");
		goto FREE_PBUF;
	}

	return ret;

FREE_PBUF:
#if ETH_PAD_SIZE
	pbuf_header(pb, -ETH_PAD_SIZE);
#endif
	pbuf_free(pb);
	return ret;
}

/**
 * Should be called at the beginning of the program to set up the
 * network interface. It calls the function low_level_init() to do the
 * actual setup of the hardware.
 *
 * This function should be passed as a parameter to netif_add().
 *
 * @param netif the lwip network interface structure for this ethernetif
 * @return ERR_OK if the loopif is initialized
 *         ERR_MEM if private data couldn't be allocated
 *         any other err_t on error
 */
static err_t ether_netif_init(struct netif *netif)
{
	NDK_ASSERT(netif);
	NDK_ASSERT(netif->state);

#if LWIP_NETIF_HOSTNAME
	/* Initialize interface hostname */
	netif->hostname = ndk_gethostname();
#endif /* LWIP_NETIF_HOSTNAME */

	/*
	 * Initialize the snmp variables and counters inside the struct netif.
	 * The last argument should be replaced with your link speed, in units
	 * of bits per second.
	 */
	NETIF_INIT_SNMP(netif, snmp_ifType_ethernet_csmacd, LINK_SPEED_OF_YOUR_NETIF_IN_BPS);

	netif->name[0] = IFNAME0;
	netif->name[1] = IFNAME1;
	/* We directly use etharp_output() here to save a function call.
	 * You can instead declare your own function an call etharp_output()
	 * from it if you have to do some checks before sending (e.g. if link
	 * is available...) */
	netif->output = etharp_output;
	netif->linkoutput = low_level_output;

	/* initialize the hardware */
	low_level_init(netif);

	return ERR_OK;
}

int netif_register_device(struct net_device *netdev)
{
	struct netif * ni;
	if (!netdev || netdev->netif)
		ndk_err_return(-1);

	ni = (struct netif *)ndk_mem_zalloc(sizeof(struct netif));
	if (!ni)
		ndk_err_return(-1);
	netdev->netif = ni;

	ip_addr_t ipaddr, netmask, gw;
	ipaddr.addr = htonl(netif_static_ipaddr);
	netmask.addr = htonl(0xFFFFFF00);
	gw.addr = 0x00; //htonl(0xC0A80001);

	if (netif_add(ni, &ipaddr, &netmask, &gw, netdev, ether_netif_init, tcpip_input) == NULL) {
		ndk_err_return(-1);
	}
	netif_set_up(ni);

	netif_static_ipaddr -= 0x100;
	return 0;
}

int netif_unregister_device(struct net_device *netdev)
{
	struct netif * ni;
	if (!netdev || !netdev->netif)
		ndk_err_return(-1);

	ni = netdev->netif;
	netif_set_down(ni);
	netif_remove(ni);
	ndk_mem_free(ni);
	ni = NULL;

	return 0;
}

static struct netif *find_netif(const char* ifname)
{
	struct netif *ni;
	struct net_device *netdev;

	netdev = __netdev_getbyname(ifname);
	ni = netdev ? (struct netif*)netdev->netif : netif_find((char*)ifname);

	return ni;
}

static int ioctl_siocgifconf(struct ifconf *ifc)
{
	struct ifreq  *ifr;
	struct netif  *ni;
	struct sockaddr_in *sin;
	int  len  = ifc->ifc_len;
	int  done = 0;
	const char *devname;

	// count netif number
	ifr = (struct ifreq *)ifc->ifc_ifcu.ifcu_buf;
	for (ni = netif_list; ni; ni = ni->next) {
		if (!ifr) {
			// Calculate size
			done += sizeof(struct ifreq);
			continue;
		}
		if (len < (int)sizeof(struct ifreq))
			break;

		memset(ifr, 0, sizeof(struct ifreq));

		devname = netif_get_devname(ni);
		if (devname) {
			strncpy(ifr->ifr_ifrn.ifrn_name, devname, IFNAMSIZ);
		}
		else {
			ifr->ifr_ifrn.ifrn_name[0] = ni->name[0];
			ifr->ifr_ifrn.ifrn_name[1] = ni->name[1];
			ifr->ifr_ifrn.ifrn_name[2] = ni->num + '0';
		}

		sin = (struct sockaddr_in*)&ifr->ifr_ifru.ifru_addr;
		sin->sin_family = AF_INET;
		sin->sin_addr.s_addr = ni->ip_addr.addr;

		++ifr;
		len  -= sizeof(struct ifreq);
		done += sizeof(struct ifreq);
	}

	ifc->ifc_len = done;
	return 0;
}

int netif_ioctl(int cmd, void *arg)
{
	int err = 0;

	if (cmd == SIOCGIFCONF) {
		err = ioctl_siocgifconf((struct ifconf*)arg);
		ndk_errno_return(err, "cmd=%x", cmd);
	}

	struct ifreq *ifr = (struct ifreq *)arg;
	struct netif *ni  = find_netif(ifr->ifr_ifrn.ifrn_name);

	if (!ni) {
		ndk_errno_return(-1, "cmd=%x dev=%s", cmd, ifr->ifr_ifrn.ifrn_name);
	}
	//profLogPrintf(0, "%s ioctl: 0x%x", ifr->ifr_ifrn.ifrn_name, cmd);

	switch (cmd) {
	case SIOCGIFADDR:	/* Get interface address */
	//case SIOCGIFBRDADDR:	/* Get the broadcast address */
	//case SIOCGIFDSTADDR:	/* Get the destination address */
	case SIOCGIFNETMASK: {	/* Get the netmask for the interface */
		ifr->ifr_ifru.ifru_sin.sin_family = AF_INET;
		ifr->ifr_ifru.ifru_sin.sin_addr.s_addr = (cmd == SIOCGIFADDR) ? ni->ip_addr.addr : ni->netmask.addr;
		break; }

	case SIOCADDRT: // gateway
	case SIOCSIFADDR:
	case SIOCSIFNETMASK: {
		ip_addr_t ipa, nm, gw;

		ipa.addr = cmd == SIOCSIFADDR	 ? ifr->ifr_ifru.ifru_sin.sin_addr.s_addr : ni->ip_addr.addr;
		nm.addr  = cmd == SIOCSIFNETMASK ? ifr->ifr_ifru.ifru_sin.sin_addr.s_addr : ni->netmask.addr;
		gw.addr  = cmd == SIOCADDRT	 ? ifr->ifr_ifru.ifru_sin.sin_addr.s_addr : ni->gw.addr;

		netif_set_addr(ni, &ipa, &nm, &gw);
		break; }

	case SIOCGIFHWADDR:
		memset(&ifr->ifr_ifru.ifru_hwaddr, 0, sizeof(ifr->ifr_ifru.ifru_hwaddr));
		memcpy(ifr->ifr_ifru.ifru_hwaddr.sa_data, ni->hwaddr, sizeof(ni->hwaddr));
		break;

	case SIOCGIFFLAGS:
		ifr->ifr_ifru.ifru_flags = 0;
		if (ni->flags & NETIF_FLAG_UP)
			ifr->ifr_ifru.ifru_flags |= IFF_UP | IFF_RUNNING;
		if (ni->flags & NETIF_FLAG_BROADCAST)
			ifr->ifr_ifru.ifru_flags |= IFF_BROADCAST;
		if (ni->flags & NETIF_FLAG_POINTTOPOINT)
			ifr->ifr_ifru.ifru_flags |= IFF_POINTOPOINT;
		if (!(ni->flags & NETIF_FLAG_ETHARP))
			ifr->ifr_ifru.ifru_flags |= IFF_NOARP;
		break;

	case SIOCGIFINDEX: {
		ifr->ifr_ifru.ifru_ivalue = if_nametoindex(ifr->ifr_ifrn.ifrn_name);
		break; }

	case SIOCSIFFLAGS: {
		err = -1;
		struct net_device *netdev = (struct net_device *)ni->state;
		if (!netdev)
			break;

		if (ifr->ifr_ifru.ifru_flags & IFF_UP) {
			set_bit(__LINK_STATE_START, &netdev->state);

			if (netdev->netdev_ops && netdev->netdev_ops->ndo_open)
				err = netdev->netdev_ops->ndo_open(netdev);

			if (err) {
				clear_bit(__LINK_STATE_START, &netdev->state);
			}
		}
		else {
			clear_bit(__LINK_STATE_START, &netdev->state);

			if (netdev->netdev_ops && netdev->netdev_ops->ndo_stop)
				err = netdev->netdev_ops->ndo_stop(netdev);
		}
		break; }

	default:
		err = netdev_ioctl(cmd, (struct net_device *)ni->state, arg);
		break;
	}

	ndk_errno_return(err, "cmd=%x", cmd);
}

const char *netif_get_devname(void *netif)
{
	if (netif) {
		unsigned index = 0;
		struct net_device *netdev;

		while ((netdev = netdev_getbyindex(++index)) != NULL) {
			if (netdev->netif == netif)
				return netdev->name;
		}
	}

	return NULL;
}

static void do_arp_request(struct netifapi_msg_msg *msg)
{
	etharp_remove_addr(msg->netif, msg->msg.add.ipaddr);
	msg->err = etharp_request(msg->netif, msg->msg.add.ipaddr);
	TCPIP_NETIFAPI_ACK(msg);
}

static void do_arp_query(struct netifapi_msg_msg *msg)
{
	ip_addr_t *ia_ret;
	struct eth_addr *ea;

	if (etharp_find_addr(msg->netif, msg->msg.add.ipaddr, &ea, &ia_ret) >= 0) {
		memcpy(msg->msg.add.state, ea, sizeof(*ea));
		msg->err = ERR_OK;
	}
	else
		msg->err = ERR_IF;
	TCPIP_NETIFAPI_ACK(msg);
}

int netif_priv_ioctl(int cmd, struct ifreq *ifr)
{
	struct netif *ni;

	if (cmd == NETIF_IOCG_DEFAULT) {
		ni = netif_get_default();
		if (ni) {
			struct net_device *netdev = (struct net_device *)ni->state;
			if (netdev) {
				ifreq_set_name(ifr, netdev->name);
			}
			else {
				ifr->ifr_ifrn.ifrn_name[0] = ni->name[0];
				ifr->ifr_ifrn.ifrn_name[1] = ni->name[1];
				ifr->ifr_ifrn.ifrn_name[2] = ni->num + '0';
				ifr->ifr_ifrn.ifrn_name[3] = 0;
			}
		}
		else
			ifreq_set_name(ifr, "");

		return 0;
	}

	ni = find_netif(ifr->ifr_ifrn.ifrn_name);
	ip_addr_t *ipaddr = (ip_addr_t*)&ifr->ifr_ifru.ifru_sin.sin_addr;
	int ret = 0;

	if (!ni)
		ndk_errno_return(-1, "cmd=%x netif not found", cmd);

	switch (cmd) {
	case NETIF_IOCS_UP:
		netif_set_up(ni);
		break;

	case NETIF_IOCS_DOWN:
		netif_set_down(ni);
		break;

	case NETIF_IOCS_ARP_REQUEST: {
		//profLogPrintf(0, "NDK_IOCS_ARP_REQ: 0x%08x", ipaddr->addr);
		struct netifapi_msg msg;
		msg.function = do_arp_request;
		msg.msg.netif = ni;
		msg.msg.msg.add.ipaddr = ipaddr;
		TCPIP_NETIFAPI(&msg);
		break; }

	case NETIF_IOCS_ARP_QUERY: {
		//profLogPrintf(0, "NETIF_IOCS_ARP_QUERY: 0x%08x", ipaddr->addr);
		struct netifapi_msg msg;
		msg.function = do_arp_query;
		msg.msg.netif = ni;
		msg.msg.msg.add.ipaddr = ipaddr;
		msg.msg.msg.add.state = ifr->ifr_ifru.ifru_addr.sa_data;
		if (TCPIP_NETIFAPI(&msg) != ERR_OK)
			ret = -1;
		break; }

	case NETIF_IOCS_DEFAULT:
		netif_set_default(ni);
		break;

	case NETIF_IOCS_PCAP_START:
		if (!pcap_handler) {
			NDKPcapArg *arg = (NDKPcapArg*)ifr->ifr_ifru.ifru_data;
			ret = ndk_pcap_init(&pcap_handler, arg);
		}
		break;

	case NETIF_IOCS_PCAP_STOP:
		if (pcap_handler) {
			ndk_pcap_destroy(&pcap_handler);
			pcap_handler = NULL;
		}
		break;

	case NETIF_IOCG_GATEWAY: // gateway
		ifr->ifr_ifru.ifru_sin.sin_addr.s_addr = ni->gw.addr;
		break;

	case NETIF_IOCG_CARRIER:
		ifr->ifr_ifru.ifru_ivalue = (ni->flags & NETIF_FLAG_LINK_UP) ? 1 : 0;
		break;

	default:
		ret = -1;
		break;
	}

	ndk_errno_return(ret, "cmd=%x", cmd);
}

struct net_device *netdev_getbyname(const char* name)
{
	struct net_device* dev = __netdev_getbyname(name);
	if (!dev) {
		struct netif *ni = netif_find((char*)name);
		if (ni)
			dev = (struct net_device*)ni->state;
	}

	return dev;
}

int netif_rx(struct sk_buff *skb)
{
	NDK_ASSERT(skb);
	NDK_ASSERT(skb->dev);

	if (netif_dbg_cutoff) { // Cut off Rx
		dev_kfree_skb(skb);
		return 0;
	}

#if NETCFG_SKBC_SUPPORT_CLONE
	if (skb->cloned) {
		pskb_t nskb = netdev_alloc_skb(skb->dev, skb->len+2);
		skb_reserve(nskb, 2); // Align the iphdr address to the boundary of 4.
		memcpy(skb_put(nskb, skb->len), skb->data, skb->len);
		SKBC_FREE(skb);
		skb = nskb;
	}
#endif

	if (netif_dbg_rx > 0) {
		ndk_hexdump("netif rx:", skb->data, skb->len, netif_dbg_rx);
		//skbc_sanity_check((pskbc_t)skb);
	}

	struct netif *ni = (struct netif *)skb->dev->netif;
	if (ni->flags & NETIF_FLAG_LINK_UP) {
		struct pbuf* pb = container_of((pbuf_skb_room_t*)skb, struct pbuf, skb_room);
		pb->next    = NULL;
		pb->payload = skb->data;
		pb->len     = skb->len;
		pb->tot_len = skb->len;
		//NDK_ASSERT_EXPR(pb->ref == 1, printf("pb=%p, skb=%p %d\n", pb, skb, pb->ref));

		++netif_sta_rx_count;
		netif_sta_rx_bytes += pb->len;

		if (pcap_handler) {
			int err = ndk_pcap_write(&pcap_handler, skb->data, skb->len);
			if (err) {
				ndk_error("netif pcap fail %d", err);
				ndk_pcap_destroy(&pcap_handler);
			}
		}

		NDK_TCHART_S(IFRX);
		int ret = ether_netif_input(skb->dev->netif, pb);
		NDK_TCHART_E(IFRX);
		return ret;
	}
	else {
		dev_kfree_skb(skb);
		ndk_err_return(-1);
	}
	return -1;
}

void netif_carrier_on(struct net_device * dev)
{
	NDK_ASSERT(dev);
	if (dev && dev->netif) {
		struct netif *nif = (struct netif*)dev->netif;
		nif->flags |= NETIF_FLAG_LINK_UP;
		profLogPrintf(0, "netif %c%c%d carrier on", nif->name[0], nif->name[1], nif->num);
	}
}

void netif_carrier_off(struct net_device * dev)
{
	NDK_ASSERT(dev);
	if (dev && dev->netif) {
		struct netif *nif = (struct netif*)dev->netif;
		nif->flags &= ~NETIF_FLAG_LINK_UP;
		profLogPrintf(0, "netif %c%c%d carrier off", nif->name[0], nif->name[1], nif->num);
	}
}

int _ndk_dhcp_start(const char *ifname)
{
	struct netif *ni = find_netif(ifname);
	if (!ni)
		ndk_err_return(-1);

	return netifapi_dhcp_start(ni);
}

void _ndk_dhcp_stop(const char *ifname)
{
	struct netif *ni = find_netif(ifname);
	if (ni)
		netifapi_dhcp_stop(ni);
}

