#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include "api/sp5k_os_api.h"
#include "api/sp5k_global_api.h"

#include <lwip/sockets.h>
#include <ndk/common.h>
#include "arch/sys_arch.h"
#include "lwip/sys.h"

static SP5K_MUTEX g_sysarch_mutex;

#undef  NDK_ASSERT
#define NDK_ASSERT(...) ((void)0)

void sys_init(void)
{
	if (sp5kOsMutexCreate(&g_sysarch_mutex, "sysarch", 0) != 0)
		NDK_ASSERT(0);
}

/*
 * Mutex
 */
err_t sys_mutex_new(sys_mutex_t *mutex)
{
	if (sp5kOsMutexCreate(mutex, "lwipmutex", 0) != 0)
		return ERR_MEM;
	NDK_ASSERT(!(*mutex & 0x01));
	return ERR_OK;
}

void sys_mutex_lock(sys_mutex_t *mutex)
{
	NDK_ASSERT(mutex && *mutex && !(*mutex & 0x01));
	sp5kOsMutexGet(mutex, TX_WAIT_FOREVER);
}

int sys_mutex_trylock(sys_mutex_t *mutex)
{
	NDK_ASSERT(mutex && *mutex && !(*mutex & 0x01));
	if (sp5kOsMutexGet(mutex, TX_NO_WAIT) == SUCCESS)
		return 1;
	return 0;
}

void sys_mutex_unlock(sys_mutex_t *mutex)
{
	NDK_ASSERT(mutex && *mutex && !(*mutex & 0x01));
	sp5kOsMutexPut(mutex);
}

void sys_mutex_free(sys_mutex_t *mutex)
{
	NDK_ASSERT(mutex && *mutex);
	sys_mutex_t m = *mutex & ~0x01;
	sp5kOsMutexDelete(&m);
}

int sys_mutex_valid(sys_mutex_t *mutex)
{
	NDK_ASSERT(mutex && *mutex);
	if (!mutex)
		return 0;
	return *mutex && ((*mutex & 0x01) ? 0 : 1);
}

void sys_mutex_set_invalid(sys_mutex_t *mutex)
{
	NDK_ASSERT(mutex);
	*mutex |= 0x01;
}

/*
 * Semaphore
 * It's used as event.
 */
err_t sys_sem_new(sys_sem_t *sem, u8_t count)
{
	NDK_ASSERT(count == 0);

	if (sp5kOsEventFlagsCreate(sem, "lwipevt") != 0)
		return ERR_MEM;

#ifdef SEM_DEBUG
	unsigned long *event_flags;
	unsigned int offset = sizeof(unsigned long) + sizeof(char*);
	event_flags = (unsigned long*)((*sem) + offset);
	NDK_ASSERT(!(*event_flags & 0x01));
#endif

	return ERR_OK;
}

void sys_sem_signal(sys_sem_t *sem)
{
	NDK_ASSERT(sem && *sem);

	sp5kOsEventFlagsSet(sem, 0x01, TX_OR);

#ifdef SEM_DEBUG
	unsigned long *event_flags;
	unsigned int offset = sizeof(unsigned long) + sizeof(char*);

	event_flags = (unsigned long*)((*sem) + offset);
	ndk_info("[%s:%d] sema:0x%x\n", __FUNCTION__, __LINE__, *event_flags);
	NDK_ASSERT(!(*event_flags & 0x01));
#endif
}

// @see http://lwip.wikia.com/wiki/Porting_for_an_OS
u32_t sys_arch_sem_wait(sys_sem_t* sem, u32_t timeout)
{
	NDK_ASSERT(sem && *sem);

	struct timeval t0, t1;
	u32_t r, to = timeout, flag = 0;
	long elapse;

#ifdef SEM_DEBUG
	unsigned long *event_flags;
	unsigned int offset = sizeof(unsigned long) + sizeof(char*);

	event_flags = (unsigned long*)((*sem) + offset);
	ndk_info("[%s:%d] sema:0x%x\n", __FUNCTION__, __LINE__, *event_flags);
#endif

	tmrTimeStampGet(&t0);
AGAIN:
	r = sp5kOsEventFlagsGet(sem, 0x01, TX_OR_CLEAR, &flag, (to == 0) ? TX_WAIT_FOREVER : to);
	tmrTimeStampGet(&t1);
	elapse = timeval_diff_ms(&t1, &t0);

	if (r != SUCCESS && elapse < timeout) {
		to = timeout - elapse;
		goto AGAIN;
	}

	return (r == SUCCESS && flag == 0x01) ? elapse : SYS_ARCH_TIMEOUT;
}

void sys_sem_free(sys_sem_t *sem)
{
	NDK_ASSERT(sem && *sem);

	sp5kOsEventFlagsDelete(sem);
}

int sys_sem_valid(sys_sem_t *sem)
{
	if (!*sem)
		return 0;

	return 1;
}

void sys_sem_set_invalid(sys_sem_t *sem)
{
	sem = NULL;
}

/*
 * Mailbox
 */

err_t sys_mbox_new(sys_mbox_t *mbox, int size)
{
	NDK_ASSERT(size == 0);

	if (sp5kOsQueueCreate(mbox, "lwipmbox", 1, NULL, LWIP_MBOX_SIZE*sizeof(void*)) != 0)
		return ERR_MEM;
	NDK_ASSERT(!(*mbox & 0x01));

	return ERR_OK;
}

void sys_mbox_post(sys_mbox_t *mbox, void *msg)
{
	NDK_ASSERT(mbox && *mbox && !(*mbox & 0x01));
	sp5kOsQueueSend(mbox, &msg, TX_WAIT_FOREVER);
}

err_t sys_mbox_trypost(sys_mbox_t *mbox, void *msg)
{
	NDK_ASSERT(mbox && *mbox && !(*mbox & 0x01));
	UINT32 r = sp5kOsQueueSend(mbox, &msg, TX_NO_WAIT);
	if (r != 0)
		NDK_TCHART_P(LWIP, "mbox");
	return (r == 0) ? ERR_OK : ERR_TIMEOUT;
}

// @see http://lwip.wikia.com/wiki/Porting_for_an_OS
u32_t sys_arch_mbox_fetch(sys_mbox_t *mbox, void **msg, u32_t timeout)
{
	NDK_ASSERT(mbox && *mbox && !(*mbox & 0x01));

	struct timeval t0, t1;
	long elapse;
	u32_t r, to = timeout;

	tmrTimeStampGet(&t0);
AGAIN:
	r = sp5kOsQueueReceive(mbox, msg, (to == 0) ? TX_WAIT_FOREVER : to);
	tmrTimeStampGet(&t1);
	elapse = timeval_diff_ms(&t1, &t0);

	if (r != SUCCESS && elapse < timeout) {
		to = timeout - elapse;
		goto AGAIN;
	}

	return (r == SUCCESS) ? elapse : SYS_ARCH_TIMEOUT;
}

u32_t sys_arch_mbox_tryfetch(sys_mbox_t *mbox, void **msg)
{
	NDK_ASSERT(mbox && *mbox && !(*mbox & 0x01));
	UINT32 r;

	r = sp5kOsQueueReceive(mbox, msg, TX_NO_WAIT);
	if (r == 0) {
		return 0;
	}
	else {
		return SYS_MBOX_EMPTY;
	}
}

void sys_mbox_free(sys_mbox_t *mbox)
{
	NDK_ASSERT(mbox && *mbox);
	sys_mbox_t m = *mbox & ~0x01;
	sp5kOsQueueDelete(&m);
}

int sys_mbox_valid(sys_mbox_t *mbox)
{
	NDK_ASSERT(mbox && *mbox);
	if (!mbox)
		return 0;
	return *mbox && ((*mbox & 0x01) ? 0 : 1);
}

void sys_mbox_set_invalid(sys_mbox_t *mbox)
{
	NDK_ASSERT(mbox);
	*mbox |= 0x01;
}

sys_thread_t sys_thread_new(const char *name, lwip_thread_fn thread, void *arg, int stacksize, int prio)
{
	void *tid = ndk_thread_create(name, 0, thread, arg, 4096-256);
	NDK_ASSERT(tid);
	return (sys_thread_t)tid;
}

sys_prot_t sys_arch_protect()
{
	sp5kOsMutexGet(&g_sysarch_mutex, TX_WAIT_FOREVER);
	return 0;
}

void sys_arch_unprotect(sys_prot_t pval)
{
	sp5kOsMutexPut(&g_sysarch_mutex);
}


#ifdef LWIP_DEBUG

unsigned int lwip_dbg_flags = LWIP_DBG_LEVEL_SERIOUS | LWIP_DBG_LEVEL_SEVERE;

#define LWIP_DBG_DEF(s, f) else if (!strcasecmp(name, s)) { if (on) lwip_dbg_flags |= (f); else lwip_dbg_flags &= ~(f); }

void lwip_set_dbg(const char* name, char on)
{
	if (0) {}
	LWIP_DBG_DEF("PBUF"      , PBUF_DEBUG            )
	LWIP_DBG_DEF("API_LIB"   , API_LIB_DEBUG         )
	LWIP_DBG_DEF("API_MSG"   , API_MSG_DEBUG         )
	LWIP_DBG_DEF("TCPIP"     , TCPIP_DEBUG           )
	LWIP_DBG_DEF("NETIF"     , NETIF_DEBUG           )
	LWIP_DBG_DEF("SOCKETS"   , SOCKETS_DEBUG         )
	LWIP_DBG_DEF("DEMO"      , DEMO_DEBUG            )
	LWIP_DBG_DEF("IP_DEBUG"  , IP_DEBUG              )
	LWIP_DBG_DEF("IP_REASS"  , IP_REASS_DEBUG        )
	LWIP_DBG_DEF("ICMP"      , ICMP_DEBUG            )
	LWIP_DBG_DEF("UDP"       , UDP_DEBUG             )
	LWIP_DBG_DEF("TCP"       , TCP_DEBUG             )
	LWIP_DBG_DEF("TCP_INPUT" , TCP_INPUT_DEBUG       )
	LWIP_DBG_DEF("TCP_OUTPUT", TCP_OUTPUT_DEBUG      )
	LWIP_DBG_DEF("TCP_RTO"   , TCP_RTO_DEBUG         )
	LWIP_DBG_DEF("TCP_CWND"  , TCP_CWND_DEBUG        )
	LWIP_DBG_DEF("TCP_WND"   , TCP_WND_DEBUG         )
	LWIP_DBG_DEF("TCP_FR"    , TCP_FR_DEBUG          )
	LWIP_DBG_DEF("TCP_QLEN"  , TCP_QLEN_DEBUG        )
	LWIP_DBG_DEF("TCP_RST"   , TCP_RST_DEBUG         )
	LWIP_DBG_DEF("DHCP"      , DHCP_DEBUG            )
	LWIP_DBG_DEF("MEM"       , MEM_DEBUG             )
	LWIP_DBG_DEF("MEMP"      , MEMP_DEBUG            )
	LWIP_DBG_DEF("PPP"       , PPP_DEBUG             )
	LWIP_DBG_DEF("ON"        , LWIP_DBG_ON           )
	LWIP_DBG_DEF("TRACE"     , LWIP_DBG_TRACE        )
	LWIP_DBG_DEF("STATE"     , LWIP_DBG_STATE        )
	LWIP_DBG_DEF("WARN"      , LWIP_DBG_LEVEL_WARNING)
	LWIP_DBG_DEF("SERI"      , LWIP_DBG_LEVEL_SERIOUS)
	LWIP_DBG_DEF("SEVE"      , LWIP_DBG_LEVEL_SEVERE )
	LWIP_DBG_DEF("all"       , 0xFFFFFFFF            )
	else if (!strcasecmp(name, "normal")) {
		if (on)
			lwip_dbg_flags |= (TCPIP_DEBUG | IP_DEBUG | SOCKETS_DEBUG);
		else
			lwip_dbg_flags &= ~(TCPIP_DEBUG | IP_DEBUG | SOCKETS_DEBUG);
	}
	else {
		ndk_puts(
			"PBUF           API_LIB         API_MSG         TCPIP           NETIF\n"
			"SOCKETS        DEMO            IP_DEBUG        IP_REASS        ICMP\n"
			"UDP            TCP             TCP_INPUT       TCP_OUTPUT      TCP_RTO\n"
			"TCP_CWND       TCP_WND         TCP_FR          TCP_QLEN        TCP_RST\n"
			"DHCP           MEM             MEMP            PPP             ON\n"
			"TRACE          STATE           WARN            SERI            SEVE\n"
			"all\n"
		);
	}

	printf("%s=>%d, current flags=0x%X\n", name, on, lwip_dbg_flags);
}
#endif

UINT32 lwip_sta_mem_alloced = 0;
UINT32 lwip_sta_mem_freed = 0;

void lwip_mem_free(void *p)
{
	++lwip_sta_mem_freed;
	ndk_mem_free(p);
}

void *lwip_mem_malloc(size_t size)
{
	++lwip_sta_mem_alloced;
	return ndk_mem_malloc(size);
}

