/*
 * SPI testing utility (using spidev driver)
 *
 * Copyright (c) 2007  MontaVista Software, Inc.
 * Copyright (c) 2007  Anton Vorontsov <avorontsov@ru.mvista.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License.
 *
 * Cross-compile with cross-gcc -I/path/to/cross-kernel/include
 */

#include <stdint.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/types.h>
#include <linux/spi/spidev.h>

#define SPI_IOC_EEPROM_READ		_IOWR(SPI_IOC_MAGIC, 5, __u32)
#define SPI_IOC_FLASHID_READ	_IOWR(SPI_IOC_MAGIC, 6, __u32)
#define SPI_IOC_EEPROM_WRITE	_IOWR(SPI_IOC_MAGIC, 7, __u32)

#define ARRAY_SIZE(a) (sizeof(a) / sizeof((a)[0]))
typedef unsigned char u8;


static unsigned char BitReverse(unsigned char byData)
{
	unsigned char byOut;

	byOut = ((byData<<7)&0x80) | ((byData<<5)&0x40) | ((byData<<3)&0x20) | ((byData<<1)&0x10) |
			((byData>>1)&0x08) | ((byData>>3)&0x04) | ((byData>>5)&0x02) | ((byData>>7)&0x01);

	return byOut;
}

static void pabort(const char *s)
{
	perror(s);
	abort();
}

static const char *device = "/dev/spidev0.1";
static uint8_t mode;
static uint8_t bits = 8;
static uint32_t speed = 5000000;
static uint16_t delay;

static void transfer(int fd)
{
	int ret;
	uint8_t tx[] = {
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0x40, 0x00, 0x00, 0x00, 0x00, 0x95,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xDE, 0xAD, 0xBE, 0xEF, 0xBA, 0xAD,
		0xF0, 0x0D,
	};
	uint8_t rx[ARRAY_SIZE(tx)] = {0};
	struct spi_ioc_transfer tr = {
		.tx_buf = (unsigned long)tx,
		.rx_buf = (unsigned long)rx,
		.len = ARRAY_SIZE(tx),
		.delay_usecs = delay,
		.speed_hz = speed,
		.bits_per_word = bits,
	};

	ret = ioctl(fd, SPI_IOC_MESSAGE(1), &tr);
	if (ret < 1)
		pabort("can't send spi message");

	for (ret = 0; ret < ARRAY_SIZE(tx); ret++) {
		if (!(ret % 6))
			puts("");
		printf("%.2X ", rx[ret]);
	}
	puts("");
}

static void print_usage(const char *prog)
{
	printf("Usage: %s [-DsbdlHOLC3]\n", prog);
	puts("  -D --device   device to use (default /dev/spidev1.1)\n"
	     "  -s --speed    max speed (Hz)\n"
	     "  -d --delay    delay (usec)\n"
	     "  -b --bpw      bits per word \n"
	     "  -l --loop     loopback\n"
	     "  -H --cpha     clock phase\n"
	     "  -O --cpol     clock polarity\n"
	     "  -L --lsb      least significant bit first\n"
	     "  -C --cs-high  chip select active high\n"
	     "  -3 --3wire    SI/SO signals shared\n");
	exit(1);
}

static void parse_opts(int argc, char *argv[])
{
	while (1) {
		static const struct option lopts[] = {
			{ "device",  1, 0, 'D' },
			{ "speed",   1, 0, 's' },
			{ "delay",   1, 0, 'd' },
			{ "bpw",     1, 0, 'b' },
			{ "loop",    0, 0, 'l' },
			{ "cpha",    0, 0, 'H' },
			{ "cpol",    0, 0, 'O' },
			{ "lsb",     0, 0, 'L' },
			{ "cs-high", 0, 0, 'C' },
			{ "3wire",   0, 0, '3' },
			{ "no-cs",   0, 0, 'N' },
			{ "ready",   0, 0, 'R' },
			{ NULL, 0, 0, 0 },
		};
		int c;

		c = getopt_long(argc, argv, "D:s:d:b:lHOLC3NR", lopts, NULL);

		if (c == -1)
			break;

		switch (c) {
		case 'D':
			device = optarg;
			break;
		case 's':
			speed = atoi(optarg);
			break;
		case 'd':
			delay = atoi(optarg);
			break;
		case 'b':
			bits = atoi(optarg);
			break;
		case 'l':
			mode |= SPI_LOOP;
			break;
		case 'H':
			mode |= SPI_CPHA;
			break;
		case 'O':
			mode |= SPI_CPOL;
			break;
		case 'L':
			mode |= SPI_LSB_FIRST;
			break;
		case 'C':
			mode |= SPI_CS_HIGH;
			break;
		case '3':
			mode |= SPI_3WIRE;
			break;
		case 'N':
			mode |= SPI_NO_CS;
			break;
		case 'R':
			mode |= SPI_READY;
			break;
		default:
			print_usage(argv[0]);
			break;
		}
	}
}

int fd;
int imx_write(int fd, unsigned short address,unsigned short data)
{
        int st=0;
        u8 writecmd[4]={0};

        writecmd[0] =BitReverse((u8)(address & 0xff));
        writecmd[1] =BitReverse((u8)((address >> 8) & 0xff));
        writecmd[2] =BitReverse(data&0xff);
        writecmd[3] =BitReverse((data>>8)&0xff);
        //printf("writecmd[0]=%02x,writecmd[1]=%02x,writecmd[2]=%02x,writecmd[3]=%02x\n",writecmd[0],writecmd[1],writecmd[2],writecmd[3]);

        //write(fd, writecmd, sizeof(writecmd) );
	ioctl(fd, SPI_IOC_EEPROM_WRITE, writecmd );		

        return st;
}

int imx_read(int fd, unsigned short address,unsigned short *data)
{
        u8 readcmd[2]={0};
        u8 readdata[2]={0};
	u8 arg[4] = {0, 0, 0, 0};

        printf("[%04x]:", address);
        readcmd[0] =BitReverse((address|0x80) & 0xff);
        readcmd[1] =BitReverse(((address)>>8) & 0xff);

	arg[0] =  readcmd[0];
	arg[1] =  readcmd[1];
	ioctl(fd, SPI_IOC_EEPROM_READ, arg);		
	readdata[0] = arg[0];
	readdata[1] = arg[1];

        *(u8 *)data= BitReverse(readdata[0]);
        *((u8 *)data+1)=BitReverse(readdata[1]);

        printf("%02x\n", (char)data[0]);

        return 0;
}

void imx238_test(void)
{
        unsigned short data=0x0;
        unsigned short address=0x0;

        //imx_write(fd, address,0x0760);

        address=0x0002;
        imx_read(fd, address,&data);
        address=0x0102;
        imx_read(fd, address,&data);
        address=0x0202;
        imx_read(fd, address,&data);
        address=0x0302;
        imx_read(fd, address,&data);
        address=0x0402;
        imx_read(fd, address,&data);
        address=0x0502;
        imx_read(fd, address,&data);
        address=0x0602;
        imx_read(fd, address,&data);
        address=0x0702;
        imx_read(fd, address,&data);
        address=0x0802;
        imx_read(fd, address,&data);
        address=0x0902;
        imx_read(fd, address,&data);
        address=0x0a02;
        imx_read(fd, address,&data);
        printf("w[%04x]\n", address);
	imx_write(fd, address,0x55);
        address=0x0a02;
        imx_read(fd, address,&data);

        address=0x0003;
        imx_read(fd, address,&data);
        address=0x0103;
        imx_read(fd, address,&data);
        address=0x0203;
        imx_read(fd, address,&data);
        address=0x0303;
        imx_read(fd, address,&data);
        address=0x0403;
        imx_read(fd, address,&data);
        address=0x0503;
        imx_read(fd, address,&data);
        address=0x0603;
        imx_read(fd, address,&data);
		
        address=0xDC04;
        imx_read(fd, address,&data);

        address=0x0005;
        imx_read(fd, address,&data);
        address=0x8505;
        imx_read(fd, address,&data);
        address=0x8405;
        imx_read(fd, address,&data);

        address=0x0407;
        imx_read(fd, address,&data);
}
void flash_test(void)
{
	char id[4] = {0x9f, 0, 0, 0};

	ioctl(fd, SPI_IOC_FLASHID_READ, id);		

	printf("flash ID %x %x %x\n", id[0], id[1], id[2]);
}
char * mn_rcom(unsigned short addr)
{
	static char ret[4] = {0}; 
        ret[0] = BitReverse( (addr>>0)&0xff );
        ret[1] = BitReverse( (addr>>8)|0x80 );
	ioctl(fd, SPI_IOC_EEPROM_READ, ret );		
        ret[0] = BitReverse( ret[0] );
        ret[1] = BitReverse( ret[1] );
	//printf("return data %x %x\n", ret[0], ret[1]);
	return ret;
}
char * mn_wcom(unsigned short addr, unsigned short data)
{
	static char ret[4] = {0}; 
        ret[0] = BitReverse( (addr>>0)&0xff );
        ret[1] = BitReverse( (addr>>8)&0x7f );
        ret[2] = BitReverse( (data>>0)&0xff );
        ret[3] = BitReverse( (data>>8)&0xff );
	return ret;
}
void mn34031pl_test(void)
{
	mn_rcom(0x0);		
	mn_rcom(0x1);		
	mn_rcom(0x2);		
	mn_rcom(0x3);		
	mn_rcom(0x4);		
	mn_rcom(0x5);		
	mn_rcom(0x6);		
	mn_rcom(0x7);		
	ioctl(fd, SPI_IOC_EEPROM_WRITE, mn_wcom(0x3, 0x4206) );		
}

int main(int argc, char *argv[])
{
	int ret = 0;

	parse_opts(argc, argv);

	fd = open(device, O_RDWR);
	if (fd < 0)
		pabort("can't open device");

	//mn34031pl_test();
	imx238_test();
	//flash_test();

	return 0;
	/*
	 * spi mode
	 */
	ret = ioctl(fd, SPI_IOC_WR_MODE, &mode);
	if (ret == -1)
		pabort("can't set spi mode");

	ret = ioctl(fd, SPI_IOC_RD_MODE, &mode);
	if (ret == -1)
		pabort("can't get spi mode");

	/*
	 * bits per word
	 */
	ret = ioctl(fd, SPI_IOC_WR_BITS_PER_WORD, &bits);
	if (ret == -1)
		pabort("can't set bits per word");

	ret = ioctl(fd, SPI_IOC_RD_BITS_PER_WORD, &bits);
	if (ret == -1)
		pabort("can't get bits per word");

	/*
	 * max speed hz
	 */
	ret = ioctl(fd, SPI_IOC_WR_MAX_SPEED_HZ, &speed);
	if (ret == -1)
		pabort("can't set max speed hz");

	ret = ioctl(fd, SPI_IOC_RD_MAX_SPEED_HZ, &speed);
	if (ret == -1)
		pabort("can't get max speed hz");

	printf("spi mode: %d\n", mode);
	printf("bits per word: %d\n", bits);
	printf("max speed: %d Hz (%d KHz)\n", speed, speed/1000);

	transfer(fd);

	close(fd);

	return ret;
}
