/*
 * $Header: /rd_2/project/Rossini/Components/JEBE/Device_Driver/JEBE/vma_jebe_driver.c 10    15/12/25 1:18p Dy.lu $
 *
 * vma_jebe
 * Driver for VMA JEBE
 *
 * Copyright (C) 2015-2020  VATICS INC.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $History: vma_jebe_driver.c $
 * 
 * *****************  Version 10  *****************
 * User: Dy.lu        Date: 15/12/25   Time: 1:18p
 * Updated in $/rd_2/project/Rossini/Components/JEBE/Device_Driver/JEBE
 * 
 * *****************  Version 9  *****************
 * User: Dy.lu        Date: 15/11/27   Time: 5:08p
 * Updated in $/rd_2/project/Rossini/Components/JEBE/Device_Driver/JEBE
 *
 */

/* ============================================================================================== */
#ifndef __KERNEL__
	#define __KERNEL__
#endif //!__KERNEL__

#ifndef MODULE
//	#define MODULE
#endif //!MODULE

#ifndef MODULE
	static int OtusOpenCnt = 0;
#endif

/* ============================================================================================== */
#include <asm/cacheflush.h>
#include <linux/dma-mapping.h>
#include "vma_jebe_driver.h"

/* ============================================================================================== */
const CHAR VMA_JEBE_ID[] = "$Version: "VMA_JEBE_ID_VERSION"  (VMA_JEBE) $";
static TVMAJEBESharedInfo *gptSharedInfo = NULL;
static SDWORD gsdwMajor = 0;

/* ============================================================================================== */
MODULE_AUTHOR ("VATICS Inc.");
MODULE_LICENSE ("GPL");
module_param (gsdwMajor, int, 0644);
MODULE_PARM_DESC (gsdwMajor, "Major number for VMA_JEBE module");

#ifdef __USE_SWAIT__
#warning "Use Simple Wait Queue!"
#endif

/* ============================================================================================== */
static irqreturn_t ISR(int irq, void *dev_id)
{
	DWORD dwReadIndex = VMA_JEBE_ISR(gptSharedInfo->hDevInfo);

	gptSharedInfo->abIntr[dwReadIndex] = TRUE;
#ifdef __USE_SWAIT__
	swake_up(&gptSharedInfo->atWaitQueueHead[dwReadIndex]);
#else
	wake_up(&gptSharedInfo->atWaitQueueHead[dwReadIndex]);
#endif

	return IRQ_HANDLED;
}

/* ============================================================================================== */
static SCODE Start(TVMAJEBEObjInfo *ptObjInfo)
{
	DWORD dwWriteIndex;

	dwWriteIndex = VMA_JEBE_StartHead(gptSharedInfo->hDevInfo);

	if (gptSharedInfo->abWriteEn[dwWriteIndex])
	{
		gptSharedInfo->abWriteEn[dwWriteIndex] = FALSE;

		ptObjInfo->dwWriteIndex = dwWriteIndex;

		VMA_JEBE_StartTail(gptSharedInfo->hDevInfo, dwWriteIndex, ptObjInfo->ptMMRInfo);


		return S_OK;
	}
	return S_FAIL;
}

/* ============================================================================================== */
static void WaitComplete(TVMAJEBEObjInfo *ptObjInfo)
{
	DWORD dwWriteIndex;

	PDEBUG("Enter WaitComplete function...\n");

	dwWriteIndex = ptObjInfo->dwWriteIndex;

#ifdef __USE_SWAIT__
	swait_event(gptSharedInfo->atWaitQueueHead[dwWriteIndex], (gptSharedInfo->abIntr[dwWriteIndex]==TRUE));
#else
	wait_event(gptSharedInfo->atWaitQueueHead[dwWriteIndex], (gptSharedInfo->abIntr[dwWriteIndex]==TRUE));
#endif

	gptSharedInfo->abIntr[dwWriteIndex] = FALSE;

	gptSharedInfo->abWriteEn[dwWriteIndex] = TRUE;
	ptObjInfo->dwWriteIndex = 0xFFFFFFFF;

#ifdef __PROFILE__
	VMA_JEBE_GetProfileInfo(gptSharedInfo->hDevInfo, ptObjInfo, dwWriteIndex);
#endif

	PDEBUG("Exit WaitComplete function !!\n");
}

/* ============================================================================================== */
static int Close(struct inode *pinode, struct file *pfile)
{
	TVMAJEBEObjInfo *ptObjInfo = (TVMAJEBEObjInfo *)pfile->private_data;

	PDEBUG("Enter Close function...\n");

	if (ptObjInfo != NULL)
	{
		if (ptObjInfo->dwWriteIndex != 0xFFFFFFFF)
		{
			while (1)
			{
				if (gptSharedInfo->abIntr[ptObjInfo->dwWriteIndex] == TRUE)
				{
					break;
				}
			}
			gptSharedInfo->abIntr[ptObjInfo->dwWriteIndex] = FALSE;
			gptSharedInfo->abWriteEn[ptObjInfo->dwWriteIndex] = TRUE;
		}

		iounmap(ptObjInfo->ptMMRInfo);
		release_mem_region(ptObjInfo->dwMMRInfoPhyAddr, sizeof(TVMAJEBEInfo));

		kfree(ptObjInfo);

		pfile->private_data = NULL;
	}

	mutex_lock(&gptSharedInfo->ioctl_lock);
#ifdef MODULE
	module_put(THIS_MODULE);

	if (module_refcount(THIS_MODULE) == 0)
#else
	if ((OtusOpenCnt > 0) && (--OtusOpenCnt == 0))
#endif
	{
		clk_enable(gptSharedInfo->pClk);

		VMA_JEBE_IntrClear(gptSharedInfo->hDevInfo);
		VMA_JEBE_IntrDisable(gptSharedInfo->hDevInfo);

		if (gptSharedInfo->dwIrq != (DWORD)NULL)
		{
			free_irq(gptSharedInfo->dwIrq, gptSharedInfo);
		}

		gptSharedInfo->dwIrq = (DWORD)NULL;
		clk_disable(gptSharedInfo->pClk);
	}
	mutex_unlock(&gptSharedInfo->ioctl_lock);

	PDEBUG("Exit Close function !!\n");

	return 0;
}

/* ============================================================================================== */
static int Open(struct inode *pinode, struct file *pfile)
{
	TVMAJEBEObjInfo *ptObjInfo;
	int scResult;
	DWORD i;
	int virq;

	mutex_lock(&gptSharedInfo->ioctl_lock);
#ifdef MODULE
	i = module_refcount(THIS_MODULE);
#else
	i = OtusOpenCnt;
#endif
	if (i == VMA_JEBE_MMR_BUFF_NUM)
	{
		printk("[vma_jebe]: only support %d handles\n", VMA_JEBE_MMR_BUFF_NUM);
		mutex_unlock(&gptSharedInfo->ioctl_lock);
		return -EFAULT;
	}

	if (pfile->private_data == NULL)
	{
		if ((pfile->private_data=(TVMAJEBEObjInfo *)kmalloc(sizeof(TVMAJEBEObjInfo), GFP_KERNEL)) == NULL)
		{
			PDEBUG("Insufficient kernel memory space !!\n");
			PDEBUG("Exit Open function !!\n");
			mutex_unlock(&gptSharedInfo->ioctl_lock);
			return -ENOMEM;
		}

		ptObjInfo = (TVMAJEBEObjInfo *)pfile->private_data;
		ptObjInfo->dwWriteIndex = 0xFFFFFFFF;
		ptObjInfo->ptMMRInfo = NULL;

#ifdef __PROFILE__
		if (VMA_JEBE_InitProfileInfo(ptObjInfo) != S_OK)
		{
			PDEBUG("Fail to initialize profile info !!\n");
			PDEBUG("Exit Open function !!\n");
			mutex_unlock(&gptSharedInfo->ioctl_lock);
			return -EFAULT;
		}
#endif
	}
	else
	{
		PDEBUG("Exit Open function !!\n");
		mutex_unlock(&gptSharedInfo->ioctl_lock);
		return -EBUSY;
	}

	if (i == 0)
	{
		clk_enable(gptSharedInfo->pClk);

		//VMA_JEBE_Reset(gptSharedInfo->hDevInfo); 

		if (gptSharedInfo->dwIrq == (DWORD)NULL)
		{
			VMA_JEBE_IntrDisable(gptSharedInfo->hDevInfo);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,24)
			scResult = request_irq(VMA_JEBE_IRQ_NUM, &ISR, SA_INTERRUPT, "vma_jebe", gptSharedInfo);
#else
			virq = irq_create_mapping(NULL, VMA_JEBE_IRQ_NUM);
			scResult = request_irq(virq, &ISR, IRQF_DISABLED | IRQF_TRIGGER_HIGH, "vma_jebe", gptSharedInfo);
#endif

			if (scResult < 0)
			{
				PDEBUG("Cannot get irq %d !!\n", VMA_JEBE_IRQ_NUM);
				mutex_unlock(&gptSharedInfo->ioctl_lock);
				Close(pinode, pfile);
				PDEBUG("Exit Open function !!\n");
				return scResult;
			}

			gptSharedInfo->dwIrq = virq;
			disable_irq(gptSharedInfo->dwIrq);
			enable_irq(gptSharedInfo->dwIrq);

			VMA_JEBE_IntrClear(gptSharedInfo->hDevInfo);
			VMA_JEBE_IntrEnable(gptSharedInfo->hDevInfo);

			VMA_JEBE_Open(gptSharedInfo->hDevInfo);

			for (i=0; i<VMA_JEBE_MMR_BUFF_NUM; i++)
			{
#ifdef __USE_SWAIT__
				init_swait_queue_head(&gptSharedInfo->atWaitQueueHead[i]);
#else
				init_waitqueue_head(&gptSharedInfo->atWaitQueueHead[i]);
#endif
				gptSharedInfo->abIntr[i] = FALSE;
				gptSharedInfo->abWriteEn[i] = TRUE;
			}
		}
		clk_disable(gptSharedInfo->pClk);
	}

#ifdef MODULE
	scResult = (try_module_get(THIS_MODULE) == 1)? 0:-EBUSY;
#else
	OtusOpenCnt++;
	scResult = 0;
#endif
	mutex_unlock(&gptSharedInfo->ioctl_lock);

	return scResult;
}

/* ============================================================================================== */
static long Ioctl(struct file *pfile, unsigned int dwCmd, unsigned long dwArg)
{
	TVMAJEBEObjInfo *ptObjInfo = (TVMAJEBEObjInfo *)pfile->private_data;
	DWORD dwVersionNum;
	int scError;
	CHAR acMemRegionName[16];

	PDEBUG("Enter Ioctl function...\n");
    
    //mutex_lock(&gptSharedInfo->ioctl_lock);
    
	if (pfile->private_data == NULL)
	{
		PDEBUG("Device does not exist !!\n");
		PDEBUG("Exit Ioctl function !!\n");
		//mutex_unlock(&gptSharedInfo->ioctl_lock);
		return -ENODEV;
	}

	if ((_IOC_TYPE(dwCmd)!=VMA_JEBE_IOC_MAGIC) || (_IOC_NR(dwCmd)>VMA_JEBE_IOC_MAX_NUMBER))
	{
		PDEBUG("Incorrect ioctl command !!\n");
		PDEBUG("Exit Ioctl function !!\n");
        //mutex_unlock(&gptSharedInfo->ioctl_lock);
		return -ENOTTY;
	}

	if (_IOC_DIR(dwCmd) & _IOC_READ)
	{
		scError = !access_ok(VERIFY_WRITE, (void *)dwArg, _IOC_SIZE(dwCmd));
	}
	else if (_IOC_DIR(dwCmd) & _IOC_WRITE)
	{
		scError = !access_ok(VERIFY_READ, (void *)dwArg, _IOC_SIZE(dwCmd));
	}
	else
	{
		scError = 0;
	}

	if (scError != 0)
	{
		PDEBUG("Unsupport ioctl command %d !!\n", dwCmd);
		PDEBUG("Exit Ioctl function !!\n");
        //mutex_unlock(&gptSharedInfo->ioctl_lock);
		return -EFAULT;
	}

	switch (dwCmd)
	{
		case VMA_JEBE_IOC_START:
			mutex_lock(&gptSharedInfo->ioctl_lock);
			if (Start(ptObjInfo) != S_OK)
			{
				PDEBUG("Exit Ioctl function !!\n");
				scError = S_FAIL;
				dwArg = TRUE;
			}
			mutex_unlock(&gptSharedInfo->ioctl_lock);
			if (dwArg)
				break;
		case VMA_JEBE_IOC_WAIT_COMPLETE:
			WaitComplete(ptObjInfo);			
			/* TODO: Maybe another leak for official SDK */
			dmac_map_area(ptObjInfo->ptMMRInfo, sizeof(TVMAJEBEInfo), DMA_BIDIRECTIONAL);
		break;
		case VMA_JEBE_IOC_SHARE_MMR_INFO_SPACE:
			ptObjInfo->dwMMRInfoPhyAddr = dwArg;
			sprintf(acMemRegionName, "%d", (int)ptObjInfo->dwMMRInfoPhyAddr);
			request_mem_region(ptObjInfo->dwMMRInfoPhyAddr, sizeof(TVMAJEBEInfo), acMemRegionName);
			ptObjInfo->ptMMRInfo = (TVMAJEBEInfo *)ioremap_cached((int)ptObjInfo->dwMMRInfoPhyAddr, sizeof(TVMAJEBEInfo));
		break;
		case VMA_JEBE_IOC_GET_VERSION_NUMBER:
			dwVersionNum = VMA_JEBE_VERSION;
			scError = copy_to_user((DWORD *)dwArg, &dwVersionNum, sizeof(DWORD));
			if (scError != 0)
			{
				PDEBUG("Exit Ioctl function !!\n");
				scError = -EFAULT;
			}
		break;
#ifdef __PROFILE__
		case VMA_JEBE_IOC_MASTER_0_GET_BANDWIDTH:
			VMA_JEBE_SetupProfile(ptObjInfo, dwArg, dwCmd);
		break;
		case VMA_JEBE_IOC_MASTER_0_GET_RG_INTERVAL:
			VMA_JEBE_SetupProfile(ptObjInfo, dwArg, dwCmd);
		break;
		case VMA_JEBE_IOC_MASTER_0_GET_REQ_TIMES:
			VMA_JEBE_SetupProfile(ptObjInfo, dwArg, dwCmd);
		break;
		case VMA_JEBE_IOC_MASTER_0_CLEAR_PROFILE:
			VMA_JEBE_SetupProfile(ptObjInfo, dwArg, dwCmd);
		break;
		case VMA_JEBE_IOC_MASTER_1_GET_BANDWIDTH:
			VMA_JEBE_SetupProfile(ptObjInfo, dwArg, dwCmd);
		break;
		case VMA_JEBE_IOC_MASTER_1_GET_RG_INTERVAL:
			VMA_JEBE_SetupProfile(ptObjInfo, dwArg, dwCmd);
		break;
		case VMA_JEBE_IOC_MASTER_1_GET_REQ_TIMES:
			VMA_JEBE_SetupProfile(ptObjInfo, dwArg, dwCmd);
		break;
		case VMA_JEBE_IOC_MASTER_1_CLEAR_PROFILE:
			VMA_JEBE_SetupProfile(ptObjInfo, dwArg, dwCmd);
		break;
#endif
		default:
			PDEBUG("Exit Ioctl function !!\n");
			scError = -ENOTTY;
	}

	PDEBUG("Exit Ioctl function !!\n");

	//mutex_unlock(&gptSharedInfo->ioctl_lock);
	return scError;
}
/* ============================================================================================== */
/*static int MMap(struct file *file, struct vm_area_struct *vma)
{
	DWORD dwSize;

	dwSize = vma->vm_end - vma->vm_start;

	vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
	vma->vm_flags |= VM_DONTEXPAND | VM_DONTDUMP;

	if (remap_pfn_range(vma, vma->vm_start, (VMA_JEBE_MMR_BASE>>PAGE_SHIFT), dwSize, vma->vm_page_prot))
	{
		return -EAGAIN;
	}

	PDEBUG("Start address = 0x%08lX, end address = 0x%08lX\n", vma->vm_start, vma->vm_end);

	return 0;
}*/

/* ============================================================================================== */
struct file_operations vma_jebe_fops =
{
	unlocked_ioctl:		Ioctl,
	//mmap:				MMap,
	open:				Open,
	release:			Close,
};

/* ============================================================================================== */
static void CleanupModule(void)
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,24)
	int scResult;
#endif
	TVMAJEBEInfo *ptMMRInfo;

	PDEBUG("Enter CleanupModule function...\n");

	if (gptSharedInfo != NULL)
	{
		if (gptSharedInfo->hDevInfo != NULL)
		{
			if (gsdwMajor != 0)
			{
			    vma_unregister_device(MKDEV(gsdwMajor, 0));
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,24)
				scResult = unregister_chrdev(gsdwMajor, "vma_jebe");

				if (scResult < 0)
				{
					PDEBUG("Cannot release major number %d !!\n", (int)gsdwMajor);
					PDEBUG("Exit CleanupModule function !!\n");
					return;
				}
#else
				unregister_chrdev(gsdwMajor, "vma_jebe");
#endif
			}

#ifdef __PROFILE__
			VMA_JEBE_CloseProfile(gptSharedInfo->hDevInfo);
#endif
			ptMMRInfo = (TVMAJEBEInfo *)VMA_JEBE_GetMMRInfo(gptSharedInfo->hDevInfo);

			if (ptMMRInfo != NULL)
			{
				iounmap(ptMMRInfo);
				release_mem_region(VMA_JEBE_MMR_BASE, sizeof(TVMAJEBEInfo));
			}

			kfree(gptSharedInfo->hDevInfo);
		}
		
		if (!IS_ERR(gptSharedInfo->pClk))
		{
			clk_unprepare(gptSharedInfo->pClk);
			clk_put(gptSharedInfo->pClk);
		}

		kfree(gptSharedInfo);
	}

	PDEBUG("Exit CleanupModule function !!\n");

	return;
}

/* ============================================================================================== */
static int InitialModule(void)
{
	int scResult;
	DWORD dwDevInfoSize, dwVersionNum;
	volatile TVMAJEBEInfo *ptMMRInfo = (TVMAJEBEInfo *)ioremap((int)VMA_JEBE_MMR_BASE, sizeof(TVMAJEBEInfo));
	//volatile DWORD *pdwClkEnMmr = (DWORD *)(IO_ADDRESS(VPL_SYSC_MMR_BASE)+0x94);
	volatile DWORD *pdwRstEnMmr = (DWORD *)(IO_ADDRESS(VPL_SYSC_MMR_BASE)+0x24);

	PDEBUG("Enter InitialModule function...\n");

	if ((gptSharedInfo=(TVMAJEBESharedInfo *)kmalloc(sizeof(TVMAJEBESharedInfo), GFP_KERNEL)) == NULL)
	{
		PDEBUG("Allocate shared info buffer fail !!\n");
		scResult = -ENOMEM;
		goto FAIL;
	}
	memset(gptSharedInfo, 0, sizeof(TVMAJEBESharedInfo));

	dwDevInfoSize = VMA_JEBE_GetDevInfoSize();

	if ((gptSharedInfo->hDevInfo=(HANDLE)kmalloc(dwDevInfoSize, GFP_KERNEL)) == NULL)
	{
		PDEBUG("Allocate device info buffer fail !!\n");
		scResult = -ENOMEM;
		goto FAIL;
	}
	memset(gptSharedInfo->hDevInfo, 0, dwDevInfoSize);

	mutex_init(&gptSharedInfo->ioctl_lock);

	gptSharedInfo->pClk = clk_get(NULL, "vma_jebe");
	if (IS_ERR(gptSharedInfo->pClk))
	{
		PDEBUG("Fail to get vma jebe clock !!\n");
		scResult = PTR_ERR(gptSharedInfo->pClk);
		goto FAIL;
	}

	clk_prepare_enable(gptSharedInfo->pClk);
	
	request_mem_region(VMA_JEBE_MMR_BASE, sizeof(TVMAJEBEInfo), "VMA_JEBE");

	//if (VMA_JEBE_SetMMRInfo(gptSharedInfo->hDevInfo, ptMMRInfo, pdwClkEnMmr) != S_OK)
	if (VMA_JEBE_SetMMRInfo(gptSharedInfo->hDevInfo, ptMMRInfo, gptSharedInfo->pClk, pdwRstEnMmr) != S_OK)
	{
		scResult = -ENODEV;
		goto FAIL_CLK;
	}

#ifdef __PROFILE__
	VMA_JEBE_InitProfile(gptSharedInfo->hDevInfo);
#endif

	scResult = register_chrdev(gsdwMajor, "vma_jebe", &vma_jebe_fops);

	if (scResult < 0)
	{
		PDEBUG("Cannot get major number %d !!\n", (int)gsdwMajor);
		goto FAIL_CLK;
	}

	if (gsdwMajor == 0)
	{
		gsdwMajor = scResult;
	}
	
	vma_register_device(MKDEV(gsdwMajor, 0), NULL, "vma_jebe");

	dwVersionNum = VMA_JEBE_GetVersion(gptSharedInfo->hDevInfo);

	printk("Install VMA_JEBE device driver version %d.%d.%d.%d on VMA_JEBE hardware version %d.%d.%d.%d complete !!\n",
	       (int)(VMA_JEBE_VERSION&0xFF),
	       (int)((VMA_JEBE_VERSION>>8)&0xFF),
	       (int)((VMA_JEBE_VERSION>>16)&0xFF),
	       (int)((VMA_JEBE_VERSION>>24)&0xFF),
	       (int)(dwVersionNum>>24)&0xFF,
	       (int)(dwVersionNum>>16)&0xFF,
	       (int)(dwVersionNum>>8)&0xFF,
	       (int)dwVersionNum&0xFF);

    clk_disable(gptSharedInfo->pClk);
    
	PDEBUG("Exit InitialModule function !!\n");

	return 0;

FAIL_CLK:
	clk_disable(gptSharedInfo->pClk);
FAIL:
	CleanupModule();
	PDEBUG("Exit InitialModule function !!\n");

	return scResult;
}

/* ============================================================================================== */
module_init(InitialModule);
module_exit(CleanupModule);

/* ============================================================================================== */
