/*
 * $Header: /rd_2/project/Rossini/Components/DMAC/Device_Driver/DMAC/vpl_dmac_driver.c 9     15/12/24 2:56p Yiming.liu $
 *
 * vpl_dmac
 * Driver for VPL DMAC
 *
 * Copyright (C) 2015-2020  VATICS INC.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $History: vpl_dmac_driver.c $
 * 
 * *****************  Version 9  *****************
 * User: Yiming.liu   Date: 15/12/24   Time: 2:56p
 * Updated in $/rd_2/project/Rossini/Components/DMAC/Device_Driver/DMAC
 * 
 * *****************  Version 8  *****************
 * User: Yiming.liu   Date: 15/11/27   Time: 2:33p
 * Updated in $/rd_2/project/Rossini/Components/DMAC/Device_Driver/DMAC
 * 
 * *****************  Version 7  *****************
 * User: Yiming.liu   Date: 15/10/16   Time: 5:59p
 * Updated in $/rd_2/project/Rossini/Components/DMAC/Device_Driver/DMAC
 * 
 * *****************  Version 6  *****************
 * User: Yiming.liu   Date: 15/10/16   Time: 5:36p
 * Updated in $/rd_2/project/Rossini/Components/DMAC/Device_Driver/DMAC
 *
 */

/* ============================================================================================== */
#ifndef __KERNEL__
	#define __KERNEL__
#endif //!__KERNEL__

#ifndef MODULE
//	#define MODULE
#endif //!MODULE

#ifndef MODULE
	static int OtusOpenCnt = 0;
#endif

/* ============================================================================================== */
#include <asm/cacheflush.h>
#include <linux/dma-mapping.h>
#include "vpl_dmac_driver.h"

/* ============================================================================================== */
const CHAR VPL_DMAC_ID[] = "$Version: "VPL_DMAC_ID_VERSION"  (VPL_DMAC) $";
static TVPLDMACSharedInfo *gptSharedInfo = NULL;
static SDWORD gsdwMajor = 0;

/* ============================================================================================== */
MODULE_AUTHOR ("VATICS Inc.");
MODULE_LICENSE ("GPL");
module_param (gsdwMajor, int, 0644);
MODULE_PARM_DESC (gsdwMajor, "Major number for VPL_DMAC module");

#ifdef __USE_SWAIT__
#warning "Use Simple Wait Queue!"
#endif

/* ============================================================================================== */
static irqreturn_t ISR(int irq, void *dev_id)
{
	DWORD dwReadIndex = VPL_DMAC_ISR(gptSharedInfo->hDevInfo);

	gptSharedInfo->abIntr[dwReadIndex] = TRUE;
#ifdef __USE_SWAIT__
	swake_up(&gptSharedInfo->atWaitQueueHead[dwReadIndex]);
#else
	wake_up(&gptSharedInfo->atWaitQueueHead[dwReadIndex]);
#endif

	return IRQ_HANDLED;
}

/* ============================================================================================== */
static SCODE Start(TVPLDMACObjInfo *ptObjInfo)
{
	DWORD dwWriteIndex;

	dwWriteIndex = VPL_DMAC_StartHead(gptSharedInfo->hDevInfo);

	if (gptSharedInfo->abWriteEn[dwWriteIndex])
	{
		gptSharedInfo->abWriteEn[dwWriteIndex] = FALSE;

		ptObjInfo->dwWriteIndex = dwWriteIndex;

		VPL_DMAC_StartTail(gptSharedInfo->hDevInfo, dwWriteIndex, ptObjInfo->ptMMRInfo);

		return S_OK;
	}
	return S_FAIL;
}

/* ============================================================================================== */
static void WaitComplete(TVPLDMACObjInfo *ptObjInfo)
{
	DWORD dwWriteIndex;

	PDEBUG("Enter WaitComplete function...\n");

	dwWriteIndex = ptObjInfo->dwWriteIndex;

#ifdef __USE_SWAIT__
	swait_event(gptSharedInfo->atWaitQueueHead[dwWriteIndex], (gptSharedInfo->abIntr[dwWriteIndex]==TRUE));
#else
	wait_event(gptSharedInfo->atWaitQueueHead[dwWriteIndex], (gptSharedInfo->abIntr[dwWriteIndex]==TRUE));
#endif

	gptSharedInfo->abIntr[dwWriteIndex] = FALSE;

	gptSharedInfo->abWriteEn[dwWriteIndex] = TRUE;
	ptObjInfo->dwWriteIndex = 0xFFFFFFFF;

#ifdef __PROFILE__
	VPL_DMAC_GetProfileInfo(gptSharedInfo->hDevInfo, ptObjInfo, dwWriteIndex);
#endif

	PDEBUG("Exit WaitComplete function !!\n");
}

/* ============================================================================================== */
static int Close(struct inode *pinode, struct file *pfile)
{
	TVPLDMACObjInfo *ptObjInfo = (TVPLDMACObjInfo *)pfile->private_data;

	PDEBUG("Enter Close function...\n");

	if (ptObjInfo != NULL)
	{
		if (ptObjInfo->dwWriteIndex != 0xFFFFFFFF)
		{
			while (1)
			{
				if (gptSharedInfo->abIntr[ptObjInfo->dwWriteIndex] == TRUE)
				{
					break;
				}
			}
			gptSharedInfo->abIntr[ptObjInfo->dwWriteIndex] = FALSE;
			gptSharedInfo->abWriteEn[ptObjInfo->dwWriteIndex] = TRUE;
		}

		if (ptObjInfo->ptMMRInfo)
		{
			iounmap(ptObjInfo->ptMMRInfo);
			release_mem_region(ptObjInfo->dwMMRInfoPhyAddr, sizeof(TVPLDMACInfo));
		}

		kfree(ptObjInfo);

		pfile->private_data = NULL;
	}

	mutex_lock(&gptSharedInfo->ioctl_lock);
#ifdef MODULE
	module_put(THIS_MODULE);

	if (module_refcount(THIS_MODULE) == 0)
#else
	if ((OtusOpenCnt > 0) && (--OtusOpenCnt == 0))
#endif
	{
		clk_enable(gptSharedInfo->pClk);

		VPL_DMAC_IntrClear(gptSharedInfo->hDevInfo);
		VPL_DMAC_IntrDisable(gptSharedInfo->hDevInfo);

		if (gptSharedInfo->dwIrq != (DWORD)NULL)
		{
			free_irq(gptSharedInfo->dwIrq, gptSharedInfo);
		}

		gptSharedInfo->dwIrq = (DWORD)NULL;
		clk_disable(gptSharedInfo->pClk);
	}
	mutex_unlock(&gptSharedInfo->ioctl_lock);

	PDEBUG("Exit Close function !!\n");

	return 0;
}

/* ============================================================================================== */
static int Open(struct inode *pinode, struct file *pfile)
{
	TVPLDMACObjInfo *ptObjInfo;
	int scResult;
	DWORD i;
	int virq;

	mutex_lock(&gptSharedInfo->ioctl_lock);
#ifdef MODULE
	i = module_refcount(THIS_MODULE);
#else
	i = OtusOpenCnt;
	scResult = 0;
#endif
	if (i == VPL_DMAC_MMR_BUFF_NUM)
	{
		printk("[vpl_dmac]: only support %d handles\n", VPL_DMAC_MMR_BUFF_NUM);
		mutex_unlock(&gptSharedInfo->ioctl_lock);
		return -EFAULT;
	}

	if (pfile->private_data == NULL)
	{
		if ((pfile->private_data=(TVPLDMACObjInfo *)kmalloc(sizeof(TVPLDMACObjInfo), GFP_KERNEL)) == NULL)
		{
			PDEBUG("Insufficient kernel memory space !!\n");
			PDEBUG("Exit Open function !!\n");
			mutex_unlock(&gptSharedInfo->ioctl_lock);
			return -ENOMEM;
		}

		ptObjInfo = (TVPLDMACObjInfo *)pfile->private_data;
		ptObjInfo->dwWriteIndex = 0xFFFFFFFF;
/* Version 1.0.0.3 modification, 2015.11.27 */
		ptObjInfo->ptMMRInfo = NULL;
/* ======================================== */

#ifdef __PROFILE__
		if (VPL_DMAC_InitProfileInfo(ptObjInfo) != S_OK)
		{
			PDEBUG("Fail to initialize profile info !!\n");
			PDEBUG("Exit Open function !!\n");
			mutex_unlock(&gptSharedInfo->ioctl_lock);
			return -EFAULT;
		}
#endif
	}
	else
	{
		PDEBUG("Exit Open function !!\n");
		mutex_unlock(&gptSharedInfo->ioctl_lock);
		return -EBUSY;
	}

	if (i == 0)
	{
		clk_enable(gptSharedInfo->pClk);

/* Version 1.0.0.4 modification, 2015.12.24 */
		//VPL_DMAC_Reset(gptSharedInfo->hDevInfo);
/* ======================================== */

		if (gptSharedInfo->dwIrq == (DWORD)NULL)
		{
			VPL_DMAC_IntrDisable(gptSharedInfo->hDevInfo);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,24)
			scResult = request_irq(VPL_DMAC_IRQ_NUM, &ISR, SA_INTERRUPT, "vpl_dmac", gptSharedInfo);
#else
			virq = irq_create_mapping(NULL, VPL_DMAC_IRQ_NUM);
			scResult = request_irq(virq, &ISR, IRQF_DISABLED | IRQF_TRIGGER_HIGH, "vpl_dmac", gptSharedInfo);
#endif

			if (scResult < 0)
			{
				PDEBUG("Cannot get irq %d !!\n", VPL_DMAC_IRQ_NUM);
				mutex_unlock(&gptSharedInfo->ioctl_lock);
				Close(pinode, pfile);
				PDEBUG("Exit Open function !!\n");
				return scResult;
			}

			gptSharedInfo->dwIrq = virq;
			disable_irq(gptSharedInfo->dwIrq);
			enable_irq(gptSharedInfo->dwIrq);

			VPL_DMAC_IntrClear(gptSharedInfo->hDevInfo);
			VPL_DMAC_IntrEnable(gptSharedInfo->hDevInfo);

			VPL_DMAC_Open(gptSharedInfo->hDevInfo);

			for (i=0; i<VPL_DMAC_MMR_BUFF_NUM; i++)
			{
#ifdef __USE_SWAIT__
				init_swait_queue_head(&gptSharedInfo->atWaitQueueHead[i]);
#else
				init_waitqueue_head(&gptSharedInfo->atWaitQueueHead[i]);
#endif
				gptSharedInfo->abIntr[i] = FALSE;
				gptSharedInfo->abWriteEn[i] = TRUE;
			}
		}
		clk_disable(gptSharedInfo->pClk);
	}

#ifdef MODULE
	scResult = (try_module_get(THIS_MODULE) == 1)? 0:-EBUSY;
#else
	OtusOpenCnt++;
#endif
	mutex_unlock(&gptSharedInfo->ioctl_lock);

	return scResult;
}

/* ============================================================================================== */
static long Ioctl(struct file *pfile, unsigned int dwCmd, unsigned long dwArg)
{
	TVPLDMACObjInfo *ptObjInfo = (TVPLDMACObjInfo *)pfile->private_data;
	DWORD dwVersionNum;
	int scError;
	CHAR acMemRegionName[16];

	PDEBUG("Enter Ioctl function...\n");

	if (pfile->private_data == NULL)
	{
		PDEBUG("Device does not exist !!\n");
		PDEBUG("Exit Ioctl function !!\n");
		return -ENODEV;
	}

	if ((_IOC_TYPE(dwCmd)!=VPL_DMAC_IOC_MAGIC) || (_IOC_NR(dwCmd)>VPL_DMAC_IOC_MAX_NUMBER))
	{
		PDEBUG("Incorrect ioctl command !!\n");
		PDEBUG("Exit Ioctl function !!\n");
		return -ENOTTY;
	}

	if (_IOC_DIR(dwCmd) & _IOC_READ)
	{
		scError = !access_ok(VERIFY_WRITE, (void *)dwArg, _IOC_SIZE(dwCmd));
	}
	else if (_IOC_DIR(dwCmd) & _IOC_WRITE)
	{
		scError = !access_ok(VERIFY_READ, (void *)dwArg, _IOC_SIZE(dwCmd));
	}
	else
	{
		scError = 0;
	}

	if (scError != 0)
	{
		PDEBUG("Unsupport ioctl command %d !!\n", dwCmd);
		PDEBUG("Exit Ioctl function !!\n");
		return -EFAULT;
	}

	switch (dwCmd)
	{
		case VPL_DMAC_IOC_START:
			mutex_lock(&gptSharedInfo->ioctl_lock);
			if (Start(ptObjInfo) != S_OK)
			{
				PDEBUG("Exit Ioctl function !!\n");
				scError = S_FAIL;
				dwArg = TRUE;
			}
			mutex_unlock(&gptSharedInfo->ioctl_lock);
		if (dwArg)
			break;
		case VPL_DMAC_IOC_WAIT_COMPLETE:
			WaitComplete(ptObjInfo);
			/* TODO: Maybe another leak for official SDK */
			dmac_map_area(ptObjInfo->ptMMRInfo, sizeof(TVPLDMACInfo), DMA_BIDIRECTIONAL);
		break;
		case VPL_DMAC_IOC_SHARE_MMR_INFO_SPACE:
			ptObjInfo->dwMMRInfoPhyAddr = dwArg;
			sprintf(acMemRegionName, "%d", (int)ptObjInfo->dwMMRInfoPhyAddr);
			request_mem_region(ptObjInfo->dwMMRInfoPhyAddr, sizeof(TVPLDMACInfo), acMemRegionName);
			ptObjInfo->ptMMRInfo = (TVPLDMACInfo *)ioremap_cached((int)ptObjInfo->dwMMRInfoPhyAddr, sizeof(TVPLDMACInfo));
		break;
		case VPL_DMAC_IOC_GET_VERSION_NUMBER:
			dwVersionNum = VPL_DMAC_VERSION;
			scError = copy_to_user((DWORD *)dwArg, &dwVersionNum, sizeof(DWORD));
			if (scError != 0)
			{
				PDEBUG("Exit Ioctl function !!\n");
				scError = -EFAULT;
			}
		break;
#ifdef __PROFILE__
		case VPL_DMAC_IOC_MASTER_0_GET_BANDWIDTH:
			VPL_DMAC_SetupProfile(ptObjInfo, dwArg, dwCmd);
		break;
		case VPL_DMAC_IOC_MASTER_0_GET_RG_INTERVAL:
			VPL_DMAC_SetupProfile(ptObjInfo, dwArg, dwCmd);
		break;
		case VPL_DMAC_IOC_MASTER_0_GET_REQ_TIMES:
			VPL_DMAC_SetupProfile(ptObjInfo, dwArg, dwCmd);
		break;
		case VPL_DMAC_IOC_MASTER_0_CLEAR_PROFILE:
			VPL_DMAC_SetupProfile(ptObjInfo, dwArg, dwCmd);
		break;
		case VPL_DMAC_IOC_MASTER_1_GET_BANDWIDTH:
			VPL_DMAC_SetupProfile(ptObjInfo, dwArg, dwCmd);
		break;
		case VPL_DMAC_IOC_MASTER_1_GET_RG_INTERVAL:
			VPL_DMAC_SetupProfile(ptObjInfo, dwArg, dwCmd);
		break;
		case VPL_DMAC_IOC_MASTER_1_GET_REQ_TIMES:
			VPL_DMAC_SetupProfile(ptObjInfo, dwArg, dwCmd);
		break;
		case VPL_DMAC_IOC_MASTER_1_CLEAR_PROFILE:
			VPL_DMAC_SetupProfile(ptObjInfo, dwArg, dwCmd);
		break;
#endif
		default:
			PDEBUG("Exit Ioctl function !!\n");
			scError = -ENOTTY;
	}

	PDEBUG("Exit Ioctl function !!\n");

	return scError;
}

/* ============================================================================================== */
/*static int MMap(struct file *file, struct vm_area_struct *vma)
{
	DWORD dwSize;

	dwSize = vma->vm_end - vma->vm_start;

	vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
	vma->vm_flags |= VM_DONTEXPAND | VM_DONTDUMP;

	if (remap_pfn_range(vma, vma->vm_start, (VPL_DMAC_MMR_BASE>>PAGE_SHIFT), dwSize, vma->vm_page_prot))
	{
		return -EAGAIN;
	}

	PDEBUG("Start address = 0x%08lX, end address = 0x%08lX\n", vma->vm_start, vma->vm_end);

	return 0;
}*/

/* ============================================================================================== */
struct file_operations vpl_dmac_fops =
{
	unlocked_ioctl:		Ioctl,
	//mmap:				MMap,
	open:				Open,
	release:			Close,
};

/* ============================================================================================== */
static void CleanupModule(void)
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,24)
	int scResult;
#endif
	TVPLDMACInfo *ptMMRInfo;

	PDEBUG("Enter CleanupModule function...\n");

	if (gptSharedInfo != NULL)
	{
		if (gptSharedInfo->hDevInfo != NULL)
		{
			if (gsdwMajor != 0)
			{
				vma_unregister_device(MKDEV(gsdwMajor, 0));
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,24)
				scResult = unregister_chrdev(gsdwMajor, "vpl_dmac");

				if (scResult < 0)
				{
					PDEBUG("Cannot release major number %d !!\n", (int)gsdwMajor);
					PDEBUG("Exit CleanupModule function !!\n");
					return;
				}
#else
				unregister_chrdev(gsdwMajor, "vpl_dmac");
#endif
			}

#ifdef __PROFILE__
			VPL_DMAC_CloseProfile(gptSharedInfo->hDevInfo);
#endif
			ptMMRInfo = (TVPLDMACInfo *)VPL_DMAC_GetMMRInfo(gptSharedInfo->hDevInfo);

			if (ptMMRInfo != NULL)
			{
				iounmap(ptMMRInfo);
				release_mem_region(VPL_DMAC_MMR_BASE, sizeof(TVPLDMACInfo));
			}

			kfree(gptSharedInfo->hDevInfo);
		}

		if (!IS_ERR(gptSharedInfo->pClk))
		{
			clk_unprepare(gptSharedInfo->pClk);
			clk_put(gptSharedInfo->pClk);
		}

		kfree(gptSharedInfo);
	}

	PDEBUG("Exit CleanupModule function !!\n");

	return;
}

/* ============================================================================================== */
static int InitialModule(void)
{
	int scResult;
	DWORD dwDevInfoSize, dwVersionNum;
	volatile TVPLDMACInfo *ptMMRInfo = (TVPLDMACInfo *)ioremap((int)VPL_DMAC_MMR_BASE, sizeof(TVPLDMACInfo));
	volatile DWORD *pdwRstEnMmr = (DWORD *)(IO_ADDRESS(VPL_SYSC_MMR_BASE)+0x24);

	PDEBUG("Enter InitialModule function...\n");

	if ((gptSharedInfo=(TVPLDMACSharedInfo *)kmalloc(sizeof(TVPLDMACSharedInfo), GFP_KERNEL)) == NULL)
	{
		PDEBUG("Allocate shared info buffer fail !!\n");
		scResult = -ENOMEM;
		goto FAIL;
	}
	memset(gptSharedInfo, 0, sizeof(TVPLDMACSharedInfo));

	dwDevInfoSize = VPL_DMAC_GetDevInfoSize();

	if ((gptSharedInfo->hDevInfo=(HANDLE)kmalloc(dwDevInfoSize, GFP_KERNEL)) == NULL)
	{
		PDEBUG("Allocate device info buffer fail !!\n");
		scResult = -ENOMEM;
		goto FAIL;
	}
	memset(gptSharedInfo->hDevInfo, 0, dwDevInfoSize);

	mutex_init(&gptSharedInfo->ioctl_lock);

	gptSharedInfo->pClk = clk_get(NULL, "vpl_dmac");
	if (IS_ERR(gptSharedInfo->pClk))
	{
		PDEBUG("Fail to get vpl_dmac clock !!\n");
		scResult = PTR_ERR(gptSharedInfo->pClk);
		goto FAIL;
	}

	clk_prepare_enable(gptSharedInfo->pClk);

	request_mem_region(VPL_DMAC_MMR_BASE, sizeof(TVPLDMACInfo), "VPL_DMAC");

	if (VPL_DMAC_SetMMRInfo(gptSharedInfo->hDevInfo, ptMMRInfo, gptSharedInfo->pClk, pdwRstEnMmr) != S_OK)
	{
		scResult = -ENODEV;
		goto FAIL_CLK;
	}

#ifdef __PROFILE__
	VPL_DMAC_InitProfile(gptSharedInfo->hDevInfo);
#endif

	scResult = register_chrdev(gsdwMajor, "vpl_dmac", &vpl_dmac_fops);

	if (scResult < 0)
	{
		PDEBUG("Cannot get major number %d !!\n", (int)gsdwMajor);
		goto FAIL_CLK;
	}

	if (gsdwMajor == 0)
	{
		gsdwMajor = scResult;
	}

	vma_register_device(MKDEV(gsdwMajor, 0), NULL, "vpl_dmac");

	dwVersionNum = VPL_DMAC_GetVersion(gptSharedInfo->hDevInfo);

	printk("Install VPL_DMAC device driver version %d.%d.%d.%d on VPL_DMAC hardware version %d.%d.%d.%d complete !!\n",
	       (int)(VPL_DMAC_VERSION&0xFF),
	       (int)((VPL_DMAC_VERSION>>8)&0xFF),
	       (int)((VPL_DMAC_VERSION>>16)&0xFF),
	       (int)((VPL_DMAC_VERSION>>24)&0xFF),
	       (int)(dwVersionNum>>24)&0xFF,
	       (int)(dwVersionNum>>16)&0xFF,
	       (int)(dwVersionNum>>8)&0xFF,
	       (int)dwVersionNum&0xFF);

	clk_disable(gptSharedInfo->pClk);

	PDEBUG("Exit InitialModule function !!\n");

	return 0;

FAIL_CLK:
	clk_disable(gptSharedInfo->pClk);
FAIL:
	CleanupModule();
	PDEBUG("Exit InitialModule function !!\n");

	return scResult;
}

/* ============================================================================================== */
module_init(InitialModule);
module_exit(CleanupModule);

/* ============================================================================================== */
