/*
 * $Header:$
 *
 *
 * Driver for DesignWare HDMI Transmitter
 *
 * Copyright (C) 2007-2013  VATICS Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $History:$
 *
 */

/* ============================================================================================== */

#ifndef __KERNEL__
    #define __KERNEL__
#endif

#ifndef MODULE
//    #define MODULE
#endif

#include "vpl_hdmitc.h"
#include <linux/irq.h>
#include <linux/irqdomain.h>
#include <linux/slab.h>

#ifdef OTUS_MODIFY
#include <linux/proc_fs.h>
#include <asm/uaccess.h>

#define PROC_NAME "HDMI"
#define HPD_TIME  HZ/2
#endif

const char VPL_HDMITC_ID[] = "$Version: "VPL_HDMITC_ID_VERSION"  (VPL_HDMITC) $";

MODULE_AUTHOR("Vatics Inc.");
MODULE_DESCRIPTION("HDMI Transmitter Driver");
MODULE_LICENSE("GPL");

struct vpl_hdmitc_info *hdmitc_info;
#ifdef POLLING_HPD
struct timer_list hpd_timer;
#endif
/* ============================================================================================== */
static void hdmitc_get_hw_feature(struct vpl_hdmitc_info *hdmi, struct hw_config_cap *hw_cap)
{
	u32 reg;

	reg = hdmitc_readl(hdmi, HDMITC_CONFIG0_ID);
	hw_cap->prepen = (reg & HDMITC_HW_FEAT_PREPEN) >> 7;
	hw_cap->audspdif = (reg & HDMITC_HW_FEAT_AUDSPDIF) >> 5;
	hw_cap->audi2s = (reg & HDMITC_HW_FEAT_AUDI2S) >> 4;
	hw_cap->hdmi14 = (reg & HDMITC_HW_FEAT_HDMI14) >> 3;
	hw_cap->csc = (reg & HDMITC_HW_FEAT_CSC) >> 2;
	hw_cap->cec = (reg & HDMITC_HW_FEAT_CEC) >> 1;
	hw_cap->hdcp = (reg & HDMITC_HW_FEAT_HDCP);

	reg = hdmitc_readl(hdmi, HDMITC_CONFIG1_ID);
	hw_cap->confapb = (reg & HDMITC_HW_FEAT_CONFAPB) >> 1;

	reg = hdmitc_readl(hdmi, HDMITC_CONFIG2_ID);
	hw_cap->phytype = reg;
	reg = hdmitc_readl(hdmi, HDMITC_CONFIG3_ID);
	hw_cap->confahbauddma = (reg & HDMITC_HW_FEAT_CONFAHBAUDDMA) >> 1;
	hw_cap->confgpaud = (reg & HDMITC_HW_FEAT_CONFPGAUD);

	printk("Internal pixel repetition %s\n", (hw_cap->prepen) ? "Y" : "N");
	printk("S/PDIF interface is present %s\n", (hw_cap->audspdif) ? "Y" : "N");
	printk("I2S interface is present %s\n", (hw_cap->audi2s) ? "Y" : "N");
	printk("HDMI 1.4 features are present %s\n", (hw_cap->hdmi14) ? "Y" : "N");
	printk("Color space conversion is present %s\n", (hw_cap->csc) ? "Y" : "N");
	printk("CEC is present %s\n", (hw_cap->cec) ? "Y" : "N");
	printk("HDCP is present %s\n", (hw_cap->hdcp) ? "Y" : "N");
	printk("Configuration interface is APB interface %s\n", (hw_cap->confapb) ? "Y" : "N");

	switch (hw_cap->phytype) {
		case 0x00:
			printk("PHY interface is Legacy PHY (HDMI TX PHY)\n");
			break;
		case 0xB2:
			printk("PHY interface is PHY_HDMI-MHL COMBO with HEAC\n");
			break;
		case 0xC2:
			printk("PHY interface is PHY_HDMI-MHL COMBO\n");
			break;
		case 0xF2:
			printk("PHY interface is PHY_GEN2 (HDMI 3D TX PHY)\n");
			break;
		case 0xE2:
			printk("PHY_GEN2 (HDMI 3D TX PHY) + HEAC PHY\n");
			break;
		default:
			printk("Oops....Unknown PHY interface\n");
	}

	printk("Audio interface is AHB AUD DMA %s\n", (hw_cap->confahbauddma) ? "Y" : "N");
	printk("Audio interface is Generic Parallel Audio (GPAUD) %s\n", (hw_cap->confgpaud) ? "Y" : "N");
}

/* ============================================================================================== */
static void hdmitc_int_maskall(struct vpl_hdmitc_info *hdmitc_info)
{
	printk("    mute interrupts\n");

	/* Mute all interrupts */
	hdmitc_writel(hdmitc_info, 0x03, HDMITC_IH_MUTE);

	printk("    mask out interrupts\n");
	/* Mask all interrupts */
	hdmitc_writel(hdmitc_info, 0xFF, HDMITC_VP_MASK);
	hdmitc_writel(hdmitc_info, 0xE7, HDMITC_FC_MASK0);
	hdmitc_writel(hdmitc_info, 0xFF, HDMITC_FC_MASK1);
	hdmitc_writel(hdmitc_info, 0x03, HDMITC_FC_MASK2);
	hdmitc_writel(hdmitc_info, 0xF3, HDMITC_PHY_MASK0);

	//hdmitc_writel(hdmitc_info, 0x0F, HDMITC_PHY_I2CM_INT);
	//hdmitc_writel(hdmitc_info, 0xFF, HDMITC_PHY_I2CM_CTLINT);

	hdmitc_writel(hdmitc_info, 0x0C, HDMITC_AUD_INT);
	hdmitc_writel(hdmitc_info, 0x10, HDMITC_AUD_INT1);

	//hdmitc_writel(hdmitc_info, 0x0C, HDMITC_AUD_SPDIFINT);
	//hdmitc_writel(hdmitc_info, 0x10, HDMITC_AUD_SPDIFINT1);

	hdmitc_writel(hdmitc_info, 0x7F, HDMITC_CEC_MASK);
#ifdef OTUS_MODIFY
	hdmitc_writel(hdmitc_info, 0x7F, HDMITC_IH_MUTE_CEC_STAT0);
#endif

	hdmitc_writel(hdmitc_info, 0x0F, HDMITC_I2CM_INT);
	hdmitc_writel(hdmitc_info, 0xFF, HDMITC_I2CM_CTLINT);
}

/* ============================================================================================== */
static void hdmitc_int_clearall(struct vpl_hdmitc_info *hdmitc_info)
{
	printk("    clear all interrupt status\n");
	hdmitc_writel(hdmitc_info, 0xFF, HDMITC_IH_FC_STAT0);
	hdmitc_writel(hdmitc_info, 0xFF, HDMITC_IH_FC_STAT1);
	hdmitc_writel(hdmitc_info, 0xFF, HDMITC_IH_FC_STAT2);
	hdmitc_writel(hdmitc_info, 0xFF, HDMITC_IH_AS_STAT0);
	hdmitc_writel(hdmitc_info, 0xFF, HDMITC_IH_PHY_STAT0);
	hdmitc_writel(hdmitc_info, 0xFF, HDMITC_IH_I2CM_STAT0);
	hdmitc_writel(hdmitc_info, 0xFF, HDMITC_IH_CEC_STAT0);
	hdmitc_writel(hdmitc_info, 0xFF, HDMITC_IH_VP_STAT0);
}

/* ============================================================================================== */
static inline void hdmitc_fc_packets_disableall(struct vpl_hdmitc_info *hdmitc_info)
{
	/* Disable all auto packet insertion */
	u32 datauto0 = ~(FC_DATAUTO0_SPD_AUTO | FC_DATAUTO0_VSD_AUTO |
			 FC_DATAUTO0_ISCR2_AUTO | FC_DATAUTO0_ISCR1_AUTO |
			 FC_DATAUTO0_ACP_AUTO);
	printk("    disable all packets\n");
	hdmitc_writel(hdmitc_info, datauto0, HDMITC_FC_DATAUTO0);
}

/* ============================================================================================== */
static inline void hdmitc_mc_gate_init(struct vpl_hdmitc_info *hdmitc_info)
{
	/* Disable all clock gate */
	u32 disclk = MC_CLKDIS_HDCPCLK | MC_CLKDIS_CECCLK | MC_CLKDIS_CSCCLK |
		     MC_CLKDIS_AUDCLK | MC_CLKDIS_PREPCLK | MC_CLKDIS_TMDSCLK |
		     MC_CLKDIS_PIXELCLK;
	printk("    disable all clock gates\n");
	hdmitc_writel(hdmitc_info, disclk, HDMITC_MC_CLKDIS);
}

/* ============================================================================================== */
static inline void hdmitc_mc_configure(struct vpl_hdmitc_info *hdmitc_info)
{
	/* Disable all clock gate */
	u32 disclk = MC_CLKDIS_HDCPCLK | MC_CLKDIS_CSCCLK | MC_CLKDIS_PREPCLK;
	printk("    enable tmsc and pixel clock gates\n");
	if (hdmitc_info->video_param.InputColor != hdmitc_info->video_param.OutputColor) {
		hdmitc_writel(hdmitc_info, 1, HDMITC_MC_FLOWCTRL); //color space bypass
		disclk = MC_CLKDIS_HDCPCLK | MC_CLKDIS_PREPCLK;
	}
	hdmitc_writel(hdmitc_info, disclk, HDMITC_MC_CLKDIS);
}

/* ============================================================================================== */
static void hdmitc_video_framecomposer(struct vpl_hdmitc_info *hdmitc_info)
{
	struct display_timing timing;
	struct vpl_hdmitc_video *video_param = &(hdmitc_info->video_param);
	u32 value = 0;
/*
	timing.vic_code = 2;
	timing.interlaced = 0;
	timing.pixelrep = 0;
	timing.HActive = 720;
	timing.VActive = 480;

	timing.HBlank = 138;
	timing.HSyncFront = 16;
	timing.HSyncWidth = 62;

	timing.VBlank = 45;
	timing.VSyncFront = 9;
	timing.VSyncWidth = 6;

	timing.HSyncPol = 0;
	timing.VSyncPol = 0;
*/
	/*/
	timing.vic_code = 4;
	timing.interlaced = 0;
	timing.pixelrep = 0;
	timing.HActive = 1280;
	timing.VActive = 720;

	timing.HBlank = 370;
	timing.HSyncFront = 110;
	timing.HSyncWidth = 40;

	timing.VBlank = 20;
	timing.VSyncFront = 5;
	timing.VSyncWidth = 5;

	timing.HSyncPol = 1;
	timing.VSyncPol = 1;
*/
#ifndef HDMI_1080I_60FPS
	timing.vic_code = 16;
	timing.interlaced = 0;
	timing.pixelrep = 0;
	timing.HActive = 1920;
	timing.VActive = 1080;

	timing.HBlank = 280;
	timing.HSyncFront = 88;
	timing.HSyncWidth = 44;

	timing.VBlank = 45;
	timing.VSyncFront = 4;
	timing.VSyncWidth = 5;
#else
	timing.vic_code = 5;
	timing.interlaced = 1;
	timing.pixelrep = 0;
	timing.HActive = 1920;
	timing.VActive = 540;

	timing.HBlank = 280;
	timing.HSyncFront = 88;
	timing.HSyncWidth = 44;

	timing.VBlank = 22;
	timing.VSyncFront = 2;
	timing.VSyncWidth = 5;
#endif

	timing.HSyncPol = 1;
	timing.VSyncPol = 1;

	if (!hdmitc_info->video_param.hdmi_mode && hdmitc_info->video_param.pix_repetition) {
		printk("damn, DVI mode with pixel repetition: not transmmit\n");
	}
	printk("    frame composer set dtd timing to fix 720x480p @ 59.94/60Hz, DVI\n");

	/* signal polarity */
	value = hdmitc_set_field(value, timing.VSyncPol<<6, FC_INVIDCONF_VSYNC_IN_POLARITY);
	value = hdmitc_set_field(value, timing.HSyncPol<<5, FC_INVIDCONF_HSYNC_IN_POLARITY);
	value = hdmitc_set_field(value, 1<<4, FC_INVIDCONF_DE_IN_POLARITY); /*sync with phy and voc*/
	value = hdmitc_set_field(value, video_param->hdmi_mode << 3, FC_INVIDCONF_DVI_MODEZ); /*DVI now*/

	/* vsync oscillation for interlace except VID 39 */
	if ((timing.interlaced) && (timing.vic_code != 39))
		value = hdmitc_set_field(value, 1<<1, FC_INVIDCONF_VBLANKOSC);
	else
		value = hdmitc_set_field(value, 0<<1, FC_INVIDCONF_VBLANKOSC);

	/* prograssive / interlace*/
	value = hdmitc_set_field(value, timing.interlaced, FC_INVIDCONF_IN_I_P);

	hdmitc_writel(hdmitc_info, value, HDMITC_FC_INVIDCONF);

	/* H Active */
	hdmitc_writel(hdmitc_info, (timing.HActive & 0xFF), HDMITC_FC_INHACTIV0);
	hdmitc_writel(hdmitc_info, (timing.HActive >> 8), HDMITC_FC_INHACTIV1);

	/* H Blank*/
	hdmitc_writel(hdmitc_info, (timing.HBlank & 0xFF), HDMITC_FC_INHBLANK0);
	hdmitc_writel(hdmitc_info, (timing.HBlank >> 8), HDMITC_FC_INHBLANK1);

	/* V Active*/
	hdmitc_writel(hdmitc_info, (timing.VActive & 0xFF), HDMITC_FC_INVACTIV0);
	hdmitc_writel(hdmitc_info, (timing.VActive >> 8), HDMITC_FC_INVACTIV1);

	/* V Blank */
	hdmitc_writel(hdmitc_info, timing.VBlank, HDMITC_FC_INVBLANK);

	/* H Sync Delay/Front Porch */
	hdmitc_writel(hdmitc_info, timing.HSyncFront, HDMITC_FC_HSYNCINDELAY0);
	hdmitc_writel(hdmitc_info, (timing.HSyncFront >> 8), HDMITC_FC_HSYNCINDELAY1);

	/* H Sync Pulse Width */
	hdmitc_writel(hdmitc_info, timing.HSyncWidth, HDMITC_FC_HSYNCINWIDTH0);
	hdmitc_writel(hdmitc_info, (timing.HSyncWidth>> 8), HDMITC_FC_HSYNCINWIDTH1);

	/* V Sync Delay/Front Porch */
	hdmitc_writel(hdmitc_info, timing.VSyncFront, HDMITC_FC_VSYNCINDELAY);

	/* V Sync Pulse Width */
	hdmitc_writel(hdmitc_info, timing.VSyncWidth, HDMITC_FC_VSYNCINWIDTH);

	/* Control Period Duration HDMI1.4b, min 12 pxl */
	hdmitc_writel(hdmitc_info, 12, HDMITC_FC_CTRLDUR);

	/* Extended Control Period Duration HDMI1.4b, min 32 pxl */
	hdmitc_writel(hdmitc_info, 32, HDMITC_FC_EXCTRLDUR);

	/*
	 * Extended Control Period Maximum Spacing
	 * spacing < 256^2 * config / tmdsClock, spacing <= 50ms
	 * worst case: tmdsClock == 25MHz => config <= 19
	 */
	hdmitc_writel(hdmitc_info, 1, HDMITC_FC_EXCTRLSPAC);

	/* channel non-preamble data */
	hdmitc_writel(hdmitc_info, 11, HDMITC_FC_CH0PREAM);
	hdmitc_writel(hdmitc_info, 22, HDMITC_FC_CH1PREAM);
	hdmitc_writel(hdmitc_info, 33, HDMITC_FC_CH2PREAM);

	/* Input Pixel Repetition, always No repetition(sent only once), VOC doesn't support repetition */
	hdmitc_writel_mask(hdmitc_info,	FC_PROCONF_IN_PR_FACTOR_0, HDMITC_FC_PRCONF, FC_PROCONF_IN_PR_FACTOR_MASK);
}

/* ============================================================================================== */
static void hdmitc_video_packetizer(struct vpl_hdmitc_info *hdmitc_info)
{
	pixel_fm_t output = hdmitc_info->video_param.OutputColor;
	u32 vp_conf = 0;
	u32 pixelrep = 0;

	/*
	 * Clarify control, to read databook Figure 2-4 Video Packetizer Functional Diagram.
	 * Pixrep flow defined by pr_en and bypass_select
	 * Output_select:
	 * if RGB or YCC444
	 * 	if colordeep 8: bypass en and output_select: bypass
	 * 	if colordeep 10, 12, 16: pp_en and output_select: pixel packet.
	 * if YUV422
	 * 	if colordeep 8: ccremap:00, ycc422 en and output_select: ycc 422 remap block
	 * 	if colordeep 10, 12, 16: ycc422 remap, ycc422 en, out ycc 422
	 * BUT VOC only colordeep 8 (24Bit), skip all pixel packing configure check in driver.
	 */

	printk("    video packetizer RGB24Bit in, RGB24 Bit out, no pixrep\n");

	/* 1. Output pixel repetition, VOC doesn't support input repetition, do repetition by HDMITC */
	hdmitc_writel_mask(hdmitc_info, (pixelrep << 0) & 0xF, HDMITC_VP_PR_CD, VP_PR_CD_DESIRED_PR_FACTOR_MASK);
	hdmitc_writel_mask(hdmitc_info, VP_STUFF_PR_STUFFING, HDMITC_VP_STUFF, VP_STUFF_PR_STUFFING);
	if  (pixelrep) {
		hdmitc_writel_mask(hdmitc_info, VP_CONF_PR_EN, HDMITC_VP_CONF, VP_CONF_PR_EN);
		//hdmitc_writel_mask(hdmitc_info, VP_CONF_BYPASS_SELECT_PR, HDMITC_VP_CONF, VP_CONF_BYPASS_SELECT_VP);
		vp_conf = VP_CONF_BYPASS_SELECT_PR;
	} else {
		hdmitc_writel_mask(hdmitc_info, 0, HDMITC_VP_CONF, VP_CONF_PR_EN);
		//hdmitc_writel_mask(hdmitc_info, VP_CONF_BYPASS_SELECT_VP, HDMITC_VP_CONF, VP_CONF_BYPASS_SELECT_VP);
		vp_conf = VP_CONF_BYPASS_SELECT_VP;
	}

	/* 2. Color depth, always 24 bits */
	hdmitc_writel_mask(hdmitc_info, VP_PR_CD_COLOR_DEPTH_24, HDMITC_VP_PR_CD, VP_PR_CD_COLOR_DEPTH_24_MASK);

	/* 3. Packing flow, look Figure 2-4 Video Packetizer Functional Diagram will clearly */
	if (output == RGB || output == YUV444) {
		vp_conf |= VP_CONF_BYPASS_EN | VP_CONF_OUTPUT_SELECT_8_BYPASS;
	} else if (output == YUV422) {
		vp_conf |= VP_CONF_YCC422_EN | VP_CONF_OUTPUT_SELECT_YCC422;
	}
	hdmitc_writel(hdmitc_info, vp_conf, HDMITC_VP_CONF);

	hdmitc_writel(hdmitc_info, VP_REMAP_YCC422_8, HDMITC_VP_REMAP);
	hdmitc_writel_mask(hdmitc_info, VP_STUFF_PP_STUFFING, HDMITC_VP_STUFF, VP_STUFF_PP_STUFFING);
	hdmitc_writel_mask(hdmitc_info, VP_STUFF_YCC422_STUFFING, HDMITC_VP_STUFF, VP_STUFF_YCC422_STUFFING);
}
/* ============================================================================================== */
static void hdmitc_video_sampler(struct vpl_hdmitc_info *hdmitc_info)
{
	pixel_fm_t input = hdmitc_info->video_param.InputColor;
	u32 mapcode = 0;
	u32 gydata = 0, rcrdata = 0, bcbdata = 0;

	if (input == RGB || input == YUV444) {
		/* only 8 bit in Rossini */
		printk("    video sampler RGB or YUV444 8bit\n");
		mapcode = TX_INVID0_VIDEO_MAP_RGB_8BITS;
	} else if (input == YUV422) {
		printk("    video sampler YUV422 8bit\n");
		mapcode = TX_INVID0_VIDEO_MAP_YUV422_8BITS;
	}
	hdmitc_writel(hdmitc_info, mapcode, HDMITC_TX_INVID0);

	/*
	 * Input data stuff when DE is low,
	 * can sample from data input or
	 * sepecified registers
	 */
	hdmitc_writel(hdmitc_info, gydata, HDMITC_TX_GYDATA0);
	hdmitc_writel(hdmitc_info, gydata >> 8, HDMITC_TX_GYDATA1);

	hdmitc_writel(hdmitc_info, rcrdata, HDMITC_TX_RCRDATA0);
	hdmitc_writel(hdmitc_info, rcrdata >> 8, HDMITC_TX_RCRDATA1);

	hdmitc_writel(hdmitc_info, bcbdata, HDMITC_TX_BCBDATA0);
	hdmitc_writel(hdmitc_info, bcbdata >> 8, HDMITC_TX_BCBDATA1);

	hdmitc_writel(hdmitc_info, TX_INSTUFFING_GY | TX_INSTUFFING_RCR | TX_INSTUFFING_BCB, HDMITC_TX_INSTUFFING);
}

/* ============================================================================================== */
static void hdmitc_config_hdmi_packets(struct vpl_hdmitc_info *hdmitc_info)
{
	/* Vendor-Specific InfoFrame, ex.3D, only in HDMI 1.4a */
	/* Source Product Descriptor InfoFrame, Skip now */
	printk("    config avi packate\n");
	//hdmitc_writel(hdmitc_info, 1, HDMITC_FC_DATAUTO1);
	//hdmitc_writel(hdmitc_info, 1<<4 | 1, HDMITC_FC_DATAUTO2);
	/* Gamut Metadata Packet */

	/* AVI InfoFrame */
	  if (hdmitc_info->video_param.OutputColor == RGB)
		hdmitc_writel(hdmitc_info, 0, HDMITC_FC_AVICONF0);
	  else if (hdmitc_info->video_param.OutputColor == YUV422)
		hdmitc_writel(hdmitc_info, 1, HDMITC_FC_AVICONF0);
	  else if (hdmitc_info->video_param.OutputColor == YUV444)
	 	hdmitc_writel(hdmitc_info, 2, HDMITC_FC_AVICONF0);

#ifndef HDMI_1080I_60FPS
	hdmitc_writel(hdmitc_info, 3, HDMITC_FC_AVIVID);
#else
	hdmitc_writel(hdmitc_info, 5, HDMITC_FC_AVIVID);
#endif
}

/* ============================================================================================== */
#if 0
static void hdmitc_audio_init(struct vpl_hdmitc_info *hdmitc_info)
{

	hdmitc_writel(hdmitc_info, 0xA1, HDMITC_AUD_CONF0);	// [5]I2S interface, [0]I2Sdata0 enabled, [7]Audio FIFO reset
	hdmitc_writel(hdmitc_info, 0x10, HDMITC_AUD_CONF1);	//[4:0]data width: 16bits, [7:5]I2S standard mode
	hdmitc_writel(hdmitc_info, ~0, HDMITC_AUD_INT);	//FIFO empty/FIFO full interrupt mask
	// TMDS=27MHz, sampling rate=48KHz ==> N=6144=h'1800, CTS=27000=h'6978
	hdmitc_writel(hdmitc_info, 0x10, HDMITC_AUD_CTS3);	// [4]: CTS manual, [7:5]: No shift of N', [3:0]: CTS[19:16]
	hdmitc_writel(hdmitc_info, 0x69, HDMITC_AUD_CTS2);	// CTS[15:8]
	hdmitc_writel(hdmitc_info, 0x78, HDMITC_AUD_CTS1);	// CTS[7:0]
	hdmitc_writel(hdmitc_info, 0x00, HDMITC_AUD_N3);	// [7]: auto-write, [3:0]: N[19:16]
	hdmitc_writel(hdmitc_info, 0x18, HDMITC_AUD_N2);	// N[15:8]
	hdmitc_writel(hdmitc_info, 0x00, HDMITC_AUD_N1);	// N[7:0]
	hdmitc_writel(hdmitc_info, /*0x01*/4, HDMITC_AUD_INPUTCLKFS);	// [2:0]: Ifsfactor is 256xFs
	hdmitc_writel(hdmitc_info, 0x11, HDMITC_FC_AUDICONF0);
	hdmitc_writel(hdmitc_info, 0x13, HDMITC_FC_AUDICONF1);
	hdmitc_writel(hdmitc_info, 0xee, HDMITC_FC_AUDSV);
	hdmitc_writel(hdmitc_info, 0x01, HDMITC_FC_AUDSCHNLS2);
	hdmitc_writel(hdmitc_info, 0x12, HDMITC_FC_AUDSCHNLS7);
	hdmitc_writel(hdmitc_info, 0x22, HDMITC_FC_AUDSCHNLS8);

#if 0
	hdmitc_writel(hdmitc_info, 0x11, HDMITC_FC_DBGFORCE);
	hdmitc_writel(hdmitc_info, 0x55, HDMITC_FC_DBGAUD0CHN0);
	hdmitc_writel(hdmitc_info, 0xAA, HDMITC_FC_DBGAUD1CHN0);
	hdmitc_writel(hdmitc_info, 0xff, HDMITC_FC_DBGAUD2CHN0);
#endif


}
#else

/* ============================================================================================== */
static void hdmitc_audio_init(struct vpl_hdmitc_info *hdmitc_info)
{

	hdmitc_writel(hdmitc_info, hdmitc_readl(hdmitc_info, HDMITC_MC_FLOWCTRL)&~(1<<3), HDMITC_MC_FLOWCTRL);

	// audio initialize
	hdmitc_writel(hdmitc_info, 0xFF, HDMITC_AUD_DMA_MASK);	// audio DMA interrupt mask
	hdmitc_writel(hdmitc_info, 0xFF, HDMITC_AUD_DMA_BUFMASK);	// audio buffer interrupt mask
	hdmitc_writel(hdmitc_info, 0xF0, HDMITC_FC_AUDSCONF);	// audio mute

	hdmitc_writel(hdmitc_info, 0x21, HDMITC_AUD_CONF0);	// [5]I2S interface, [0]I2Sdata0 enabled, [7]Audio FIFO reset
	hdmitc_writel(hdmitc_info, 0x10, HDMITC_AUD_CONF1);	//[4:0]data width: 16bits, [7:5]I2S standard mode
	hdmitc_writel(hdmitc_info, 0x0c, HDMITC_AUD_INT);	//FIFO empty/FIFO full interrupt mask
	hdmitc_writel(hdmitc_info, 0xA1, HDMITC_AUD_CONF0);	// [7]Audio FIFO reset
	hdmitc_writel(hdmitc_info, 0x80, 0x3300);	// [7]Audio FIFO reset
	hdmitc_writel(hdmitc_info, 0x80, 0x3400);	// [7]Audio FIFO reset
	// TMDS=27MHz, sampling rate=48KHz ==> N=6144=h'1800, CTS=27000=h'6978
	hdmitc_writel(hdmitc_info, 0x00, HDMITC_AUD_N3);	// [7]: auto-write, [3:0]: N[19:16]

	//TMDS = 148.5 MHz
	//#define S32K
#ifdef S32K
	hdmitc_writel(hdmitc_info, 0x10, HDMITC_AUD_N2);	// N[15:8]
	hdmitc_writel(hdmitc_info, 0x00, HDMITC_AUD_N1);	// N[7:0]
#else // S48K
	hdmitc_writel(hdmitc_info, 0x18, HDMITC_AUD_N2);// N[15:8]
	hdmitc_writel(hdmitc_info, 0x00, HDMITC_AUD_N1);// N[7:0]
#endif

#ifndef HDMI_1080I_60FPS
#if 1
	hdmitc_writel(hdmitc_info, 0x12, HDMITC_AUD_CTS3);	// [4]: CTS manual, [7:5]: No shift of N', [3:0]: CTS[19:16]
	hdmitc_writel(hdmitc_info, 0x44, HDMITC_AUD_CTS2);	// CTS[15:8]
	hdmitc_writel(hdmitc_info, 0x14, HDMITC_AUD_CTS1);	// CTS[7:0]
#else
	hdmitc_writel(hdmitc_info, 0x00, HDMITC_AUD_CTS3);	// [4]: CTS auto
	hdmitc_writel(hdmitc_info, 0x00, HDMITC_AUD_CTS2);
	hdmitc_writel(hdmitc_info, 0x00, HDMITC_AUD_CTS1);
#endif
#else
	// Fixed SONY KDL-46NX720 no sound
	hdmitc_writel(hdmitc_info, 0x11, HDMITC_AUD_CTS3);	// [4]: CTS manual, [7:5]: No shift of N', [3:0]: CTS[19:16]
	hdmitc_writel(hdmitc_info, 0x22, HDMITC_AUD_CTS2);	// CTS[15:8]
	hdmitc_writel(hdmitc_info, 0x0a, HDMITC_AUD_CTS1);	// CTS[7:0]
#endif
	hdmitc_writel(hdmitc_info, 0x04, HDMITC_AUD_INPUTCLKFS);	// [2:0]: Ifsfactor is 64xFs
	hdmitc_writel(hdmitc_info, 0x10, HDMITC_FC_AUDICONF0);
	hdmitc_writel(hdmitc_info, 0x00, HDMITC_FC_AUDICONF1);
	hdmitc_writel(hdmitc_info, 0x00, HDMITC_FC_AUDICONF2);
	hdmitc_writel(hdmitc_info, 0x11, HDMITC_FC_AUDSV);
	hdmitc_writel(hdmitc_info, 0x31, HDMITC_FC_AUDSCHNLS0);
	hdmitc_writel(hdmitc_info, 0x11, HDMITC_FC_AUDSCHNLS2);
	hdmitc_writel(hdmitc_info, 0x42, HDMITC_FC_AUDSCHNLS3);
	hdmitc_writel(hdmitc_info, 0x86, HDMITC_FC_AUDSCHNLS4);
	hdmitc_writel(hdmitc_info, 0x31, HDMITC_FC_AUDSCHNLS5);
	hdmitc_writel(hdmitc_info, 0x75, HDMITC_FC_AUDSCHNLS6);
	hdmitc_writel(hdmitc_info, 0x02, HDMITC_FC_AUDSCHNLS7);
	hdmitc_writel(hdmitc_info, 0xD2, HDMITC_FC_AUDSCHNLS8);		//orignal sampling frequency is set to 48K too

#if 0
	hdmitc_writel(hdmitc_info, 0x11, HDMITC_FC_DBGFORCE);
	hdmitc_writel(hdmitc_info, 0x55, HDMITC_FC_DBGAUD0CHN0);
	hdmitc_writel(hdmitc_info, 0xAA, HDMITC_FC_DBGAUD1CHN0);
	hdmitc_writel(hdmitc_info, 0xff, HDMITC_FC_DBGAUD2CHN0);
#endif
	hdmitc_writel(hdmitc_info, 0x00, HDMITC_FC_AUDSCONF);	// audio unmute
	hdmitc_writel(hdmitc_info, hdmitc_readl(hdmitc_info, HDMITC_MC_FLOWCTRL)|(1<<3), HDMITC_MC_FLOWCTRL);

}

#endif

/* ============================================================================================== */
static void hdmitc_read_edid_work_fn(struct work_struct *work)
{

}

/* ============================================================================================== */
#ifdef POLLING_HPD
void hpd_pin_detect(unsigned long data)
{
	u32 hpd_pol;
	u32 hpd_stat;
	u32 phy_int_stat;

	hpd_pol = hdmitc_readl(hdmitc_info, HDMITC_PHY_POL0) & 0x2;
	hpd_stat = hdmitc_readl(hdmitc_info, HDMITC_PHY_STAT0) & 0x2;
	if (hpd_pol == hpd_stat) {
		hdmitc_writel_mask(hdmitc_info, ~hpd_pol, HDMITC_PHY_POL0, PHY_POL0_HPD_MASK);
		phy_int_stat = hdmitc_readl(hdmitc_info, HDMITC_IH_PHY_STAT0);
		hdmitc_writel(hdmitc_info, phy_int_stat, HDMITC_IH_PHY_STAT0);

		printk("hpd_pin_detect hpd_pol %d hpd_stat %d \n", hpd_pol, hpd_stat);
        if (hpd_pol == 2)
			hdmitc_info->connected = 1;
		else
			hdmitc_info->connected = 0;
    }

	hpd_timer.expires = jiffies + HPD_TIME;
	add_timer(&hpd_timer);
}
#endif

/* ============================================================================================== */
static irqreturn_t hdmitc_isr(int irq, void *dev_id)
{
	u8 ih_decode;
	u32 hpd_pol;
	u32 hpd_stat;
	u32 i2cm_int_stat;
	u32 phy_int_stat;


	ih_decode = hdmitc_readl(hdmitc_info, HDMITC_IH_DECODE);

	if (ih_decode & IH_DECODE_PHY) {
		printk("PHY interrupt \n");
		hpd_pol = hdmitc_readl(hdmitc_info, HDMITC_PHY_POL0) & 0x2;
		hpd_stat = hdmitc_readl(hdmitc_info, HDMITC_PHY_STAT0) & 0x2;
		if (hpd_pol == hpd_stat) {
		#ifdef OTUS_MODIFY
			printk("PHY interrupt hpd_pol %d hpd_stat %d \n", hpd_pol, hpd_stat);
			if (hpd_pol == 2)
				hdmitc_info->connected = 1;
			else
				hdmitc_info->connected = 0;
		#endif
			hdmitc_writel_mask(hdmitc_info, ~hpd_pol, HDMITC_PHY_POL0, PHY_POL0_HPD_MASK);
		}
		phy_int_stat = hdmitc_readl(hdmitc_info, HDMITC_IH_PHY_STAT0);
		hdmitc_writel(hdmitc_info, phy_int_stat, HDMITC_IH_PHY_STAT0);
	}

	if (ih_decode & IH_DECODE_I2CM_STAT0) {
		printk("I2CM interrupt \n");
		i2cm_int_stat = hdmitc_readl(hdmitc_info, HDMITC_I2CM_INT);
		hdmitc_writel(hdmitc_info, i2cm_int_stat, HDMITC_I2CM_INT);
	}

	if (ih_decode & IH_DECODE_CEC_STAT0) {
		printk("CEC interrupt \n");
		cec_int_handle(hdmitc_info);
	}

	return IRQ_HANDLED;
}

#ifdef OTUS_MODIFY
/* ============================================================================================== */
int read_proc(struct file *filp, char *buf, size_t count, loff_t *offp) 
{
    if (*offp > 0)
        return 0;

    if(hdmitc_info->connected)
    {
        memcpy(buf, "1", 2);
    }
    else
    {
        memcpy(buf, "0", 2);
    }

    return 2;
}
/* ============================================================================================== */
struct file_operations proc_fops = {
	read:   read_proc
};
#endif
/* ============================================================================================== */
static int __init hdmitc_mod_init(void)
{
	int ret = 0;
	int limit = 10;
	int virq;

	/* TODO: Sysc clock enable */
	void __iomem *sysc_base = ioremap(VPL_SYSC_MMR_BASE, SZ_4K);
	writel(readl(sysc_base+0x94) | 1 << 20, sysc_base+0x94);
	iounmap(sysc_base);

	hdmitc_info = kzalloc(sizeof(*hdmitc_info), GFP_KERNEL);
	if (!hdmitc_info) {
		printk("Cannot allocate device data\n");
		return -ENOMEM;
	}

	hdmitc_info->base = ioremap(VPL_HDMITC_MMR_BASE, SZ_128K);
	if (!hdmitc_info->base) {
		printk("Request hdmitc register region fail\n");
		ret = -ENOMEM;
		goto emap;
	}

	printk("DesignWare Core HDMI Transmitter Version 0x%02x:0x%02x\n",
	       hdmitc_readl(hdmitc_info, HDMITC_DESIGN_ID),
	       hdmitc_readl(hdmitc_info, HDMITC_REVISION_ID));
	printk("Product ID 0x%02x:0x%02x\n",
	       hdmitc_readl(hdmitc_info, HDMITC_PRODUCT_ID0),
	       hdmitc_readl(hdmitc_info, HDMITC_PRODUCT_ID1));

	hdmitc_get_hw_feature(hdmitc_info, &hdmitc_info->hw_cap);

	hdmitc_info->video_param.hdmi_mode = 1;
	hdmitc_info->video_param.InputColor = YUV422;	//evetest
	hdmitc_info->video_param.OutputColor = RGB;
#ifdef OTUS_MODIFY
	hdmitc_info->proc_dir = NULL;
	hdmitc_info->connected = 0;
#endif

	printk("Init stage\n");
	/* mask out all interrupt */
	hdmitc_int_maskall(hdmitc_info);

	/* Initialize all block */
	hdmitc_fc_packets_disableall(hdmitc_info);

	hdmitc_mc_gate_init(hdmitc_info);

	hdmitc_mc_configure(hdmitc_info);

	hdmitc_phy_init(hdmitc_info);

	hdmitc_int_clearall(hdmitc_info);

	printk("Vido configure stage\n");
	/* Video Path Configure */
	hdmitc_dbgvideo(hdmitc_info, 1);

	hdmitc_video_framecomposer(hdmitc_info);

	hdmitc_video_packetizer(hdmitc_info);

	hdmitc_video_csc(hdmitc_info);

	hdmitc_video_sampler(hdmitc_info);

	if (hdmitc_info->video_param.hdmi_mode == 1) {
		hdmitc_config_hdmi_packets(hdmitc_info);
	}

	/* Audio Configure */
	printk("Audio configure stage\n");
	hdmitc_audio_init(hdmitc_info);

	/* Star active */
	/* control */
	hdmitc_mc_configure(hdmitc_info);

	hdmitc_phy_configure(hdmitc_info);

	while (limit--) {
		if (hdmitc_readl(hdmitc_info, HDMITC_PHY_STAT0) & PHY_STAT0_TX_PHY_LOCK)
			break;
		mdelay(50);
		printk("[OOPS], PHY PLL try Locked %x\n", hdmitc_readl(hdmitc_info, HDMITC_PHY_STAT0));
	}
	if (limit < 0)
		printk("[OOPS], PHY PLL Can't Locked\n");

	hdmitc_dbgvideo(hdmitc_info, 0);

	INIT_DELAYED_WORK(&hdmitc_info->edid_work, hdmitc_read_edid_work_fn);

	virq = irq_create_mapping(NULL, VPL_HDMITC_IRQ_NUM);
	ret = request_irq(virq, &hdmitc_isr, IRQF_DISABLED | IRQF_TRIGGER_HIGH, "vpl_hdmitc", hdmitc_info);
	if (ret < 0) {
		printk("Unable to request irq %d !!\n", VPL_HDMITC_IRQ_NUM);
		goto ereirq;
	}

#ifndef POLLING_HPD
	/* enable HPD interrupt */
	hdmitc_writel(hdmitc_info, ~0x2, HDMITC_PHY_MASK0);
#endif
	hdmitc_writel(hdmitc_info, 0x0, HDMITC_IH_MUTE);
#ifdef OTUS_MODIFY
	hdmitc_writel_mask(hdmitc_info, 0, HDMITC_PHY_POL0, PHY_POL0_HPD_MASK);
#endif

	edid_init(hdmitc_info);

#ifndef OTUS_MODIFY
	hdmitc_cec_init(hdmitc_info, 0x1000);
#endif
#ifdef OTUS_MODIFY
	hdmitc_info->proc_dir = proc_create_data(PROC_NAME, 0, NULL, &proc_fops, NULL);

	if (!hdmitc_info->proc_dir)
	{
		printk("proc file create fail\n");
		ret = -ENOMEM;
		goto ereirq;
	}
	printk("HDMITC_CEC_MASK = 0x%x\n", hdmitc_readl(hdmitc_info, HDMITC_CEC_MASK));
	printk("HDMITC_IH_MUTE_CEC_STAT0 = 0x%x\n", hdmitc_readl(hdmitc_info, HDMITC_IH_MUTE_CEC_STAT0));
#ifdef POLLING_HPD
	// use a timer to detect HPD
	init_timer(&hpd_timer);
	hpd_timer.function = hpd_pin_detect;
	hpd_timer.expires = jiffies + HPD_TIME;
	hpd_timer.data = ((unsigned long) 0);
	add_timer(&hpd_timer);
#endif
#endif
	return 0;

ereirq:
#ifdef OTUS_MODIFY
	if (hdmitc_info->base)
	{
		iounmap(hdmitc_info->base);
		hdmitc_info->base = NULL;
	}
#else
	iounmap(hdmitc_info->base);
#endif
emap:
	return ret;
}

static void __exit hdmitc_mod_exit(void)
{
#ifdef OTUS_MODIFY
	if (hdmitc_info)
	{
		if (hdmitc_info->proc_dir)
			remove_proc_entry(PROC_NAME, NULL);

		free_irq(irq_find_mapping(NULL, VPL_HDMITC_IRQ_NUM), hdmitc_info);

		if (hdmitc_info->base)
		{
			iounmap(hdmitc_info->base);
			hdmitc_info->base = NULL;
		}

		kfree(hdmitc_info);
		hdmitc_info = NULL;
#ifdef POLLING_HPD
		del_timer(&hpd_timer);
#endif
	}
#else
	free_irq(irq_find_mapping(NULL, VPL_HDMITC_IRQ_NUM), hdmitc_info);
	iounmap(hdmitc_info->base);
#endif
}

module_init(hdmitc_mod_init);
module_exit(hdmitc_mod_exit);
