#ifndef __KERNEL__
    #define __KERNEL__
#endif

#ifndef MODULE
//    #define MODULE
#endif

#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/dmaengine.h>
#include <linux/init.h>
#include <linux/kthread.h>
#include <linux/module.h>
#include <linux/gfp.h>
#include <linux/random.h>

#include <mach/platform.h>
#include <asm/sizes.h>
#include "vpl_tvec.h"
#include "typedef.h"
#include "tv_encoder.h"

MODULE_AUTHOR("Vatics Inc.");
MODULE_DESCRIPTION("TV Encoder driver");
MODULE_LICENSE("GPL");

#define VDAC_BG_CTRL  0x04
#define VDAC_LDO_CTRL 0x08
#define VDAC_CTRL     0x0C
#define VDAC_DATA     0x10

#define PLLC_4_CTRL 0x24
#define PLLC_4_DIV  0x28

#define SYSC_VOC_CTRL 0x88
static struct clk *clk;
static void __iomem *vdac_base;
static void __iomem *pllc_base;
static void __iomem *sysc_base;

static u32 *tvec_base;

const char VPL_TVEC_ID[] = "$Version: "VPL_TVEC_ID_VERSION"  (TVEC DRIVER) $";

static void vpl_tvec_write_block(u32 *base, const u8 *data, unsigned int len)
{
	while (len >= 2) {
		writel(data[1], base + data[0]);
		len -= 2;
		data += 2;
	}
}

static const unsigned char  init_common[] = {
	0x05, 0x1,	/* 0x2 clrbar, 0x1 bpass_yclamp */
	0x06, 0x09,	/* rgb_setup, rgb_sync, yc_delay, cvbs_enable */
	0x07, 0x00,	/* chroma_bw[1], comp_yuv, compchgain */
	0x0d, 0x01,	/* slave_thresh, slave_mode */
	0x25, 0,	/* tint */
};

static const unsigned char init_ntsc[] = {
	0x00, 0x21,
	0x01, 0xf0,
	0x02, 0x7c,
	0x03, 0x1f,
	0x04, 0,
	0x08, 126,		/* hsync_width */
	0x09, 68,		/* burst_width */
	0x0a, 118,		/* back_porch */
	0x0b, 67,		/* cb_burst_level */
	0x0c, 0,		/* cr_burst_level */
	0x0e, (282 >> 2),	/* black_level msb-8 */
	0x0f, (282 & 0x3),	/* black_level lsb-2 */
	0x10, (240 >> 2),	/* blank_level msb-8 */
	0x11, (240 & 0x3),	/* blank_level lsb-2 */
	0x17, (525 >> 2),	/* num_lines msb-8 */
	0x18, (525 & 0x3),	/* num_lines lsb-2 */
	0x1e, (800 >> 2),	/* white_level msb-8 */
	0x1f, (800 & 0x3),	/* white_level lsb-2 */
	0x20, 156,		/* cb_gain */
	0x22, 156,		/* cr_gain */
	0x29, 22,		/* breeze_way */
	0x2c, 32,		/* front_porch */
	0x31, (1440 >> 3),	/* activeline msb-8, full slave is auto update */
	0x32, (1440 & 0x7),	/* activeline lsb-3 */
	0x33, 21,		/* firstvideoline */
	0x34, (2 << 1),		/* uv_order, pal_mode, chroma_bw[0], invert_top, sys625_50, cphase_rst, vsync5 */
	0x35, 16,		/* sync_level */
	0x3c, (240 >> 2),	/* vbi_blank_level msb-8 */
	0x3d, (240 & 0x3),	/* vbi_blank_level lsb-2 */
};

static const unsigned char init_pal[] = {
	0x00, 0x2a,
	0x01, 0x09,
	0x02, 0x8a,
	0x03, 0xcb,
	0x04, 0,
	0x08, 126,		/* hsync_width */
	0x09, 64,		/* burst_width */
	0x0a, 138,		/* back_porch */
	0x0b, 47,		/* cb_burst_level */
	0x0c, 35,		/* cr_burst_level */
	0x0e, (251 >> 2),	/* black_level msb-8 */
	0x0f, (251 & 0x3),	/* black_level lsb-2 */
	0x10, (251 >> 2),	/* blank_level msb-8 */
	0x11, (251 & 0x3),	/* blank_level lsb-2 */
	0x17, (625 >> 2),	/* num_lines msb-8 */
	0x18, (625 & 0x3),	/* num_lines lsb-2 */
	0x1e, (800 >> 2),	/* white_level msb-8 */
	0x1f, (800 & 0x3),	/* white_level lsb-2 */
	0x20, 165,		/* cb_gain */
	0x22, 165,		/* cr_gain */
	0x29, 26,		/* breeze_way */
	0x2c, 24,		/* front_porch */
	0x31, (1440 >> 3),	/* activeline msb-8, full slave is auto update */
	0x32, (1440 & 0x7),	/* activeline lsb-3 */
	0x33, 22,		/* firstvideoline */
	0x34, (1 | 1<<3 | 1<<6),/* uv_order, pal_mode, chroma_bw[0], invert_top, sys625_50, cphase_rst, vsync5 */
	0x35, 16,		/* sync_level */
	0x3c, (251 >> 2),	/* vbi_blank_level msb-8 */
	0x3d, (251 & 0x3),	/* vbi_blank_level lsb-2 */
	0x41, 0x1,
};

static const unsigned char init_wntsc[] = {
	0x00, 0x19,
	0x01, 0x74,
	0x02, 0x5d,
	0x03, 0x17,
	0x04, 0,
	0x08, 168,		/* hsync_width. */
	0x09, 68,		/* burst_width */
	0x0a, 158,		/* back_porch. */
	0x0b, 67,		/* cb_burst_level */
	0x0c, 0,		/* cr_burst_level */
	0x0e, (282 >> 2),	/* black_level msb-8 */
	0x0f, (282 & 0x3),	/* black_level lsb-2 */
	0x10, (240 >> 2),	/* blank_level msb-8 */
	0x11, (240 & 0x3),	/* blank_level lsb-2 */
	0x17, (525 >> 2),	/* num_lines msb-8 */
	0x18, (525 & 0x3),	/* num_lines lsb-2 */
	0x1e, (800 >> 2),	/* white_level msb-8 */
	0x1f, (800 & 0x3),	/* white_level lsb-2 */
	0x20, 156,		/* cb_gain */
	0x22, 156,		/* cr_gain */
	0x29, 22,		/* breeze_way */
	0x2c, 42,		/* front_porch. */
	0x31, (1920 >> 3),	/* activeline msb-8., full slave is auto update */
	0x32, (1920 & 0x7),	/* activeline lsb-3. */
	0x33, 21,		/* firstvideoline */
	0x34, (2 << 1),		/* uv_order, pal_mode, chroma_bw[0], invert_top, sys625_50, cphase_rst, vsync5 */
	0x35, 16,		/* sync_level */
	0x3c, (240 >> 2),	/* vbi_blank_level msb-8 */
	0x3d, (240 & 0x3),	/* vbi_blank_level lsb-2 */
};

static const unsigned char init_wpal[] = {
	0x00, 0x1f,
	0x01, 0x87,
	0x02, 0x28,
	0x03, 0x18,
	0x04, 0,
	0x08, 126,		/* hsync_width. */
	0x09, 64,		/* burst_width */
	0x0a, 138,		/* back_porch. */
	0x0b, 50,		/* cb_burst_level */
	0x0c, 35,		/* cr_burst_level */
	0x0e, (251 >> 2),	/* black_level msb-8 */
	0x0f, (251 & 0x3),	/* black_level lsb-2 */
	0x10, (251 >> 2),	/* blank_level msb-8 */
	0x11, (251 & 0x3),	/* blank_level lsb-2 */
	0x17, (625 >> 2),	/* num_lines msb-8 */
	0x18, (625 & 0x3),	/* num_lines lsb-2 */
	0x1e, (800 >> 2),	/* white_level msb-8 */
	0x1f, (800 & 0x3),	/* white_level lsb-2 */
	0x20, 166,		/* cb_gain */
	0x22, 166,		/* cr_gain */
	0x29, 26,		/* breeze_way */
	0x2c, 32,		/* front_porch. */
	0x31, (1920 >> 3),	/* activeline msb-8., full slave is auto update */
	0x32, (1920 & 0x7),	/* activeline lsb-3. */
	0x33, 22,		/* firstvideoline */
	0x34, (1 | 1<<3 | 1<<6),/* uv_order, pal_mode, chroma_bw[0], invert_top, sys625_50, cphase_rst, vsync5 */
	0x35, 16,		/* sync_level */
	0x3c, (251 >> 2),	/* vbi_blank_level msb-8 */
	0x3d, (251 & 0x3),	/* vbi_blank_level lsb-2 */
};

static int vpl_tvec_open(int pal_mode_en)
{
	/* VDAC */
	printk("tvec %x\n", readl(vdac_base));
	writel(0x1, vdac_base + VDAC_BG_CTRL);

	/* use external Bias*/
	writel(0x7, vdac_base + VDAC_CTRL);

	/* TVEC mode */
	//Require 27MHz for VDAC
	writel(0x1, sysc_base + 0x7c);

	/* must hold reset when change timing */
	writel(0x01, tvec_base + 0x3e);   // Resync and reset
	mdelay(50);

	vpl_tvec_write_block(tvec_base, init_common, sizeof(init_common));

	/* TVEC timing */
	switch (pal_mode_en) {
		case 0:
			printk("ntsc\n");
			vpl_tvec_write_block(tvec_base, init_ntsc, sizeof(init_ntsc));
			break;
		case 1:
			printk("pal\n");
			vpl_tvec_write_block(tvec_base, init_pal, sizeof(init_pal));
			break;
		case 2:
			printk("w-ntsc 960H\n");
			vpl_tvec_write_block(tvec_base, init_wntsc, sizeof(init_wntsc));
			break;
		case 3:
			printk("w-pal 960H\n");
			vpl_tvec_write_block(tvec_base, init_wpal, sizeof(init_wpal));
			break;
		default:
			printk("unknown mode\n");
	}

	mdelay(50);
	writel(0x0, tvec_base + 0x3e); // Resybc without reset

	return 0;
}

static void vpl_tvec_close(void)
{
	/* close dac */
	writel(0x0, vdac_base + VDAC_BG_CTRL);

	/* hold reset */
	writel(0x1, tvec_base + 0x3e); // Resybc without reset
}

static int __init tvec_mod_init(void)
{
	clk = clk_get(NULL, "vpl_tvec");

	if (IS_ERR(clk)) {
		printk("Fail to get device clk !!\n");
	} else {
		clk_prepare_enable(clk);
	}

	vdac_base = ioremap(VPL_VDAC_MMR_BASE, SZ_4K);
	pllc_base = ioremap(VPL_PLLC_MMR_BASE, SZ_4K);
	sysc_base = ioremap(VPL_SYSC_MMR_BASE, SZ_4K);

	request_mem_region(VPL_TVEC_MMR_BASE, 0x13C, "TV_ENCODER");
	tvec_base = (u32 *)ioremap((int)VPL_TVEC_MMR_BASE, 0x13C);

	writel(0x1, tvec_base + 0x3e);   // Resync and reset
	mdelay(50);

	return 0;
}

static void __exit tvec_mod_exit(void)
{
	clk_disable_unprepare(clk);
	iounmap(vdac_base);
	iounmap(pllc_base);
	iounmap(tvec_base);
	release_mem_region(VPL_TVEC_MMR_BASE, 0x13C);
}

struct tv_encoder_device tv_encoder_ops = {
	.open = vpl_tvec_open,
	.release = vpl_tvec_close,
};
EXPORT_SYMBOL(tv_encoder_ops);

module_init(tvec_mod_init);
module_exit(tvec_mod_exit);
