/*
 * $Header:$
 *
 * vpl_voc
 * Driver for VPL VOC
 *
 * Copyright (C) 2007-2012  VATICS Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $History:$
 *
 */

/* ============================================================================================== */
#ifndef __KERNEL__
	#define __KERNEL__
#endif //!__KERNEL__

#ifndef MODULE
//	#define MODULE
#endif //!MODULE

/* ============================================================================================== */
#include "vpl_voc_driver.h"

/* ============================================================================================== */
const CHAR VPL_VOC_ID[] = "$Version: "VPL_VOC_ID_VERSION"  (VPL_VOC) $";
static TVPLVOCDevInfo *gptDevInfo = NULL;
static HANDLE hDevInfo;
static SDWORD gsdwMajor = 0;
static bool keep_boot_settings = 0;
/* ============================================================================================== */
MODULE_AUTHOR ("VATICS Inc.");
MODULE_LICENSE ("GPL");
module_param (gsdwMajor, int, 0644);
MODULE_PARM_DESC (gsdwMajor, "Major number for VPL_VOC module");
module_param (keep_boot_settings, bool, 0644);
MODULE_PARM_DESC (keep_boot_settings, "Keep settings when insert driver");

/* ============================================================================================== */
static irqreturn_t ISR(int irq, void *dev_id)
{
	PDEBUG("Enter ISR function !!\n");

	if (gptDevInfo->ptMMRInfo->dwStat & 0x10)
		printk("VOC underflow\n");

	if (!gptDevInfo->bfirst_int && (gptDevInfo->ptCur_Frm != gptDevInfo->ptNext_Frm))
	{
		gptDevInfo->ptCur_Frm->state = BUF_STATE_DONE;
		list_add_tail(&gptDevInfo->ptCur_Frm->queue, &gptDevInfo->done_list); //move to done_list
		gptDevInfo->ptCur_Frm = gptDevInfo->ptNext_Frm;
		PDEBUG("add done buffer for dequeue\n");
	}

	gptDevInfo->bfirst_int = 0;
	if (!list_empty(&gptDevInfo->queued_list))
	{
		gptDevInfo->ptNext_Frm = list_first_entry(&gptDevInfo->queued_list, TVPLVOCBufInfo, queue);

		list_del_init(&gptDevInfo->ptNext_Frm->queue);
		gptDevInfo->ptNext_Frm->state = BUF_STATE_ACTIVE;

		gptDevInfo->ptMMRInfo->dwYBuff1Addr = (DWORD)(gptDevInfo->ptNext_Frm->pbyYFrame);
		gptDevInfo->ptMMRInfo->dwCbBuff1Addr = (DWORD)(gptDevInfo->ptNext_Frm->pbyCbFrame);
		gptDevInfo->ptMMRInfo->dwCrBuff1Addr = (DWORD)(gptDevInfo->ptNext_Frm->pbyCrFrame);

		gptDevInfo->ptNext_Frm->dwFrameCount = gptDevInfo->dwFrameCnt;
	}

	if (gptDevInfo->bpip_on) {
		if (!gptDevInfo->bfirst_int_pip && (gptDevInfo->ptCur_Frm_pip != gptDevInfo->ptNext_Frm_pip)) {
			gptDevInfo->ptCur_Frm_pip->state = BUF_STATE_DONE;
			list_add_tail(&gptDevInfo->ptCur_Frm_pip->queue, &gptDevInfo->done_list_pip); //move to done_list
			gptDevInfo->ptCur_Frm_pip = gptDevInfo->ptNext_Frm_pip;
		}

		gptDevInfo->bfirst_int_pip = 0;
		if (!list_empty(&gptDevInfo->queued_list_pip)) {
			gptDevInfo->ptNext_Frm_pip = list_first_entry(&gptDevInfo->queued_list_pip, TVPLVOCBufInfo, queue);

			list_del_init(&gptDevInfo->ptNext_Frm_pip->queue);
			gptDevInfo->ptNext_Frm_pip->state = BUF_STATE_ACTIVE;

			gptDevInfo->ptMMRInfo->dwPIPYBuff1Addr = (DWORD)(gptDevInfo->ptNext_Frm_pip->pbyYFrame);
			gptDevInfo->ptMMRInfo->dwPIPCbBuff1Addr = (DWORD)(gptDevInfo->ptNext_Frm_pip->pbyCbFrame);
			gptDevInfo->ptMMRInfo->dwPIPCrBuff1Addr = (DWORD)(gptDevInfo->ptNext_Frm_pip->pbyCrFrame);

			gptDevInfo->ptCur_Frm_pip->dwFrameCount = gptDevInfo->dwFrameCnt_pip;
		}
	}

	VPL_VOC_IntrClear(hDevInfo);

	gptDevInfo->dwFrameCnt++;

	wake_up_interruptible(&gptDevInfo->wq);

	if (gptDevInfo->bpip_on) {
		gptDevInfo->dwFrameCnt_pip++;

		wake_up_interruptible(&gptDevInfo->wq_pip);
	}

	PDEBUG("Exit ISR function !!\n");
	return IRQ_HANDLED;
}

/* ============================================================================================== */
static SCODE Start(TVPLVOCDevInfo *ptDevInfo)
{
	return VPL_VOC_Start(hDevInfo);
}

/* ============================================================================================== */
static SCODE WaitComplete(TVPLVOCDevInfo *ptDevInfo)
{
	return S_OK;
}

/* ============================================================================================== */
static int Close(struct inode *pinode, struct file *pfile)
{
	PDEBUG("Enter Close function...\n");

	if (pfile->private_data != NULL)
	{
		pfile->private_data = NULL;
	}


	//writel(readl(SYSC_CLK_EN_MMR)|(0x1<<VPL_VOC_CLK_EN_NUM), SYSC_CLK_EN_MMR);

	VPL_VOC_Stop(hDevInfo);
	VPL_VOC_IntrClear(hDevInfo);
	VPL_VOC_IntrDisable(hDevInfo);
	VPL_VOC_ClearQBuf(hDevInfo);
	VPL_VOC_PIP_ClearQBuf(hDevInfo);
	//VPL_VOC_Stop(hDevInfo);
	if (gptDevInfo->dwIrq != (DWORD)NULL)
	{
		free_irq(irq_find_mapping(NULL, VPL_VOC_IRQ_NUM), gptDevInfo);
	}

	gptDevInfo->dwIrq = (DWORD)NULL;

	clk_disable(gptDevInfo->clk);

	gptDevInfo->open_count = 0;

	PDEBUG("Exit Close function !!\n");

	return 0;
}

/* ============================================================================================== */
static int Open(struct inode *pinode, struct file *pfile)
{
	int scResult;
	int virq;

	PDEBUG("Enter Open function...\n");

	if (gptDevInfo->open_count)
	{
		PDEBUG("Exit Open function !!\n");
		return -EBUSY;
	}

	gptDevInfo->open_count = 1;
	if (VPL_VOC_InitProfileInfo(gptDevInfo) != S_OK)
	{
		PDEBUG("Fail to initialize profile info !!\n");
		PDEBUG("Exit Open function !!\n");
		return -EFAULT;
	}

	clk_enable(gptDevInfo->clk);

	VPL_VOC_Reset(hDevInfo);

	if (gptDevInfo->dwIrq == (DWORD)NULL)
	{
		VPL_VOC_IntrDisable(hDevInfo);

		virq = irq_create_mapping(NULL, VPL_VOC_IRQ_NUM);
		scResult = request_irq(virq, ISR, IRQF_TRIGGER_HIGH, "vpl_voc", gptDevInfo);

		if (scResult < 0)
		{
			PDEBUG("Cannot get irq %d !!\n", VPL_VOC_IRQ_NUM);
//			if (try_module_get(THIS_MODULE) == 0)
//			{
//				PDEBUG("Exit Open function !!\n");
//				return -EBUSY;
//			}

			Close(pinode, pfile);
			PDEBUG("Exit Open function !!\n");
			return scResult;
		}

		disable_irq(VPL_VOC_IRQ_NUM);
		enable_irq(VPL_VOC_IRQ_NUM);
		gptDevInfo->dwIrq = virq;

		VPL_VOC_IntrClear(hDevInfo);
		VPL_VOC_IntrEnable(hDevInfo);

		VPL_VOC_Open(hDevInfo);
	}
	//writel(readl(SYSC_CLK_EN_MMR)&(~(0x1<<VPL_VOC_CLK_EN_NUM)), SYSC_CLK_EN_MMR);

	pfile->private_data = gptDevInfo;

	PDEBUG("Exit Open function !!\n");

	return 0;
}

/* ============================================================================================== */
static long Ioctl(struct file *pfile, unsigned int dwCmd, unsigned long dwArg)
{
	TVPLVOCDevInfo *ptVOCDevInfo = (TVPLVOCDevInfo *)pfile->private_data;
	TVOCInitOptions tVOCInitOptions;
	DWORD dwVersionNum;
	int scError;
	TVideoBuffer tVideoBuf_User;
	TPIPInfo tPIPInfo_User;
	PDEBUG("Enter Ioctl function...\n");

	if (pfile->private_data == NULL)
	{
		PDEBUG("Device does not exist !!\n");
		PDEBUG("Exit Ioctl function !!\n");
		return -ENODEV;
	}

	if ((_IOC_TYPE(dwCmd)!=VPL_VOC_IOC_MAGIC) || (_IOC_NR(dwCmd)>VPL_VOC_IOC_MAX_NUMBER))
	{
		PDEBUG("Incorrect ioctl command !!\n");
		PDEBUG("Exit Ioctl function !!\n");

		return -ENOTTY;
	}

	if (_IOC_DIR(dwCmd) & _IOC_READ)
	{
		scError = !access_ok(VERIFY_WRITE, (void *)dwArg, _IOC_SIZE(dwCmd));
	}
	else if (_IOC_DIR(dwCmd) & _IOC_WRITE)
	{
		scError = !access_ok(VERIFY_READ, (void *)dwArg, _IOC_SIZE(dwCmd));
	}
	else
	{
		scError = 0;
	}

	if (scError != 0)
	{
		PDEBUG("Unsupport ioctl command %d !!\n", dwCmd);
		PDEBUG("Exit Ioctl function !!\n");

		return -EFAULT;
	}

	switch (dwCmd)
	{
		case VPL_VOC_IOC_START:
			if (Start(ptVOCDevInfo) != S_OK)
			{
				PDEBUG("Exit Ioctl function !!\n");
				return S_FAIL;
			}
			break;
		case VPL_VOC_IOC_WAIT_COMPLETE:
			if (WaitComplete(ptVOCDevInfo) != S_OK)
			{
				PDEBUG("Exit Ioctl function !!\n");
				return S_FAIL;
			}
			break;
		case VPL_VOC_IOC_GET_VERSION_NUMBER:
			dwVersionNum = VPL_VOC_VERSION;
			scError = copy_to_user((DWORD *)dwArg, &dwVersionNum, sizeof(DWORD));
			break;
		case VPL_VOC_IOC_GET_BANDWIDTH:
			VPL_VOC_SetupProfile(ptVOCDevInfo, dwArg, dwCmd);
			break;
		case VPL_VOC_IOC_GET_RG_INTERVAL:
			VPL_VOC_SetupProfile(ptVOCDevInfo, dwArg, dwCmd);
			break;
		case VPL_VOC_IOC_GET_REQ_TIMES:
			VPL_VOC_SetupProfile(ptVOCDevInfo, dwArg, dwCmd);
			break;
		case VPL_VOC_IOC_CLEAR_PROFILE:
			VPL_VOC_SetupProfile(ptVOCDevInfo, dwArg, dwCmd);
			break;
		case VPL_VOC_IOC_GET_FRAME_CNT:
			scError = copy_to_user((DWORD *)dwArg, (DWORD *)&(ptVOCDevInfo->dwFrameCnt), sizeof(DWORD));
			break;
		case VPL_VOC_IOC_INITIAL:
			scError = copy_from_user(&tVOCInitOptions, (TVOCInitOptions *)dwArg, sizeof(TVOCInitOptions));
			VPL_VOC_Init(hDevInfo, &tVOCInitOptions, keep_boot_settings);
			keep_boot_settings = false;
			break;
		case VPL_VOC_IOC_QUEUE_BUF:
			scError = copy_from_user(&tVideoBuf_User, (void *)dwArg, sizeof(TVideoBuffer));
			VPL_VOC_QBuf(hDevInfo, &tVideoBuf_User);
			break;
		case VPL_VOC_IOC_DEQUEUE_BUF:
			VPL_VOC_DQBuf(hDevInfo, &tVideoBuf_User);
			scError = copy_to_user((DWORD *)dwArg, &tVideoBuf_User, sizeof(TVideoBuffer));
			break;
		case VPL_VOC_IOC_PIP_START:
			VPL_VOC_PIP_START(hDevInfo);
			break;
		case VPL_VOC_IOC_PIP_STOP:
			VPL_VOC_PIP_STOP(hDevInfo);
			break;
		case VPL_VOC_IOC_PIP_QUEUE_BUF:
			scError = copy_from_user(&tVideoBuf_User, (void *)dwArg, sizeof(TVideoBuffer));
			VPL_VOC_PIP_QBuf(hDevInfo, &tVideoBuf_User);
			break;
		case VPL_VOC_IOC_PIP_DEQUEUE_BUF:
			VPL_VOC_PIP_DQBuf(hDevInfo, &tVideoBuf_User);
			scError = copy_to_user((DWORD *)dwArg, &tVideoBuf_User, sizeof(TVideoBuffer));
			break;
		case VPL_VOC_IOC_PIP_SET_IMAGE:
			if (copy_from_user((&tPIPInfo_User), (void *)dwArg, sizeof(tPIPInfo_User))) {
				return -EFAULT;
			}
			VPL_VOC_Set_PIP_Image(hDevInfo, tPIPInfo_User.dwWidth, tPIPInfo_User.dwHeight, tPIPInfo_User.dwStride);
			VPL_VOC_Set_PIP_Pos(hDevInfo, tPIPInfo_User.dwXPos, tPIPInfo_User.dwYPos);
		default:
			PDEBUG("Exit Ioctl function !!\n");
			return -ENOTTY;
	}

	PDEBUG("Exit Ioctl function !!\n");

	return scError;
}

/* ============================================================================================== */
static int MMap(struct file *file, struct vm_area_struct *vma)
{
	DWORD dwSize;

	dwSize = vma->vm_end - vma->vm_start;

	vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
	vma->vm_flags |= VM_DONTEXPAND | VM_DONTDUMP;

	if (remap_pfn_range(vma, vma->vm_start, (VPL_VOC_MMR_BASE>>PAGE_SHIFT), dwSize, vma->vm_page_prot))
	{
		return -EAGAIN;
	}

	PDEBUG("Start address = 0x%08lX, end address = 0x%08lX\n", vma->vm_start, vma->vm_end);

	return 0;
}

/* ============================================================================================== */
struct file_operations vpl_voc_fops =
{
	owner: THIS_MODULE,
	unlocked_ioctl:	Ioctl,
	mmap:		MMap,
	open:		Open,
	release:	Close,
};

/* ============================================================================================== */
static void CleanupModule(void)
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,24)
	int scResult;
#endif
	TVPLVOCInfo *ptMMRInfo;

	PDEBUG("Enter CleanupModule function...\n");

//	if (gptSharedInfo != NULL)
	{
		if (gptDevInfo != NULL)
		{
			if (gsdwMajor != 0)
			{
				vocfb_do_release();

				vma_unregister_device(MKDEV(gsdwMajor, 0));
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,24)
				scResult = unregister_chrdev(gsdwMajor, "vpl_voc");

				if (scResult < 0)
				{
					PDEBUG("Cannot release major number %d !!\n", (int)gsdwMajor);
					PDEBUG("Exit CleanupModule function !!\n");
					return;
				}
#else
				unregister_chrdev(gsdwMajor, "vpl_voc");
#endif
			}

			VPL_VOC_CloseProfile(hDevInfo);
			ptMMRInfo = (TVPLVOCInfo *)VPL_VOC_GetMMRInfo(hDevInfo);

			if (ptMMRInfo != NULL)
			{
				iounmap(ptMMRInfo);
				release_mem_region(VPL_VOC_MMR_BASE, sizeof(TVPLVOCInfo));
			}

			kfree(hDevInfo);
			if (!keep_boot_settings)
				clk_disable(gptDevInfo->clk);
			clk_unprepare(gptDevInfo->clk);
			clk_put(gptDevInfo->clk);
		}
		voc_connect_mgr_remove();
//		kfree(gptSharedInfo);
	}

	PDEBUG("Exit CleanupModule function !!\n");

	return;
}

/* ============================================================================================== */
static int InitialModule(void)
{
	int scResult;
	DWORD dwDevInfoSize, dwVersionNum;
	volatile TVPLVOCInfo *ptMMRInfo = (TVPLVOCInfo *)ioremap((int)VPL_VOC_MMR_BASE, sizeof(TVPLVOCInfo));
	volatile TVPLPLLCInfo *ptPLLCMMRInfo = (TVPLPLLCInfo *) IO_ADDRESS(VPL_PLLC_MMR_BASE);
	volatile DWORD *pdwSYSCVOCCtrlMMR = (DWORD *) IO_ADDRESS(VPL_SYSC_MMR_BASE + 0x88);

	/* Set VOC AHB Bus Priority */
#if 0   /* Default set VOC AHB bus as high priority. */
    writel(0x20000, IO_ADDRESS(VPL_AHBC_2_MMR_BASE)+0x4);
#else
    writel( 0x80000 | readl(IO_ADDRESS(VPL_AHBC_1_MMR_BASE)+0x4), IO_ADDRESS(VPL_AHBC_1_MMR_BASE)+0x4);
#endif

	PDEBUG("Enter InitialModule function...\n");

	dwDevInfoSize = VPL_VOC_GetDevInfoSize();

	if ((gptDevInfo = (TVPLVOCDevInfo *)kmalloc(sizeof(TVPLVOCDevInfo), GFP_KERNEL)) == NULL)
	{
		PDEBUG("Allocate device info buffer fail !!\n");
		scResult = -ENOMEM;
		goto FAIL;
	}
	memset(gptDevInfo, 0, sizeof(TVPLVOCDevInfo));
	hDevInfo = (HANDLE)gptDevInfo;

	gptDevInfo->clk = clk_get(NULL, "vpl_voc");

	if (IS_ERR(gptDevInfo->clk)) {
		printk("no clock\n");
		return -EINVAL;
	}

	if (clk_prepare_enable(gptDevInfo->clk))
		printk("can't enable\n");

	request_mem_region(VPL_VOC_MMR_BASE, sizeof(TVPLVOCInfo), "VPL_VOC");

	if (VPL_VOC_SetMMRInfo(hDevInfo, ptMMRInfo, ptPLLCMMRInfo, pdwSYSCVOCCtrlMMR) != S_OK)
	{
		scResult = -ENODEV;
		goto FAIL;
	}

	/* Reset when install driver, clear the legacy in previous install */
	if (!keep_boot_settings)
		VPL_VOC_Reset(hDevInfo);

	VPL_VOC_InitProfile(hDevInfo);

	scResult = register_chrdev(gsdwMajor, "vpl_voc", &vpl_voc_fops);

	if (scResult < 0)
	{
		PDEBUG("Cannot get major number %d !!\n", (int)gsdwMajor);
		goto FAIL;
	}

	if (gsdwMajor == 0)
	{
		gsdwMajor = scResult;
	}

	vma_register_device(MKDEV(gsdwMajor, 0), NULL, "vpl_voc");

	dwVersionNum = VPL_VOC_GetVersion(hDevInfo);

	printk("Install VPL_VOC device driver version %d.%d.%d.%d on VPL_VOC hardware version %d.%d.%d.%d complete !!\n",
	       (int)(VPL_VOC_VERSION&0xFF),
	       (int)((VPL_VOC_VERSION>>8)&0xFF),
	       (int)((VPL_VOC_VERSION>>16)&0xFF),
	       (int)((VPL_VOC_VERSION>>24)&0xFF),
	       (int)(dwVersionNum>>24)&0xFF,
	       (int)(dwVersionNum>>16)&0xFF,
	       (int)(dwVersionNum>>8)&0xFF,
	       (int)dwVersionNum&0xFF);

	if (!keep_boot_settings)
		clk_disable(gptDevInfo->clk);

	vocfb_do_probe(hDevInfo);

	voc_connect_mgr_init();

	PDEBUG("Exit InitialModule function !!\n");

	return 0;

FAIL:
	clk_disable_unprepare(gptDevInfo->clk);
	clk_put(gptDevInfo->clk);
	CleanupModule();
	PDEBUG("Exit InitialModule function !!\n");

	return scResult;
}

/* ============================================================================================== */
module_init(InitialModule);
module_exit(CleanupModule);

/* ============================================================================================== */
