/*
 * $Header:$
 *
 * vpl_voc
 * Driver for VPL VOC
 *
 * Copyright (C) 2007-2012  VATICS Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $History:$
 *
 */

/* ============================================================================================== */
#include <linux/fb.h>
#include "vpl_voc_driver.h"
#include "vpl_voc_fb.h"
#define PALETTE_SIZE 256

/* ============================================================================================== */
struct voc_fb_device {
	struct device   *dev; /* well, plateform device */
	struct fb_info  *fb_info;
	struct vocfb_mem_info mem_info;
	u32 palette[PALETTE_SIZE];
	u32 pseudo_palette[16]; /* MEMO: For TRUECOLOR, Beethoven */
	bool osd_is_on;
	u8 padding[2];
	/*MEMO :struct xxx pannel(provide sync) */
};

static struct voc_fb_device *fbdev; /*TODO: we can hide this by platform device/driver method */
static unsigned long def_vxres = 640;
static unsigned long def_vyres = 480;

static struct fb_fix_screeninfo voc_fb_fix = {
	.id =           "vpl_voc_osd_fb",
	.type =         FB_TYPE_PACKED_PIXELS,
	.visual =       FB_VISUAL_PSEUDOCOLOR,
	.xpanstep =     0,
	.ypanstep =     0,
	.ywrapstep =    1,
	.accel =        FB_ACCEL_NONE,
};

/* MEMO:
 * fake pannel structure
 * lcd pannel @device hdmi, video encoder PA syncT
 * drivers/video/omap/lcd_inn1610.c
 */
/*
struct voc_subdevice
{
	.config ex.  OMAP_LCDC_PANEL_TFT | OMAP_LCDC_INV_VSYNC | OMAP_LCDC_INV_HSYNC | OMAP_LCDC_HSVS_RISING_EDGE | OMAP_LCDC_HSVS_OPPOSITE
	.init
	.sync_spec (xres, yres, pixel_clock, hsw, vfp, vbp, pcd)
	.get_caps
	data_lines;  Lines on LCD HW interface? --should means 8bit, 16bit, 24bit data pin
}
*/

/* ============================================================================================== */
static int vocfb_open(struct fb_info *info, int user)
{
	return 0;
}

/* ============================================================================================== */
static int vocfb_release(struct fb_info *info, int user)
{
	return 0;
}

/* ============================================================================================== */
/* According to skeletonfb.c
 * 1. See if the hardware supports
 * 2. Not alter the hardware state (fb_info, xxx_par)
 * 3. Change PASSED in var, not fb_info.var, (FBIOPUT_VSCREENINFO flow will help to copy it, see fbmem.c)
 * 4. off value:round up; larger than capability: return -EINVAL
 */
static int vocfb_check_var(struct fb_var_screeninfo *var, struct fb_info *info)
{
	int mem;
	/* Round up bits_per_pixel, xres, yres, xoffset, yoffset, bitfield */
	if (var->bits_per_pixel > 8) {
		printk("vocfb: only support 8bit PSEUDOCOLOR image\n");
		/*dev_err(fbdev->dev, "inavlid bpp\n");*/
		return -EINVAL;
	}
	else {
		var->bits_per_pixel = 8;
	}

	if (var->xres > var->xres_virtual)
		var->xres_virtual = var->xres;

	if (var->yres > var->yres_virtual)
		var->yres_virtual = var->yres;

	if ((var->yres > VOCFB_OSD_MAX_HEIGHT) || (var->xres_virtual > VOCFB_OSD_MAX_WIDTH)) {
		printk("Warning: OSD resolution out of HW capabiliy\n");
		return -EINVAL;
	}

	/* TODO:Check whether is larger than current major display mode? */

	var->xoffset = 0;
	var->yoffset = 0;
	var->vmode = FB_VMODE_NONINTERLACED;

	/* Make sure we don't exceed our allocated memory */
	mem = var->yres_virtual*var->xres_virtual*(var->bits_per_pixel>>3);
	if (mem > info->fix.smem_len) {
		printk("vocfb: not enough framebuffer memory (%d bytes required, %d allocated)", mem, info->fix.smem_len);
		return -EINVAL;
	}

	switch (var->bits_per_pixel) {
		case 8:
			var->red   = (struct fb_bitfield) {0, 8, 0};      /* offset, length, msb-right */
			var->green = (struct fb_bitfield) {0, 8, 0};
			var->blue  = (struct fb_bitfield) {0, 8, 0};
			var->transp = (struct fb_bitfield) {0, 8, 0};
			break;
	}

	/*if (var->lef_margin + var->xres > )*/
	/*if (var->upper_margin + var->yres > )*/
	return 0;
}

/* ============================================================================================== */
/*
 * Change "fix" and applied to Hardware with var settings.
 * The "var" should be checked by check_var function, then call set_par.
 * This fix can change only here.
 */
static int vocfb_set_par(struct fb_info *info)
{
	info->fix.line_length = (info->var.xres_virtual * info->var.bits_per_pixel) >> 3;
	VPL_VOC_Set_OSD_Image(info->par, info->var.xres, info->var.yres, info->var.xres_virtual);
	return 0;
}

/* ============================================================================================== */
static int vocfb_setcolreg(unsigned regno, unsigned red, unsigned green,
		unsigned blue, unsigned transp, struct fb_info *info)
{
	int Y, Cb, Cr;
	int A, R, G, B; /* for signed operation and 16bit->8bit */

	/* HW limitation, can't change palette when OSD ON */
	if (fbdev->osd_is_on)
		return -EINVAL;

	if (info->var.grayscale) {
		/* grayscale = 0.30*R + 0.59*G + 0.11*B */
		red = green = blue = (red * 77 + green * 151 + blue * 28) >> 8;
	}

	switch(info->fix.visual) {
		case FB_VISUAL_PSEUDOCOLOR:
			if (regno >= PALETTE_SIZE)
				return -EINVAL;
			/*
			 * Y  =  (0.257 * r) + (0.504 * g) + (0.098 * b) + 16;
			 * Cb = -(0.148 * r) - (0.291 * g) + (0.439 * b) + 128;
			 * Cr =  (0.439 * r) - (0.368 * g) - (0.071 * b) + 128;
			 *
			 * SCALEBITS_IN 8
			 * FIX_IN(x)  ((u16)((x) * (1L<<SCALEBITS_IN) + 0.5))
			 *  Y  =  ((FIX_IN(0.257) * r) + (FIX_IN(0.504) * g) + (FIX_IN(0.098) * b) + (1L<<SCALEBITS_IN)/2) >> SCALEBITS_IN + 16;
			 *  Cb = (-(FIX_IN(0.148) * r) - (FIX_IN(0.291) * g) + (FIX_IN(0.439) * b) + (1L<<SCALEBITS_IN)/2) >> SCALEBITS_IN + 128;
			 *  Cr =  ((FIX_IN(0.439) * r) - (FIX_IN(0.368) * g) - (FIX_IN(0.071) * b) + (1L<<SCALEBITS_IN)/2) >> SCALEBITS_IN + 128;
			 */
			A = 0xff - (transp >> 8);
			R = (red >> 8);
			G = (green >> 8);
			B = (blue >> 8);

			Y = (((66 * R + 129 * G +  25 * B) + 128) >> 8) + 16;
			Cb =(((-38 * R - 74 * G + 112 * B) + 128) >> 8) + 128;
			Cr =(((112 * R - 94 * G - 18 * B) + 128) >> 8) + 128;

			/* clip */
			Y = (Y < 0) ? 0 : ((Y > 255) ? 255 : Y);
			Cb = (Cb < 0) ? 0 : ((Cb > 255) ? 255 : Cb);
			Cr = (Cr < 0) ? 0 : ((Cr > 255) ? 255 : Cr);
			break;
		default:
			return -EINVAL;
	}

	printk("[%03d] alpha (%d), rgb (%d, %d, %d) -> ycbcr(%d, %d %d)\n", regno, A, R, G, B, Y, Cb, Cr);
	VPL_VOC_SetPalette(info->par, regno, A, Y, Cb, Cr);

	return 0;
}
#if 0
static int vocfb_setcmap(struct fb_cmap *cmap, struct fb_info *info)
{
	int Y, Cb, Cr;
	int R, G, B; /* for signed operation and 16bit->8bit */

	int i, start, r;
	u16 *red, *green, *blue, *transp;
	u16 trans = 0xffff;

	red = cmap->red;
	green = cmap->green;
	blue = cmap->blue;
	start = cmap->start;

	for (i = 0; i < cmap->len; i++) {
		R = (red[i] >> 8);
		G = (green[i] >> 8);
		B = (blue[i] >> 8);
		Y = (((66 * R + 129 * G +  25 * B) + 128) >> 8) + 16;
		Cb = (((-38 * R - 74 * G + 112 * B) + 128) >> 8) + 128;
		Cr = (((112 * R - 94 * G - 18 * B) + 128) >> 8) + 128;
		fbdev->palette[start++] = (Y<<16 | Cb<<8 | Cr);
	}
	start = cmap->start;
	for (i = 0; i < cmap->len; i++) {

		VPL_VOC_SetPalette(info->par, start, fbdev->palette[start++]>>16, fbdev->palette[start++], *blue++, trans, info);
	}

	return 0;
}
#endif

/* ============================================================================================== */
static int vocfb_pan_display(struct fb_var_screeninfo *var, struct fb_info *info)
{
	unsigned int osd_addr = 0;
	/* up layer in fbmem.c helps to check yoffset, yres, yres_virtual range already */
	if (var->vmode & FB_VMODE_YWRAP) {
		osd_addr = info->fix.smem_start + var->xres * var->yoffset * (info->var.bits_per_pixel / 8);
		VPL_VOC_Set_OSD_Addr(info->par, osd_addr);
	}
	return 0;
}

/* ============================================================================================== */
static int vocfb_setup_fbmem(struct fb_info *info, struct vocfb_mem_info *mem)
{
	if (info->screen_base != 0)
		iounmap(info->screen_base);

	info->screen_base = ioremap(mem->paddr, mem->size);
	if (!info->screen_base) {
		printk(KERN_ERR "vocfb: unable to map framebuffer\n");
		return -ENOMEM;
	}
	info->screen_size = mem->size;
	info->fix.smem_start = mem->paddr;
	info->fix.smem_len = mem->size;
	VPL_VOC_Set_OSD_Addr(info->par, mem->paddr);
	return 0;
}

/* ============================================================================================== */
static int vocfb_ioctl(struct fb_info *info, unsigned int cmd, unsigned long arg)
{
	int rc = 0;

	struct vocfb_mem_info mem;
	struct vocfb_osd_pos osd_pos;
	void __user *argp = (void __user *)arg;
	switch(cmd) {
		case VOCFB_IOC_SETUP_MEM:
			if (copy_from_user(&mem, argp, sizeof(mem))) {
				rc = -EFAULT;
				break;
			}
			rc = vocfb_setup_fbmem(info, &mem);
			if (rc)
				break;

			fbdev->mem_info = mem;
			break;
		case VOCFB_IOC_QUERY_MEM:
			if (copy_to_user(argp, &mem, sizeof(mem)))
				rc = -EFAULT;
			break;
		case VOCFB_IOC_OSD_ON:
			if (!fbdev->mem_info.paddr) {
				printk("vocfb: Can't start OSD without setup OSD Video RAM through VOCFB_IOC_SETUP_MEM\n");
				rc = -EFAULT;
				break;
			}
			VPL_VOC_Set_OSD_DISP(info->par, 1, argp);
			fbdev->osd_is_on = true;
			break;
		case VOCFB_IOC_OSD_OFF:
			VPL_VOC_Set_OSD_DISP(info->par, 0, 0);
			fbdev->osd_is_on = false;
			break;
		case VOCFB_IOC_SET_POS:
			if (copy_from_user(&osd_pos, argp, sizeof(osd_pos))) {
				rc = -EFAULT;
				break;
			}
			VPL_VOC_Set_OSD_Pos(info->par, (TVPLVOCOSDPOS *)&osd_pos);
			break;
		default:
			rc = -EINVAL;
	}

	return rc;
}

/* ============================================================================================== */
#if 0
vocfb_mmap(struct fb_info *info, struct vma_area_struct *vma)
{
	unsigned long start = vma->vm_start;
	unsigned long size = vma->vm_end - vma->vm_start;
	unsigned long offset = vma->vm_pgoff << PAGE_SHIFT;
	unsigned long page, pos;

	if (offset + size > info->fix.smem_len) {
		return -EINVAL;
	}

	pos = (unsigned long)info->fix.smem_start + offset;

	while (size > 0) {
		page = vmalloc_to_pfn((void *)pos);
		if (remap_pfn_range(vma, start, page, PAGE_SIZE, PAGE_SHARED)) {
			return -EAGAIN;
		}
		start += PAGE_SIZE;
		pos += PAGE_SIZE;
		if (size > PAGE_SIZE)
			size -= PAGE_SIZE;
		else
			size = 0;
	}

	return 0;
}
#endif
/* ============================================================================================== */
static struct fb_ops vocfb_ops = {
	.owner			= THIS_MODULE,
	.fb_open		= vocfb_open,
	.fb_release		= vocfb_release,
	.fb_read		= NULL,
	.fb_write		= NULL,
	.fb_check_var		= vocfb_check_var,
	.fb_set_par		= vocfb_set_par,
	.fb_setcolreg		= vocfb_setcolreg,
	.fb_setcmap		= NULL,
	.fb_blank		= NULL,//vocfb_blank,
	.fb_pan_display		= vocfb_pan_display,
	.fb_fillrect		= NULL, /* sys_fillrect */
	.fb_copyarea		= NULL, /* sys_copyarea */
	.fb_imageblit		= NULL, /* sys_imageblit */
	.fb_cursor		= NULL,
	.fb_rotate		= NULL,
	.fb_sync		= NULL,
	.fb_ioctl 		= vocfb_ioctl,
	.fb_mmap		= NULL,
	.fb_get_caps		= NULL,
};

/* ============================================================================================== */
int vocfb_do_probe(HANDLE hDevInfo)
{
	struct fb_info *info;
	struct fb_var_screeninfo *var;
	int ret = -ENOMEM;

	fbdev = kzalloc(sizeof(struct voc_fb_device), GFP_KERNEL);
	if (fbdev == NULL) {
		//dev_err(dev, "Can't allocate memory for driver state.\n");
		printk("Can't allocate memory for driver state.\n");
		return -ENOMEM;
	}

	/* MEMO: omap initialize voc device in fb, skip this, or maybe, a global skip_init can pass if voc initialize by library
	 * 1.setup_color_conv_coef
	 * 2.setup datalin, i.e data pin number , maybe include shift
	 * 3.setup x_res, y_res, i.e in_widht, out_width, check with 1<<regbitsnumb only?
	 * 4.setup timing by info from registered pannel
	 */
	/* MEMO: set default mode same with the display manager(main video display) */

	/* allocat framebuffer info */
	info = framebuffer_alloc(sizeof(int), fbdev->dev);
	if (info == NULL)
		goto err_fballoc;

	info->par = hDevInfo; /* private data area */

	fbdev->fb_info = info;
	info->fbops = &vocfb_ops;
	info->flags = FBINFO_DEFAULT;
	info->pseudo_palette  = fbdev->pseudo_palette;
	/* set fix */
	info->fix = voc_fb_fix;

	/*MEMO: Video fb find default mode and major mode control; In OSD, is not a real mode control*/

	/*set var*/
	var = &info->var;
	var->xres = def_vxres;
	var->yres = def_vyres;
	var->xres_virtual = def_vxres;
	var->yres_virtual = def_vyres;
	var->bits_per_pixel = 8;
	var->activate = FB_ACTIVATE_NOW;
	ret = fb_alloc_cmap(&info->cmap, PALETTE_SIZE, 1);
	if (ret < 0) {
		dev_err(fbdev->dev, "unable to allocate color map memory\n");
		goto err_cmap;
	}

	ret = register_framebuffer(fbdev->fb_info);
	if (ret < 0) {
		dev_err(fbdev->dev, "register framebuffer failed\n");
		goto err_reg;
	}

	printk(KERN_INFO "fb%d: Virtual frame buffer device probed\n", info->node);

	return 0;
err_reg:
	fb_dealloc_cmap(&info->cmap);
err_cmap:
	framebuffer_release(info);
err_fballoc:
	kfree(fbdev);

	return ret;
}

/* ============================================================================================== */
void vocfb_do_release(void)
{
	struct fb_info *info = fbdev->fb_info;
	if (info) {
		unregister_framebuffer(info);

		if (info->cmap.len)
			fb_dealloc_cmap(&info->cmap);

		if (info->screen_base)
			iounmap(info->screen_base);

		framebuffer_release(info);

		if (fbdev)
			kfree(fbdev);
	}
}

/* ============================================================================================== */
