/*
 * Video Display Subdevice Manager
 *
 * Copyright (C) 2013-2018  VATICS Inc.
 *
 * Author: ChangHsien Ho <vincent.ho@vatics.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/err.h>
#include <linux/string.h>
#include "vpl_voc_driver.h"
#include "vpl_voc_sub.h"
#include "vpl_voc_mgr.h"

static DEFINE_MUTEX(vocmgr_lock);
struct voc_device_mgr voc_dev_mgr;
EXPORT_SYMBOL(voc_dev_mgr);

static int voc_mode_vrefresh(const struct videomode *vm)
{
	int refresh = 0;
	unsigned int calc_val;
	int htotal, vtotal;

	htotal = vm->hactive + vm->hback_porch + vm->hfront_porch + vm->hsync_len;
	vtotal = vm->vactive + vm->vback_porch + vm->vfront_porch + vm->vsync_len;
	calc_val = vm->pixelclock;

	if (htotal > 0 && vtotal > 0) {
		/* work out vrefresh the value will be x1000 */
		calc_val /= htotal;
		refresh = (calc_val + vtotal / 2) / vtotal;

		if (vm->flags & DISPLAY_FLAGS_INTERLACED)
			refresh *= 2;
		/*
		if (mode->flags & DRM_MODE_FLAG_DBLSCAN)
			refresh /= 2;
		if (mode->vscan > 1)
			refresh /= mode->vscan;
		*/
	}
	return refresh;
}

void videomode_to_voc_timing(const struct videomode *vm, struct vpl_voc_timing *voc_timing)
{
	char VocConnectorName[VOC_MODE_LEN];

	memcpy(VocConnectorName, voc_timing->ConnectorName, VOC_MODE_LEN);
	memset(voc_timing, 0, sizeof(*voc_timing));

	voc_timing->dwActiveWidth = vm->hactive;
	voc_timing->dwActiveHeight = vm->vactive;
	voc_timing->dwRefresh = voc_mode_vrefresh(vm);
	voc_timing->dwPixClock = vm->pixelclock;

	voc_timing->dwOutWidth = vm->hactive + vm->hback_porch + vm->hfront_porch + vm->hsync_len;
	voc_timing->dwOutHeight = vm->vactive + vm->vback_porch + vm->vfront_porch + vm->vsync_len;

	voc_timing->dwHsyncFrontPorch = vm->hfront_porch;
	voc_timing->dwHsyncBackPorch = vm->hback_porch;
	voc_timing->dwVsyncFrontPorch = vm->vfront_porch;
	voc_timing->dwVsyncBackPorch = vm->vback_porch;

	if (vm->flags & DISPLAY_FLAGS_HSYNC_HIGH)
		voc_timing->dwHSyncPol = ACTIVE_HIGH;
	else if (vm->flags & DISPLAY_FLAGS_HSYNC_LOW)
		voc_timing->dwHSyncPol = ACTIVE_LOW;

	if (vm->flags & DISPLAY_FLAGS_VSYNC_HIGH)
		voc_timing->dwVSyncPol = ACTIVE_HIGH;
	else if (vm->flags & DISPLAY_FLAGS_VSYNC_LOW)
		voc_timing->dwVSyncPol = ACTIVE_LOW;

	if (vm->flags & DISPLAY_FLAGS_DE_HIGH)
		voc_timing->dwBlankPol = ACTIVE_LOW; /* DE Active High == Blank Active Low */
	else if (vm->flags & DISPLAY_FLAGS_DE_LOW)
		voc_timing->dwBlankPol = ACTIVE_HIGH; /* DE Active Low == Blank Active High */

	if (vm->flags & DISPLAY_FLAGS_PIXDATA_POSEDGE)
		voc_timing->dwPClkPol = POSITIVE_EDGE_ALIGNED;
	else if (vm->flags & DISPLAY_FLAGS_PIXDATA_NEGEDGE)
		voc_timing->dwPClkPol = NEGATIVE_EDGE_ALIGNED;

	if (vm->flags & DISPLAY_FLAGS_INTERLACED)
		voc_timing->bInterlaced = 1;

	if (voc_timing->bInterlaced) {
		if (strcasecmp(VocConnectorName, "Composite") == 0) {
			voc_timing->dwInterOutSPH = 0;
			voc_timing->dwInterF0Overlap = 2;
			voc_timing->dwInterF1Overlap = 3;
			voc_timing->dwInterF0Height = (voc_timing->dwOutHeight >> 1);
		}
		else {
		#ifndef HDMI_1080i_60FPS
			voc_timing->dwInterOutSPH = 0;
			voc_timing->dwInterF0Overlap = 2;
			voc_timing->dwInterF1Overlap = 3;
			voc_timing->dwInterF0Height = (voc_timing->dwOutHeight >> 1);
		#else
	    	voc_timing->dwOutHeight = vm->vactive +
	    	        		((vm->vback_porch + vm->vfront_porch
	    	                + vm->vsync_len) << 1) + 1;
	    	voc_timing->dwInterOutSPH = (voc_timing->dwOutWidth >> 1);

	    	/* HDMI interlace tricky, can't set FOverlap and F1Ovarlap for
	    	 * interlace BT1120, otherwise the separate sync signals
	    	 * will be abnormal shift
	    	 * */
	    	voc_timing->dwInterF0Overlap = 0;
	    	voc_timing->dwInterF1Overlap = 0;
			voc_timing->dwInterF0Height = (voc_timing->dwOutHeight >> 1);
		#endif
		}
	}

	if (strcasecmp(VocConnectorName, "Composite") == 0) {
		snprintf(voc_timing->name, VOC_MODE_LEN, "%dx%d%s",
			 vm->hactive, vm->vactive,
			 voc_timing->bInterlaced ? "i" : "");
	}
	else {
	#ifndef HDMI_1080i_60FPS
		snprintf(voc_timing->name, VOC_MODE_LEN, "%dx%d%s",
			 vm->hactive, vm->vactive,
			 voc_timing->bInterlaced ? "i" : "");
	#else
		snprintf(voc_timing->name, VOC_MODE_LEN, "%dx%d",
		        vm->hactive, vm->vactive);
	#endif
	}

	PDEBUG("timings %d %d %d %d %d %d %d %d %d %d %d %d %d %d %s %s %s %s %d\n",
		voc_timing->dwActiveWidth,
		voc_timing->dwActiveHeight,
		voc_timing->dwRefresh,
	       	voc_timing->dwPixClock,
		voc_timing->dwOutWidth,
	      	voc_timing->dwOutHeight,
		voc_timing->dwHsyncFrontPorch,
        	voc_timing->dwHsyncBackPorch,
		voc_timing->dwVsyncFrontPorch,
		voc_timing->dwVsyncBackPorch,
		voc_timing->dwInterOutSPH,
		voc_timing->dwInterF0Overlap,
		voc_timing->dwInterF1Overlap,
		voc_timing->dwInterF0Height,
		(voc_timing->dwHSyncPol == ACTIVE_HIGH)?"ACTIVE_HIGH":"ACTIVE_LOW",
		(voc_timing->dwVSyncPol == ACTIVE_HIGH)?"ACTIVE_HIGH":"ACTIVE_LOW",
		(voc_timing->dwBlankPol == ACTIVE_HIGH)?"ACTIVE_HIGH":"ACTIVE_LOW",
		(voc_timing->dwPClkPol == POSITIVE_EDGE_ALIGNED)?"POSITIVE_EDGE_ALIGNED":"NEGATIVE_EDGE_ALIGNED",
		voc_timing->bInterlaced
		);

}

int voc_register_connector(struct voc_connector *connector)
{
	struct voc_device_mgr *mgr = &voc_dev_mgr;

	if (connector->if_version != SUBCONNECT_IF_VERSION) {
		PDEBUG("interface version err %d\n", connector->if_version);
		return -EINVAL;
	}
	if (mgr->num_connector > VOC_MAX_CONNECTOR)
		return -EINVAL;

	if ((connector->num_modes == 0) || (connector->modes == NULL))
		return -EINVAL;

	mutex_lock(&vocmgr_lock);

	mgr->connectors[mgr->num_connector++] = connector;
	if (mgr->num_connector == 1) {
		mgr->curt_connector = connector;
		strncpy(mgr->curt_mode.ConnectorName, mgr->curt_connector->name, VOC_MODE_LEN);
		videomode_to_voc_timing(&mgr->curt_connector->modes[0], &mgr->curt_mode);
	}

	mutex_unlock(&vocmgr_lock);

	voc_sysfs_add_connector(connector);

	return 0;
}
EXPORT_SYMBOL(voc_register_connector);

int voc_unregister_connector(struct voc_connector *connector)
{
	struct voc_device_mgr *mgr = &voc_dev_mgr;
	int i,j;

	mutex_lock(&vocmgr_lock);

	for (i = 0; i < mgr->num_connector; i++) {
		if (connector == mgr->connectors[i]) {
			mgr->connectors[i] = NULL;
			mgr->num_connector--;
			for (j = i; j < mgr->num_connector; j++) {
				mgr->connectors[j] = mgr->connectors[j + 1];
			}
			break;
		}
	}

	if ((mgr->num_connector) && (mgr->curt_connector == connector)) {
		mgr->curt_connector = mgr->connectors[mgr->num_connector-1];
		strncpy(mgr->curt_mode.ConnectorName, mgr->curt_connector->name, VOC_MODE_LEN);
		videomode_to_voc_timing(&mgr->curt_connector->modes[0], &mgr->curt_mode);
	} else if (mgr->num_connector == 0) {
		mgr->curt_connector = NULL;
	}

	mutex_unlock(&vocmgr_lock);

	device_unregister(&connector->dev);

	return 0;
}
EXPORT_SYMBOL(voc_unregister_connector);

/*
 * Cause the new register API also need to modiy HDMI and TVEC driver to work with it.
 * Before it done, we need to keep them work with a self registered info.
 * This function is for temp compatiable with HDMI and TV connector
 */

const struct videomode hdmi_modes[] = {
	/*         pixclk,   hctv,   hf,   hb,   hsync,   vctv, vf, vb, vsync, flags */
	/* 640x480p @ 59.94/60 Hz (Format 1) */
	[0] = {  25200000,    640,   16,   48,      96,    480, 10, 33,    2,  DISPLAY_FLAGS_HSYNC_LOW | DISPLAY_FLAGS_VSYNC_LOW | DISPLAY_FLAGS_PIXDATA_NEGEDGE },
	/* 720x480p @ 59.94/60 Hz (Formats 2 & 3) */
	[1] = {  27000000,    720,   16,   60,      62,    480,  9,  30,   6,  DISPLAY_FLAGS_HSYNC_LOW | DISPLAY_FLAGS_VSYNC_LOW | DISPLAY_FLAGS_PIXDATA_NEGEDGE },
	/* 1280x720p @ 59.94/60 Hz (Format 4) */
	[2] = {  74250000,   1280,  110,  220,      40,    720,  5,  20,   5,  DISPLAY_FLAGS_HSYNC_HIGH | DISPLAY_FLAGS_VSYNC_HIGH | DISPLAY_FLAGS_PIXDATA_NEGEDGE },
#ifndef HDMI_1080i_60FPS
	/* 1920X1080p @ 59.94/60 Hz (Format 16) */
	[3] = { 148500000,   1920,   88,  148,      44,   1080,  4,  36,   5,  DISPLAY_FLAGS_HSYNC_HIGH | DISPLAY_FLAGS_VSYNC_HIGH | DISPLAY_FLAGS_PIXDATA_NEGEDGE },
	/* 720X576p @ 50 Hz (Formats 17 & 18) */
	[4] = {  27000000,    720,   12,   68,      64,    576,  5,  39,   5,  DISPLAY_FLAGS_HSYNC_LOW | DISPLAY_FLAGS_VSYNC_LOW | DISPLAY_FLAGS_PIXDATA_NEGEDGE },
	/* 1920X1080p @ 23.97/24 Hz (Format 32) */
	[5] = {  74250000,   1920,  638,  148,      44,   1080,  4, 36,    5,  DISPLAY_FLAGS_HSYNC_HIGH | DISPLAY_FLAGS_VSYNC_HIGH | DISPLAY_FLAGS_PIXDATA_NEGEDGE },
	/* 1920X1080p @ 29.97/30 Hz (Format 34) */
	[6] = {  74250000,   1920,   88,  148,      44,   1080,  4, 36,    5,  DISPLAY_FLAGS_HSYNC_HIGH | DISPLAY_FLAGS_VSYNC_HIGH | DISPLAY_FLAGS_PIXDATA_NEGEDGE },
#else
	/* 1920X1080i @ 59.94/60 Hz (Format 16) */
	[3] = {  74250000,   1920,   88,  148,      44,   1080,  2,  15,   5,  DISPLAY_FLAGS_HSYNC_HIGH | DISPLAY_FLAGS_VSYNC_HIGH | DISPLAY_FLAGS_PIXDATA_NEGEDGE | DISPLAY_FLAGS_INTERLACED },
#endif
};

static struct voc_connector vpl_hdmitc = {
	.if_version = SUBCONNECT_IF_VERSION,
	.name = "HDMI-B",
	.data_arrangement = ARRANGEMENT_POS_1,
#ifndef HDMI_1080i_60FPS
	.signal_format = VIDEO_SIGNAL_FORMAT_PROGRESSIVE_BT1120,
	.num_modes = 7,
#else
	.signal_format = VIDEO_SIGNAL_FORMAT_INTERLACE_RAW_16BITS,
	.num_modes = 4,
#endif
	.modes = hdmi_modes,
	.setup = NULL,
};

const struct videomode tv_modes[] = {
	/*         pixclk,   hctv,   hf,   hb,   hsync,   vctv, vf, vb, vsync, flags */
	[0] = {  13500000,    720,   16,    2,     120,    480,  9,  0,   36,  DISPLAY_FLAGS_HSYNC_HIGH | DISPLAY_FLAGS_VSYNC_HIGH | DISPLAY_FLAGS_PIXDATA_NEGEDGE | DISPLAY_FLAGS_INTERLACED },
	[1] = {  13500000,    720,   12,    2,     130,    576,  5,  0,   44,  DISPLAY_FLAGS_HSYNC_HIGH | DISPLAY_FLAGS_VSYNC_HIGH | DISPLAY_FLAGS_PIXDATA_NEGEDGE | DISPLAY_FLAGS_INTERLACED },
#if 0 /* experiment, not standardlize SONY 960H */
	[2] = {  18000000,    960,   21,    2,     161,    480,  9,  0,   36,  DISPLAY_FLAGS_HSYNC_HIGH | DISPLAY_FLAGS_VSYNC_HIGH | DISPLAY_FLAGS_PIXDATA_NEGEDGE | DISPLAY_FLAGS_INTERLACED },
	[3] = {  18000000,    960,   16,    2,     174,    576,  9,  0,   40,  DISPLAY_FLAGS_HSYNC_HIGH | DISPLAY_FLAGS_VSYNC_HIGH | DISPLAY_FLAGS_PIXDATA_NEGEDGE | DISPLAY_FLAGS_INTERLACED },
#endif
};

static struct voc_connector vpl_tvec = {
	.if_version = SUBCONNECT_IF_VERSION,
	.name = "Composite",
	.signal_format = VIDEO_SIGNAL_FORMAT_INTERLACE_RAW_16BITS,
	.data_arrangement = ARRANGEMENT_POS_1,
	.num_modes = 2,
	.modes = tv_modes,
	.setup = NULL,
};

int voc_connect_mgr_init(void)
{
	struct voc_device_mgr *mgr = &voc_dev_mgr;
	memset((void *)mgr, 0, sizeof(struct voc_device_mgr));

	voc_class_init();

	voc_register_connector(&vpl_hdmitc);
	voc_register_connector(&vpl_tvec);
	return 0;
}

void voc_connect_mgr_remove(void)
{
	voc_unregister_connector(&vpl_hdmitc);
	voc_unregister_connector(&vpl_tvec);
	voc_class_exit();
}
