/*
 * Video Display Sysfs
 *
 * Copyright (C) 2013-2018  VATICS Inc.
 *
 * Author: ChangHsien Ho <vincent.ho@vatics.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/gfp.h>
#include <linux/ctype.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/sysfs.h>
#include <linux/stat.h>
#include "vpl_voc_mgr.h"

extern struct voc_device_mgr voc_dev_mgr;
/* class attribute */
static ssize_t version_show(struct class *class,
                           struct class_attribute *attr,
                           char *buf)
{
	return sprintf(buf, "0x%08x\n", SUBCONNECT_IF_VERSION);
}

static ssize_t connectors_show(struct class *class,
                           struct class_attribute *attr,
                           char *buf)
{
	struct voc_device_mgr *mgr = &voc_dev_mgr;
	int written = 0;
	int i;

	if (mgr->num_connector == 0)
		return 0;

	for (i = 0; i < mgr->num_connector; i++) {
		written += snprintf(buf + written, PAGE_SIZE - written,  "%s\n", (char *)&mgr->connectors[i]->name);
	}
	return written;
}

static ssize_t curt_connector_show(struct class *class,
                           struct class_attribute *attr,
                           char *buf)
{
	struct voc_device_mgr *mgr = &voc_dev_mgr;
	if (mgr->num_connector == 0)
		return 0;

	return snprintf(buf, PAGE_SIZE, "%s\n", mgr->curt_connector->name);
}

static ssize_t curt_connector_store(struct class *class,
                           struct class_attribute *attr,
                           const char *buf, size_t count)
{
	struct voc_device_mgr *mgr = &voc_dev_mgr;
	int i;

	for (i = 0; i < mgr->num_connector; i++) {
		if (sysfs_streq(buf, (char *)&mgr->connectors[i]->name)) {
			mgr->curt_connector = mgr->connectors[i];
			strncpy(mgr->curt_mode.ConnectorName, mgr->curt_connector->name, VOC_MODE_LEN);
			videomode_to_voc_timing(&mgr->curt_connector->modes[0], &mgr->curt_mode);
			return count;
		}
	}

	return -EINVAL;
}

static ssize_t curt_mode_show(struct class *class,
                           struct class_attribute *attr,
                           char *buf)
{
	struct voc_device_mgr *mgr = &voc_dev_mgr;
	if (mgr->num_connector == 0)
		return 0;

	return snprintf(buf, PAGE_SIZE, "%s\n", mgr->curt_mode.name);
}

static ssize_t curt_mode_store(struct class *class,
                           struct class_attribute *attr,
                           const char *buf, size_t count)
{
	struct voc_device_mgr *mgr = &voc_dev_mgr;
	int i;
	const struct videomode *modes = mgr->curt_connector->modes;
	int num_modes = mgr->curt_connector->num_modes;
	char name[32];
	bool interlaced;

	if (mgr->num_connector == 0)
		return 0;

	for (i = 0; i < num_modes; i++) {
		interlaced = !!(modes[i].flags & DISPLAY_FLAGS_INTERLACED);
		if (strcasecmp(mgr->curt_connector->name, "Composite") == 0) {
			snprintf(name, 32, "%dx%d%s", modes[i].hactive, modes[i].vactive, interlaced ? "i" : "");
		}
		else {
		#ifndef HDMI_1080i_60FPS
			snprintf(name, 32, "%dx%d%s", modes[i].hactive, modes[i].vactive, interlaced ? "i" : "");
		#else
			snprintf(name, 32, "%dx%d", modes[i].hactive, modes[i].vactive);
		#endif
		}

		if (sysfs_streq(buf, name)) {
			strncpy(mgr->curt_mode.ConnectorName, mgr->curt_connector->name, VOC_MODE_LEN);
			videomode_to_voc_timing(&modes[i], &mgr->curt_mode);
			return count;
		}
	}

	return -EINVAL;
}

static struct class_attribute voc_class_attrs[] = {
	__ATTR_RO(version),
	__ATTR_RO(connectors),
	__ATTR(curt_connector, S_IRWXUGO, curt_connector_show, curt_connector_store),
	__ATTR(curt_mode, S_IRWXUGO, curt_mode_show, curt_mode_store),
	__ATTR_NULL
};

/* Device attribute */
#define to_voc_connector(cd) container_of(cd, struct voc_connector, dev)

static ssize_t modes_show(struct device *dev,
                           struct device_attribute *attr,
                           char *buf)
{
	struct voc_connector *connector = to_voc_connector(dev);
	int written = 0;
	const struct videomode *modes = connector->modes;
	int num_modes = connector->num_modes;
	int i;
	char name[32];
	bool interlaced;

	for (i = 0; i < num_modes; i++) {
		interlaced = !!(modes[i].flags & DISPLAY_FLAGS_INTERLACED);
		snprintf(name, 32, "%dx%d%s", modes[i].hactive, modes[i].vactive, interlaced ? "i" : "");
		written += snprintf(buf + written, PAGE_SIZE - written, "%s\n", name);
	}

        return written;
}

static struct device_attribute voc_device_attrs[] = {
	__ATTR_RO(modes),
	__ATTR_NULL
};

static struct class voc_class = {
	.name = "vpl_voc",
	.owner = THIS_MODULE,
	.class_attrs = voc_class_attrs,
	.dev_attrs = voc_device_attrs,
};

static void voc_connector_device_release(struct device *dev) {};

int voc_sysfs_add_connector(struct voc_connector *connector)
{
	int ret;
	connector->dev.class = &voc_class;

	dev_set_name(&connector->dev, "%s", connector->name);
	ret = device_register(&connector->dev);
	if (ret < 0) {
		printk(KERN_ERR "%s: device_register failed\n", __func__);
		return ret;
	}
	/* Register the release callback that will be called when the last
	   reference to the device goes away. */
	connector->dev.release = voc_connector_device_release;

	return 0;
}

int voc_class_init(void)
{
	return class_register(&voc_class);
}

void voc_class_exit(void)
{
	class_unregister(&voc_class);
}
