/*
 * drivers/clk/vatics/clk-pll.c
 *
 * Copyright (C) 2013-2018  VATICS Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
//#define DEBUG
#include <linux/kernel.h>
#include <linux/clk.h>
#include <linux/clk-provider.h>
#include <linux/clkdev.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/clk-private.h>
#include <mach/platform.h>
#include "clk.h"

#define PLL_BASE_LOCK    BIT(4)
#define PLL_BASE_BYPASS  BIT(3)
#define PLL_BASE_STANDBY BIT(2)
#define PLL_BASE_PWRDOWN BIT(1)
#define PLL_BASE_UPDATE  BIT(0)

#define PLL_BASE_DIVP_SHIFT 8
#define PLL_BASE_DIVP_WIDTH 5
#define PLL_BASE_DIVN_SHIFT 0
#define PLL_BASE_DIVN_WIDTH 6
#define PLL_BASE_DIVM_SHIFT 16
#define PLL_BASE_DIVM_WIDTH 16

#define PLL_1_BASE_DIVP_SHIFT 8
#define PLL_1_BASE_DIVP_WIDTH 5
#define PLL_1_BASE_DIVN_SHIFT 0
#define PLL_1_BASE_DIVN_WIDTH 6
#define PLL_1_BASE_DIVM_SHIFT 16
#define PLL_1_BASE_DIVM_WIDTH 11


#define pll_readl_ctrl(p) readl(p->clk_base)
#define pll_readl_div(p) readl(p->clk_base+0x4)
#define pll_writel_ctrl(val, p) writel(val, p->clk_base)
#define pll_writel_div(val, p) writel(val, p->clk_base+0x4)
#define mask(w) ((1 << (w)) - 1)
#define divm_mask(p) mask(p->divm_width)
#define divn_mask(p) mask(p->divn_width)
#define divp_mask(p) mask(p->divp_width)

static struct pesaro_clk_pll *_pesaro_init_pll(void __iomem *clk_base,
		unsigned long fixed_rate,
		struct pesaro_clk_pll_params *pll_params, u32 pll_flags,
		struct pesaro_clk_pll_freq_table *freq_table, spinlock_t *lock)
{
	struct pesaro_clk_pll *pll;

	pll = kzalloc(sizeof(*pll), GFP_KERNEL);
	if (!pll)
		return ERR_PTR(-ENOMEM);

	pll->clk_base = clk_base;

	pll->freq_table = freq_table;
	pll->params = pll_params;
	pll->fixed_rate = fixed_rate;
	pll->flags = pll_flags;
	pll->lock = lock;

	if (clk_base == PLLC_1) {
		pll->divp_shift = PLL_1_BASE_DIVP_SHIFT;
		pll->divp_width = PLL_1_BASE_DIVP_WIDTH;
		pll->divn_shift = PLL_1_BASE_DIVN_SHIFT;
		pll->divn_width = PLL_1_BASE_DIVN_WIDTH;
		pll->divm_shift = PLL_1_BASE_DIVM_SHIFT;
		pll->divm_width = PLL_1_BASE_DIVM_WIDTH;
	} else {
		pll->divp_shift = PLL_BASE_DIVP_SHIFT;
		pll->divp_width = PLL_BASE_DIVP_WIDTH;
		pll->divn_shift = PLL_BASE_DIVN_SHIFT;
		pll->divn_width = PLL_BASE_DIVN_WIDTH;
		pll->divm_shift = PLL_BASE_DIVM_SHIFT;
		pll->divm_width = PLL_BASE_DIVM_WIDTH;
	}

	return pll;
}

static void _get_pll_mnp(struct pesaro_clk_pll *pll,
			 struct pesaro_clk_pll_freq_table *cfg)
{
	u32 val;

	val = pll_readl_div(pll);

	cfg->m = (val >> pll->divm_shift) & (divm_mask(pll));
	cfg->n = (val >> pll->divn_shift) & (divn_mask(pll));
	cfg->p = (val >> pll->divp_shift) & (divp_mask(pll));
	pr_debug("%s, pll:0x%x, m:%d, n:%d, p:%d\n",__func__, val, cfg->m,
								cfg->n, cfg->p);
}

static int clk_pll_enable(struct clk_hw *hw)
{
	/*  for current hw design, we enable all pll at loader, and
	 *  we dont turn it off
	 */
	pr_debug("%s\n", __func__);
	return 0;
}

static void clk_pll_disable(struct clk_hw *hw)
{
	/*  for current hw design, we enable all pll at loader, and
	 *  we dont turn it off
	 */
	pr_debug("%s\n", __func__);
	return;
}


static int _program_pll(struct clk_hw *hw, struct pesaro_clk_pll_freq_table *cfg,
		unsigned long rate)
{
        struct pesaro_clk_pll *pll = to_clk_pll(hw);
	u32 val = 0;
	val |= ((cfg->m << pll->divm_shift) |
		(cfg->n << pll->divn_shift) |
		(cfg->p << pll->divp_shift));

	pr_debug("%s, pll:0x%x, m:%d, n:%d, p:%d\n",__func__, val, cfg->m,
								cfg->n, cfg->p);

	pll_writel_div(val, pll);
	val = pll_readl_ctrl(pll);
	val |= 0x1;
	pll_writel_ctrl(val, pll);
	while (pll_readl_ctrl(pll) != 0x30){
		// do nothing
	};
	return 0;
}

static int _get_table_rate(struct clk_hw *hw,
		struct pesaro_clk_pll_freq_table *cfg,
		unsigned long rate, unsigned long parent_rate)
{
	struct pesaro_clk_pll *pll = to_clk_pll(hw);
	struct pesaro_clk_pll_freq_table *sel;

	for (sel = pll->freq_table; sel->input_rate != 0; sel++)
		if (sel->input_rate == parent_rate &&
				sel->output_rate == rate)
			break;

	if (sel->input_rate == 0)
		return -EINVAL;

	cfg->input_rate = sel->input_rate;
	cfg->output_rate = sel->output_rate;
	cfg->m = sel->m;
	cfg->n = sel->n;
	cfg->p = sel->p;

	return 0;
}

static long clk_pll_round_rate(struct clk_hw *hw, unsigned long rate,
			unsigned long *prate)
{
	struct pesaro_clk_pll_freq_table cfg;
	int ret;
	ret = _get_table_rate(hw, &cfg, rate, *prate);
	if (ret) {
		/*  TODO: do calculate round rate, but we
		 *        use predefine rate table first
		 */
		pr_warning("clk:%s, NO target rate in freq table\n",
				hw->clk->name);
		return 0;
	}

	pr_debug("%s, in:%lu out:%lu, m:%u, n:%u, p:%u\n",
				__func__, cfg.input_rate, cfg.output_rate,
				cfg.m, cfg.n, cfg.p);
	return cfg.output_rate;
}

static unsigned long clk_pll_recalc_rate(struct clk_hw *hw,
					 unsigned long parent_rate)
{
        struct pesaro_clk_pll *pll = to_clk_pll(hw);
	struct pesaro_clk_pll_freq_table cfg;
	unsigned long rate;

	/* init stage, we dont care, we'll later reparent clk's parent */
	if (parent_rate == 0)
		return 0;
	_get_pll_mnp(pll, &cfg);
	rate = parent_rate / (cfg.n+1) / (cfg.p+1) * (cfg.m+1);
	pr_debug("%s, parent_rate:%lu, rate:%lu\n", __func__, parent_rate, rate);
	return rate;
}

static int clk_pll_set_rate(struct clk_hw *hw, unsigned long rate,
		unsigned long parent_rate)
{
	struct pesaro_clk_pll *pll = to_clk_pll(hw);
	struct pesaro_clk_pll_freq_table cfg, old_cfg;
	unsigned long flags = 0;
	int ret = 0;
	cfg.m = cfg.n = cfg.p = 0;
	pr_debug("%s, rate:%ld, parent_rate:%ld\n", __func__, rate, parent_rate);

	if (_get_table_rate(hw, &cfg, rate, parent_rate))
		return -EINVAL;

	if (pll->lock)
		spin_lock_irqsave(pll->lock, flags);

	_get_pll_mnp(pll, &old_cfg);
	if (old_cfg.m != cfg.m || old_cfg.n != cfg.n || old_cfg.p != cfg.p)
		ret = _program_pll(hw, &cfg, rate);

	if (pll->lock)
		spin_unlock_irqrestore(pll->lock, flags);

	return ret;
}

const struct clk_ops pesaro_clk_pll_ops = {
//	.is_enabled = clk_pll_is_enabled,
	.enable = clk_pll_enable,
	.disable = clk_pll_disable,
	.recalc_rate = clk_pll_recalc_rate,
	.round_rate = clk_pll_round_rate,
	.set_rate = clk_pll_set_rate,
};

static struct clk *_pesaro_clk_register_pll(struct pesaro_clk_pll *pll,
		const char *name, const char *parent_name, unsigned long flags,
		const struct clk_ops *ops)
{
	struct clk_init_data init;

	init.name = name;
	init.ops = ops;
	init.flags = flags;
	init.parent_names = (parent_name ? &parent_name : NULL);
	init.num_parents = (parent_name ? 1 : 0);

	/* Data in .init is copied by clk_register(), so stack variable OK */
	pll->hw.init = &init;

	return clk_register(NULL, &pll->hw);
}

struct clk *pesaro_clk_register_pll(const char *name, const char *parent_name,
		void __iomem *clk_base,
		unsigned long flags, unsigned long fixed_rate,
		struct pesaro_clk_pll_params *pll_params, u32 pll_flags,
		struct pesaro_clk_pll_freq_table *freq_table, spinlock_t *lock)
{
	struct pesaro_clk_pll *pll;
	struct clk* clk;
	pll = _pesaro_init_pll(clk_base, fixed_rate, pll_params, pll_flags,
			       freq_table, lock);
	clk = _pesaro_clk_register_pll(pll, name, parent_name, flags,
				      &pesaro_clk_pll_ops);
	return clk;
}
