/*
 * drivers/clk/vatics/clk.c.c
 *
 * Copyright (C) 2013-2018  VATICS Inc.
 *
 * Author: ChangHsien Ho <vincent.ho@vatics.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/kernel.h>
#include <linux/clk.h>
#include <linux/clk-provider.h>
#include <linux/clkdev.h>
#include <linux/clk-provider.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <mach/platform.h>
#include "clk-gate.h"
#include "clk.h"

enum pesaro_clk {
osc_12m = 0, osc_20m, apb_pclk,
vpl_acdcc = 10, vpl_agpoc, vma_cce, vma_dce, vpl_dmac , vpl_gmac,
gmac_rmii = 16, vpl_gpadc, vpl_gpioc_0, vpl_gpioc_1, vpl_gpioc_2,
vma_h4de = 21, vma_h4ee, vpl_hdmitc_0, vpl_i2cc_0, vpl_i2cc_1,
vpl_i2cc_2 = 26, vpl_i2ssc, vpl_irdac, vma_ispe, vma_jebe,
vma_meae = 31, arsn_mipirc, vpl_mshc_0,

vpl_mshc_1 = 40, vpl_nfc, vpl_pllc, vpl_rtcc, vpl_ssic, vpl_tvec,
vpl_uartc_0 = 46, vpl_uartc_1, vpl_uartc_2, vpl_uartc_3, vpl_usbc,
vpl_vdac = 51, vpl_vic_dev_0, vpl_vic_dev_1, vpl_vic_mux, vpl_voc,
vpl_wdtc = 56,

vpl_mshc_0_div = 57, vpl_mshc_1_div,

pll_0 = 60, pll_1, pll_2, pll_3, pll_4, pll_5,
vpl_uartgate_0 = 67, vpl_uartgate_1, vpl_uartgate_2, vpl_uartgate_3,
vpl_mshc0_1200m, vpl_mshc1_1200m,
clk_max
};

struct clk *clks[clk_max];
struct clk_onecell_data clk_data;

DEFINE_SPINLOCK(gating_lock_0);
DEFINE_SPINLOCK(gating_lock_1);
DEFINE_SPINLOCK(pll_lock);
DEFINE_SPINLOCK(div_lock);
DEFINE_SPINLOCK(uart_lock);

struct sysc_clk_gate_data {
	const char *name;
	const char *parent_name;
	u8 bit_idx;
	unsigned long flags;
};
/* F(pllout) = F(ref) x (M / N) x 1/P */

/* pll_1 */
struct pesaro_clk_pll_freq_table freq_tb_pll1[] = {
	{12000000,1200000000, 299, 2,  0},   // /3 x 300 / 1 , Default
	{12000000,1600000000, 399, 2,  0},   // /3 x 400 / 1 , Overclock
};

/* pll_2 */
struct pesaro_clk_pll_freq_table freq_tb_ethernet[] = {
	{12000000,500000000, 249, 2,  1},
};

/* pll_3 */
struct pesaro_clk_pll_freq_table freq_tb_vic[] = {
	{12000000, 20000000, 239, 2, 23},
	{12000000, 24000000,  95, 0, 23},
	{12000000, 27000000, 197, 1, 21},
	{12000000, 37125000, 197, 1, 15},
	{12000000,150000000, 299, 2,  3},
	{12000000,200000000, 399, 2,  3},
};

/* pll_4 */
struct pesaro_clk_pll_freq_table freq_tb_voc[] = {
	{12000000, 13500000, 197, 1, 21},
	{12000000, 27000000, 179, 1,  9},
	{12000000, 74250000, 197, 1,  3},
	{12000000,108000000, 143, 1,  1},
	{12000000,148500000, 197, 1,  1},
	{12000000,150000000, 299, 2,  1},
	{12000000,200000000, 399, 2,  1},
};
/* pll_5 */
struct pesaro_clk_pll_freq_table freq_tb_audio[] = {
	{12000000,147456000,12287,124,  3},
	{12000000, 49152000, 2047, 24, 19},
	{12000000, 32768000,16383,124,  7},
	{12000000, 24576000, 2047, 49, 19},
	{12000000, 22579200, 9407,124,  3},
};


void __init pesaro_pll_clk_init(void)
{
	clks[osc_12m] = clk_register_fixed_rate(NULL, "osc_12m", NULL, CLK_IS_ROOT, 12000000);
        clk_register_clkdev(clks[osc_12m], "osc_12m", NULL);

	clks[vpl_mshc0_1200m] = clk_register_fixed_rate(NULL, "vpl_mshc0_1200m", "vpl_mshc_0", 0, 1200000000);
        clk_register_clkdev(clks[vpl_mshc0_1200m], "vpl_mshc0_1200m", NULL);

	clks[vpl_mshc1_1200m] = clk_register_fixed_rate(NULL, "vpl_mshc1_1200m", "vpl_mshc_1", 0, 1200000000);
        clk_register_clkdev(clks[vpl_mshc1_1200m], "vpl_mshc1_1200m", NULL);

	clks[pll_1] = pesaro_clk_register_pll("pll_1", "osc_12m", PLLC_1,
			                CLK_IGNORE_UNUSED, 0,
				        NULL, 0,
					freq_tb_pll1, &pll_lock);
        clk_register_clkdev(clks[pll_1], "pll_1", NULL);

	clks[pll_2] = pesaro_clk_register_pll("pll_2", "osc_12m", PLLC_2,
			                CLK_IGNORE_UNUSED, 0,
				        NULL, 0,
					freq_tb_ethernet, &pll_lock);
        clk_register_clkdev(clks[pll_2], "pll_2", NULL);

	clks[pll_3] = pesaro_clk_register_pll("pll_3", "osc_12m", PLLC_3,
				      CLK_IGNORE_UNUSED, 0,
				      NULL, 0,
			              freq_tb_vic, &pll_lock);

        clk_register_clkdev(clks[pll_3], "pll_3", NULL);

	clks[pll_4] = pesaro_clk_register_pll("pll_4", "osc_12m", PLLC_4,
				      CLK_IGNORE_UNUSED, 0,
				      NULL, 0,
			              freq_tb_voc, &pll_lock);
        clk_register_clkdev(clks[pll_4], "pll_4", NULL);

	clks[pll_5] = pesaro_clk_register_pll("pll_5", "osc_12m", PLLC_5,
				      CLK_IGNORE_UNUSED, 0,
				      NULL, 0,
			              freq_tb_audio, &pll_lock);
        clk_register_clkdev(clks[pll_5], "pll_5", NULL);
}

static const struct sysc_clk_gate_data clk_gates_0[] __initconst = {
	{ "vpl_mshc_0",		NULL,	MSHC_0_CLK_EN,		CLK_IGNORE_UNUSED},
	{ "arsn_mipirc",  "apb_pclk",	MIPIRC_CLK_EN,		CLK_SET_RATE_PARENT |CLK_IGNORE_UNUSED},
	{ "vma_meae",		NULL,	MEAE_CLK_EN,		0},
	{ "vma_jebe",		NULL,	JEBE_CLK_EN,		0},
	{ "vma_ispe",		NULL,	ISPE_CLK_EN,		0},
	{ "vpl_irdac",	  "apb_pclk",	IRDAC_CLK_EN,		CLK_SET_RATE_PARENT},
	{ "vpl_i2ssc",	  "apb_pclk",	I2SSC_CLK_EN,		CLK_SET_RATE_PARENT},
	{ "vpl_i2cc_2",   "apb_pclk",	I2CC_2_CLK_EN,		CLK_SET_RATE_PARENT},
	{ "vpl_i2cc_1",	  "apb_pclk",	I2CC_1_CLK_EN,		CLK_SET_RATE_PARENT},
	{ "vpl_i2cc_0",	  "apb_pclk",	I2CC_0_CLK_EN,		CLK_SET_RATE_PARENT},
	{ "vpl_hdmitc_0", "apb_pclk",	HDMITC_CLK_EN,		CLK_SET_RATE_PARENT},
	{ "vma_h4ee",		NULL,	H4EE_CLK_EN,		0},
	{ "vma_h4de",	  "vma_h4ee",	H4DE_CLK_EN,		0},
	{ "vpl_gpioc_2",  "apb_pclk",	GPIOC_2_CLK_EN,		0},
	{ "vpl_gpioc_1",  "apb_pclk",	GPIOC_1_CLK_EN,		0},
	{ "vpl_gpioc_0",  "apb_pclk",	GPIOC_0_CLK_EN,		0},
	{ "vpl_gpadc",	  "apb_pclk",	GPADC_CLK_EN,		CLK_SET_RATE_PARENT|CLK_IGNORE_UNUSED},
	{ "gmac_rmii",		NULL,	GMAC_RMII_CLK_EN,	CLK_IGNORE_UNUSED},
	{ "vpl_gmac",	  "ahb_sclk",	GMAC_CLK_EN,		CLK_IGNORE_UNUSED},
	{ "vpl_dmac",		NULL,	DMAC_CLK_EN,		0},
	{ "vma_dce",		NULL,	DCE_CLK_EN,		0},
	{ "vma_cce",		NULL,	CCE_CLK_EN,		0},
	{ "vpl_agpoc",	  "apb_pclk",	AGPOC_CLK_EN,		CLK_SET_RATE_PARENT|CLK_IGNORE_UNUSED},
	{ "vpl_acdcc",	  "apb_pclk",	ACDCC_CLK_EN,		CLK_SET_RATE_PARENT|CLK_IGNORE_UNUSED},
};

static const struct sysc_clk_gate_data clk_gates_1[] __initconst = {
	{ "vpl_wdtc",	  "apb_pclk",	WDTC_CLK_EN,		CLK_IGNORE_UNUSED},
	{ "vpl_voc",		NULL,	VOC_CLK_EN,			CLK_IGNORE_UNUSED},
	{ "vpl_vic_mux",	NULL,	VIC_MUX_CLK_EN,		0},
	{ "vpl_vic_dev_1",	NULL,	VIC_DEV_1_CLK_EN,	0},
	{ "vpl_vic_dev_0",	NULL,	VIC_DEV_0_CLK_EN,	0},
	{ "vpl_vdac",		NULL,	VDAC_CLK_EN,		CLK_IGNORE_UNUSED},
	{ "vpl_usbc",	  "apb_pclk",	USBC_CLK_EN,		CLK_SET_RATE_PARENT|CLK_IGNORE_UNUSED},
	{ "vpl_uartc_3",   "vpl_uartgate_3",	UARTC_3_CLK_EN,		CLK_SET_RATE_PARENT|CLK_IGNORE_UNUSED},
	{ "vpl_uartc_2",   "vpl_uartgate_2",	UARTC_2_CLK_EN,		CLK_SET_RATE_PARENT|CLK_IGNORE_UNUSED},
	{ "vpl_uartc_1",   "vpl_uartgate_1",	UARTC_1_CLK_EN,		CLK_SET_RATE_PARENT|CLK_IGNORE_UNUSED},
	{ "vpl_uartc_0",   "vpl_uartgate_0",	UARTC_0_CLK_EN,		CLK_SET_RATE_PARENT|CLK_IGNORE_UNUSED},
	{ "vpl_tvec",		NULL,	TVEC_CLK_EN,		0},
	{ "vpl_ssic",	  "apb_pclk",	SSIC_CLK_EN,		CLK_SET_RATE_PARENT|CLK_IGNORE_UNUSED},
	{ "vpl_rtcc",		NULL,	RTCC_CLK_EN,		CLK_IGNORE_UNUSED},
	{ "vpl_pllc",	  "apb_pclk",	PLLC_CLK_EN,		CLK_SET_RATE_PARENT|CLK_IGNORE_UNUSED},
	{ "vpl_nfc",	  "ahb_sclk",	NFC_CLK_EN,		CLK_SET_RATE_PARENT|CLK_IGNORE_UNUSED},
	{ "vpl_mshc_1",		NULL,	MSHC_1_CLK_EN,		CLK_IGNORE_UNUSED},
};

static const struct sysc_clk_gate_data uart_clk_gates[] __initconst = {
	{ "vpl_uartgate_0",	  "osc_20m",	UART_0_CLK_SRC_SEL,		0},
	{ "vpl_uartgate_1",	  "osc_20m",	UART_1_CLK_SRC_SEL,		0},
	{ "vpl_uartgate_2",	  "osc_20m",	UART_2_CLK_SRC_SEL,		0},
	{ "vpl_uartgate_3",	  "osc_20m",	UART_3_CLK_SRC_SEL,		0},	
};
void __init pesaro_gating_clk_init(void)
{
	int i, j;

	for (i = vpl_uartgate_3, j= ARRAY_SIZE(uart_clk_gates)-1; j >= 0; i++, j--) {
		clks[i] = clk_register_gate(NULL, uart_clk_gates[j].name, uart_clk_gates[j].parent_name,
					    uart_clk_gates[j].flags, SYSC_UARTC_CTRL,
					    uart_clk_gates[j].bit_idx, 0,
					    &uart_lock);
		clk_register_clkdev(clks[i], uart_clk_gates[j].name, NULL);
	}

	for (i = vpl_acdcc, j= ARRAY_SIZE(clk_gates_0)-1; j >= 0; i++, j--) {
		clks[i] = clk_register_gate(NULL, clk_gates_0[j].name, clk_gates_0[j].parent_name,
					    clk_gates_0[j].flags, SYSC_CLK_EN_CTRL_0,
					    clk_gates_0[j].bit_idx, 0,
					    &gating_lock_0);
		clk_register_clkdev(clks[i], clk_gates_0[j].name, NULL);
	}

	for (i = vpl_mshc_1, j= ARRAY_SIZE(clk_gates_1)-1; j >= 0; i++, j--) {
		clks[i] = clk_register_gate(NULL, clk_gates_1[j].name, clk_gates_1[j].parent_name,
					    clk_gates_1[j].flags, SYSC_CLK_EN_CTRL_1,
					    clk_gates_1[j].bit_idx, 0,
					    &gating_lock_1);
		clk_register_clkdev(clks[i], clk_gates_1[j].name, NULL);
	}

	for (i = 0; i < ARRAY_SIZE(clks); i++) {
		if (!clks[i])
			clks[i] = ERR_PTR(-EINVAL);
	}
}

struct pesaro_div_clock{
	unsigned int		id;
	const char		*dev_name;
	const char		*name;
	const char		*parent_name;
	unsigned long		flags;
	unsigned long		offset;
	u8			shift;
	u8			width;
	u8			div_flags;
	const char		*alias;
	struct clk_div_table	*table;
};

static struct clk_div_table mshc_div_table[] = {
	{ .val = 0, .div = 12 },  // 100 MHZ
	{ .val = 1, .div = 24 },  // 50 MHZ
	{ .val = 2, .div = 36 },  // 
	{ .val = 3, .div = 48 },  // 
	{ .val = 4, .div = 60 },  // 
	{ .val = 5, .div = 72 },  // 
	{ .val = 6, .div = 84 },  // 
	{ .val = 7, .div = 96 },  // 
};

static const struct pesaro_div_clock pesaro_div_clks[] = {
	{ vpl_mshc_0_div, NULL, "mshc_0_div", "vpl_mshc0_1200m", 0, (unsigned long)SYSC_MSHC_0_CTRL, 0, 3, 0, NULL,mshc_div_table},
	{ vpl_mshc_1_div, NULL, "mshc_1_div", "vpl_mshc1_1200m", 0, (unsigned long)SYSC_MSHC_1_CTRL, 0, 3, 0, NULL,mshc_div_table},
};

/* register a list of div clocks */
void __init pesaro_clk_register_div(void)
{
	int i;
	struct clk *clk;

	for (i = 0; i < ARRAY_SIZE(pesaro_div_clks); i++) {
		clk = clk_register_divider_table(NULL, pesaro_div_clks[i].name,
				 pesaro_div_clks[i].parent_name,
				 pesaro_div_clks[i].flags,
				 (void *)pesaro_div_clks[i].offset,
				 pesaro_div_clks[i].shift,
				 pesaro_div_clks[i].width,
				 pesaro_div_clks[i].div_flags,
				 pesaro_div_clks[i].table,
				 &div_lock);
		if (IS_ERR(clk)) {
			pr_err("%s: failed to register clock %s\n", __func__,
				pesaro_div_clks[i].name);
			continue;
		}
		clk_register_clkdev(clk, pesaro_div_clks[i].name, NULL);

		clks[pesaro_div_clks[i].id] = clk;
	}
}

/**
 * struct clk_init_tabel - clock initialization table
 * @clk_id:	clock id as mentioned in device tree bindings
 * @parent_id:	parent clock id as mentioned in device tree bindings
 * @rate:	rate to set
 * @state:	enable/disable
 */
struct pesaro_clk_init_table {
	unsigned int	clk_id;
	unsigned int	parent_id;
	unsigned long	rate;
	int		state;
};


static __initdata struct pesaro_clk_init_table init_table[] = {
	{vpl_mshc_0_div, vpl_mshc0_1200m, 100000000, 0},
	{vpl_mshc_1_div, vpl_mshc1_1200m, 100000000, 0},
	{clk_max, clk_max, 0, 0}, /* This MUST be the last entry */
};
void __init pesaro_init_from_table(struct pesaro_clk_init_table *tbl,
				  struct clk *clks[], int clk_max)
{
	struct clk *clk;

	for (; tbl->clk_id < clk_max; tbl++) {
		clk = clks[tbl->clk_id];
		if (IS_ERR_OR_NULL(clk))
			return;
#if 0
		if (tbl->parent_id < clk_max) {
			struct clk *parent = clks[tbl->parent_id];
			if (clk_set_parent(clk, parent)) {
				pr_err("%s: Failed to set parent %s of %s\n",
				       __func__, __clk_get_name(parent),
				       __clk_get_name(clk));
				WARN_ON(1);
			}
		}
#endif

		if (tbl->rate)
			if (clk_set_rate(clk, tbl->rate)) {
				pr_err("%s: Failed to set rate %lu of %s\n",
				       __func__, tbl->rate,
				       __clk_get_name(clk));
				WARN_ON(1);
			}

		if (tbl->state)
			if (clk_prepare_enable(clk)) {
				pr_err("%s: Failed to enable %s\n", __func__,
				       __clk_get_name(clk));
				WARN_ON(1);
			}
	}
}
static void __init pesaro_clk_apply_init_table(void)
{
	pesaro_init_from_table(init_table, clks, clk_max);
}

static void __init pesaro_clk_init(struct device_node *np)
{
	pesaro_gating_clk_init();
	pesaro_pll_clk_init();
	pesaro_clk_register_div();
	pesaro_clk_apply_init_table();

	clk_data.clks = clks;
	clk_data.clk_num = ARRAY_SIZE(clks);
	of_clk_add_provider(np, of_clk_src_onecell_get, &clk_data);
}

CLK_OF_DECLARE(pesaro_clk, "vatics,pesaro-clk", pesaro_clk_init);
