/*
 * VATICS SoCs system timer handling
 *  drivers/clocksource/vpl_sys_timer.c
 *
 * Copyright (C) 2013-2018  VATICS Inc.
 *
 * Author: ChangHsien Ho <vincent.ho@vatics.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/kernel.h>
#include <linux/clocksource.h>
#include <linux/clockchips.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/clk.h>
#include <linux/of.h>
#include <linux/of_irq.h>
#include <linux/delay.h>
#include <asm/mach/time.h>
#include <mach/maps.h>
#include <mach/irqs.h>
#include <mach/vpl_sysc.h>
#include <mach/vpl_tmrc.h>
#include <linux/irqchip/pesaro.h>

static void __iomem *sysc_base;
static void __iomem *tmrc_base;
#ifdef CONFIG_PESARO_N9_RTOS
static void __iomem *intc_base;
#endif
static unsigned long hz_period;

/*
 * Clocksource (system control 64-bit free running counter)
 */
/*
 * To get the value from the Global Timer Counter register proceed as follows:
 * 1. Read the upper 32-bit timer counter register
 * 2. Read the lower 32-bit timer counter register
 * 3. Read the upper 32-bit timer counter register again. If the value is
 *  different to the 32-bit upper value read previously, go back to step 2.
 *  Otherwise the 64-bit timer counter value is correct.
 */
static cycle_t sysc_cnt_read(struct clocksource *cs)
{
 	u64 counter;
	u32 lower;
	u32 upper, old_upper;

	upper = readl_relaxed(sysc_base + SYSC_CNT_HIGH);
	do {
		old_upper = upper;
		lower = readl_relaxed(sysc_base + SYSC_CNT_LOW);
		upper = readl_relaxed(sysc_base + SYSC_CNT_HIGH);
	} while (upper != old_upper);

	counter = upper;
	counter <<= 32;
	counter |= lower;

	return counter;
}

static void sysc_cnt_reset(int divider)
{
	u32 ctl;

	ctl = SYSC_CNT_CTRL_CNT_CLEAR | (divider & SYSC_CNT_CTRL_CNT_DIV_MASK);
	writel(ctl, sysc_base + SYSC_CNT_CTRL);
	/* For newer version, reset doesn't self clean */
	ctl &= ~SYSC_CNT_CTRL_CNT_CLEAR;
	writel(ctl, sysc_base + SYSC_CNT_CTRL);
}

static struct clocksource sysc_cnt_clocksource = {
	.name	= "sysc_freerun_counter",
	.rating	= 300,
	.read	= sysc_cnt_read,
	.mask	= CLOCKSOURCE_MASK(64),
	.flags	= CLOCK_SOURCE_IS_CONTINUOUS,
};

/*
 * Clockevents (tmrc0 interrupting timer)
 */
static int tmrc_set_next_event(unsigned long next, struct clock_event_device *evt)
{
	u32 ctrl = readl(tmrc_base + TMRC_CTRL);

	/* Disable Timer */
	ctrl &= ~TMRC_CTRL_TM3_EN;
	writel(ctrl, tmrc_base + TMRC_CTRL);
	/* Write new count */
	writel(next, tmrc_base + TMRC_TIMER3_COUNTER);
	writel(next, tmrc_base + TMRC_TIMER3_AUTO_RELOAD_VALUE);
	/* Enable Timer and Interrupt */
	ctrl |= TMRC_CTRL_TM3_EN | TMRC_CTRL_TM3_OF_EN;
	writel(ctrl, tmrc_base + TMRC_CTRL);

	return 0;
}

static void tmrc_set_mode(enum clock_event_mode mode, struct clock_event_device *evt)
{
	u32 ctrl = readl(tmrc_base + TMRC_CTRL);

	switch (mode) {
	case CLOCK_EVT_MODE_PERIODIC:
		ctrl &= ~TMRC_CTRL_TM3_EN;
		writel(ctrl, tmrc_base + TMRC_CTRL);
		writel(hz_period, tmrc_base + TMRC_TIMER3_COUNTER);
		writel(hz_period, tmrc_base + TMRC_TIMER3_AUTO_RELOAD_VALUE);
	        ctrl |= TMRC_CTRL_TM3_EN | TMRC_CTRL_TM3_OF_EN;
		break;
	case CLOCK_EVT_MODE_RESUME:
		ctrl |= TMRC_CTRL_TM3_EN | TMRC_CTRL_TM3_OF_EN;
		break;
	case CLOCK_EVT_MODE_ONESHOT:
		/* period set, and timer enabled in 'next_event' hook */
		writel(0, tmrc_base + TMRC_TIMER3_AUTO_RELOAD_VALUE);
	case CLOCK_EVT_MODE_SHUTDOWN:
	case CLOCK_EVT_MODE_UNUSED:
	default:
		ctrl &= ~(TMRC_CTRL_TM3_EN | TMRC_CTRL_TM3_OF_EN);
		break;
	}
	writel(ctrl, tmrc_base + TMRC_CTRL);
}

static struct clock_event_device tmrc_clockevent = {
	.name	= "vpl_sys_timer_tick",
	.features = CLOCK_EVT_FEAT_PERIODIC | CLOCK_EVT_FEAT_ONESHOT,
	.rating = 300,
	.set_next_event = tmrc_set_next_event,
	.set_mode = tmrc_set_mode,
};
#ifdef CONFIG_PESARO_N9_RTOS
extern int n9_start(void);
#endif
static irqreturn_t vpl_sys_timer_isr(int irq, void *dev_id)
{
	struct clock_event_device *evt = dev_id;
	u32 stat;

	stat = readl(tmrc_base + TMRC_STATUS); 
	if (!(stat & TMRC_STATUS_TM3_INTR))
		return IRQ_NONE;
	writel(readl(tmrc_base + TMRC_STATUS) &~ TMRC_STATUS_TM3_INTR, tmrc_base + TMRC_STATUS);
#if  defined(CONFIG_PESARO_N9_RTOS) && !defined(CONFIG_PESARO_N9_TICKLESS)
	if (likely(n9_start())) {
		writel(readl(intc_base + INTC_SET_HI)
				| 0x1<< (TMRC_TM0_IRQ_NUM - 32),
				intc_base + INTC_SET_HI);
	}
#endif
	evt->event_handler(evt);

	return IRQ_HANDLED;
}

static struct irqaction vpl_sys_timer_irq = {
	.name       = "VPL System Timer Tick",
	.handler    = vpl_sys_timer_isr,
	.flags      = IRQF_DISABLED | IRQF_TIMER | IRQF_TRIGGER_RISING | IRQF_SHARED,
	.dev_id     = &tmrc_clockevent,
};

static inline void tmrc_reset(void)
{
	/* decrementing, no interrupt */
	writel(TMRC_CTRL_TM3_CNT_DIR, tmrc_base + TMRC_CTRL);
	writel(0, tmrc_base + TMRC_TIMER3_COUNTER);
	writel(0, tmrc_base + TMRC_TIMER3_AUTO_RELOAD_VALUE);
	writel(0, tmrc_base + TMRC_TIMER3_MATCH_VALUE);
}

static void __init vpl_sys_timer_init(struct device_node *np)
{
	unsigned long sys_cnt_clk_rate, apb_clk_rate;
	struct clk *clk_sys_cnt, *clk_apb;
	u32 divider;
	int virq;

	sysc_base = ioremap(VPL_SYSC_MMR_BASE, SZ_4K);
	if (IS_ERR(sysc_base))
		panic("%s: invald sysc base address\n", np->name);

	tmrc_base = ioremap(VPL_TMRC_MMR_BASE, SZ_4K);
	if (IS_ERR(tmrc_base))
		panic("%s: invald tmrc base address\n", np->name);
	
#ifdef CONFIG_PESARO_N9_RTOS
	intc_base = ioremap(VPL_INTC_MMR_BASE, SZ_4K);
	if (IS_ERR(intc_base))
		panic("%s: invald intc base address\n", np->name);
#endif

	/* sysc counter base clock */
	clk_sys_cnt = of_clk_get(np, 0);
	if (IS_ERR(clk_sys_cnt))
		panic("%s: unable to get sysc counter clock\n", np->name);

	sys_cnt_clk_rate = clk_get_rate(clk_sys_cnt);
	pr_info("SYSC counter at clock %ld\n", sys_cnt_clk_rate);

	/* tmrc base clock */
	clk_apb = of_clk_get(np, 1);
	if (IS_ERR(clk_apb))
		panic("%s: unable to get sysc counter clock\n", np->name);

	apb_clk_rate = clk_get_rate(clk_apb);
	pr_info("TMRC at APB bus clock %ld\n", apb_clk_rate);

	/*
	 * On FPGA 12MHz timer domain updated by APB 10MHz domain, will casue to problem.
	 * Divide to 1MHz as default
	 */
	divider = sys_cnt_clk_rate / 1000000;

	/*
	 * Set up clocksource
	 * Do not change divider or precision muliplier.
	 * The frequency is the same as APB,
	 * when 100MHz, 1 tick is 10ns, maximum precision.
	 */
	sysc_cnt_reset(divider-1);
	if (clocksource_register_hz(&sysc_cnt_clocksource, 1000000))
		pr_warn( "register clocksource %s fail\n", sysc_cnt_clocksource.name);

	/*
	 * Set up clockevent
	 */
	virq = irq_of_parse_and_map(np, 0);
	if (virq <= 0)
		panic("Can't parse IRQ");

	printk("%s virq %d\n", __func__, virq);

	hz_period = DIV_ROUND_CLOSEST(apb_clk_rate, HZ);
	tmrc_reset();
	setup_irq(virq, &vpl_sys_timer_irq);
	clockevents_config_and_register(&tmrc_clockevent, apb_clk_rate, 500, 0xFFFFFFFF);
}

CLOCKSOURCE_OF_DECLARE(vpl_sys_timer, "vatics,vpl-sys-timer", vpl_sys_timer_init);
