/*
 * drivers/gpio/gpio-vpl.c
 *
 * Copyright (C) 2013-2018  VATICS Inc.
 *
 * Author: ChangHsien Ho <vincent.ho@vatics.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/clk.h>
#include <linux/gpio.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/irqdomain.h>
#include <linux/irqchip/chained_irq.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/spinlock.h>
#include <mach/platform.h>
#include <mach/vpl_gpioc.h>

struct vpl_gpio_bank {
	struct gpio_chip 	chip;
	void __iomem		*base;
	spinlock_t		lock;
	struct device		*dev;
	struct clk 		*clk;
	int			clk_rate;
	int 			virq;
	struct irq_domain	*irqdomain;
};

static int vpl_gpio_request(struct gpio_chip *chip, unsigned offset)
{
	return pinctrl_request_gpio(chip->base + offset);
}

static void vpl_gpio_free(struct gpio_chip *chip, unsigned offset)
{
	pinctrl_free_gpio(chip->base + offset);
}

static int vpl_gpio_direction_input(struct gpio_chip *chip, unsigned offset)
{
	struct vpl_gpio_bank *bank = container_of(chip, struct vpl_gpio_bank, chip);
	void __iomem *base = bank->base;
	unsigned long flags;
	u32 val;

	spin_lock_irqsave(&bank->lock, flags);

	val = readl_relaxed(base + GPIOC_PIN_DIR);
	writel_relaxed(val & ~BIT(offset), base + GPIOC_PIN_DIR);

	spin_unlock_irqrestore(&bank->lock, flags);

	return 0;
}

static int vpl_gpio_get(struct gpio_chip *chip, unsigned offset)
{
	struct vpl_gpio_bank *bank = container_of(chip, struct vpl_gpio_bank, chip);
	void __iomem *base = bank->base;
	u32 data;

	if (readl_relaxed(base + GPIOC_PIN_DIR) & BIT(offset))
		data = readl_relaxed(base + GPIOC_DATA_OUT);
	else
		data = readl_relaxed(base + GPIOC_DATA_IN);

	return (data >> offset) & 1;
}

static int vpl_gpio_direction_output(struct gpio_chip *chip, unsigned offset, int value)
{
	struct vpl_gpio_bank *bank = container_of(chip, struct vpl_gpio_bank, chip);
	void __iomem *base = bank->base;
	unsigned long flags;
	u32 val;

	spin_lock_irqsave(&bank->lock, flags);

	val = readl_relaxed(base + GPIOC_PIN_DIR);
	if (value) {
		writel_relaxed(BIT(offset), base + GPIOC_DATA_SET);
	} else {
		writel_relaxed(BIT(offset), base + GPIOC_DATA_CLR);
	}
	writel_relaxed(val | BIT(offset), base + GPIOC_PIN_DIR);

	spin_unlock_irqrestore(&bank->lock, flags);

	return 0;
}

static void vpl_gpio_set(struct gpio_chip *chip, unsigned offset, int value)
{
	struct vpl_gpio_bank *bank = container_of(chip, struct vpl_gpio_bank, chip);
	void __iomem *base = bank->base;
	unsigned long flags;

	spin_lock_irqsave(&bank->lock, flags);

	if (value) {
		writel_relaxed(BIT(offset), base + GPIOC_DATA_SET);
	} else {
		writel_relaxed(BIT(offset), base + GPIOC_DATA_CLR);
	}

	spin_unlock_irqrestore(&bank->lock, flags);
}

static int vpl_gpio_set_debounce(struct gpio_chip *chip, unsigned offset, unsigned debounce)
{
	struct vpl_gpio_bank *bank = container_of(chip, struct vpl_gpio_bank, chip);
	void __iomem *base = bank->base;
	unsigned long flags;
	unsigned long ticks_per_usec;
	u32 val, max, ticks;

	/* debounce is microsecond */
	ticks_per_usec = (bank->clk_rate / USEC_PER_SEC);
	max = BIT(GPIOC_DEBOUNCE_PERIOD_BIT) / ticks_per_usec;
	if (debounce > max) {
		dev_err(bank->dev, "Debounce value %d not in range (max %d)\n", debounce, max);
		return -EINVAL;
	}

	ticks = debounce * ticks_per_usec;

	spin_lock_irqsave(&bank->lock, flags);

	writel_relaxed(ticks, base + GPIOC_DEBOUNCE_PERIOD);

	val = readl_relaxed(base + GPIOC_DEBOUNCE_ENABLE);
	if (debounce) {
		val |= BIT(offset);
	} else {
		val &= ~BIT(offset);
	}
	writel_relaxed(val, base + GPIOC_DEBOUNCE_ENABLE);

	spin_unlock_irqrestore(&bank->lock, flags);

	return 0;
}

static int vpl_gpio_to_irq(struct gpio_chip *chip, unsigned offset)
{
	struct vpl_gpio_bank *bank = container_of(chip, struct vpl_gpio_bank, chip);

	return irq_find_mapping(bank->irqdomain, offset);
}

/* interrupt handler */
static void vpl_gpio_irq_handler(unsigned int irq, struct irq_desc *desc)
{
	struct vpl_gpio_bank *bank = irq_get_handler_data(irq);
	void __iomem *base = bank->base;
	int pin;
	int unmasked = 0;
	struct irq_chip *chip = irq_desc_get_chip(desc);
	unsigned long status;
	u32 lvl;

	chained_irq_enter(chip, desc);

	/* ack any irqs */
	status = readl_relaxed(base + GPIOC_INTR_MASK_STATE);
	lvl = ~readl_relaxed(base + GPIOC_INTR_TRIGGER_TYPE);
	/* now demux them to the right lowlevel handler */
	for_each_set_bit(pin, &status, 32) {
		writel_relaxed(1 << pin, base + GPIOC_INTR_CLEAR);
		/*
		 * if gpio is edge triggered, clear condition
		 * before executing the hander so that we don't
		 * miss edges
		 */
		if (lvl & (1 << pin)) {
			unmasked = 1;
			chained_irq_exit(chip, desc);
		}

		generic_handle_irq(vpl_gpio_to_irq(&bank->chip, pin));
	}

	if (!unmasked)
		chained_irq_exit(chip, desc);
}

static void vpl_gpio_irq_enable(struct irq_data *d)
{
	struct vpl_gpio_bank *bank = irq_data_get_irq_chip_data(d);
	void __iomem *base = bank->base;
	u32 mask = (u32) irq_data_get_irq_handler_data(d);
	unsigned long flags;

	spin_lock_irqsave(&bank->lock, flags);
	writel_relaxed(readl_relaxed(base + GPIOC_INTR_ENABLE) | mask, base + GPIOC_INTR_ENABLE);
	spin_unlock_irqrestore(&bank->lock, flags);
}

static void vpl_gpio_irq_disable(struct irq_data *d)
{
	struct vpl_gpio_bank *bank = irq_data_get_irq_chip_data(d);
	void __iomem *base = bank->base;
	u32 mask = (u32) irq_data_get_irq_handler_data(d);
	unsigned long flags;

	spin_lock_irqsave(&bank->lock, flags);
	writel_relaxed(readl_relaxed(base + GPIOC_INTR_ENABLE) & ~mask, base + GPIOC_INTR_ENABLE);
	spin_unlock_irqrestore(&bank->lock, flags);
}

static void vpl_gpio_irq_ack(struct irq_data *d)
{
	struct vpl_gpio_bank *bank = irq_data_get_irq_chip_data(d);
	void __iomem *base = bank->base;
	u32 mask = (u32) irq_data_get_irq_handler_data(d);
	unsigned long flags;

	spin_lock_irqsave(&bank->lock, flags);
	writel_relaxed(mask, base + GPIOC_INTR_CLEAR);
	spin_unlock_irqrestore(&bank->lock, flags);
}

static void vpl_gpio_irq_mask(struct irq_data *d)
{
	struct vpl_gpio_bank *bank = irq_data_get_irq_chip_data(d);
	void __iomem *base = bank->base;
	u32 mask = (u32) irq_data_get_irq_handler_data(d);
	unsigned long flags;

	spin_lock_irqsave(&bank->lock, flags);
	writel_relaxed(readl_relaxed(base + GPIOC_INTR_MASK) | mask, base + GPIOC_INTR_MASK);
	spin_unlock_irqrestore(&bank->lock, flags);
}

static void vpl_gpio_irq_unmask(struct irq_data *d)
{
	struct vpl_gpio_bank *bank = irq_data_get_irq_chip_data(d);
	void __iomem *base = bank->base;
	u32 mask = (u32) irq_data_get_irq_handler_data(d);
	unsigned long flags;

	spin_lock_irqsave(&bank->lock, flags);
	writel_relaxed(readl_relaxed(base + GPIOC_INTR_MASK) & ~mask, base + GPIOC_INTR_MASK);
	spin_unlock_irqrestore(&bank->lock, flags);
}

static int vpl_gpio_irq_type(struct irq_data *d, unsigned int type)
{
	struct vpl_gpio_bank *bank = irq_data_get_irq_chip_data(d);
	void __iomem *base = bank->base;
	u32 mask = (u32) irq_data_get_irq_handler_data(d);
	unsigned long flags;

	if (type & IRQ_TYPE_LEVEL_MASK) {
		dev_err(bank->dev, "irq %d: unsupported type %d\n", d->irq, type);
		return -EINVAL;
	}
	spin_lock_irqsave(&bank->lock, flags);

	writel_relaxed(readl_relaxed(base + GPIOC_INTR_TRIGGER_TYPE) & ~mask, base + GPIOC_INTR_TRIGGER_TYPE);

	switch (type) {
	case IRQ_TYPE_EDGE_BOTH: 	// double edge trigger
		writel_relaxed(readl_relaxed(base + GPIOC_INTR_BOTH) | mask, base + GPIOC_INTR_BOTH);
		break;
	case IRQ_TYPE_EDGE_RISING: 	// rising edge trigger
		writel_relaxed(readl_relaxed(base + GPIOC_INTR_BOTH) & ~mask, base + GPIOC_INTR_BOTH);
		writel_relaxed(readl_relaxed(base + GPIOC_INTR_DIR) & ~mask, base + GPIOC_INTR_DIR);
		break;
	case IRQ_TYPE_EDGE_FALLING: 	// rising edge trigger
		writel_relaxed(readl_relaxed(base + GPIOC_INTR_BOTH) & ~mask, base + GPIOC_INTR_BOTH);
		writel_relaxed(readl_relaxed(base + GPIOC_INTR_DIR) | mask, base + GPIOC_INTR_DIR);
		break;
	default:
		break;
	}
	spin_unlock_irqrestore(&bank->lock, flags);

	return 0;
}

static struct irq_chip gpio_irqchip = {
	.name		= "PESARO_GPIO",
	.irq_enable	= vpl_gpio_irq_enable,
	.irq_disable	= vpl_gpio_irq_disable,
	.irq_ack	= vpl_gpio_irq_ack,
	.irq_mask	= vpl_gpio_irq_mask,
	.irq_unmask	= vpl_gpio_irq_unmask,
	.irq_set_type	= vpl_gpio_irq_type,
};

/* This lock class tells lockdep that GPIO irqs are in a different
 * category than their parents, so it won't report false recursion.
 */
static struct lock_class_key gpio_lock_class;

static int vpl_gpio_probe(struct platform_device *pdev)
{
	struct vpl_gpio_bank *priv_data;
	struct device *dev = &pdev->dev;
	struct resource *res;
	int id, ret;
	int gpio;

	priv_data = devm_kzalloc(dev, sizeof(struct vpl_gpio_bank), GFP_KERNEL);
	if (!priv_data) {
		dev_err(dev, "Unable to alloacate driver data\n");
		return -ENOMEM;
	}

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	priv_data->base = devm_ioremap_resource(dev, res);
	if (IS_ERR(priv_data->base))
		return PTR_ERR(priv_data->base);

	priv_data->virq = platform_get_irq(pdev, 0);
	if (priv_data->virq < 0) {
		dev_err(dev, "no irq resource specified\n");
		return priv_data->virq;
	}

	id = of_alias_get_id(pdev->dev.of_node, "gpio");
	if (id < 0) {
		dev_err(dev, "Couldn't get OF id\n");
		return id;
	}

	priv_data->clk = devm_clk_get(dev, NULL);
	if (IS_ERR(priv_data->clk))
		return PTR_ERR(priv_data->clk);
	clk_prepare_enable(priv_data->clk);

	priv_data->clk_rate = clk_get_rate(priv_data->clk);
	priv_data->dev = dev;

	platform_set_drvdata(pdev, priv_data);

	/* chip setup */
	priv_data->chip.label = dev_name(dev);
	priv_data->chip.dev = dev;
	priv_data->chip.owner = THIS_MODULE;
	priv_data->chip.request = vpl_gpio_request;
	priv_data->chip.free = vpl_gpio_free;
	priv_data->chip.direction_input = vpl_gpio_direction_input;
	priv_data->chip.get = vpl_gpio_get;
	priv_data->chip.direction_output = vpl_gpio_direction_output;
	priv_data->chip.set = vpl_gpio_set;
	priv_data->chip.set_debounce = vpl_gpio_set_debounce;
	priv_data->chip.to_irq = vpl_gpio_to_irq;
	priv_data->chip.base = id * MAX_GPIO_PER_BANK;
	priv_data->chip.ngpio = MAX_GPIO_PER_BANK;

	spin_lock_init(&priv_data->lock);

	ret = gpiochip_add(&priv_data->chip);
	if (ret) {
		dev_err(dev, "Could not register gpiochip");
		clk_disable_unprepare(priv_data->clk);
		clk_put(priv_data->clk);
		return ret;
	}

	/* vpl_gpio_irqchip_add */
	priv_data->irqdomain = irq_domain_add_linear(dev->of_node, priv_data->chip.ngpio,
			                             &irq_domain_simple_ops, priv_data);
	if (!priv_data->irqdomain)
		return -EINVAL;

	for (gpio = 0; gpio < priv_data->chip.ngpio; gpio++) {
		int irq = irq_create_mapping(priv_data->irqdomain, gpio);

		irq_set_lockdep_class(irq, &gpio_lock_class);
		irq_set_chip_data(irq, priv_data);
		irq_set_chip_and_handler(irq, &gpio_irqchip, handle_simple_irq);
		irq_set_handler_data(irq, (void *)BIT(gpio));
		set_irq_flags(irq, IRQF_VALID);
	}

	irq_set_chained_handler(priv_data->virq, vpl_gpio_irq_handler);
	irq_set_handler_data(priv_data->virq, priv_data);

	return 0;
}

static const struct of_device_id vpl_gpio_of_match[] = {
	{ .compatible = "vatics,vpl-gpio", },
	{}
};
MODULE_DEVICE_TABLE(of, vpl_gpio_of_match);

static struct platform_driver vpl_gpio_driver = {
	.probe = vpl_gpio_probe,
	.driver = {
		.name = "vpl_gpio",
		.owner = THIS_MODULE,
		.of_match_table = vpl_gpio_of_match,
	},
};
module_platform_driver(vpl_gpio_driver);

MODULE_AUTHOR("Vincent Ho <vincent.ho@vatics.com>");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("VATICS VPL GPIOC driver");
