/*
 *  drivers/irqchip/irq-pesaro.c
 *
 *  Copyright (C) 2007-2014  VATICS Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/irq.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>
#include <linux/irqdomain.h>
#include <linux/irqchip/pesaro.h>
#include <asm/exception.h>

#include "irqchip.h"

struct pesaro_intc_data {
	void __iomem *base;
	struct irq_domain *domain;
};

static struct pesaro_intc_data intc __read_mostly;

static void pesaro_irq_ack(struct irq_data *d)
{
	irq_hw_number_t hwirq = irqd_to_hwirq(d);
	u32 mask = (1 << (hwirq & 31));

	if (hwirq > 31)
		writel(mask, intc.base + INTC_CLEAR_HI);
	else
		writel(mask, intc.base + INTC_CLEAR_LO);
}

static void pesaro_irq_mask(struct irq_data *d)
{
	irq_hw_number_t hwirq = irqd_to_hwirq(d);
	u32 mask = (1 << (hwirq & 31));

	if (hwirq > 31)
		writel(readl(intc.base + INTC_HOSTU_0_MASK_HI) & ~mask, intc.base + INTC_HOSTU_0_MASK_HI);
	else
		writel(readl(intc.base + INTC_HOSTU_0_MASK_LO) & ~mask, intc.base + INTC_HOSTU_0_MASK_LO);
}

static void pesaro_irq_unmask(struct irq_data *d)
{
	irq_hw_number_t hwirq = irqd_to_hwirq(d);
	u32 mask = (1 << (hwirq & 31));

	if (hwirq > 31)
		writel(readl(intc.base + INTC_HOSTU_0_MASK_HI) | mask, intc.base + INTC_HOSTU_0_MASK_HI);
	else
		writel(readl(intc.base + INTC_HOSTU_0_MASK_LO) | mask, intc.base + INTC_HOSTU_0_MASK_LO);
}

static int pesaro_irq_set_type(struct irq_data *d, unsigned int flow_type)
{
	irq_hw_number_t hwirq = irqd_to_hwirq(d);
	u32 mask = (1 << (hwirq & 31));

	switch(flow_type) {
		case IRQ_TYPE_EDGE_RISING:
			if (hwirq > 31)
				writel(readl(intc.base + INTC_TRIGGER_MODE_HI) | mask, intc.base + INTC_TRIGGER_MODE_HI);
			else
				writel(readl(intc.base + INTC_TRIGGER_MODE_LO) | mask, intc.base + INTC_TRIGGER_MODE_LO);

			__irq_set_handler_locked(d->irq, handle_edge_irq);
			break;
		case IRQ_TYPE_LEVEL_HIGH:
			if (hwirq > 31)
				writel(readl(intc.base + INTC_TRIGGER_MODE_HI) & ~mask, intc.base + INTC_TRIGGER_MODE_HI);
			else
				writel(readl(intc.base + INTC_TRIGGER_MODE_LO) & ~mask, intc.base + INTC_TRIGGER_MODE_LO);

			 __irq_set_handler_locked(d->irq, handle_level_irq);
			break;
		default:
			printk(KERN_ERR "Unsupported irq type %d\n", flow_type);
			return -EINVAL;
	}

	return 0;
}

static struct irq_chip pesaro_intc = {
	.name = "PERASO_INTC",
	.irq_disable = pesaro_irq_mask,
	.irq_ack = pesaro_irq_ack,
	.irq_mask = pesaro_irq_mask,
	.irq_unmask = pesaro_irq_unmask,
	.irq_set_type = pesaro_irq_set_type,
};

static int pesaro_intc_irq_map(struct irq_domain *h,
			       unsigned int virq, irq_hw_number_t hw)
{
	irq_set_chip_and_handler(virq, &pesaro_intc, handle_level_irq);
	set_irq_flags(virq, IRQF_VALID | IRQF_PROBE);
	return 0;
}

static struct irq_domain_ops pesaro_intc_irq_domain_ops = {
	.xlate = irq_domain_xlate_onecell,
	.map = pesaro_intc_irq_map,
};

static asmlinkage void __exception_irq_entry pesaro_handle_irq(struct pt_regs *regs)
{
	u32 irqnr;

	irqnr = readl(intc.base + INTC_HOSTU_0_IRQ_NUM);

	if (irqnr != 0) {
		handle_IRQ(irq_find_mapping(intc.domain, irqnr), regs);
	}
};

static int __init pesaro_intc_of_init(struct device_node *node,
				      struct device_node *parent)
{
	int i;

	intc.base = of_iomap(node, 0);
	if (!intc.base)
		panic("%s: unable to map INTC registers\n", node->full_name);

	intc.domain = irq_domain_add_linear(node, 64, &pesaro_intc_irq_domain_ops, NULL);
	if (!intc.domain)
		panic("%s: unable to add irq domain\n", node->full_name);

	/*
	 * convenient for irq_create_mapping() with NULL domain
	 * for some hard coded interrupt not represented in the device tree.
	 * ex. vma, vpl drivers
	 */
	irq_set_default_host(intc.domain);

	/* Mask all interrupts */
	writel(0, intc.base + INTC_HOSTU_0_MASK_LO);
	writel(0, intc.base + INTC_HOSTU_0_MASK_HI);
	/* Clear all interrupts */
	writel(0xffffffff, intc.base + INTC_CLEAR_LO);
	writel(0xffffffff, intc.base + INTC_CLEAR_HI);

	/* Set up priority */
	for (i = 0; i <= 64; i++ ) {
		writel(i, intc.base+INTC_INTR_PRI_START+(i*4));
	}

	set_handle_irq(pesaro_handle_irq);
	return 0;
}

IRQCHIP_DECLARE(pesaro_intc, "vatics,pesaro-intc", pesaro_intc_of_init);
