/*
 *
 * Copyright (C) 2014, VATICS
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */
//#define DEBUG
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/clk.h>
#include <linux/slab.h>
#include <linux/proc_fs.h>
#include <linux/mmc/host.h>
#include <linux/mmc/dw_mmc.h>
#include <linux/mmc/mmc.h>
#include <linux/mmc/host.h>
#include <linux/mmc/card.h>
#include <linux/of.h>
#include <linux/of_gpio.h>
#include <linux/of_address.h>
#include <linux/delay.h>

#include <mach/maps.h>
#include <mach/hardware.h>

#include "dw_mmc.h"
#include "dw_mmc-pltfm.h"
#define SYSC_BASE VPL_SYSC_MMR_BASE

struct dw_mci_pesaro_priv_data {
	int uhs_gpio;
	int dev_id;
};
enum {
	PAD_OUTPUT_STRENGTH_2_MA,
	PAD_OUTPUT_STRENGTH_4_MA,
	PAD_OUTPUT_STRENGTH_8_MA,
	PAD_OUTPUT_STRENGTH_12_MA
};

enum {
	PAD_SLEW_RATE_SLOW,
	PAD_SLEW_RATE_FAST
};


void *sysc_mmc_base;

static s32 switch_voltage_delay = 50;
static s32 force_rx_delay_ctrl = 0;
static s32 force_rx_delay = 180;

struct dw_mci_delay_control {
	char * str;
	u32 tx_delay;
	u32 clk_in;
	u32 slew_rate;
	u32 driving_strength;
};

/*
Delay contrl index ,
#define MMC_TIMING_LEGACY	0
#define MMC_TIMING_MMC_HS	1
#define MMC_TIMING_SD_HS	2
#define MMC_TIMING_UHS_SDR12	3
#define MMC_TIMING_UHS_SDR25	4
#define MMC_TIMING_UHS_SDR50	5
#define MMC_TIMING_UHS_SDR104	6
#define MMC_TIMING_UHS_DDR50	7
#define MMC_TIMING_MMC_HS200	8
*/

static struct dw_mci_delay_control delay_control_array[] = {
	{
		.str = "legacy",
		.tx_delay = 180, .clk_in =  50000000, 
		.slew_rate = PAD_SLEW_RATE_SLOW, 
		.driving_strength = 2 
	},
	{
		.str = "mmchs",
		.tx_delay =  90, .clk_in =  50000000, 
		.slew_rate = PAD_SLEW_RATE_SLOW, 
		.driving_strength = 2 
	},
	{
		.str = "sdhs",
		.tx_delay =  90, .clk_in =  50000000, 
		.slew_rate = PAD_SLEW_RATE_SLOW, 
		.driving_strength = 2 
	},
	{
		.str = "sdr12",
		.tx_delay = 180, .clk_in =  50000000, 
		.slew_rate = PAD_SLEW_RATE_SLOW, 
		.driving_strength = 2 
	},
	{
		.str = "sdr25",
		.tx_delay =  90, .clk_in =  50000000, 
		.slew_rate = PAD_SLEW_RATE_SLOW, 
		.driving_strength = 2 
	},
	{
		.str = "sdr50",
		.tx_delay =  90, .clk_in = 100000000, 
		.slew_rate = PAD_SLEW_RATE_FAST, 
		.driving_strength = 2 
	},
	{
		.str = "sdr104",
		.tx_delay = 180, .clk_in = 200000000, 
		.slew_rate = PAD_SLEW_RATE_FAST, 
		.driving_strength = 2 
	},
	{
		.str = "ddr50",
		.tx_delay =  90, .clk_in =  50000000, 
		.slew_rate = PAD_SLEW_RATE_SLOW, 
		.driving_strength = 2 
	},
	{ 
		.str = "mmchs200",
		.tx_delay = 180, .clk_in = 100000000, 
		.slew_rate = PAD_SLEW_RATE_FAST, 
		.driving_strength = 2 
	}
};

static int pesaro_get_dev_index(struct dw_mci * host);

static int pesaro_voltage_switch(struct mmc_host *mmc, struct mmc_ios *ios)
{
	struct dw_mci_slot *slot = mmc_priv(mmc);
	struct dw_mci *host = slot->host;
	struct dw_mci_pesaro_priv_data *priv = host->priv;
	u32 * membase;
	u32 slot_index;
	volatile u32 reg_val;


	if (priv->uhs_gpio < 0)
		return -ENODEV;
	dev_dbg(&mmc->class_dev,
		"uhs-gpio:%d, voltage:%s\n", priv->uhs_gpio, 
		mmc->ios.signal_voltage == MMC_SIGNAL_VOLTAGE_330 ? "v330" : "v180");

	slot_index = pesaro_get_dev_index(host);
	if (slot_index == 0)
		membase = sysc_mmc_base + 0x60;
	else
		membase = sysc_mmc_base + 0x64;
	reg_val = readl(membase);

	if (mmc->ios.signal_voltage == MMC_SIGNAL_VOLTAGE_330) {
		gpio_direction_input(priv->uhs_gpio);
		reg_val &= ~(0x1 << 20);
	} else {
		gpio_direction_output(priv->uhs_gpio, 0);
		reg_val |= (0x1 << 20);
	}
	writel(reg_val, membase);

	/*  from the measure, board ver 1.2 with other power IC, 
	 *  the 1.8 v stable needs at least 30ms. 
	 *  although 20 ms works well,
	 *  but we want to make sure it's 100% stable. so we choose 30ms delay
	 */
	mdelay(switch_voltage_delay);
	return 0;
}
static int pesaro_get_dev_index(struct dw_mci * host) 
{
	struct dw_mci_pesaro_priv_data *priv = host->priv;
	return priv->dev_id;
}
static void pesaro_set_tx_clock_phase(struct dw_mci * host, int degree) 
{
	u32 * membase;
	u32 slot_index;
	volatile u32 reg_val;

	slot_index = pesaro_get_dev_index(host);
	if (slot_index == 0)
		membase = sysc_mmc_base + 0x60;
	else
		membase = sysc_mmc_base + 0x64;
	reg_val = readl(membase);


	switch (degree) {
	case 0:
		if (((reg_val >> 8) & 0x3) == (0x0)) 
			return;
		reg_val &= ~(0x3 << 8);
		reg_val |= (0x0 << 8);
		break;
	case 90:
		if (((reg_val >> 8) & 0x3) == (0x1)) 
			return;
		reg_val &= ~(0x3 << 8);
		reg_val |= (0x1 << 8);
		break;
	case 180:
		if (((reg_val >> 8) & 0x3) == (0x2)) 
			return;
		reg_val &= ~(0x3 << 8);
		reg_val |= (0x2 << 8);
		break;
	case 270:
		if (((reg_val >> 8) & 0x3) == (0x3)) 
			return;
		reg_val &= ~(0x3 << 8);
		reg_val |= (0x3 << 8);
		break;
	default:
		dev_warn(host->dev, "switch to illegal %d degree\n", degree);
		return;
	}
	writel(reg_val, membase);
	dev_dbg(host->dev, "switch TX clock to %d degree, SYSC val %08x\n", 
		degree, reg_val);
	mdelay(30);
}
static void pesaro_set_rx_clock_phase(struct dw_mci * host, int degree) 
{
	u32 * membase;
	u32 slot_index;
	volatile u32 reg_val;

	slot_index = pesaro_get_dev_index(host);
	if (slot_index == 0)
		membase = sysc_mmc_base + 0x60;
	else
		membase = sysc_mmc_base + 0x64;
	reg_val = readl(membase);

	switch (degree) {
	case 0:
		if (((reg_val >> 4) & 0x3) == (0x0)) 
			return;
		reg_val &= ~(0x3 << 4);
		reg_val |= (0x0 << 4);
		break;
	case 90:
		if (((reg_val >> 4) & 0x3) == (0x1)) 
			return;
		reg_val &= ~(0x3 << 4);
		reg_val |= (0x1 << 4);
		break;
	case 180:
		if (((reg_val >> 4) & 0x3) == (0x2)) 
			return;
		reg_val &= ~(0x3 << 4);
		reg_val |= (0x2 << 4);
		break;
	case 270:
		if (((reg_val >> 4) & 0x3) == (0x3)) 
			return;
		reg_val &= ~(0x3 << 4);
		reg_val |= (0x3 << 4);
		break;
	default:
		dev_warn(host->dev, "switch to illegal %d degree\n", degree);
		return;
	}
	writel(reg_val, membase);
	dev_dbg(host->dev, "switch RX clock to %d degree, SYSC val %08x\n", 
		degree, reg_val);
	mdelay(30);
}

static void pesaro_set_pad_slew_rate(struct dw_mci * host, int isfast) 
{
	u32 * membase;
	u32 slot_index;
	volatile u32 reg_val;


	slot_index = pesaro_get_dev_index(host);
	if (slot_index == 0)
		membase = sysc_mmc_base + 0x60;
	else
		membase = sysc_mmc_base + 0x64;
	reg_val = readl(membase);

	if (((reg_val >> 16) & 0x1) == ((isfast > 0)?1:0))
		return;

	reg_val &= ~(0x1 << 16);
	mb();
	reg_val |= (((isfast > 0)? 1: 0) << 16);

	writel(reg_val, membase);

	dev_dbg(host->dev, "set pad slew rate to %s SYSC val %08x\n",
		(isfast)?"fast":"slow" , reg_val);
}
static void pesaro_set_output_driving_strength(struct dw_mci * host, int strength) 
{
	u32 * membase;
	u32 slot_index;
	volatile u32 reg_val;
	u32 set_val;

	switch (strength) {
	case 2:
		set_val = PAD_OUTPUT_STRENGTH_2_MA;
		break;
	case 4:
		set_val = PAD_OUTPUT_STRENGTH_4_MA;
		break;
	case 8:
		set_val = PAD_OUTPUT_STRENGTH_8_MA;
		break;
	case 12:
		set_val = PAD_OUTPUT_STRENGTH_12_MA;
		break;
	default:
		dev_warn(host->dev, "set illegal output driving strength %d\n", strength);
		return;
	}


	slot_index = pesaro_get_dev_index(host);
	if (slot_index == 0)
		membase = sysc_mmc_base + 0x60;
	else
		membase = sysc_mmc_base + 0x64;
	reg_val = readl(membase);
	if (((reg_val >> 24) & 0x3) == set_val)
		return;

	reg_val &= ~(0x3 << 24);
	reg_val |= (set_val << 24);
	mb();

	writel(reg_val, membase);

	dev_dbg(host->dev, "set output driving %d SYSC val %08x\n", 
		strength, reg_val);
}

static int compare_pattern(u8* data,struct dw_mci_tuning_data *tuning_data )
{
	int i;
	for (i = 0 ; i < tuning_data->blksz ; i++) {
		if (data[i] != tuning_data->blk_pattern[i]){
			return -1;
		}
	}
	return 0;
}


static int pesaro_execute_tuning_run(struct mmc_host *mmc, u32 opcode,
					struct dw_mci_tuning_data *tuning_data)
{
	struct dw_mci_slot *slot = mmc_priv(mmc);
	struct dw_mci *host = slot->host;
	__maybe_unused int i;
	struct mmc_request mrq = {NULL};
	struct mmc_command cmd = {0};
	struct mmc_data data = {0};
	struct scatterlist sg;
	u8 *card_data;
	int ret = 0;
	int tuning_loop_counter = 40; 
	unsigned long start_time;
	unsigned int timeout = 0;

	dev_dbg(host->dev, "try to read. opcode %u\n", opcode);

	card_data = kmalloc(tuning_data->blksz, GFP_KERNEL);

	/*
	 * Issue CMD19 repeatedly till Execute Tuning is set to 0 or the number
	 * of loops reaches 40 times or a timeout of 150ms occurs.
	 */
	if (opcode != MMC_SEND_TUNING_BLOCK_HS200 && 
		opcode != MMC_SEND_TUNING_BLOCK)
		tuning_loop_counter =1;

	start_time = jiffies;
	do {
		mrq.cmd = &cmd;
		mrq.data = &data;

		cmd.opcode = opcode;
		cmd.arg = 0;
		cmd.flags = MMC_RSP_R1 | MMC_CMD_ADTC;

		data.blksz = tuning_data->blksz;
		data.blocks = 1;
		data.flags = MMC_DATA_READ;
		data.sg = &sg;
		data.sg_len = 1;

		memset(card_data, 0x0, tuning_data->blksz);
		sg_init_one(&sg, card_data, tuning_data->blksz);

		if (mmc->card)
			mmc_set_data_timeout(&data, mmc->card);

		mmc_wait_for_req(mmc, &mrq);

		if (cmd.error)
			dev_dbg(host->dev, "SEND_TUNING_BLOCK cmd error %u\n", cmd.error);
		if (data.error)
			dev_dbg(host->dev, "SEND_TUNING BLOCK data error %u\n", data.error);

		if (cmd.error || data.error)  {
			ret = -EIO;

			dev_dbg(host->dev, "Send Stop Command \n");
			mrq.cmd = &cmd;
			mrq.data = NULL;

			cmd.opcode = MMC_STOP_TRANSMISSION;
			cmd.arg = 0;
			cmd.flags = MMC_RSP_R1B | MMC_CMD_AC;
			mmc_wait_for_req(mmc, &mrq);
		}

		if (ret == 0) {
			if (tuning_data->blk_pattern) {
#ifdef DEBUG
				pr_debug("tuning result:\n");
				for (i = 0 ; i < tuning_data->blksz ; i=i+4) {
					pr_debug("%02x%02x%02x%02x\n",
						 card_data[i], card_data[i+1],
						 card_data[i+2], card_data[i+3]);
				}
#endif

				if (compare_pattern(card_data, tuning_data)) {
					dev_dbg(host->dev, "pattern incorrect, tuning pattern\n");
					ret = -1;
					break;
					/*TODO, do tuning here*/
				}
				else {
					dev_dbg(host->dev, "pattern correct\n");
				}
			} else {
				dev_dbg(host->dev, "tunning command OK!\n");
			}
		}
		timeout = jiffies_to_msecs(jiffies - start_time);
		tuning_loop_counter--;
		pr_debug("timeout: %u loop_counter %d\n", timeout, tuning_loop_counter);
	} while (timeout < 160 && tuning_loop_counter > 0 && ret == 0);

	/*
	 * The Host Driver has exhausted the maximum number of loops allowed,
	 * so use fixed sampling frequency.
	 */
	kfree(card_data);
	return ret;
}

static int pesaro_execute_tuning(struct mmc_host *mmc, u32 opcode,
					struct dw_mci_tuning_data *tuning_data)
{
	struct dw_mci_slot *slot = mmc_priv(mmc);
	struct dw_mci *host = slot->host;
	int degree = 0;
	int good_start = -1;
	int good_end = -1;

	dev_dbg(host->dev, "Tunning sample opcode %d\n", opcode);
	for (degree = 0; degree <= 270 ; degree +=90) {
		pesaro_set_rx_clock_phase(host, degree);
		if (pesaro_execute_tuning_run(mmc, opcode, tuning_data)) {
			if (good_start == -1)
				continue;
			else
				break;
		}
		if (good_start == -1)
			good_start = degree;
		good_end = degree;
	}

	if (good_start == -1) {
		degree = 180;
		if (force_rx_delay_ctrl)
			degree = force_rx_delay;

		pesaro_set_rx_clock_phase(host, degree);
		dev_err(host->dev,
			"Tuning sampling failed! Use default Rx clock shift %d\n", degree);
		return -1;
	}
#define DEGREE_PREFER_HIGH
	/* Take the degree closer upbound for Toshiba UHS card */
	if ((good_end - good_start == 180) || (good_end - good_start == 270)) {
#ifdef DEGREE_PREFER_HIGH
		degree = good_end - 90;
#else
		degree = good_start + 90;
#endif
	} else {
#ifdef DEGREE_PREFER_HIGH
		degree = good_end;
#else
		degree = good_start;
#endif
	}

	if (force_rx_delay_ctrl)
		degree = force_rx_delay;

	pesaro_set_rx_clock_phase(host, degree);
	dev_info(host->dev, "Tuning ...... Select RX clock shift %d between %d and %d\n", 
			 degree, good_start, good_end);
	return 0;
}


static int dw_mci_pesaro_setup_clock(struct dw_mci *host)
{
	/* Set to Maximum at initial. 100mHZ */
	if (host->ciu_clk) {
		host->bus_hz = 100000000;
		clk_set_rate(host->ciu_clk, host->bus_hz);
	}

	return 0;
}

static void dw_mci_pesaro_set_ios(struct dw_mci *host, struct mmc_ios *ios)
{
	/* Delay_D, refer to Table 10-7 */
	dev_dbg(host->dev, "timing %d, host->bus_hz %u\n", __func__, ios->timing, host->bus_hz);

	pesaro_set_tx_clock_phase(host, delay_control_array[ios->timing].tx_delay);
	if (host->bus_hz != delay_control_array[ios->timing].clk_in) {
		host->bus_hz = delay_control_array[ios->timing].clk_in;
		clk_set_rate(host->ciu_clk, host->bus_hz);
		dev_dbg(host->dev, "switch bus_hz to %u\n", host->bus_hz);
	}

	pesaro_set_pad_slew_rate(host, delay_control_array[ios->timing].slew_rate); 

	/* EVM use 2mAh, depending on board characteristics */
	pesaro_set_output_driving_strength(host, delay_control_array[ios->timing].driving_strength);

}

static int dw_mci_pesaro_parse_dt(struct dw_mci *host)
{
	struct dw_mci_pesaro_priv_data *priv = host->priv;
	struct device_node *np = host->dev->of_node;
	const __be32 *reg;
	int len;
	u64 addr;

	if (of_property_read_u32(np, "uhs-gpio", &priv->uhs_gpio)) {
		priv->uhs_gpio = -1;
	}

	if (priv->uhs_gpio >= 0) {
		dev_info(host->dev,"using uhs-gpio:%d\n", priv->uhs_gpio);
		gpio_request(priv->uhs_gpio, "uhs_gpio");
		gpio_direction_input(priv->uhs_gpio);
	}

	reg = of_get_property(np, "reg", &len);
	if (reg) {
		addr = of_translate_address(np, reg);
		if (addr == VPL_MSHC_0_MMR_BASE)
			priv->dev_id = 0;
		else if (addr== VPL_MSHC_1_MMR_BASE)
			priv->dev_id = 1;

		dev_info(host->dev, "dev-id: %d\n", priv->dev_id);
	} else {
		dev_info(host->dev, "Can't get reg property\n");
	}


	return 0;
}

static int dw_mci_pesaro_priv_init(struct dw_mci *host)
{
	struct dw_mci_pesaro_priv_data *priv;

	priv = devm_kzalloc(host->dev, sizeof(*priv), GFP_KERNEL);
	if (!priv) {
		dev_err(host->dev, "mem alloc failed for private data\n");
		return -ENOMEM;
	}

	host->priv = priv;


	return 0;
}



static int get_control_index(const char *name) 
{
	s32 i = 0;
	s32 str_oft = -1;
	if (strncmp(name, "clk_in_", strlen("clk_in_")) == 0) 
		str_oft = strlen("clk_in_");
	else if (strncmp(name, "tx_delay_", strlen("tx_delay_")) == 0) 
		str_oft = strlen("tx_delay_");
	else if (strncmp(name, "slew_rate_", strlen("slew_rate_")) == 0) 
		str_oft = strlen("slew_rate_");
	else if (strncmp(name, "driving_strength_", strlen("driving_strength_")) == 0) 
		str_oft = strlen("driving_strength_");

	if (str_oft >= 0) {
		for (i = 0; i < sizeof(delay_control_array); i++) {
			if (strcmp(name+str_oft, delay_control_array[i].str) == 0) {
				return i;
			}
		}
	} 

	return -1;

}
static ssize_t ctrl_show(struct device *dev,
		       struct device_attribute *attr, char *buf)
{
	s32 index;

	index = get_control_index(attr->attr.name);
	if (index >= 0) {
		if (strncmp(attr->attr.name, "clk_in_", strlen("clk_in_")) == 0)
			return sprintf(buf, "%u\n", delay_control_array[index].clk_in);
		else if (strncmp(attr->attr.name, "tx_delay_", strlen("tx_delay_")) == 0)
			return sprintf(buf, "%u\n", delay_control_array[index].tx_delay);
		else if (strncmp(attr->attr.name, "slew_rate_", strlen("slew_rate_")) == 0)
			return sprintf(buf, "%u %s\n", delay_control_array[index].slew_rate,
				       (delay_control_array[index].slew_rate == PAD_SLEW_RATE_FAST)? "fast":"slow");
		else if (strncmp(attr->attr.name, "driving_strength_", strlen("driving_strength_")) == 0)
			return sprintf(buf, "%u mAh\n", delay_control_array[index].driving_strength);
	}
	return 0;
}

static ssize_t ctrl_store(struct device *dev,
			struct device_attribute *attr,
			const char *buf, size_t count)
{
	u32 i;
	s32 index;

	if (count > 0 && sscanf(buf, "%u", &i) > 0) {
		index = get_control_index(attr->attr.name);
		if (index >= 0) {
			if (strncmp(attr->attr.name, "clk_in_", strlen("clk_in_")) == 0)
				delay_control_array[index].clk_in = i;
			else if (strncmp(attr->attr.name, "tx_delay_", strlen("tx_delay_")) == 0) {
				if (i <= 270 && i % 90 == 0)
					delay_control_array[index].tx_delay = i;
			} else if (strncmp(attr->attr.name, "slew_rate_", strlen("slew_rate_")) == 0) {
				if (i <= 1) 
					delay_control_array[index].slew_rate = i;
			} else if (strncmp(attr->attr.name, "driving_strength_", strlen("driving_strength_")) == 0) {
				if (i == 2 || i == 4 || i == 8 || i == 12) {
					delay_control_array[index].driving_strength = i;
				}
			}
		}
	}
	return count;
}
static DEVICE_ATTR(clk_in_legacy, S_IRUGO | S_IWUGO  , ctrl_show, ctrl_store);
static DEVICE_ATTR(clk_in_sdhs,   S_IRUGO | S_IWUGO  , ctrl_show, ctrl_store);
static DEVICE_ATTR(clk_in_mmchs,  S_IRUGO | S_IWUGO  , ctrl_show, ctrl_store);
static DEVICE_ATTR(clk_in_mmchs200,  S_IRUGO | S_IWUGO  , ctrl_show, ctrl_store);
static DEVICE_ATTR(clk_in_sdr12,  S_IRUGO | S_IWUGO , ctrl_show, ctrl_store);
static DEVICE_ATTR(clk_in_sdr25,  S_IRUGO | S_IWUGO, ctrl_show, ctrl_store);
static DEVICE_ATTR(clk_in_sdr50,  S_IRUGO | S_IWUGO, ctrl_show, ctrl_store);
static DEVICE_ATTR(clk_in_sdr104,  S_IRUGO | S_IWUGO, ctrl_show, ctrl_store);
static DEVICE_ATTR(clk_in_ddr50,  S_IRUGO | S_IWUGO, ctrl_show, ctrl_store);

static DEVICE_ATTR(tx_delay_legacy, S_IRUGO | S_IWUGO  , ctrl_show, ctrl_store);
static DEVICE_ATTR(tx_delay_sdhs,   S_IRUGO | S_IWUGO  , ctrl_show, ctrl_store);
static DEVICE_ATTR(tx_delay_mmchs,  S_IRUGO | S_IWUGO  , ctrl_show, ctrl_store);
static DEVICE_ATTR(tx_delay_mmchs200,  S_IRUGO | S_IWUGO  , ctrl_show, ctrl_store);
static DEVICE_ATTR(tx_delay_sdr12,  S_IRUGO | S_IWUGO , ctrl_show, ctrl_store);
static DEVICE_ATTR(tx_delay_sdr25,  S_IRUGO | S_IWUGO, ctrl_show, ctrl_store);
static DEVICE_ATTR(tx_delay_sdr50,  S_IRUGO | S_IWUGO, ctrl_show, ctrl_store);
static DEVICE_ATTR(tx_delay_sdr104,  S_IRUGO | S_IWUGO, ctrl_show, ctrl_store);
static DEVICE_ATTR(tx_delay_ddr50,  S_IRUGO | S_IWUGO, ctrl_show, ctrl_store);

static DEVICE_ATTR(slew_rate_legacy, S_IRUGO | S_IWUGO  , ctrl_show, ctrl_store);
static DEVICE_ATTR(slew_rate_sdhs,   S_IRUGO | S_IWUGO  , ctrl_show, ctrl_store);
static DEVICE_ATTR(slew_rate_mmchs,  S_IRUGO | S_IWUGO  , ctrl_show, ctrl_store);
static DEVICE_ATTR(slew_rate_mmchs200,  S_IRUGO | S_IWUGO  , ctrl_show, ctrl_store);
static DEVICE_ATTR(slew_rate_sdr12,  S_IRUGO | S_IWUGO , ctrl_show, ctrl_store);
static DEVICE_ATTR(slew_rate_sdr25,  S_IRUGO | S_IWUGO, ctrl_show, ctrl_store);
static DEVICE_ATTR(slew_rate_sdr50,  S_IRUGO | S_IWUGO, ctrl_show, ctrl_store);
static DEVICE_ATTR(slew_rate_sdr104,  S_IRUGO | S_IWUGO, ctrl_show, ctrl_store);
static DEVICE_ATTR(slew_rate_ddr50,  S_IRUGO | S_IWUGO, ctrl_show, ctrl_store);

static DEVICE_ATTR(driving_strength_legacy, S_IRUGO | S_IWUGO  , ctrl_show, ctrl_store);
static DEVICE_ATTR(driving_strength_sdhs,   S_IRUGO | S_IWUGO  , ctrl_show, ctrl_store);
static DEVICE_ATTR(driving_strength_mmchs,  S_IRUGO | S_IWUGO  , ctrl_show, ctrl_store);
static DEVICE_ATTR(driving_strength_mmchs200,  S_IRUGO | S_IWUGO  , ctrl_show, ctrl_store);
static DEVICE_ATTR(driving_strength_sdr12,  S_IRUGO | S_IWUGO , ctrl_show, ctrl_store);
static DEVICE_ATTR(driving_strength_sdr25,  S_IRUGO | S_IWUGO, ctrl_show, ctrl_store);
static DEVICE_ATTR(driving_strength_sdr50,  S_IRUGO | S_IWUGO, ctrl_show, ctrl_store);
static DEVICE_ATTR(driving_strength_sdr104,  S_IRUGO | S_IWUGO, ctrl_show, ctrl_store);
static DEVICE_ATTR(driving_strength_ddr50,  S_IRUGO | S_IWUGO, ctrl_show, ctrl_store);



static ssize_t other_show(struct device *dev,
		       struct device_attribute *attr, char *buf)
{
	if (strcmp(attr->attr.name, "force_rx_delay_ctrl") == 0)
		return sprintf(buf, "%d\n", force_rx_delay_ctrl);
	else if (strcmp(attr->attr.name, "switch_voltage_delay") == 0)
		return sprintf(buf, "%d\n", switch_voltage_delay);
	else if (strcmp(attr->attr.name, "force_rx_delay") == 0)
		return sprintf(buf, "%d\n", force_rx_delay);
	else
		return 0;
}

static ssize_t other_store(struct device *dev,
			struct device_attribute *attr,
			const char *buf, size_t count)
{
	s32 i;

	if (count > 0 && sscanf(buf, "%d", &i) > 0) {
		if (strcmp(attr->attr.name, "force_rx_delay_ctrl") == 0)
			force_rx_delay_ctrl = i;
		if (strcmp(attr->attr.name, "force_rx_delay") == 0)
			force_rx_delay = i;
		else if (strcmp(attr->attr.name, "switch_voltage_delay") == 0)
			switch_voltage_delay = i;
	}

	return count;
}

static DEVICE_ATTR(force_rx_delay_ctrl, S_IRUGO | S_IWUGO, other_show, other_store);
static DEVICE_ATTR(force_rx_delay, S_IRUGO | S_IWUGO, other_show, other_store);
static DEVICE_ATTR(switch_voltage_delay, S_IRUGO | S_IWUGO, other_show, other_store);

static ssize_t dmasz_show(struct device *dev,
		       struct device_attribute *attr, char *buf)
{
	int len = 0;

	struct platform_device *pdev = to_platform_device(dev);
	struct dw_mci *host = platform_get_drvdata(pdev);

	len = sprintf(buf,      "512K:     %lu\n", host->stats_dmasz_512k);
	len += sprintf(buf+len, "256K:     %lu\n", host->stats_dmasz_256k);
	len += sprintf(buf+len, "128K:     %lu\n", host->stats_dmasz_128k);
	len += sprintf(buf+len, " 64K:     %lu\n", host->stats_dmasz_64k);
	len += sprintf(buf+len, " 32K:     %lu\n", host->stats_dmasz_32k);
	len += sprintf(buf+len, "  4K:     %lu\n", host->stats_dmasz_4k);
	len += sprintf(buf+len, " oth:     %lu\n", host->stats_dmasz_other);
	return len;
}
static DEVICE_ATTR(stats_dmasz, S_IRUGO , dmasz_show, NULL);


static int proc_init (struct platform_device *pdev) 
{
	device_create_file(&pdev->dev, &dev_attr_stats_dmasz);

	device_create_file(&pdev->dev, &dev_attr_tx_delay_ddr50);
	device_create_file(&pdev->dev, &dev_attr_tx_delay_sdr104);
	device_create_file(&pdev->dev, &dev_attr_tx_delay_sdr50);
	device_create_file(&pdev->dev, &dev_attr_tx_delay_sdr25);
	device_create_file(&pdev->dev, &dev_attr_tx_delay_sdr12);
	device_create_file(&pdev->dev, &dev_attr_tx_delay_legacy);
	device_create_file(&pdev->dev, &dev_attr_tx_delay_sdhs);
	device_create_file(&pdev->dev, &dev_attr_tx_delay_mmchs);
	device_create_file(&pdev->dev, &dev_attr_tx_delay_mmchs200);

	device_create_file(&pdev->dev, &dev_attr_clk_in_ddr50);
	device_create_file(&pdev->dev, &dev_attr_clk_in_sdr104);
	device_create_file(&pdev->dev, &dev_attr_clk_in_sdr50);
	device_create_file(&pdev->dev, &dev_attr_clk_in_sdr25);
	device_create_file(&pdev->dev, &dev_attr_clk_in_sdr12);
	device_create_file(&pdev->dev, &dev_attr_clk_in_legacy);
	device_create_file(&pdev->dev, &dev_attr_clk_in_sdhs);
	device_create_file(&pdev->dev, &dev_attr_clk_in_mmchs);
	device_create_file(&pdev->dev, &dev_attr_clk_in_mmchs200);

	device_create_file(&pdev->dev, &dev_attr_slew_rate_ddr50);
	device_create_file(&pdev->dev, &dev_attr_slew_rate_sdr104);
	device_create_file(&pdev->dev, &dev_attr_slew_rate_sdr50);
	device_create_file(&pdev->dev, &dev_attr_slew_rate_sdr25);
	device_create_file(&pdev->dev, &dev_attr_slew_rate_sdr12);
	device_create_file(&pdev->dev, &dev_attr_slew_rate_legacy);
	device_create_file(&pdev->dev, &dev_attr_slew_rate_sdhs);
	device_create_file(&pdev->dev, &dev_attr_slew_rate_mmchs);
	device_create_file(&pdev->dev, &dev_attr_slew_rate_mmchs200);

	device_create_file(&pdev->dev, &dev_attr_driving_strength_ddr50);
	device_create_file(&pdev->dev, &dev_attr_driving_strength_sdr104);
	device_create_file(&pdev->dev, &dev_attr_driving_strength_sdr50);
	device_create_file(&pdev->dev, &dev_attr_driving_strength_sdr25);
	device_create_file(&pdev->dev, &dev_attr_driving_strength_sdr12);
	device_create_file(&pdev->dev, &dev_attr_driving_strength_legacy);
	device_create_file(&pdev->dev, &dev_attr_driving_strength_sdhs);
	device_create_file(&pdev->dev, &dev_attr_driving_strength_mmchs);
	device_create_file(&pdev->dev, &dev_attr_driving_strength_mmchs200);


	device_create_file(&pdev->dev, &dev_attr_force_rx_delay_ctrl);
	device_create_file(&pdev->dev, &dev_attr_force_rx_delay);
	device_create_file(&pdev->dev, &dev_attr_switch_voltage_delay);

	return 0;
}
static void proc_cleanup(struct platform_device * pdev) 
{
	device_remove_file(&pdev->dev, &dev_attr_tx_delay_ddr50);
	device_remove_file(&pdev->dev, &dev_attr_tx_delay_sdr104);
	device_remove_file(&pdev->dev, &dev_attr_tx_delay_sdr50);
	device_remove_file(&pdev->dev, &dev_attr_tx_delay_sdr25);
	device_remove_file(&pdev->dev, &dev_attr_tx_delay_sdr12);
	device_remove_file(&pdev->dev, &dev_attr_tx_delay_legacy);
	device_remove_file(&pdev->dev, &dev_attr_tx_delay_sdhs);
	device_remove_file(&pdev->dev, &dev_attr_tx_delay_mmchs);
	device_remove_file(&pdev->dev, &dev_attr_tx_delay_mmchs200);

	device_remove_file(&pdev->dev, &dev_attr_clk_in_ddr50);
	device_remove_file(&pdev->dev, &dev_attr_clk_in_sdr104);
	device_remove_file(&pdev->dev, &dev_attr_clk_in_sdr50);
	device_remove_file(&pdev->dev, &dev_attr_clk_in_sdr25);
	device_remove_file(&pdev->dev, &dev_attr_clk_in_sdr12);
	device_remove_file(&pdev->dev, &dev_attr_clk_in_legacy);
	device_remove_file(&pdev->dev, &dev_attr_clk_in_sdhs);
	device_remove_file(&pdev->dev, &dev_attr_clk_in_mmchs);
	device_remove_file(&pdev->dev, &dev_attr_clk_in_mmchs200);

	device_remove_file(&pdev->dev, &dev_attr_slew_rate_ddr50);
	device_remove_file(&pdev->dev, &dev_attr_slew_rate_sdr104);
	device_remove_file(&pdev->dev, &dev_attr_slew_rate_sdr50);
	device_remove_file(&pdev->dev, &dev_attr_slew_rate_sdr25);
	device_remove_file(&pdev->dev, &dev_attr_slew_rate_sdr12);
	device_remove_file(&pdev->dev, &dev_attr_slew_rate_legacy);
	device_remove_file(&pdev->dev, &dev_attr_slew_rate_sdhs);
	device_remove_file(&pdev->dev, &dev_attr_slew_rate_mmchs);
	device_remove_file(&pdev->dev, &dev_attr_slew_rate_mmchs200);

	device_remove_file(&pdev->dev, &dev_attr_driving_strength_ddr50);
	device_remove_file(&pdev->dev, &dev_attr_driving_strength_sdr104);
	device_remove_file(&pdev->dev, &dev_attr_driving_strength_sdr50);
	device_remove_file(&pdev->dev, &dev_attr_driving_strength_sdr25);
	device_remove_file(&pdev->dev, &dev_attr_driving_strength_sdr12);
	device_remove_file(&pdev->dev, &dev_attr_driving_strength_legacy);
	device_remove_file(&pdev->dev, &dev_attr_driving_strength_sdhs);
	device_remove_file(&pdev->dev, &dev_attr_driving_strength_mmchs);
	device_remove_file(&pdev->dev, &dev_attr_driving_strength_mmchs200);


	device_remove_file(&pdev->dev, &dev_attr_force_rx_delay_ctrl);
	device_remove_file(&pdev->dev, &dev_attr_force_rx_delay);
	device_remove_file(&pdev->dev, &dev_attr_switch_voltage_delay);

}
static const struct dw_mci_drv_data pesaro_drv_data = {
	.init                   = dw_mci_pesaro_priv_init,
	.setup_clock            = dw_mci_pesaro_setup_clock,
	.parse_dt		= dw_mci_pesaro_parse_dt,
	.execute_tuning         = pesaro_execute_tuning,
	.voltage_switch         = pesaro_voltage_switch,
	.set_ios		= dw_mci_pesaro_set_ios,
};

static const struct of_device_id dw_mci_pesaro_match[] = {
	{ .compatible = "vpl,pesaro-dw_mshc",
			.data = &pesaro_drv_data, },
	{},
};
MODULE_DEVICE_TABLE(of, dw_mci_pesaro_match);

static int dw_mci_pesaro_probe(struct platform_device *pdev)
{
	const struct dw_mci_drv_data *drv_data;
	const struct of_device_id *match;

	match = of_match_node(dw_mci_pesaro_match, pdev->dev.of_node);
	drv_data = match->data;

	request_mem_region(SYSC_BASE, 0x100, "SYSC-MMC");
	sysc_mmc_base = ioremap(SYSC_BASE, 0x100);

	proc_init(pdev);

	return dw_mci_pltfm_register(pdev, drv_data);
}
static int dw_mci_pesaro_remove(struct platform_device *pdev)
{
	release_mem_region(SYSC_BASE,0x100);
	iounmap(sysc_mmc_base);

	proc_cleanup(pdev);

	dw_mci_pltfm_remove(pdev);
	return 0;
}

static struct platform_driver dw_mci_pesaro_pltfm_driver = {
	.probe		= dw_mci_pesaro_probe,
	.remove		= dw_mci_pesaro_remove,
	.driver		= {
		.name		= "dwmmc_pesaro",
		.of_match_table	= dw_mci_pesaro_match,
	},
};

module_platform_driver(dw_mci_pesaro_pltfm_driver);

MODULE_LICENSE("GPL v2");
MODULE_ALIAS("platform:dwmmc-pesaro");
