#include <linux/module.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/ioport.h>
#include <linux/platform_device.h>
#include <linux/delay.h>
#include <linux/err.h>
#include <linux/slab.h>
#include <linux/clk.h>

#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/nand_ecc.h>
#include <linux/mtd/partitions.h>

#include <asm/io.h>
#include <asm/cacheflush.h>
#include <linux/interrupt.h>
#include <mach/hardware.h>
#include <linux/dma-mapping.h>
#include <linux/of.h>

//-------------------------------
/*
 ********************************************************
 *                 NFC MMRs
 ********************************************************
 */
#define VPL_NFC_MMR_CONF        (0x00)
#define VPL_NFC_MMR_CTRL        (0x04)
#define VPL_NFC_MMR_COMM        (0x08)
#define VPL_NFC_MMR_ADDR0L      (0x0c)
#define VPL_NFC_MMR_ADDR1L      (0x10)
#define VPL_NFC_MMR_DATA        (0x14)
#define VPL_NFC_MMR_PB0         (0x18)
#define VPL_NFC_MMR_STATE       (0x1c)
#define VPL_NFC_MMR_FLookUp0    (0x20)
#define VPL_NFC_MMR_FLookUp1    (0x24)
#define VPL_NFC_MMR_FLookUp2    (0x28)
#define VPL_NFC_MMR_FLookUp3    (0x2c)
#define VPL_NFC_MMR_FLookUp4    (0x30)
#define VPL_NFC_MMR_FLookUp5    (0x34)
#define VPL_NFC_MMR_FLookUp6    (0x38)
#define VPL_NFC_MMR_FLookUp7    (0x3c)
#define VPL_NFC_MMR_ECCState    (0x40)
#define VPL_NFC_MMR_ADDR0H      (0x44)
#define VPL_NFC_MMR_ADDR1H      (0x48)
#define VPL_NFC_MMR_LookUpEN    (0x4c)

//--NANDDMA MMR--
#define VPL_NFC_MMR_DMAADDR    (0x80)
#define VPL_NFC_MMR_DMACTRL    (0x84)
#define VPL_NFC_MMR_DMACNTR    (0x88)

//--NAND DMA SETTING
#define VPL_NFC_ADDR_MASK        0x0fffffff
#define VPL_NFC_DMA_ENABLE       0x01000000
#define VPL_NFC_DMA_DIR_BUF2AHB  0x10000
#define VPL_NFC_DMA_DIR_AHB2BUF  0x0
#define VPL_NFC_DMA_SIZE         (0 << 14)
#define VPL_NFC_DMA_BURST        (2 << 11)

//--NAND Useful Flag
#define VPL_NF_TRANSFER_READ  0x01
#define VPL_NF_TRANSFER_WRITE 0x00

//--NAND Errors
#define VPL_NF_ERR_NO_ERRORS                0x00u
#define VPL_NF_ERR_INVALID_PARAMETER        0x01u /** error - invalid parameter */
#define VPL_NF_ERR_INVALID_ADDRESS          0x03u /** error - invalid address */
#define VPL_NF_ERR_DMA_TRANSFER             0x04u /** DMA transmission error */
#define VPL_NF_ERR_MEMORY_BUSY              0x07u /** error - memory doesn't response on request */
#define VPL_NF_ERR_PROTECTED                0x08u /** error - try Write or Erase protected area */
#define VPL_NF_ERR_ECC_CODE_WRONG           0x0Au /** error - read data are corrupted */
#define VPL_NF_ERR_CURR_PAGE_OP_ERR         0x0Eu
#define VPL_NF_ERR_PRIOR_CURR_PAGE_OP_FAIL  0x10u /** erase/program operation of current and prior page/block failed */

#define BRC_MEM_BASE            0x40000000 //0x00000000 befor remap
#define SRAM_CTRL_MMR_BASE      (0x1800008)//0x4004)
#define SRAM_MMR_BASE           (0x1000000)//0x1000)

#define CAST_BUF_SIZE		(4096 + 96)
struct nand_buf {
	int head;
	int tail;
	uint8_t buf[CAST_BUF_SIZE];
	dma_addr_t addr;
};
struct cast_nfc_nand {
	struct mtd_info		mtd;

	void __iomem		*io;
	int				irq;

	void __iomem		*brc;	
	void __iomem		*buf;		
	struct nand_buf dmabuf;
	//chip settings
	u32 nf_chip_timing;// micron : 0x01860822; samsung : 0x01020821
	//DMA-mode / PIO-mode
	int dma_mode;

	int buf_mode_rw_op;
	unsigned long pageaddr;
	unsigned long status;
	struct mutex nand_erasecheck_mutex;
	unsigned char* nand_erasecheck_pagebuf;
};

#define mtd_to_castnfc(_mtd)	container_of(_mtd, struct cast_nfc_nand, mtd)
/*------------------------------------------------------
 *                    Global Variables Part
 *------------------------------------------------------
 * We put global variables, macros, and structures in this part.
 */
#define cast_writel(dev, reg, value)			\
	__raw_writel((value), (dev)->io + reg)
#define cast_readl(dev, reg)			\
	__raw_readl((dev)->io + reg)


#ifdef CONFIG_MTD_CMDLINE_PARTS
static const char *part_probes[] = { "cmdlinepart", NULL };
#endif

DECLARE_COMPLETION(NFC_TRANSFER_COMPLETE);

/*------------------------------------------------------
 *                    redundant part
 *------------------------------------------------------
 * We declare these redundant functions because when HW ECC mode is enabled,
 * kernel would check if these functions are already implemented.
 * But in our hw ecc mode, we do not use these functions.
 */
int	vpl_nand_calculate(struct mtd_info *mtd, const uint8_t *dat, uint8_t *ecc_code)
{
	printk("[jam ERR] Our NFC does not need this, you should not call vpl_nand_calculate()\n");
	return -1;
}

int vpl_nand_correct(struct mtd_info *mtd, uint8_t *dat, uint8_t *read_ecc, uint8_t *calc_ecc)
{
	printk("[jam ERR] Our NFC does not need this, you should not call vpl_nand_correct()\n");
	return -1;
}

void vpl_nand_hwctl(struct mtd_info *mtd, int mode)
{
	printk("[jam ERR] Our NFC does not need this, you should not call vpl_nand_hwctl()\n");
	return;
}

static void vpl_nand_hwcontrol(struct mtd_info *mtd, int cmd, unsigned int ctrl)
{
	printk("[jam ERR] Our NFC does not need this, you should not call vpl_nand_hwcontrol()\n");
	return;
}

/*------------------------------------------------------
 *                    Misc Part
 *------------------------------------------------------
 * This is a important part. We collect set functions, command function, and status functions here.
 */
#define NF_FLCTRL_BLOCK_64PAGES  (0 << 12)
#define NF_FLCTRL_BLOCK_128PAGES (1 << 12)
#define NF_FLCTRL_PAGESIZE_4KB   (0 << 11)
#define NF_FLCTRL_PAGESIZE_2KB   (1 << 11)
#define NF_FLCTRL_ECC_DISABLE    (0 <<  9)
#define NF_FLCTRL_ECC_ENABLE     (1 <<  9)
#define NF_FLCTRL_GLOBAL_INT_DISABLE (0 << 8)
#define NF_FLCTRL_GLOBAL_INT_ENABLE  (1 << 8)
#define NF_FLCTRL_DMA_TRIGGER_DISABLE (0 << 3)
#define NF_FLCTRL_DMA_TRIGGER_ENABLE  (1 << 3)
#define NF_FLCTRL_TRANSFER_COMPLETE_INT_DISABLE (0 << 2)
#define NF_FLCTRL_TRANSFER_COMPLETE_INT_ENABLE  (1 << 2)
#define NF_FLCTRL_ADDRCYCLE_4    (0 <<  0)
#define NF_FLCTRL_ADDRCYCLE_5    (1 <<  0)

void NFC_setDmaTriggerMode(struct cast_nfc_nand* cast_nand)
{
    volatile unsigned long flctrl;

    if(!(cast_nand->dma_mode)) {
        printk("[ERR] It means that you cannot call this function when not PAGEPROG or PAGEREAD in dma mode!!\n");
        return;
    }

    flctrl = cast_readl(cast_nand, VPL_NFC_MMR_CTRL);
    flctrl |= NF_FLCTRL_GLOBAL_INT_ENABLE;
    flctrl |= NF_FLCTRL_TRANSFER_COMPLETE_INT_ENABLE;
    flctrl |= NF_FLCTRL_DMA_TRIGGER_ENABLE;
    cast_writel(cast_nand, VPL_NFC_MMR_CTRL, flctrl);
}

void vpl_nand_select_chip(struct mtd_info *mtd, int chipnumber)
{
    /* We not only select the chip, but also config it. */

	struct nand_chip *chip = mtd->priv;
	struct cast_nfc_nand *cast_nand = mtd_to_castnfc(mtd);
	unsigned long setting = 0x0;

	//In usual case, we only use at most 2 nand flash chips.
	//So we assign 3 to chip-select field when dis-select.
	if (chipnumber < 0) {
		chipnumber = 0x3;
		setting = (chipnumber << 16);
		cast_writel(cast_nand, VPL_NFC_MMR_CTRL, setting);
		return;
	}
	//step1. set timing
	cast_writel(cast_nand, VPL_NFC_MMR_CONF, cast_nand->nf_chip_timing);

	//setp2. set ctrl settings
	setting |= (chipnumber << 16) | NF_FLCTRL_ECC_ENABLE | (1 << 1);
	if (mtd->writesize == 2048) {
		setting |= NF_FLCTRL_PAGESIZE_2KB;
	}
	else if (mtd->writesize == 4096) {
		setting |= NF_FLCTRL_PAGESIZE_4KB;
	}

	//step3. set cycle#
	if (chip->chipsize <= (128 << 20)) {
		setting |= NF_FLCTRL_ADDRCYCLE_4;
	}
	else {
		setting |= NF_FLCTRL_ADDRCYCLE_5;
	}

	//step4. write setting
	cast_writel(cast_nand, VPL_NFC_MMR_CTRL, setting);

	return;
}

//VPL NFC status bit field
#define VPL_NFC_RnB_READY 0x2
#define VPL_NFC_RnB_BUSY_MASK  (0x60 | 0X2)
#define VPL_NFC_DMA_BUSY_MASK  0x80
static int vpl_nand_dev_ready(struct mtd_info *mtd)
{
    struct cast_nfc_nand *cast_nand = mtd_to_castnfc(mtd);

    //Because TRANSFER-COMPLETE-INTERRUPT is only triggered when :
    //  1. READ/WRITE to nand flash chip(the operation must operate the nand flash chip directly)
    //  2. the command must be PageRead or PageProgram
    //So we should add "else condition" to handle other cases.
	if ((cast_nand->buf_mode_rw_op == 1) && (cast_nand->dma_mode)) {
		wait_for_completion(&NFC_TRANSFER_COMPLETE);
		cast_nand->buf_mode_rw_op = 0;
		return 1;
	}
	else {
		volatile unsigned long status = cast_readl(cast_nand, VPL_NFC_MMR_STATE);
		return ((status & VPL_NFC_RnB_BUSY_MASK) == VPL_NFC_RnB_READY);
	}
}

static int vpl_nand_program_erase_waitfunc(struct mtd_info *mtd, struct nand_chip *chip)
{
    while (1) {
        if (chip->dev_ready(mtd))
            break;
    }

    chip->cmdfunc(mtd, NAND_CMD_STATUS, -1, -1);
    return ((int)chip->read_byte(mtd));
}

static void vpl_nand_command_waitfunc(struct mtd_info *mtd)
{
    nand_wait_ready(mtd);
}

static void vpl_nand_dma_ready(struct cast_nfc_nand *cast_nand)
{
	volatile unsigned long status;
	do {
		status = cast_readl(cast_nand, VPL_NFC_MMR_STATE);
	} while((status & VPL_NFC_DMA_BUSY_MASK) == VPL_NFC_DMA_BUSY_MASK);
}

//TODO: phase-out NFC_set_status()
static long NFC_set_status(int set, unsigned long val)
{
    static unsigned long status = 0;
    if(set) {
        status = val;
    }

    return status;
}

static void vpl_nand_internalbuf_CPUMode(struct cast_nfc_nand* cast_nand)
{
	__raw_writel(0, cast_nand->brc);
	return;
}

static void vpl_nand_internalbuf_NFCMode(struct cast_nfc_nand* cast_nand)
{
  	__raw_writel(1, cast_nand->brc);
	return;
}

static void vpl_nand_cmd_raw_read(struct mtd_info *mtd, int column, int page_num)
{
	register struct nand_chip *chip = mtd->priv;


	struct cast_nfc_nand *cast_nand = mtd_to_castnfc(mtd);
	cast_nand->buf_mode_rw_op = 0;

	// clear status
	cast_writel(cast_nand, VPL_NFC_MMR_STATE, 0);

	// write address
	cast_writel(cast_nand, VPL_NFC_MMR_ADDR0L, column | (page_num << (chip->page_shift + 1)));	// We should add 1 to chip->page_shift
	cast_writel(cast_nand, VPL_NFC_MMR_ADDR0H, (chip->chipsize > (128 << 20)) ? (page_num >> 16) : 0);	// One more address cycle for devices > 128MiB

	cast_writel(cast_nand, VPL_NFC_MMR_COMM, NAND_CMD_PAGEREAD1);	//[jam] we use page-read#1 here

	//make sure that the action is completed.
	vpl_nand_command_waitfunc(mtd);
	cast_writel(cast_nand, VPL_NFC_MMR_STATE, 0);
	vpl_nand_internalbuf_CPUMode(cast_nand);

	return;
}

static int vpl_nand_read_page_raw(struct mtd_info *mtd, struct nand_chip *chip,
			      uint8_t *buf, int oob_required, int page)
{
       vpl_nand_cmd_raw_read(mtd, 0x00, page);//give addr and command
	chip->read_buf(mtd, buf, mtd->writesize);
	if (oob_required)
		chip->read_buf(mtd, chip->oob_poi, mtd->oobsize);
	return 0;
}
static void vpl_nand_command(struct mtd_info *mtd, unsigned int command, int column, int page_num)
{
	register struct nand_chip *chip = mtd->priv;
	unsigned long low_addr = 0x0, high_addr = 0x0;

	struct cast_nfc_nand *cast_nand = mtd_to_castnfc(mtd);
	cast_nand->buf_mode_rw_op = 0;

	if (command == NAND_CMD_ERASE2)
		return;
	if (command == NAND_CMD_READ0) {
		if (cast_nand->dma_mode)
			return;
		else
			command = NAND_CMD_PAGEREAD_BUFMODE;
	}
	/* Emulate NAND_CMD_READOOB */
	if (command == NAND_CMD_READOOB) {
		vpl_nand_cmd_raw_read(mtd, column+mtd->writesize, page_num);
		return;
	}

	// Step0. Clear status
	cast_writel(cast_nand, VPL_NFC_MMR_STATE, 0);

	// Step1. Write Address
	if (column != -1 || page_num != -1) {
		/* Serially input address */
		if (column != -1)
			low_addr = column;

		if (page_num != -1) {
			// We should add 1 to chip->page_shift
			low_addr = low_addr | (page_num << (chip->page_shift + 1));

			// One more address cycle for devices > 128MiB
			if (chip->chipsize > (128 << 20))
				high_addr = high_addr | (page_num >> 16);			
		}

		// READ and PROGRAM uses different sets of address registers
		if (command == NAND_CMD_PAGEREAD_BUFMODE) {
			cast_writel(cast_nand, VPL_NFC_MMR_ADDR0L, low_addr);
			cast_writel(cast_nand, VPL_NFC_MMR_ADDR0H, high_addr);
		}
		else if ((command == NAND_CMD_PAGEPROG1) || (command == NAND_CMD_ERASE1) || (command == NAND_CMD_PAGEPROG_BUFMODE)) {
			//printk(KERN_DEBUG "wlo = 0x%lx, whi = 0x%lx\n", low_addr, high_addr);
			cast_writel(cast_nand, VPL_NFC_MMR_ADDR1L, low_addr);
			cast_writel(cast_nand, VPL_NFC_MMR_ADDR1H, high_addr);
		}
	}

	//Step2. Write command
	if ((command == NAND_CMD_PAGEREAD_BUFMODE) || (command == NAND_CMD_PAGEPROG_BUFMODE)) {
		vpl_nand_internalbuf_NFCMode(cast_nand);
		cast_nand->buf_mode_rw_op = 1;
	}
	if (command == NAND_CMD_PAGEPROG_BUFMODE)
		command = NAND_CMD_SEQIN;
	cast_writel(cast_nand, VPL_NFC_MMR_COMM, command);

	//Step3. Make sure that the action is completed.
	switch (command) {
	case NAND_CMD_CACHEDPROG: //[jam trace] it is cancelled now, see comments in nand_write_page()
	case NAND_CMD_RNDIN:
	case NAND_CMD_ERASE2:
	case NAND_CMD_SEQIN:
		return;

	case NAND_CMD_PAGEPROG:
	case NAND_CMD_PAGEPROG1:
	case NAND_CMD_STATUS:
	case NAND_CMD_ERASE1:
		return;

	case NAND_CMD_RESET:
		if (chip->dev_ready)
			break;
		udelay(chip->chip_delay);
		vpl_nand_command_waitfunc(mtd);//defined in nand_base.c
		cast_writel(cast_nand, VPL_NFC_MMR_STATE, 0);
		return;

	case NAND_CMD_PAGEREAD1:
	case NAND_CMD_PAGEREAD_BUFMODE:
		vpl_nand_command_waitfunc(mtd);//defined in nand_base.c
		cast_writel(cast_nand, VPL_NFC_MMR_STATE, 0);
		vpl_nand_internalbuf_CPUMode(cast_nand);
		return;

	/* This applies to read commands */
	default:
	/*
	* If we don't have access to the busy pin, we apply the given
	* command delay
	*/
		if (!chip->dev_ready) {
			udelay(chip->chip_delay);
			return;
		}
	}

	/* Apply this short delay always to ensure that we do wait tWB in
	 * any case on any machine. */
	ndelay(100);
}

/*------------------------------------------------------
 *                    Internal-DMA part
 *------------------------------------------------------
 */

// We read/write data from/to internal buffer at address 0
static void vpl_nand_dma_set_counter(struct cast_nfc_nand *cast_nand,  unsigned long counter)
{
	volatile unsigned long reg = cast_readl(cast_nand, VPL_NFC_MMR_DMACNTR);
	counter = counter & 0xffff;
	reg = (reg & 0x0000ffff) | (counter << 16);
	cast_writel(cast_nand, VPL_NFC_MMR_DMACNTR, reg);
	return;
}

static void vpl_nand_dma_set_srambase(struct cast_nfc_nand *cast_nand, unsigned long addr)
{
	volatile unsigned long reg = cast_readl(cast_nand, VPL_NFC_MMR_DMACNTR);
	addr = addr & 0xffff;
	reg = (reg & 0xffff0000) | addr;
	cast_writel(cast_nand, VPL_NFC_MMR_DMACNTR, reg);
	return;
}

#define VPL_NFC_DMA_BURSTTYPE    (5 << 11) 
#define VPL_NFC_DMA_TRANSFERSIZE (3 << 14) //32-bits transfer size
#define VPL_NFC_DMA_TRANSFER_AHB2SRAM (0 << 16)
#define VPL_NFC_DMA_TRANSFER_SRAM2AHB (1 << 16)
#define VPL_NFC_DMA_DMA_ENABLE   (1 << 24)
void vpl_nand_dma_start_transfer(struct cast_nfc_nand *cast_nand, unsigned long dir)
{
	unsigned long val = VPL_NFC_DMA_BURSTTYPE | VPL_NFC_DMA_TRANSFERSIZE | dir | VPL_NFC_DMA_DMA_ENABLE;
	cast_writel(cast_nand, VPL_NFC_MMR_DMACTRL, val);
	return;
}

void vpl_nand_dma_end_transfer(struct cast_nfc_nand *cast_nand)
{
	cast_writel(cast_nand, VPL_NFC_MMR_DMACTRL, 0);
	vpl_nand_internalbuf_CPUMode(cast_nand);
	return;
}

int vpl_nand_dma_error(struct cast_nfc_nand *cast_nand)
{
	volatile unsigned long reg = cast_readl(cast_nand, VPL_NFC_MMR_DMACTRL);
	return ((reg & 0x2) == 0x2) ? 1 : 0;//0x2 means dma error
}

int vpl_nand_dma_transfer(struct mtd_info *mtd, u32 busaddr, unsigned long sram_addr, u32 len, unsigned long dir, int oobmode)
{
	struct nand_chip *chip = mtd->priv;
	struct cast_nfc_nand *cast_nand = mtd_to_castnfc(mtd);
	unsigned long val, status;

	//reset
	vpl_nand_internalbuf_NFCMode(cast_nand);
	cast_writel(cast_nand, VPL_NFC_MMR_DMACTRL, 0);

	//wait for dma ready
	vpl_nand_dma_ready(cast_nand);

	//start dma transferring...
	cast_writel(cast_nand, VPL_NFC_MMR_DMAADDR, busaddr);
	// [Note] the unit of dma counter is data size
	vpl_nand_dma_set_counter(cast_nand, (unsigned long)(len/4)); // #define VPL_NFC_DMA_TRANSFERSIZE (3 << 14) //32-bits transfer size
	vpl_nand_dma_set_srambase(cast_nand, sram_addr);

	//Because we will move "data-part" between buffer and nand flash chip,
	//but we will read "oob-part" to the buffer of struct nand_chip.
	//And because the data of oob-part is few, we can just use dma-polling-mode to handle this.
	if (!oobmode) {
		NFC_setDmaTriggerMode(cast_nand);
		val = VPL_NFC_DMA_BURSTTYPE | VPL_NFC_DMA_TRANSFERSIZE | dir;
		cast_writel(cast_nand, VPL_NFC_MMR_DMACTRL, val);

		if (dir == VPL_NFC_DMA_TRANSFER_SRAM2AHB) {
			vpl_nand_command(mtd, NAND_CMD_PAGEREAD_BUFMODE, 0x00, cast_nand->pageaddr);//give addr and command
		} else if (dir == VPL_NFC_DMA_TRANSFER_AHB2SRAM) {
			vpl_nand_command(mtd, NAND_CMD_PAGEPROG_BUFMODE, 0x00, cast_nand->pageaddr);//give addr and command
			status = (unsigned long)vpl_nand_program_erase_waitfunc(mtd, chip);
		} else {
			printk("[ERROR] Wrong direction of dma transfer!\n");
		}
	}
	else {
		vpl_nand_dma_start_transfer(cast_nand, dir);

		//wait for dma ready
		vpl_nand_dma_ready(cast_nand);

		if (vpl_nand_dma_error(cast_nand)) {
			if (dir == VPL_NFC_DMA_TRANSFER_AHB2SRAM) {
				printk("[ERR] NFC-DMA Transfer from AHB to SRAM failed!!\n");
			}
			else {
				printk("[ERR] NFC-DMA Transfer from SRAM to AHB failed!!\n");
			}
			return -1;
		}
	}

	//clear status and disable dma
	vpl_nand_dma_end_transfer(cast_nand);
	cast_writel(cast_nand, VPL_NFC_MMR_STATE, 0);

	return 0;
}

/*------------------------------------------------------
 *                    read part
 *------------------------------------------------------
 */
/*
 * This function is only for READ-OPERATION.
 */
static int vpl_nand_read_eccstatus(struct mtd_info *mtd)
{
	struct cast_nfc_nand *cast_nand = mtd_to_castnfc(mtd);
	struct nand_chip *chip = mtd->priv;
	int i;
	unsigned long mask;
	volatile unsigned long eccstatus, error_flag, error_correct;

	mask = 1 << (mtd->writesize / 512);
	mask -= 1;

	eccstatus = cast_readl(cast_nand, VPL_NFC_MMR_ECCState);
	error_flag = (eccstatus >> 16) & mask;
	error_correct = eccstatus & mask;
	error_correct &= error_flag; // the only valid bits are those for which corresponding ecc error bits are set.

	if (likely(error_flag == 0)) {	// No Error Occurs
		return 0;
	}
	else {	// Error Occurs
		/* Error Corrected Successfully */
		if (error_flag == error_correct) {
			int n = 0;
			while(error_correct != 0) {
				n += (error_correct & 1);
				error_correct >>= 1;
			}
			printk("correct %d errors\n", n);
			return n;
		}
		else {
			/* check if it is a freshly-erased block
			 * (oob filled with 0xff), in this case the ECC will not come out right.
			 * We'll suppress the error and tell the caller everything's
			 * OK. Because it is. */
			int emptymatch = 1;
			for (i = 0; i < mtd->oobsize; i++) {
				if(chip->oob_poi[i] == 0xff)
					continue;
				emptymatch = 0;
				break;
			}
			/* if emptymatch == 1, the block has just been erased, return OK */
			return ((emptymatch) ? 0 : -1);
		}
	}
}

static int vpl_nand_ecc_read_page_hwecc(struct mtd_info *mtd, struct nand_chip *chip, uint8_t *buf, int oob_required, int page)
{
	struct cast_nfc_nand *cast_nand = mtd_to_castnfc(mtd);
	int stat;
	int i, j;
	unsigned long tmp;

	if (!(cast_nand->dma_mode)) {
		vpl_nand_internalbuf_CPUMode(cast_nand);

		i = 0;
		while(i < mtd->writesize) {
			tmp = __raw_readl(cast_nand->buf+i);
			for (j = 0; j < 4; j++) {
				unsigned long mask = 0xff;
				int shift = j * 8;
				buf[i] = (tmp & (mask << shift)) >> shift;
				i++;
			}
		}

		i = 0;
		while(i < mtd->oobsize) {
			tmp = __raw_readl(cast_nand->buf+mtd->writesize+i);
			for (j = 0; j < 4; j++) {
				unsigned long mask = 0xff;
				int shift = j * 8;
				chip->oob_poi[i] = (tmp & (mask << shift)) >> shift;
				i++;
			}
		}
	}
	else {
		dma_sync_single_for_device(NULL, cast_nand->dmabuf.addr, cast_nand->mtd.writesize + cast_nand->mtd.oobsize, DMA_FROM_DEVICE);
		cast_nand->pageaddr = page;
		vpl_nand_dma_transfer(mtd, cast_nand->dmabuf.addr, 0, mtd->writesize, VPL_NFC_DMA_TRANSFER_SRAM2AHB, 0);
		vpl_nand_dma_transfer(mtd, cast_nand->dmabuf.addr+mtd->writesize, mtd->writesize, mtd->oobsize, VPL_NFC_DMA_TRANSFER_SRAM2AHB, 1);
		dma_sync_single_for_cpu(NULL, cast_nand->dmabuf.addr, cast_nand->mtd.writesize + cast_nand->mtd.oobsize, DMA_FROM_DEVICE);
		memcpy(buf, cast_nand->dmabuf.buf , mtd->writesize);		
		memcpy(chip->oob_poi, cast_nand->dmabuf.buf+mtd->writesize , mtd->oobsize);				
	}

	stat = vpl_nand_read_eccstatus(mtd);
	if (stat == -1)
		mtd->ecc_stats.failed++;
	else
		mtd->ecc_stats.corrected += stat;
	return 0;
}


/*------------------------------------------------------
 *                    write part
 *------------------------------------------------------
 */

static int vpl_nand_ecc_write_page_hwecc(struct mtd_info *mtd, struct nand_chip *chip, const uint8_t *buf, int oob_required)
{
	struct cast_nfc_nand *cast_nand = mtd_to_castnfc(mtd);

	if (!(cast_nand->dma_mode)) {
		int i = 0, j = 0;
		unsigned long tmp;
		uint32_t *buf32 = (uint32_t *)buf;

		vpl_nand_internalbuf_CPUMode(cast_nand);

		while(i < mtd->writesize) {
			tmp = buf32[j];
			__raw_writel(tmp, cast_nand->buf + i);
			i+=4;
			j++;
		}
	}
	else{
		memcpy(cast_nand->dmabuf.buf, buf, mtd->writesize);
		dma_sync_single_for_device(NULL, cast_nand->dmabuf.addr, cast_nand->mtd.writesize + cast_nand->mtd.oobsize, DMA_TO_DEVICE);
		vpl_nand_dma_transfer(mtd, cast_nand->dmabuf.addr, (unsigned long)0, mtd->writesize, VPL_NFC_DMA_TRANSFER_AHB2SRAM, 0);
		dma_sync_single_for_cpu(NULL, cast_nand->dmabuf.addr, cast_nand->mtd.writesize + cast_nand->mtd.oobsize, DMA_TO_DEVICE);		
	}
	return 0;
}

static int vpl_nand_write_page(struct mtd_info *mtd, struct nand_chip *chip,
		uint32_t offset, int data_len, const uint8_t *buf, 
		int oob_required, int page, int cached, int raw)
{
	int status;
	struct cast_nfc_nand *cast_nand = mtd_to_castnfc(mtd);

	if(cast_nand->dma_mode) {
		if (unlikely(raw)) {
			chip->ecc.write_page_raw(mtd, chip, buf, oob_required);  //[jam]write data
			chip->cmdfunc(mtd, NAND_CMD_PAGEPROG_BUFMODE, 0x00, page);//[jam]write page command
		} else {
			cast_nand->pageaddr = page;
			chip->ecc.write_page(mtd, chip, buf, oob_required);
		}
		status = NFC_set_status(0, 0); //TODO: phase-out NFC_set_status()
	}
	else {
		if (unlikely(raw)) 
			chip->ecc.write_page_raw(mtd, chip, buf, oob_required);  //[jam]write data
		else 
			chip->ecc.write_page(mtd, chip, buf, oob_required);
		chip->cmdfunc(mtd, NAND_CMD_PAGEPROG_BUFMODE, 0x00, page);//[jam]write page command
		status = chip->waitfunc(mtd, chip);
	}

	/*
	* See if operation failed and additional status checks are
	* available
	*/
	if ((status & NAND_STATUS_FAIL) && (chip->errstat)) {
		status = chip->errstat(mtd, chip, FL_WRITING, status, page);
	}

	if (status & NAND_STATUS_FAIL) {
		return -EIO;
	}

	return 0;
}

/*------------------------------------------------------
 *                    remove part
 *------------------------------------------------------
 */
static int vpl_nand_remove(struct platform_device *pdev)
{
//TODO
	return 0;
}

/*------------------------------------------------------
 *                    ISR Part
 *------------------------------------------------------
 */
static irqreturn_t NFC_ISR(int int_num, void *dev_id)
{
	volatile unsigned long status;
	struct cast_nfc_nand* cast_nand = dev_id;

	status = cast_readl(cast_nand, VPL_NFC_MMR_STATE);
	if (((status&VPL_NFC_RnB_READY) == VPL_NFC_RnB_READY) &&
		((status&VPL_NFC_DMA_BUSY_MASK) != VPL_NFC_DMA_BUSY_MASK)) {
		// DMA reset
		vpl_nand_internalbuf_NFCMode(cast_nand);
		cast_writel(cast_nand, VPL_NFC_MMR_DMACTRL, 0);
		// DMA clear status
		cast_writel(cast_nand, VPL_NFC_MMR_STATE, 0);

		complete(&NFC_TRANSFER_COMPLETE);
		return IRQ_HANDLED;
	}
	return IRQ_NONE;
}


/*------------------------------------------------------ */
static int vpl_nand_probe(struct platform_device *pdev)
{
	struct nand_chip *chip = NULL;
	struct cast_nfc_nand *cast_nand = NULL;
	struct device *dev = &pdev->dev;
	struct mtd_part_parser_data ppdata;
	struct device_node *np = dev->of_node;
	struct resource *res;
	int err = 0;

	printk("evetest~ vpl_nand_probe! refinement alpha version\n");

	cast_nand = kzalloc(sizeof(struct cast_nfc_nand), GFP_KERNEL);
	if (cast_nand == NULL) {
		printk("Unable to allocate CAST NFC NAND MTD device structure.\n");
		return -ENOMEM;
	}

	chip = kzalloc(sizeof(struct nand_chip), GFP_KERNEL);
	if (chip == NULL) {
		printk("Unable to allocate CAST NFC NAND chip device structure.\n");
		kfree(cast_nand);
		return -ENOMEM;
	}

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		dev_err(&pdev->dev, "no io memory resource defined!\n");
		err = -ENODEV;
		goto err_get_res;
	}

	/* map physical address */
	cast_nand->io = ioremap(res->start, resource_size(res));
	if (!cast_nand->io) {
		printk("ioremap to access CAST NFC NAND chip failed\n");
		err = -EIO;
		goto err_ioremap;
	}
	cast_nand->brc = ioremap(BRC_MEM_BASE+SRAM_CTRL_MMR_BASE,  SZ_4K);
	if (!cast_nand->brc) {
		printk("ioremap to access CAST NFC NAND chip BRC failed\n");
		err = -EIO;
		goto err_ioremap;
	}
	cast_nand->buf = ioremap(BRC_MEM_BASE+SRAM_MMR_BASE,  SZ_8K);
	if (!cast_nand->buf) {
		printk("ioremap to access CAST NFC NAND chip SRAM buffer failed\n");
		err = -EIO;
		goto err_ioremap;
	}
	cast_nand->dmabuf.addr = dma_map_single(NULL, cast_nand->dmabuf.buf, CAST_BUF_SIZE, DMA_BIDIRECTIONAL);
	if (dma_mapping_error(NULL, cast_nand->dmabuf.addr)) {
		printk("cast_nfc: failed to map DMA buffer\n");
		err = -EIO;
		goto err_dmabuf;
	}

	/* Link the private data with the MTD structure */
	cast_nand->mtd.priv = chip;
	cast_nand->mtd.owner = THIS_MODULE;

	platform_set_drvdata(pdev, cast_nand);

	//struct mtd_info init
	cast_nand->mtd.numeraseregions = 0;//fixed erase size

	//struct nand_chip init
	chip->options |= NAND_NO_SUBPAGE_WRITE;//[jam test]NAND_BBT_SCANEMPTY for testing...
	chip->bbt_options |=  NAND_BBT_USE_FLASH | NAND_BBT_NO_OOB_BBM | NAND_BBT_NO_OOB;//[jam test]NAND_BBT_SCANEMPTY for testing...	
	chip->IO_ADDR_W = cast_nand->io + VPL_NFC_MMR_DATA;
	chip->IO_ADDR_R = chip->IO_ADDR_W;

	chip->select_chip = vpl_nand_select_chip;
	chip->cmdfunc = vpl_nand_command;
	chip->dev_ready = vpl_nand_dev_ready;
	chip->cmd_ctrl = vpl_nand_hwcontrol;//This is used to issue command and address cycles to the chip (CLE/ALE)
	chip->write_page = vpl_nand_write_page;
	chip->waitfunc = vpl_nand_program_erase_waitfunc;

	chip->ecc.read_page_raw = vpl_nand_read_page_raw;
	chip->ecc.read_page = vpl_nand_ecc_read_page_hwecc;
	chip->ecc.write_page = vpl_nand_ecc_write_page_hwecc;
	chip->ecc.calculate = vpl_nand_calculate;
	chip->ecc.correct = vpl_nand_correct;
	chip->ecc.hwctl = vpl_nand_hwctl;
	chip->ecc.mode = NAND_ECC_HW;
	chip->ecc.bytes= 12;
	chip->ecc.size = 512;
	chip->ecc.strength = 8;

	//struct cast_nfc_nand init
	err = of_property_read_u32(np, "timings", &(cast_nand->nf_chip_timing));	
	if (err)
		goto err_timing;
	mutex_init(&cast_nand->nand_erasecheck_mutex);

	/*
	* New patched in latest version of kernel
	* Reset the chip, required by some chips (e.g. Micron MT29FxGxxxxx)
	* after power-up.(Before calling nand_get_flash_type())
	*/
	//chip->cmdfunc(mtd, NAND_CMD_RESET, 0x00, -1);

	//[patch add]
	//Because in nand_scan_bbt(), it will use vmalloc() to alloc a buffer > 128KB.
	//DMA cannot handle its un-continutive space, so we use CPUMode.
	cast_nand->dma_mode = 0;
	err = nand_scan(&cast_nand->mtd, 1);
	if(err) 
		goto err_scan;
	
	cast_nand->nand_erasecheck_pagebuf = kmalloc(cast_nand->mtd.writesize, GFP_KERNEL);
	if(!cast_nand->nand_erasecheck_pagebuf) {
		err = -ENOMEM;
		goto err_alloc_erasebuf;
	}
		
#if defined CONFIG_MTD_NAND_CAST_DMAMODE
	cast_nand->dma_mode = 1;
#else
	cast_nand->dma_mode = 0;
#endif
	cast_nand->irq = platform_get_irq(pdev, 0);
	if (cast_nand->irq < 0) {
		dev_err(&pdev->dev, "failed to get platform irq\n");
		err = -EINVAL;
		goto err_irq_req;
	}

	if (request_irq(cast_nand->irq, (irq_handler_t)&NFC_ISR,
			IRQF_DISABLED, pdev->name, cast_nand)) {
		dev_err(&pdev->dev, "Error requesting NAND IRQ\n");
		err = -ENXIO;
		goto err_irq_req;
	}

	ppdata.of_node = np;  
	err = mtd_device_parse_register(&cast_nand->mtd, NULL, &ppdata, NULL, 0);	
	if (err)
		goto err_add;

	return 0;
	
err_add:
	nand_release(&cast_nand->mtd);
	free_irq(cast_nand->irq, cast_nand);
err_irq_req:
	kfree(cast_nand->nand_erasecheck_pagebuf);	
err_alloc_erasebuf:
err_scan:
	platform_set_drvdata(pdev, NULL);
err_timing:
	dma_unmap_single(NULL, cast_nand->dmabuf.addr, CAST_BUF_SIZE, DMA_BIDIRECTIONAL);	
err_dmabuf:
err_ioremap:
	if(cast_nand->buf)
		iounmap(cast_nand->buf);
	if(cast_nand->brc)
		iounmap(cast_nand->brc);
	if (cast_nand->io)
		iounmap(cast_nand->io);
err_get_res:
	kfree(chip);
	kfree(cast_nand);
	return err;
}


/*------------------------------------------------------
 *                    module part
 *------------------------------------------------------
 */
static const struct of_device_id cast_nand_dt_ids[] = {
	{ .compatible = "cast,cast-nfc", },
	{},
};

MODULE_DEVICE_TABLE(of, cast_nand_dt_ids);

static struct platform_driver cast_nfc_driver = {
	.remove		= vpl_nand_remove,
	.driver		= {
		.name	= "Pesaro-nand",
		.owner	= THIS_MODULE,
		.of_match_table	= of_match_ptr(cast_nand_dt_ids),		
	},
};

module_platform_driver_probe(cast_nfc_driver, vpl_nand_probe);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("James.Lin");
MODULE_DESCRIPTION("VPL NAND driver");
