/*******************************************************************************
  STMMAC Common Header File

  Copyright (C) 2007-2009  STMicroelectronics Ltd

  This program is free software; you can redistribute it and/or modify it
  under the terms and conditions of the GNU General Public License,
  version 2, as published by the Free Software Foundation.

  This program is distributed in the hope it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.

  The full GNU General Public License is included in this distribution in
  the file called "COPYING".

  Author: Giuseppe Cavallaro <peppe.cavallaro@st.com>
*******************************************************************************/

#ifndef __COMMON_H__
#define __COMMON_H__

#include <linux/etherdevice.h>
#include <linux/netdevice.h>
#include <linux/phy.h>
#include <linux/module.h>
#include <linux/init.h>
#if defined(CONFIG_VLAN_8021Q) || defined(CONFIG_VLAN_8021Q_MODULE)
//#define STMMAC_VLAN_TAG_USED
#include <linux/if_vlan.h>
#endif
#include "mmc.h"


#if 0
#define DRIVER_DEBUG
#endif

#ifdef DRIVER_DEBUG
#define PDEBUG(fmt, args...) pr_info(fmt, ## args)
#else
#define PDEBUG(x...) do { } while (0)
#endif

#undef CHIP_DEBUG_PRINT
/* Turn-on extra printk debug for MAC core, dma and descriptors */
/* #define CHIP_DEBUG_PRINT */

#ifdef CHIP_DEBUG_PRINT
#define CHIP_DBG(fmt, args...)  printk(fmt, ## args)
#else
#define CHIP_DBG(fmt, args...)  do { } while (0)
#endif

#undef FRAME_FILTER_DEBUG
/* #define FRAME_FILTER_DEBUG */

struct stmmac_extra_stats {
	/* Transmit errors */
	unsigned long tx_underflow ____cacheline_aligned;
	unsigned long tx_carrier;
	unsigned long tx_losscarrier;
	unsigned long vlan_tag;
	unsigned long tx_deferred;
	unsigned long tx_vlan;
	unsigned long tx_jabber;
	unsigned long tx_packet_flushed;
	unsigned long tx_payload_error;
	unsigned long tx_ip_header_error;
	/* Receive errors */
	unsigned long rx_desc;
	unsigned long sa_filter_fail;
	unsigned long overflow_error;
	unsigned long ipc_csum_error;
	unsigned long rx_collision;
	unsigned long rx_crc;
	unsigned long dribbling_bit;
	unsigned long rx_length;
	unsigned long rx_mii;
	unsigned long rx_multicast;
	unsigned long rx_gmac_overflow;
	unsigned long rx_watchdog;
	unsigned long da_rx_filter_fail;
	unsigned long sa_rx_filter_fail;
	unsigned long rx_missed_cntr;
	unsigned long rx_overflow_cntr;
	unsigned long rx_vlan;
	/* Tx/Rx IRQ errors */
	unsigned long tx_undeflow_irq;
	unsigned long tx_process_stopped_irq;
	unsigned long tx_jabber_irq;
	unsigned long rx_overflow_irq;
	unsigned long rx_buf_unav_irq;
	unsigned long rx_process_stopped_irq;
	unsigned long rx_watchdog_irq;
	unsigned long tx_early_irq;
	unsigned long fatal_bus_error_irq;
	/* Extra info */
	unsigned long threshold;
	unsigned long tx_pkt_n;
	unsigned long rx_pkt_n;
	unsigned long poll_n;
	unsigned long sched_timer_n;
	unsigned long normal_irq_n;
};

/* CSR Frequency Access Defines*/
#define CSR_F_35M	35000000
#define CSR_F_60M	60000000
#define CSR_F_100M	100000000
#define CSR_F_150M	150000000
#define CSR_F_250M	250000000
#define CSR_F_300M	300000000

#define	MAC_CSR_H_FRQ_MASK	0x20

#define HASH_TABLE_SIZE 64
#define PAUSE_TIME 0x200

/* Flow Control defines */
#define FLOW_OFF	0
#define FLOW_RX		1
#define FLOW_TX		2
#define FLOW_AUTO	(FLOW_TX | FLOW_RX)

#define SF_DMA_MODE 1 /* DMA STORE-AND-FORWARD Operation Mode */

/* MAC HW feature register fields */
#define MAC_HW_FEAT0_MIISEL		0x00000001 /* 10/100 Mbps Support */
#define MAC_HW_FEAT0_GMIISEL		0x00000002 /* 1000 Mbps Support */
#define MAC_HW_FEAT0_HDSEL		0x00000004 /* Half-Duplex Support */
#define MAC_HW_FEAT0_PCSSEL		0x00000008 /* PCS registers */
#define MAC_HW_FEAT0_VLHASH		0x00000010 /* Vlan Hash Filter */
#define MAC_HW_FEAT0_SMASEL		0x00000020 /* SMA(MDIO) Interface */
#define MAC_HW_FEAT0_RWKSEL		0x00000040 /* PMT Remote Wakeup */
#define MAC_HW_FEAT0_MGKSEL		0x00000080 /* PMT Magic Packet */
#define MAC_HW_FEAT0_MMCSEL		0x00000100 /* RMON Module */
#define MAC_HW_FEAT0_ARPOFFSEL		0x00000200 /* ARP offload */
#define MAC_HW_FEAT0_TSSEL		0x00001000 /* IEEE 1588-2008 Timestamp */
#define MAC_HW_FEAT0_EEESEL		0x00002000 /* Energy Efficient Ethernet */
#define MAC_HW_FEAT0_TXCOESEL		0x00004000 /* Checksum Offload in Tx */
#define MAC_HW_FEAT0_RXCOESEL		0x00010000 /* Checksum Offload in Rx */
#define MAC_HW_FEAT0_ADDMACADRSEL	0x00400000 /* MAC Addresses 16V31 */
#define MAC_HW_FEAT0_MACADR32SEL	0x00800000 /* MAC Addresses 32V63 */
#define MAC_HW_FEAT0_MACADR64SEL	0x01000000 /* MAC Addresses 64V127 */
#define MAC_HW_FEAT0_TSSTSSEL		0x06000000 /* Timestamp System Time Source */
#define MAC_HW_FEAT0_SAVLANINS		0x08000000 /* Source Addr or VLAN Insertion */
#define MAC_HW_FEAT0_ACTPHYIF		0x70000000 /* Active/selected PHY interface */

#define MAC_HW_FEAT1_RXFIFOSIZE		0x0000001f /* MTL Receive FIFO Size */
#define MAC_HW_FEAT1_TXFIFOSIZE		0x000007c0 /* MTL Transmit FIFO Size */
#define MAC_HW_FEAT1_ADVTHWORD		0x00002000 /* IEEE 1588 High Word Register*/
#define MAC_HW_FEAT1_DCBEN		0x00010000 /* DCB Feature Enable */
#define MAC_HW_FEAT1_SPHEN		0x00020000 /* TCP Segmentation Offload */
#define MAC_HW_FEAT1_TSOEN		0x00040000 /* Split Header Feature */
#define MAC_HW_FEAT1_DBGMEMA		0x00080000 /* DMA Debug Registers */
#define MAC_HW_FEAT1_AVSEL		0x00100000 /* AV Feature */
#define MAC_HW_FEAT1_LPMODEEN		0x00800000 /* Low Power Mode */
#define MAC_HW_FEAT1_HASHTBLSZ		0x03000000 /* Hash Table Size */
#define MAC_HW_FEAT1_L3L4FNUM		0x78000000 /* Total number of L3 or L4 Filters */

#define MAC_HW_FEAT2_RXQCNT		0x0000000f /* Number of MTL Receive Queues */
#define MAC_HW_FEAT2_TXQCNT		0x000003c0 /* Number of MTL Transmit Queues */
#define MAC_HW_FEAT2_RXCHCNT		0x0000f000 /* Number of DMA Receive Channels */
#define MAC_HW_FEAT2_TXCHCNT		0x003c0000 /* Number of DMA Transmit Channels */
#define MAC_HW_FEAT2_PPSOUTNUM		0x07000000 /* Numberof PPS Outputs */
#define MAC_HW_FEAT2_AUXSNAPNUM		0x70000000 /* Number of Auxiliary Snapshot Inputs */

#define DEFAULT_DMA_PBL		8

enum rx_frame_status { /* IPC status */
	good_frame = 0,
	discard_frame = 1,
	csum_none = 2,
	llc_snap = 4,
};

enum tx_dma_irq_status {
	tx_hard_error = 1,
	tx_hard_error_bump_tc = 2,
	handle_tx_rx = 3,
};

/* MAC HW capabilities */
struct mac_features {
	/* feature 0 */
	unsigned int mbps_10_100;
	unsigned int mbps_1000;
	unsigned int half_duplex;
	unsigned int pcs;
	unsigned int vlan_hash_filter;
	unsigned int sma_mdio;
	unsigned int pmt_remote_wake_up;
	unsigned int pmt_magic_frame;
	unsigned int rmon;
	unsigned int arpoe;
	/* IEEE 1588-2008 */
	unsigned int time_stamp;
	/* 802.3az - Energy-Efficient Ethernet (EEE) */
	unsigned int eee;
	/* TX and RX csum */
	unsigned int tx_coe;
	unsigned int rx_coe;
	unsigned int multi_addr;
	unsigned int multi_addr_32;
	unsigned int multi_addr_64;
	/* Timestamp System Time Source */
	unsigned int time_stamp_src;
	/* VLAN insertion */
	unsigned int sa_vlan_ins;
	/* PHY interface selected */
	unsigned int phy_intf_sel;

	/* feature 1 */
	unsigned int rxfifo_size;
	unsigned int txfifo_size;
	unsigned int time_stamp_hiword;
	unsigned int dcb;
	unsigned int split_header;
	unsigned int tso;
	unsigned int dma_debug;
	unsigned int av;
	unsigned int low_power;
	unsigned int hash_table_size;
	unsigned int l3l4_filter_num;

	/* feature 2 */
	/* TX and RX number of queues */
	unsigned int number_rx_queue;
	unsigned int number_tx_queue;
	/* TX and RX number of channels */
	unsigned int number_rx_channel;
	unsigned int number_tx_channel;
	/* PPS output*/
	unsigned int number_pps;
	unsigned int number_aux_snapshot;
};

/* GMAC TX FIFO is 8K, Rx FIFO is 16K */
#define BUF_SIZE_16KiB 16384
#define BUF_SIZE_8KiB 8192
#define BUF_SIZE_4KiB 4096
#define BUF_SIZE_2KiB 2048

/* Power Down and WOL */
#define PMT_NOT_SUPPORTED 0
#define PMT_SUPPORTED 1

/* Common MAC defines */
#define MAC_CTRL_REG		0x00000000	/* MAC Control */
#define MAC_ENABLE_TX		0x00000002	/* Transmitter Enable */
#define MAC_RNABLE_RX		0x00000001	/* Receiver Enable */

struct stmmac_dma_ops {
	/* DMA core initialization */
	int (*init) (void __iomem *ioaddr, int pbl, int fb, int mb,
		     int burst_len, u32 dma_tx, u32 dma_rx, u32 dma_tx_size, u32 dma_rx_size, u32 rx_bufsize);
	/* Dump DMA registers */
	void (*dump_regs) (void __iomem *ioaddr);
	/* Set tx/rx threshold in the csr6 register
	 * An invalid value enables the store-and-forward mode */
	void (*dma_mode) (void __iomem *ioaddr, int txmode, int rxmode);
	/* To track extra statistic (if supported) */
	void (*dma_diagnostic_fr) (void *data, struct stmmac_extra_stats *x,
				   void __iomem *ioaddr);
	void (*enable_dma_transmission) (void __iomem *ioaddr);
	void (*enable_dma_irq) (void __iomem *ioaddr);
	void (*disable_dma_irq) (void __iomem *ioaddr);
	void (*start_tx) (void __iomem *ioaddr);
	void (*stop_tx) (void __iomem *ioaddr);
	void (*start_rx) (void __iomem *ioaddr);
	void (*stop_rx) (void __iomem *ioaddr);
	int (*dma_interrupt) (void __iomem *ioaddr,
			      struct stmmac_extra_stats *x);
	int (*mtl_interrupt) (void __iomem *ioaddr);
};

struct stmmac_ops {
	/* MAC core initialization */
	void (*core_init) (void __iomem *ioaddr) ____cacheline_aligned;
	/* Enable and verify that the IPC module is supported */
	int (*rx_ipc) (void __iomem *ioaddr);
	/* Dump MAC registers */
	void (*dump_regs) (void __iomem *ioaddr);
	/* Handle extra events on specific interrupts hw dependent */
	void (*host_irq_status) (void __iomem *ioaddr);
	/* Multicast filter setting */
	void (*set_filter) (struct net_device *dev, int id);
	/* Flow control setting */
	void (*flow_ctrl) (void __iomem *ioaddr, unsigned int duplex,
			   unsigned int fc, unsigned int pause_time);
	/* Set power management mode (e.g. magic frame) */
	void (*pmt) (void __iomem *ioaddr, unsigned long mode);
	/* Set/Get Unicast MAC addresses */
	void (*set_umac_addr) (void __iomem *ioaddr, unsigned char *addr,
			       unsigned int reg_n);
	void (*get_umac_addr) (void __iomem *ioaddr, unsigned char *addr,
			       unsigned int reg_n);
	/* If supported then get the optional core features */
	unsigned int (*get_hw_feature) (void __iomem *ioaddr, unsigned int num);
};

struct mac_link {
	int port;
	int duplex;
	int speed;
};

struct stmmac_ring_mode_ops {
	int (*set_16kib_bfsize) (int mtu);
};

struct mac_device_info {
	const struct stmmac_ops		*mac;
	const struct stmmac_dma_ops	*dma;
	struct mac_link link;
	unsigned int synopsys_uid;
};

struct mac_device_info *dweqos_mac_setup(void __iomem *ioaddr);

extern void stmmac_set_mac_addr(void __iomem *ioaddr, u8 addr[6],
				unsigned int high, unsigned int low);
extern void stmmac_get_mac_addr(void __iomem *ioaddr, unsigned char *addr,
				unsigned int high, unsigned int low);

extern void stmmac_set_mac(void __iomem *ioaddr, bool enable);

extern void dwmac_dma_flush_tx_fifo(void __iomem *ioaddr);
extern const struct stmmac_ring_mode_ops ring_mode_ops;

#endif /* __COMMON_H__ */
