/*******************************************************************************
  Header File to describe the DMA descriptors.
  Enhanced descriptors have been in case of DWMAC1000 Cores.

  This program is free software; you can redistribute it and/or modify it
  under the terms and conditions of the GNU General Public License,
  version 2, as published by the Free Software Foundation.

  This program is distributed in the hope it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.

  The full GNU General Public License is included in this distribution in
  the file called "COPYING".

  Author: Giuseppe Cavallaro <peppe.cavallaro@st.com>
*******************************************************************************/

#ifndef __DESCS_H__
#define __DESCS_H__
#include "common.h"
struct dma_rx_desc {
	union {
		struct {
			/* DESC0 */
			u32 buffer1;
			/* DESC1 */
			u32 reserved;
			/* DESC2 */
			u32 buffer2;
			/* DESC3 */
			u32 reserved1:24;
			u32 buffer1_vailed:1;
			u32 buffer2_vailed:1;
			u32 reserved2:4;
			u32 interrupt:1;
			u32 own:1;
		} read;
		struct {
			/* DESC0 */
			u32 vlan_tag:16;
			u32 reserved1:16;
			/* DESC1 */
			u32 ip_payload_type:3;
			u32 ip_hdr_err:1;
			u32 ipv4_hdr_prsnt:1;
			u32 ipv6_hdr_prsnt:1;
			u32 ip_csum_bypassed:1;
			u32 ipc_csum_error:1;
			u32 ptp_msg_type:4;
			u32 ptp_packet_type:1;
			u32 ptp_ver:1;
			u32 timestamp_avaliable:1;
			u32 timestamp_dropped:1;
			u32 ip1c_oamtpc_macctlop:16;
			/* DESC2 */
			u32 l3_l4_hdr_length:10;
			u32 arp_pkt_rcvd:1;
			u32 arp_rply_not_gen:1;
			u32 reserved2:3;
			u32 vlan_filter_status:1;
			u32 sa_filter_fail:1;
			u32 da_filter_fail:1;
			u32 hash_filter_status:1;
			u32 mac_match_hash_value:8;
			u32 l3_filter_match:1;
			u32 l4_filter_match:1;
			u32 l3_l4_filter_no_match:3;
			/* DESC3 */
			u32 packet_length:15;
			u32 error_summary:1;
			u32 length_type_field:3;
			u32 dribbling:1;
			u32 gmii_rx_error:1;
			u32 overflow_error:1;
			u32 receive_watchdog:1;
			u32 giant_pkt:1;
			u32 crc_error:1;
			u32 rcvd_status_desc0_valid:1;
			u32 rcvd_status_desc1_valid:1;
			u32 rcvd_status_desc2_valid:1;
			u32 last_desc:1;
			u32 first_desc:1;
			u32 context_type:1;
			u32 own:1;
		} wb;
	} rx;
};


struct dma_tx_desc {
	union {
		struct {
			/* DESC0 */
			u32 buffer1;
			/* DESC1 */
			u32 buffer2;
			/* DESC2 */
			u32 buffer1_length:14;
			u32 vlan_insertion:2;
			u32 buffer2_length:14;
			u32 time_stamp_enable:1;
			u32 interrupt:1;
			/* DESC3 */
			u32 payload_length:18;
			u32 tse:1;
			u32 tcp_hdr_length:4;
			u32 sa_insertion_ctrl:3;
			u32 crc_pad_ctrl:2;
			u32 last_desc:1;
			u32 first_desc:1;
			u32 context_type:1;
			u32 own:1;
		} read;
		struct {
			/* DESC0 */
			u32 timestamp_low;
			/* DESC1 */
			u32 timestamp_high;
			/* DESC2 */
			u32 reserved;
			/* DESC3 */
			u32 ip_header_error:1;
			u32 deferred:1;
			u32 underflow_error:1;
			u32 excessive_deferral:1;
			u32 collision_count:4;
			u32 excessive_collisions:1;
			u32 late_collision:1;
			u32 no_carrier:1;
			u32 loss_carrier:1;
			u32 payload_csum_error:1; //payload_error
			u32 packet_flushed:1; //frame flushed
			u32 jabber_timeout:1;
			u32 error_summary:1;
			u32 reserved1:1;
			u32 time_stamp_status:1;
			u32 reserved2:10;
			u32 last_desc:1;
			u32 first_desc:1;
			u32 context_type:1;
			u32 own:1;
		} wb;
		struct {
			/* DESC0 */
			u32 timestamp_low;
			/* DESC1 */
			u32 timestamp_high;
			/* DESC2 */
			u32 mss:14;
			u32 reserved:2;
			u32 inner_vlan_tag:16;
			/* DESC3 */
			u32 vt:16;
			u32 vltv:1;
			u32 ivltv:1;
			u32 ivtir:2;
			u32 reserved2:3;
			u32 cde:1;
			u32 reserved3:2;
			u32 tcmssv:1;
			u32 ostc:1;
			u32 reserved4:2;
			u32 ctxt:1;
			u32 own:1;
		} cntxt;
	} tx;
};

/* Transmit checksum insertion control */
enum tdes_csum_insertion {
	cic_disabled = 0,		/* Checksum Insertion Control */
	cic_only_ip = 1,		/* Only IP header */
	cic_no_pseudoheader = 2,	/* IP header but pseudoheader
					 * is not calculated */
	cic_full = 3,			/* IP header and pseudoheader */
};

enum tdes_vlan_insertion {
	vtir_no_add = 0,	/* Do not add a VLAN tag */
	vtir_remov = 1,		/* Remove the VLAN tag */
	vtir_insert = 2,	/* Insert a VLAN tag */
	vtir_replace = 3,	/* Replace the VLAN tag */
};

/* Return the transmit status looking at the TDES3 */
static inline int desc_get_tx_status(void *data, struct stmmac_extra_stats *x,
			       struct dma_tx_desc *p, void __iomem *ioaddr)
{
	int ret = 0;
	struct net_device_stats *stats = (struct net_device_stats *)data;

	if (unlikely(p->tx.wb.error_summary)) {

		if (unlikely(p->tx.wb.ip_header_error)) {
			CHIP_DBG(KERN_ERR "\tTX IP header csum error\n");
			x->tx_ip_header_error++;
		}
		if (unlikely(p->tx.wb.jabber_timeout)) {
			CHIP_DBG(KERN_ERR "\tjabber_timeout error\n");
			x->tx_jabber++;
		}
		if (unlikely(p->tx.wb.packet_flushed)) {
			CHIP_DBG(KERN_ERR "\tpacket_flushed error\n");
			x->tx_packet_flushed++;
			dwmac_dma_flush_tx_fifo(ioaddr);
		}
		if (unlikely(p->tx.wb.payload_csum_error)) {
			CHIP_DBG(KERN_ERR "\tAddr/Payload csum error\n");
			x->tx_payload_error++;
			dwmac_dma_flush_tx_fifo(ioaddr);
		}
		if (unlikely(p->tx.wb.loss_carrier)) {
			CHIP_DBG(KERN_ERR "\tloss_carrier error\n");
			x->tx_losscarrier++;
			stats->tx_carrier_errors++;
		}
		if (unlikely(p->tx.wb.no_carrier)) {
			CHIP_DBG(KERN_ERR "\tno_carrier error\n");
			x->tx_carrier++;
			stats->tx_carrier_errors++;
		}
		if (unlikely(p->tx.wb.late_collision)) {
			CHIP_DBG(KERN_ERR "\tlate_collision error\n");
			stats->collisions += p->tx.wb.collision_count;
		}
		if (unlikely(p->tx.wb.excessive_collisions)) {
			CHIP_DBG(KERN_ERR "\texcessive_collisions\n");
			stats->collisions += p->tx.wb.collision_count;
		}
		if (unlikely(p->tx.wb.excessive_deferral)) {
			CHIP_DBG(KERN_INFO "\texcessive tx_deferral\n");
			x->tx_deferred++;
		}

		if (unlikely(p->tx.wb.underflow_error)) {
			CHIP_DBG(KERN_ERR "\tunderflow error\n");
			x->tx_underflow++;
			stats->tx_fifo_errors++;
		}

		ret = -1;
	}

	if (unlikely(p->tx.wb.deferred)) {
		CHIP_DBG(KERN_INFO "GMAC TX status: tx deferred\n");
		x->tx_deferred++;
	}
	return ret;
}

/* This function verifies if each incoming frame has some errors
 * and, if required, updates the multicast statistics.
 * In case of success, it returns good_frame because the GMAC device
 * is supposed to be able to compute the csum in HW. */
static inline int desc_get_rx_status(void *data, struct stmmac_extra_stats *x,
			       struct dma_rx_desc *p)
{
	int ret = good_frame;
	struct net_device_stats *stats = (struct net_device_stats *)data;

	if (unlikely(p->rx.wb.error_summary)) {
		if (unlikely(p->rx.wb.giant_pkt)) {
			CHIP_DBG(KERN_ERR "\tdescriptor error/Giant frame\n");
			x->rx_desc++;
			stats->rx_length_errors++;
		}
		if (unlikely(p->rx.wb.overflow_error)) {
			CHIP_DBG(KERN_ERR "\toverflow error\n");
			x->rx_gmac_overflow++;
		}
		if (unlikely(p->rx.wb.receive_watchdog)) {
			CHIP_DBG(KERN_ERR "\treceive_watchdog error\n");
			x->rx_watchdog++;
		}
		if (unlikely(p->rx.wb.gmii_rx_error)) {
			CHIP_DBG(KERN_ERR "\tReceive Error\n");
			x->rx_mii++;
		}
		if (unlikely(p->rx.wb.crc_error)) {
			CHIP_DBG(KERN_ERR "\tCRC error\n");
			x->rx_crc++;
			stats->rx_crc_errors++;
		}
		ret = discard_frame;
	}

	if (unlikely(p->rx.wb.dribbling)) {
		CHIP_DBG(KERN_ERR "\tGMAC RX: dribbling error\n");
		x->dribbling_bit++;
	}

	if (p->rx.wb.rcvd_status_desc1_valid) {
		if (unlikely(p->rx.wb.ipc_csum_error)) {
			CHIP_DBG(KERN_ERR "\tIPC Csum Error\n");
			x->ipc_csum_error++;
			ret = discard_frame;
		}
	}

	if (p->rx.wb.rcvd_status_desc2_valid) {
		if (unlikely(p->rx.wb.sa_filter_fail)) {
 			CHIP_DBG(KERN_ERR "GMAC RX : Source Address filter fail\n");
			x->sa_rx_filter_fail++;
			ret = discard_frame;
		}
		if (unlikely(p->rx.wb.da_filter_fail)) {
			CHIP_DBG(KERN_ERR "GMAC RX : Dest Address filter fail\n");
			x->da_rx_filter_fail++;
			ret = discard_frame;
		}
	}

	return ret;
}

/* DMA RX descriptor ring initialization */
static inline void desc_init_rx_desc(struct dma_rx_desc *p, unsigned int ring_size,
			       int disable_rx_ic)
{
	int i;
	for (i = 0; i < ring_size; i++) {
		p->rx.read.own = 1;
		if (!disable_rx_ic)
			p->rx.read.interrupt = 1;
		p++;
	}
}

/* DMA TX descriptor ring initialization */
static inline void desc_init_tx_desc(struct dma_tx_desc *p, unsigned int ring_size)
{
	int i;
	for (i = 0; i < ring_size; i++) {
		p->tx.read.own = 0;
		p++;
	}
}

/* Clean the tx descriptor as soon as the tx irq is received */
static inline void desc_release_tx_desc(struct dma_tx_desc *p)
{
	memset(p, 0, sizeof(struct dma_tx_desc));
}

/* Set/get the owner of the descriptor */
static inline int desc_get_tx_owner(struct dma_tx_desc *p)
{
	return p->tx.wb.own;
}

static inline void desc_set_tx_owner(struct dma_tx_desc *p)
{
	p->tx.read.own = 1;
}

/* Invoked by the xmit function to prepare the tx descriptor */
static inline void desc_prepare_tx_desc(struct dma_tx_desc *p, int is_fs, int len, int pkg_len,
				  int csum_flag, int vlan_flag)
{
	u32 value;

	p->tx.read.first_desc = is_fs;
	p->tx.read.buffer1_length = len;

	if (is_fs) {
		if (likely(vlan_flag))
			p->tx.read.vlan_insertion = vtir_insert;

		value = pkg_len & 0x7FFF;
		if (likely(csum_flag))
			value |= (cic_full << 16);

		p->tx.read.payload_length = value;
	}
}

/* Invoked by the xmit function to close the tx descriptor */
static inline void desc_close_tx_desc(struct dma_tx_desc *p)
{
	p->tx.read.last_desc = 1;
	p->tx.read.interrupt = 1;
}

/* Clear interrupt on tx frame completion. When this bit is
 * set an interrupt happens as soon as the frame is transmitted */
static inline void desc_clear_tx_ic(struct dma_tx_desc *p)
{
	p->tx.read.interrupt = 1;
}

/* Last tx segment reports the transmit status */
static inline int desc_get_tx_ls(struct dma_tx_desc *p)
{
	return p->tx.wb.last_desc;
}

//static inline void norm_set_tx_desc_len(struct dma_tx_desc *p, int len)
//{
//	if (unlikely(len > BUF_SIZE_2KiB)) {
//		p->tx.read.buffer1_length = BUF_SIZE_2KiB - 1;
//		p->tx.read.buffer2_length = len - p->tx.read.buffer1_length;
//	} else {
//		p->tx.read.buffer1_length = len;
//	}
//}

static inline int desc_get_rx_owner(struct dma_rx_desc *p)
{
	return p->rx.wb.own;
}

static inline void desc_set_rx_owner(struct dma_rx_desc *p)
{
	p->rx.wb.own = 1;
}

static inline void desc_set_rx_buf(struct dma_rx_desc *p, u32 paddr)
{
	p->rx.read.buffer1 = paddr;
	p->rx.read.buffer1_vailed = 1;
}

static inline void desc_reset_rx_desc(struct dma_rx_desc *p)
{
	memset(p, 0, sizeof(struct dma_rx_desc));
	p->rx.read.interrupt = 1;
}

/* Get the receive frame size */
static inline int desc_get_rx_frame_len(struct dma_rx_desc *p, int rx_coe_type)
{
	return p->rx.wb.packet_length;
}
#endif /* __DESCS_H__ */
