/*
 *  drivers/net/ethernet/vatics/dweqos/dweqos_main.c
 *
 *  Copyright (C) 2007-2014  VATICS Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/interrupt.h>
#include <linux/etherdevice.h>
#include <linux/platform_device.h>
#include <linux/ip.h>
#include <linux/irqdomain.h>
#include <linux/tcp.h>
#include <linux/skbuff.h>
#include <linux/ethtool.h>
#include <linux/if_ether.h>
#include <linux/crc32.h>
#include <linux/mii.h>
#include <linux/if.h>
#include <linux/if_vlan.h>
#include <linux/dma-mapping.h>
#include <linux/slab.h>
#include <linux/prefetch.h>
#ifdef CONFIG_DWEQOS_DEBUG_FS
#include <linux/debugfs.h>
#include <linux/seq_file.h>
#endif
#include "stmmac.h"
#include "dweqos.h"

#define STMMAC_ALIGN(x)	L1_CACHE_ALIGN(x)
/* Driver Configuration Settings */
#define ETH_TX_TIMEOUT	5000 /* default 5 seconds */
#define TX_DESC_CNT	L1_CACHE_ALIGN(256)
#define RX_DESC_CNT	L1_CACHE_ALIGN(256)

/* Module parameters */
static int debug = -1;		/* -1: default, 0: no output, 16:  all */
module_param(debug, int, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(debug, "Message Level (0: no output, 16: all)");

static int flow_ctrl = FLOW_OFF;
module_param(flow_ctrl, int, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(flow_ctrl, "Flow control ability [on/off]");

static int pause = PAUSE_TIME;
module_param(pause, int, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(pause, "Flow Control Pause Time");

#define TC_DEFAULT 64
static int tc = TC_DEFAULT;
module_param(tc, int, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(tc, "DMA threshold control value");

#define DMA_BUFFER_SIZE	BUF_SIZE_2KiB
static int buf_sz = DMA_BUFFER_SIZE;
module_param(buf_sz, int, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(buf_sz, "DMA buffer size");

static const u32 default_msg_level = (NETIF_MSG_DRV | NETIF_MSG_PROBE |
				      NETIF_MSG_LINK | NETIF_MSG_IFUP |
				      NETIF_MSG_IFDOWN | NETIF_MSG_TIMER);

static irqreturn_t stmmac_interrupt(int irq, void *dev_id);

/**
 * stmmac_verify_args - verify the driver parameters.
 * Description: it verifies if some wrong parameter is passed to the driver.
 * Note that wrong parameters are replaced with the default values.
 */
static void stmmac_verify_args(void)
{
	if (unlikely((buf_sz < DMA_BUFFER_SIZE) || (buf_sz > BUF_SIZE_16KiB)))
		buf_sz = DMA_BUFFER_SIZE;
	if (unlikely(flow_ctrl > 1))
		flow_ctrl = FLOW_AUTO;
	else if (likely(flow_ctrl < 0))
		flow_ctrl = FLOW_OFF;
	if (unlikely((pause < 0) || (pause > 0xffff)))
		pause = PAUSE_TIME;
}

static void dweqos_clk_csr_set(struct stmmac_priv *priv)
{
	u32 clk_rate = clk_get_rate(priv->plat->sysclk);
	printk("clk rate %d\n", clk_rate);
	if (clk_rate < CSR_F_35M)
		priv->clk_csr = DWEQOS_CSR_20_35M;
	else if ((clk_rate >= CSR_F_35M) && (clk_rate < CSR_F_60M))
		priv->clk_csr = DWEQOS_CSR_35_60M;
	else if ((clk_rate >= CSR_F_60M) && (clk_rate < CSR_F_100M))
		priv->clk_csr = DWEQOS_CSR_60_100M;
	else if ((clk_rate >= CSR_F_100M) && (clk_rate < CSR_F_150M))
		priv->clk_csr = DWEQOS_CSR_100_150M;
	else if ((clk_rate >= CSR_F_150M) && (clk_rate < CSR_F_250M))
		priv->clk_csr = DWEQOS_CSR_150_250M;
	else if ((clk_rate >= CSR_F_250M) && (clk_rate < CSR_F_300M))
		priv->clk_csr = DWEQOS_CSR_250_300M;
}

static void print_pkt(unsigned char *buf, int len)
{
	int j;
	printk("len = %d byte, buf addr: 0x%p\n", len, buf);
	for (j = 0; j < len; j++) {
		if ((j % 16) == 0)
			printk("\n %03x:", j);
		printk(" %02x", buf[j]);
	}
	printk("\n");
}

static void sys_print_pkt(struct seq_file *seq, unsigned char *buf, int len, int shift)
{
	int j;
	seq_printf(seq, "len = %d byte, buf addr: 0x%p\n", len, buf);
	for (j = 0; j < len; j++) {
		if (((j+shift) % 16) == 0)
			seq_printf(seq, "\n %03x:", j+shift);
		seq_printf(seq," %02x", buf[j]);
	}
	seq_printf(seq,"\n");
}

/* minimum number of free TX descriptors required to wake up TX process */
#define STMMAC_TX_THRESH(x)	(x->tx_desc_count/4)

static inline u32 dweqos_tx_avail(struct stmmac_priv *priv)
{
	return priv->dirty_tx + priv->tx_desc_count - priv->cur_tx - 1;
}

/* On some ST platforms, some HW system configuraton registers have to be
 * set according to the link speed negotiated.
 */
static inline void stmmac_hw_fix_mac_speed(struct stmmac_priv *priv)
{
	struct phy_device *phydev = priv->phydev;

	if (likely(priv->plat->fix_mac_speed))
		priv->plat->fix_mac_speed(priv->plat->bsp_priv,
					  phydev->speed);
}

/**
 * stmmac_adjust_link
 * @dev: net device structure
 * Description: it adjusts the link parameters.
 */
static void stmmac_adjust_link(struct net_device *dev)
{
	struct stmmac_priv *priv = netdev_priv(dev);
	struct phy_device *phydev = priv->phydev;
	unsigned long flags;
	int new_state = 0;
	unsigned int fc = priv->flow_ctrl, pause_time = priv->pause;

	if (phydev == NULL)
		return;

	spin_lock_irqsave(&priv->lock, flags);

	if (phydev->link) {
		u32 ctrl = readl(priv->ioaddr + MAC_CTRL_REG);

		/* Now we make sure that we can be in full duplex mode.
		 * If not, we operate in half-duplex mode. */
		if (phydev->duplex != priv->oldduplex) {
			new_state = 1;
			if (!(phydev->duplex))
				ctrl &= ~priv->hw->link.duplex;
			else
				ctrl |= priv->hw->link.duplex;
			priv->oldduplex = phydev->duplex;
		}
		/* Flow Control operation */
		if (phydev->pause)
			priv->hw->mac->flow_ctrl(priv->ioaddr, phydev->duplex,
						 fc, pause_time);

		if (phydev->speed != priv->speed) {
			new_state = 1;
			switch (phydev->speed) {
			case 1000:
				ctrl &= ~priv->hw->link.port;
				stmmac_hw_fix_mac_speed(priv);
				break;
			case 100:
			case 10:
				ctrl |= priv->hw->link.port;
				if (phydev->speed == SPEED_100) {
					ctrl |= priv->hw->link.speed;
				} else {
					ctrl &= ~(priv->hw->link.speed);
				}
				stmmac_hw_fix_mac_speed(priv);
				break;
			default:
				if (netif_msg_link(priv))
					pr_warning("%s: Speed (%d) is not 10"
				       " or 100!\n", dev->name, phydev->speed);
				break;
			}

			priv->speed = phydev->speed;
		}

		writel(ctrl, priv->ioaddr + MAC_CTRL_REG);

		if (!priv->oldlink) {
			new_state = 1;
			priv->oldlink = 1;
		}
	} else if (priv->oldlink) {
		new_state = 1;
		priv->oldlink = 0;
		priv->speed = 0;
		priv->oldduplex = -1;
	}

	if (new_state && netif_msg_link(priv))
		phy_print_status(phydev);

	spin_unlock_irqrestore(&priv->lock, flags);
}

/**
 * stmmac_init_phy - PHY initialization
 * @ndev: net device structure
 * Description: it initializes the driver's PHY state, and attaches the PHY
 * to the mac driver.
 *  Return value:
 *  0 on success
 */
static int dweqos_init_phy(struct net_device *ndev)
{
	struct stmmac_priv *priv = netdev_priv(ndev);
	struct phy_device *phydev;
	int interface = priv->plat->interface;
	priv->oldlink = 0;
	priv->speed = 0;
	priv->oldduplex = -1;

	if (!priv->phydev)
		return -ENODEV;

	phydev = priv->phydev;
	netdev_info(ndev, "%s: trying to attach to %s\n", __func__, dev_name(&phydev->dev));

	phydev = phy_connect(ndev, dev_name(&phydev->dev), &stmmac_adjust_link, interface);

	if (IS_ERR(phydev)) {
		netdev_err(ndev, "could not attach PHY\n");
		return PTR_ERR(phydev);
	}

	/* Stop Advertising 1000BASE Capability if interface is not GMII */
	if ((interface == PHY_INTERFACE_MODE_MII) ||
	    (interface == PHY_INTERFACE_MODE_RMII))
		phydev->advertising &= ~(SUPPORTED_1000baseT_Half |
					 SUPPORTED_1000baseT_Full);

	/*
	 * Broken HW is sometimes missing the pull-up resistor on the
	 * MDIO line, which results in reads to non-existent devices returning
	 * 0 rather than 0xffff. Catch this here and treat 0 as a non-existent
	 * device as well.
	 * Note: phydev->phy_id is the result of reading the UID PHY registers.
	 */
	if (phydev->phy_id == 0) {
		phy_disconnect(phydev);
		return -ENODEV;
	}
	netdev_info(ndev, "%s: attached to PHY (UID 0x%x)"
		 " Link = %d\n", __func__, phydev->phy_id, phydev->link);
	netdev_info(ndev, "PHY: %s\n", phydev->drv->name);

	return 0;
}

/**
 * dweqos_display_ring
 * @head: pointer to the head of the ring passed.
 * @size: size of the ring.
 * Description: display all the descriptors within the ring.
 */
static void dweqos_display_ring(void *head, int size)
{
	struct tmp_s {
		unsigned int a;
		unsigned int b;
		unsigned int c;
		unsigned int d;
	};
	int i;
	for (i = 0; i < size; i++) {
		struct tmp_s *x = (struct tmp_s *)(head + i);
		pr_info("%d [0x%08x]: 0x%08x 0x%08x 0x%08x 0x%08x\n",
		       i, (unsigned int)virt_to_phys(&x),
		       x->a, x->b,
		       x->c, x->d);
		pr_info("\n");
	}
}

static int dweqos_setup_tx_resource(struct stmmac_priv *priv)
{
	unsigned int txsize = priv->tx_desc_count;

	priv->tx_buffer_info = kcalloc(txsize, sizeof(struct dweqos_buffer), GFP_KERNEL);

	if (!priv->tx_buffer_info)
		return -ENOMEM;

	priv->dma_tx = (struct dma_tx_desc *)dma_alloc_coherent(priv->dev,
					  txsize *
					  sizeof(struct dma_tx_desc),
					  &priv->dma_tx_phy,
					  GFP_KERNEL | __GFP_ZERO);
	if (!priv->dma_tx) {
		kfree(priv->tx_buffer_info);
		return -ENOMEM;
	}

	memset(priv->dma_tx, 0, txsize * sizeof(struct dma_tx_desc));

	if (netif_msg_probe(priv))
		pr_debug("(%s) dma_tx_phy=0x%08x\n", __func__, (u32) priv->dma_tx_phy);

	priv->dirty_tx = 0;
	priv->cur_tx = 0;
	return 0;
}

static int dweqos_setup_rx_resource(struct stmmac_priv *priv)
{
	unsigned int rxsize = priv->rx_desc_count;

	priv->rx_buffer_info = kcalloc(rxsize, sizeof(struct dweqos_buffer), GFP_KERNEL);

	if (!priv->rx_buffer_info)
		return -ENOMEM;

	priv->dma_rx = (struct dma_rx_desc *)dma_alloc_coherent(priv->dev,
					  rxsize *
					  sizeof(struct dma_rx_desc),
					  &priv->dma_rx_phy,
					  GFP_KERNEL | __GFP_ZERO);
	if (!priv->dma_rx) {
		kfree(priv->rx_buffer_info);
		return -ENOMEM;
	}

	memset(priv->dma_rx, 0, rxsize * sizeof(struct dma_rx_desc));

	if (netif_msg_probe(priv))
		pr_debug("(%s) dma_rx_phy=0x%08x\n", __func__, (u32) priv->dma_rx_phy);

	priv->cur_rx = 0;
	priv->dirty_rx = 0;

	return 0;
}

static void dma_free_rx_skbufs(struct stmmac_priv *priv)
{
	int i;
	struct dweqos_buffer *buffer_info;

	for (i = 0; i < priv->rx_desc_count; i++) {
		buffer_info = priv->rx_buffer_info + i;
		if (buffer_info->skbuff) {
			dma_unmap_single(priv->dev, buffer_info->skbuff_dma,
					 priv->dma_buf_sz, DMA_FROM_DEVICE);
			dev_kfree_skb_any(buffer_info->skbuff);
		}
		buffer_info->skbuff = NULL;
	}
}

static void dma_free_tx_skbufs(struct stmmac_priv *priv)
{
	int i;
	struct dweqos_buffer *buffer_info;

	for (i = 0; i < priv->tx_desc_count; i++) {
		buffer_info = priv->tx_buffer_info + i;

		if (buffer_info->skbuff != NULL) {
			if (buffer_info->skbuff_dma)
				dma_unmap_single(priv->dev, buffer_info->skbuff_dma,
						 buffer_info->skbuff_dma_len,
						 DMA_TO_DEVICE);
			dev_kfree_skb_any(buffer_info->skbuff);
			memset(buffer_info, 0, sizeof(struct dweqos_buffer));
		}
	}
}

static void dweqos_free_tx_resource(struct stmmac_priv *priv)
{
	/* Release the DMA TX/RX socket buffers */
	dma_free_tx_skbufs(priv);
	/* Free the region of consistent memory previously allocated for
	 * the DMA */
	dma_free_coherent(priv->dev,
			  priv->tx_desc_count * sizeof(struct dma_tx_desc),
			  priv->dma_tx, priv->dma_tx_phy);

	kfree(priv->tx_buffer_info);
}

static void dweqos_free_rx_resource(struct stmmac_priv *priv)
{
	/* Release the DMA TX/RX socket buffers */
	dma_free_rx_skbufs(priv);

	/* Free the region of consistent memory previously allocated for
	 * the DMA */
	dma_free_coherent(priv->dev,
			  priv->rx_desc_count * sizeof(struct dma_rx_desc),
			  priv->dma_rx, priv->dma_rx_phy);
	kfree(priv->rx_buffer_info);
}

static void dweqos_alloc_rx_buffers(struct stmmac_priv *priv)
{
	int i;
	struct sk_buff *skb;
	struct net_device *dev = priv->netdev;
	unsigned int bfsize = priv->dma_buf_sz;
	unsigned int rxsize = priv->rx_desc_count;
	pr_debug("\tSKB addresses:\nskb\t\tskb data\tdma data\n");

	for (i = 0; i < rxsize; i++) {
		struct dma_rx_desc *p = priv->dma_rx + i;
		struct dweqos_buffer *buffer_info = priv->rx_buffer_info + i;

		skb = netdev_alloc_skb_ip_align(dev, bfsize);

		if (unlikely(skb == NULL)) {
			pr_err("%s: Rx init fails; skb is NULL\n", __func__);
			break;
		}

		buffer_info->skbuff = skb;
		buffer_info->skbuff_dma = dma_map_single(priv->dev, skb->data,
						bfsize, DMA_FROM_DEVICE);

		desc_set_rx_buf(p, buffer_info->skbuff_dma);

		if (netif_msg_probe(priv))
			pr_debug("[%p]\t[%p]\t[%x]\n", buffer_info->skbuff,
				 buffer_info->skbuff->data,
				 (unsigned int)buffer_info->skbuff_dma);
	}

}

/**
 * dweqos_init_dma_desc_rings - init the RX/TX descriptor rings
 * @dev: net device structure
 * Description:  this function initializes the DMA RX/TX descriptors ring
 * and allocates the socket buffers.
 */
static int dweqos_init_dma_desc_rings(struct net_device *ndev)
{
	struct stmmac_priv *priv = netdev_priv(ndev);
	int dis_ic = 0;
	int ret = -ENOMEM;

	/* Set the max buffer size according to the DESC mode
	 * and the MTU. Note that RING mode allows 16KiB bsize. */
	if (netif_msg_probe(priv))
		pr_debug("%s: txsize %d, rxsize %d, bfsize %d\n", __func__,
			 priv->rx_desc_count, priv->rx_desc_count, priv->dma_buf_sz);

	ret = dweqos_setup_tx_resource(priv);
	if (ret)
		goto err_setup_tx;

	ret = dweqos_setup_rx_resource(priv);
	if (ret)
		goto err_setup_rx;

	dweqos_alloc_rx_buffers(priv);

	/* Clear the Rx/Tx descriptors */
	desc_init_rx_desc(priv->dma_rx, priv->rx_desc_count, dis_ic);
	desc_init_tx_desc(priv->dma_tx, priv->tx_desc_count);

	if (netif_msg_hw(priv)) {
		pr_info("RX descriptor ring:\n");
		dweqos_display_ring((void *)priv->dma_rx, priv->rx_desc_count);
		pr_info("TX descriptor ring:\n");
		dweqos_display_ring((void *)priv->dma_tx, priv->tx_desc_count);
	}

	return ret;

err_setup_rx:
	dweqos_free_tx_resource(priv);

err_setup_tx:
	return ret;
}



static void free_dma_desc_resources(struct stmmac_priv *priv)
{
	dweqos_free_tx_resource(priv);
	dweqos_free_rx_resource(priv);
}

/**
 *  stmmac_dma_operation_mode - HW DMA operation mode
 *  @priv : pointer to the private device structure.
 *  Description: it sets the DMA operation mode: tx/rx DMA thresholds
 *  or Store-And-Forward capability.
 */
static void stmmac_dma_operation_mode(struct stmmac_priv *priv)
{
	if (likely(priv->plat->force_sf_dma_mode ||
		((priv->plat->tx_coe) && (!priv->no_csum_insertion)))) {
		/*
		 * In case of GMAC, SF mode can be enabled
		 * to perform the TX COE in HW. This depends on:
		 * 1) TX COE if actually supported
		 * 2) There is no bugged Jumbo frame support
		 *    that needs to not insert csum in the TDES.
		 */
		priv->hw->dma->dma_mode(priv->ioaddr,
					SF_DMA_MODE, SF_DMA_MODE);
		tc = SF_DMA_MODE;
	} else
		priv->hw->dma->dma_mode(priv->ioaddr, tc, SF_DMA_MODE);
}

/**
 * dweqos_tx_clean:
 * @priv: private driver structure
 * Description: it reclaims resources after transmission completes.
 */
static void dweqos_tx_clean(struct stmmac_priv *priv)
{
	unsigned int txsize = priv->tx_desc_count;

	spin_lock(&priv->tx_lock);

	while (priv->dirty_tx != priv->cur_tx) {
		int last;
		unsigned int entry = priv->dirty_tx % txsize;
		struct dma_tx_desc *p = priv->dma_tx + entry;
		struct dweqos_buffer *buffer_info = priv->tx_buffer_info + entry;
		struct sk_buff *skb = buffer_info->skbuff;

		/* Check if the descriptor is owned by the DMA. */
		if (desc_get_tx_owner(p))
			break;

		/* if context */
		if ((p->tx.cntxt.ctxt == 1) && (p->tx.cntxt.cde == 1))
			pr_err("context descript error\n");

		/* Verify tx error by looking at the last segment */
		last = desc_get_tx_ls(p);
		if (likely(last)) {
			int tx_error =
				desc_get_tx_status(&priv->netdev->stats,
							  &priv->xstats, p,
							  priv->ioaddr);
			if (likely(tx_error == 0)) {
				priv->netdev->stats.tx_packets++;
				priv->xstats.tx_pkt_n++;
			} else
				priv->netdev->stats.tx_errors++;
		}

		if (netif_msg_tx_done(priv))
			pr_debug("%s: curr %d, dirty %d\n", __func__,
				 priv->cur_tx, priv->dirty_tx);

		if (likely(buffer_info->skbuff_dma)) {
			dma_unmap_single(priv->dev,
					 buffer_info->skbuff_dma,
					 buffer_info->skbuff_dma_len,
					 DMA_TO_DEVICE);
			buffer_info->skbuff_dma = 0;
			buffer_info->skbuff_dma_len = 0;
		}

		if (likely(skb != NULL)) {
			dev_kfree_skb(skb);
			buffer_info->skbuff = NULL;
		}

		desc_release_tx_desc(p);

		priv->dirty_tx++;
	}

	if (unlikely(netif_queue_stopped(priv->netdev) &&
		     dweqos_tx_avail(priv) > STMMAC_TX_THRESH(priv))) {
		netif_tx_lock(priv->netdev);
		if (netif_queue_stopped(priv->netdev) &&
		     dweqos_tx_avail(priv) > STMMAC_TX_THRESH(priv)) {
			if (netif_msg_tx_done(priv))
				pr_debug("%s: restart transmit\n", __func__);
			netif_wake_queue(priv->netdev);
		}
		netif_tx_unlock(priv->netdev);
	}
	spin_unlock(&priv->tx_lock);
}

static inline void stmmac_enable_dma_irq(struct stmmac_priv *priv)
{
	priv->hw->dma->enable_dma_irq(priv->ioaddr);
}

static inline void stmmac_disable_dma_irq(struct stmmac_priv *priv)
{
	priv->hw->dma->disable_dma_irq(priv->ioaddr);
}

static int stmmac_has_work(struct stmmac_priv *priv)
{
	unsigned int has_work = 0;
	int rxret, tx_work = 0;

	rxret = desc_get_rx_owner(priv->dma_rx +
		(priv->cur_rx % priv->rx_desc_count));

	if (priv->dirty_tx != priv->cur_tx)
		tx_work = 1;

	if (likely(!rxret || tx_work))
		has_work = 1;

	return has_work;
}

static inline void _stmmac_schedule(struct stmmac_priv *priv)
{
	if (likely(stmmac_has_work(priv))) {
		stmmac_disable_dma_irq(priv);
		napi_schedule(&priv->napi);
	}
}

/**
 * stmmac_tx_err:
 * @priv: pointer to the private device structure
 * Description: it cleans the descriptors and restarts the transmission
 * in case of errors.
 */
static void stmmac_tx_err(struct stmmac_priv *priv)
{
	netif_stop_queue(priv->netdev);

	priv->hw->dma->stop_tx(priv->ioaddr);
	dma_free_tx_skbufs(priv);
	desc_init_tx_desc(priv->dma_tx, priv->tx_desc_count);
	priv->dirty_tx = 0;
	priv->cur_tx = 0;
	priv->hw->dma->start_tx(priv->ioaddr);

	priv->netdev->stats.tx_errors++;
	netif_wake_queue(priv->netdev);
}

static void stmmac_mtl_interrupt(struct stmmac_priv *priv)
{
	priv->hw->dma->mtl_interrupt(priv->ioaddr);
}

static void stmmac_dma_interrupt(struct stmmac_priv *priv)
{
	int status;

	status = priv->hw->dma->dma_interrupt(priv->ioaddr, &priv->xstats);
	if (likely(status == handle_tx_rx)) {
		_stmmac_schedule(priv);
	}
	else if (unlikely(status == tx_hard_error_bump_tc)) {
		/* Try to bump up the dma threshold on this failure */
		if (unlikely(tc != SF_DMA_MODE) && (tc <= 256)) {
			tc += 64;
			priv->hw->dma->dma_mode(priv->ioaddr, tc, SF_DMA_MODE);
			priv->xstats.threshold = tc;
		}
	} else if (unlikely(status == tx_hard_error)) {
		printk("tx_rx hard error\n");
		stmmac_tx_err(priv);
	}
}

static void stmmac_mmc_setup(struct stmmac_priv *priv)
{
	unsigned int mode = MMC_CNTRL_RESET_ON_READ | MMC_CNTRL_COUNTER_RESET |
			    MMC_CNTRL_PRESET | MMC_CNTRL_FULL_HALF_PRESET;

	/* Mask MMC irq, counters are managed in SW and registers
	 * are cleared on each READ eventually. */
	dwmac_mmc_intr_all_mask(priv->ioaddr);
/*
	if (priv->mac_cap.rmon) {
		dwmac_mmc_ctrl(priv->ioaddr, mode);
		memset(&priv->mmc, 0, sizeof(struct stmmac_counters));
	} else
		pr_info(" No MAC Management Counters available\n");
*/
}

static u32 stmmac_get_synopsys_id(struct stmmac_priv *priv)
{
	u32 hwid = priv->hw->synopsys_uid;

	/* Only check valid Synopsys Id because old MAC chips
	 * have no HW registers where get the ID */
	if (likely(hwid)) {
		u32 uid = ((hwid & 0x0000ff00) >> 8);
		u32 synid = (hwid & 0x000000ff);

		pr_info("stmmac - user ID: 0x%x, Synopsys ID: 0x%x\n",
			uid, synid);

		return synid;
	}
	return 0;
}

/**
 * dweqos_get_hw_features
 * @priv : private device pointer
 * Description:
 *  new GMAC chip generations have a new register to indicate the
 *  presence of the optional feature/functions.
 *  This can be also used to override the value passed through the
 *  platform and necessary for old MAC10/100 and GMAC chips.
 */
static int dweqos_get_hw_features(struct stmmac_priv *priv)
{
	u32 hw_cap = 0;
	u32 ret=0;
	if (priv->hw->mac->get_hw_feature) {
		/* HW feature 0 */
		hw_cap = priv->hw->mac->get_hw_feature(priv->ioaddr, 0);
		ret |= hw_cap;
		priv->mac_cap.mbps_10_100 = (hw_cap & MAC_HW_FEAT0_MIISEL);
		priv->mac_cap.mbps_1000 = (hw_cap & MAC_HW_FEAT0_GMIISEL) >> 1;
		priv->mac_cap.half_duplex = (hw_cap & MAC_HW_FEAT0_HDSEL) >> 2;
		priv->mac_cap.pcs = (hw_cap & MAC_HW_FEAT0_PCSSEL) >> 3;
		priv->mac_cap.vlan_hash_filter = (hw_cap & MAC_HW_FEAT0_VLHASH) >> 4;
		priv->mac_cap.sma_mdio = (hw_cap & MAC_HW_FEAT0_SMASEL) >> 5;
		priv->mac_cap.pmt_remote_wake_up =
			(hw_cap & MAC_HW_FEAT0_RWKSEL) >> 6;
		priv->mac_cap.pmt_magic_frame =
			(hw_cap & MAC_HW_FEAT0_MGKSEL) >> 7;
		/* MMC */
		priv->mac_cap.rmon = (hw_cap & MAC_HW_FEAT0_MMCSEL) >> 8;
		priv->mac_cap.arpoe = (hw_cap & MAC_HW_FEAT0_ARPOFFSEL) >> 9;

		/* IEEE 1588-2008*/
		priv->mac_cap.time_stamp =
			(hw_cap & MAC_HW_FEAT0_TSSEL) >> 12;
		/* 802.3az - Energy-Efficient Ethernet (EEE) */
		priv->mac_cap.eee = (hw_cap & MAC_HW_FEAT0_EEESEL) >> 13;
		/* TX and RX csum */
		priv->mac_cap.tx_coe = (hw_cap & MAC_HW_FEAT0_TXCOESEL) >> 14;
		priv->mac_cap.rx_coe = (hw_cap & MAC_HW_FEAT0_RXCOESEL) >> 16;
		priv->mac_cap.multi_addr =
			(hw_cap & MAC_HW_FEAT0_ADDMACADRSEL) >> 22;
		priv->mac_cap.multi_addr_32 =
			(hw_cap & MAC_HW_FEAT0_MACADR32SEL) >> 23;
		priv->mac_cap.multi_addr_64 =
			(hw_cap & MAC_HW_FEAT0_MACADR64SEL) >> 24;
		priv->mac_cap.time_stamp_src =
			(hw_cap & MAC_HW_FEAT0_TSSTSSEL) >> 25;
		priv->mac_cap.sa_vlan_ins =
			(hw_cap & MAC_HW_FEAT0_SAVLANINS) >> 27;
		//priv->mac_cap.phy_intf_sel =
		//	(hw_cap & MAC_HW_FEAT0_ACTPHYIF) >> 28;

		/* HW feature 1 */
		hw_cap = priv->hw->mac->get_hw_feature(priv->ioaddr, 1);
		ret |= hw_cap;
		priv->mac_cap.rxfifo_size = (hw_cap & MAC_HW_FEAT1_RXFIFOSIZE);
		priv->mac_cap.txfifo_size = (hw_cap & MAC_HW_FEAT1_TXFIFOSIZE) >> 6;
		priv->mac_cap.time_stamp_hiword = (hw_cap & MAC_HW_FEAT1_ADVTHWORD) >> 13;
		priv->mac_cap.dcb = (hw_cap & MAC_HW_FEAT1_DCBEN) >> 16;
		priv->mac_cap.split_header = (hw_cap & MAC_HW_FEAT1_SPHEN) >> 17;
		priv->mac_cap.tso = (hw_cap & MAC_HW_FEAT1_TSOEN) >> 18;
		priv->mac_cap.dma_debug = (hw_cap & MAC_HW_FEAT1_DBGMEMA) >> 19;
		priv->mac_cap.av = (hw_cap & MAC_HW_FEAT1_AVSEL) >> 20;
		priv->mac_cap.low_power = (hw_cap & MAC_HW_FEAT1_LPMODEEN) >> 23;
		priv->mac_cap.hash_table_size = (hw_cap & MAC_HW_FEAT1_HASHTBLSZ) >> 24;
		priv->mac_cap.l3l4_filter_num = (hw_cap & MAC_HW_FEAT1_L3L4FNUM) >> 27;

		/* HW feature 2 */
		hw_cap = priv->hw->mac->get_hw_feature(priv->ioaddr, 2);
		ret |= hw_cap;
		/* TX and RX number of queues */
		priv->mac_cap.number_rx_queue = (hw_cap & MAC_HW_FEAT2_RXQCNT);
		priv->mac_cap.number_tx_queue = (hw_cap & MAC_HW_FEAT2_TXQCNT) >> 6;
		/* TX and RX number of channels */
		priv->mac_cap.number_rx_channel = (hw_cap & MAC_HW_FEAT2_RXCHCNT) >> 12;
		priv->mac_cap.number_tx_channel = (hw_cap & MAC_HW_FEAT2_TXCHCNT) >> 18;
		/* PPS output*/
		priv->mac_cap.number_pps = (hw_cap & MAC_HW_FEAT2_PPSOUTNUM) >> 24;
		priv->mac_cap.number_aux_snapshot = (hw_cap & MAC_HW_FEAT2_AUXSNAPNUM) >> 28;
	}

	return ret;
}

static void stmmac_check_ether_addr(struct stmmac_priv *priv)
{
	/* verify if the MAC address is valid, in case of failures it
	 * generates a random MAC address */
	if (!is_valid_ether_addr(priv->netdev->dev_addr)) {
		priv->hw->mac->get_umac_addr((void __iomem *)
					     priv->netdev->base_addr,
					     priv->netdev->dev_addr, 0);
		if  (!is_valid_ether_addr(priv->netdev->dev_addr))
			random_ether_addr(priv->netdev->dev_addr);
	}
	pr_warning("%s: device MAC address %pM\n", priv->netdev->name,
						   priv->netdev->dev_addr);
}

static int stmmac_init_dma_engine(struct stmmac_priv *priv)
{
	int pbl = DEFAULT_DMA_PBL, fixed_burst = 0, burst_len = 0;
	int mixed_burst = 0;

	/* Some DMA parameters can be passed from the platform;
	 * in case of these are not passed we keep a default
	 * (good for all the chips) and init the DMA! */
	if (priv->plat->dma_cfg) {
		pbl = priv->plat->dma_cfg->pbl;
		fixed_burst = priv->plat->dma_cfg->fixed_burst;
		mixed_burst = priv->plat->dma_cfg->mixed_burst;
		burst_len = priv->plat->dma_cfg->burst_len;
	}

	return priv->hw->dma->init(priv->ioaddr, pbl, fixed_burst, mixed_burst,
				   burst_len, priv->dma_tx_phy,
				   priv->dma_rx_phy, priv->tx_desc_count, priv->rx_desc_count, priv->dma_buf_sz);
}

/**
 *  stmmac_open - open entry point of the driver
 *  @dev : pointer to the device structure.
 *  Description:
 *  This function is the open entry point of the driver.
 *  Return value:
 *  0 on success and an appropriate (-)ve integer as defined in errno.h
 *  file on failure.
 */
static int stmmac_open(struct net_device *dev)
{
	struct stmmac_priv *priv = netdev_priv(dev);
	int ret;

	stmmac_check_ether_addr(priv);

	ret = dweqos_init_phy(dev);
	if (unlikely(ret)) {
		netdev_err(dev, "%s: Cannot attach to PHY (error: %d)\n", __func__, ret);
		return ret;
	}

	/* Create and initialize the TX/RX descriptors chains. */
	priv->tx_desc_count = TX_DESC_CNT;
	priv->rx_desc_count = RX_DESC_CNT;
	priv->dma_buf_sz = STMMAC_ALIGN(buf_sz);
	priv->cur_mss = 0;
	priv->cur_vlan_ctag = 0;
	ret = dweqos_init_dma_desc_rings(dev);
	if (ret < 0) {
		pr_err("%s: DMA descriptors initialization failed\n", __func__);
		goto dma_desc_error;
	}

	/* DMA initialization and SW reset */
	ret = stmmac_init_dma_engine(priv);
	if (ret < 0) {
		pr_err("%s: DMA initialization failed\n", __func__);
		goto open_error;
	}

	stmmac_disable_dma_irq(priv);

	/* [Vincent] moved after dma init, otherwise the setting was reset */
	/* Enable the IPC (Checksum Offload) and check if the feature has been
	 * enabled during the core configuration. */
	ret = priv->hw->mac->rx_ipc(priv->ioaddr);
	if (!ret) {
		pr_warning(" RX IPC Checksum Offload not configured.\n");
		priv->plat->rx_coe = STMMAC_RX_COE_NONE;
	}

	/* Copy the MAC addr into the HW  */
	priv->hw->mac->set_umac_addr(priv->ioaddr, dev->dev_addr, 0);

	/* If required, perform hw setup of the bus. */
	if (priv->plat->bus_setup)
		priv->plat->bus_setup(priv->ioaddr);

	/* Initialize the MAC Core */
	priv->hw->mac->core_init(priv->ioaddr);

	netdev_update_features(dev);

	/* Request the IRQ lines */
	ret = request_irq(dev->irq, stmmac_interrupt,
			 IRQF_SHARED, dev->name, dev);
	if (unlikely(ret < 0)) {
		pr_err("%s: ERROR: allocating the IRQ %d (error: %d)\n",
		       __func__, dev->irq, ret);
		goto open_error;
	}

	/* Enable the MAC Rx/Tx */
	stmmac_set_mac(priv->ioaddr, true);

	/* Set the HW DMA mode and the COE */
	stmmac_dma_operation_mode(priv);

	/* Extra statistics */
	memset(&priv->xstats, 0, sizeof(struct stmmac_extra_stats));
	priv->xstats.threshold = tc;

	stmmac_mmc_setup(priv);

	/* Start the ball rolling... */
	pr_debug("%s: DMA RX/TX processes started...\n", dev->name);
	priv->hw->dma->start_tx(priv->ioaddr);
	priv->hw->dma->start_rx(priv->ioaddr);

	/* Dump DMA/MAC registers */
	if (netif_msg_hw(priv)) {
		priv->hw->mac->dump_regs(priv->ioaddr);
		priv->hw->dma->dump_regs(priv->ioaddr);
	}

	if (priv->phydev)
		phy_start(priv->phydev);

	napi_enable(&priv->napi);
	netif_start_queue(dev);

	stmmac_enable_dma_irq(priv);
	return 0;

open_error:
	free_dma_desc_resources(priv);
dma_desc_error:
	if (priv->phydev)
		phy_disconnect(priv->phydev);

	return ret;
}

/**
 *  stmmac_release - close entry point of the driver
 *  @dev : device pointer.
 *  Description:
 *  This is the stop entry point of the driver.
 */
static int stmmac_release(struct net_device *dev)
{
	struct stmmac_priv *priv = netdev_priv(dev);

	/* Stop and disconnect the PHY */
	if (priv->phydev) {
		phy_stop(priv->phydev);
		phy_disconnect(priv->phydev);
	}

	netif_stop_queue(dev);

	napi_disable(&priv->napi);

	/* Free the IRQ lines */
	free_irq(dev->irq, dev);

	/* Stop TX/RX DMA and clear the descriptors */
	priv->hw->dma->stop_tx(priv->ioaddr);
	priv->hw->dma->stop_rx(priv->ioaddr);

	/* Release and free the Rx/Tx resources */
	free_dma_desc_resources(priv);

	/* Disable the MAC Rx/Tx */
	stmmac_set_mac(priv->ioaddr, false);

	netif_carrier_off(dev);

	return 0;
}

/**
 *  dweqos_xmit_frame:
 *  @skb : the socket buffer
 *  @dev : device pointer
 *  Description : Tx entry point of the driver.
 */
static netdev_tx_t dweqos_xmit_frame(struct sk_buff *skb, struct net_device *dev)
{
	struct stmmac_priv *priv = netdev_priv(dev);
	unsigned int txsize = priv->tx_desc_count;
	unsigned int entry;
	struct dweqos_buffer *buffer_info;
	int i, csum_insertion = 0, vlan_insertion = 0;
	struct dma_tx_desc *desc, *first;
	unsigned int nopaged_len = skb_headlen(skb);
	unsigned int datalen = 0;
	unsigned int offset = 0, size;

	unsigned int nr_frags = skb_shinfo(skb)->nr_frags;
	/* packet */
	unsigned short mss;
	unsigned short vlan_ctag;
	int desc_count = 0;
	unsigned int is_tso = skb_is_gso(skb), is_vlan = vlan_tx_tag_present(skb);
	unsigned int tso_context = 0, vlan_context = 0;
	unsigned int f;

	u32 hdr_len;

	/* reserve for mss context descriptor */
	mss = skb_shinfo(skb)->gso_size;

	if (unlikely(is_tso && (mss != priv->cur_mss))) {
		tso_context = 1;
	}

	if (is_vlan) {
		vlan_ctag = vlan_tx_tag_get(skb);
		if (unlikely(vlan_ctag != priv->cur_vlan_ctag))
		vlan_context = 1;
	}

	desc_count += (tso_context | vlan_context);

	/* Reserve descriptors for nopaged skb->data */
	desc_count += TXD_USE_COUNT(nopaged_len);

	/* Reserve descriptors for frags */
	for (f = 0; f < nr_frags; f++) {
		desc_count += TXD_USE_COUNT(skb_frag_size(&skb_shinfo(skb)->frags[f]));
	}

	/* Check that there are enough descripors avaliable */
	if (unlikely(dweqos_tx_avail(priv) < desc_count)) {
		if (!netif_queue_stopped(dev)) {
			PDEBUG(" Not enougth Tx descriptors, Tx queue stopped\n");
			netif_stop_queue(dev);
		}
		return NETDEV_TX_BUSY;
	}

	/* ??move before check desc? */
	spin_lock(&priv->tx_lock);

	entry = priv->cur_tx % txsize;
	desc = priv->dma_tx + entry;
	buffer_info = priv->tx_buffer_info + entry;
	first = desc;

	if (unlikely(tso_context || vlan_context)) {
		if (tso_context) {
			PDEBUG("  TSO context descriptor, mss=%u\n", mss);
			desc->tx.cntxt.mss = mss;
			desc->tx.cntxt.ctxt = 1;
			desc->tx.cntxt.tcmssv = 1;
			priv->cur_mss = mss;
		}

		if (vlan_context) {
			PDEBUG("  VLAN context descriptor, ctag=%u\n", vlan_ctag);
			desc->tx.cntxt.ctxt = 1;
			desc->tx.cntxt.vltv = 1;
			desc->tx.cntxt.vt = vlan_ctag;
			priv->cur_vlan_ctag = vlan_ctag;
		}

		priv->cur_tx++;
		entry = priv->cur_tx % txsize;
		desc = priv->dma_tx + entry;
		buffer_info = priv->tx_buffer_info + entry;
	}

	buffer_info->skbuff = skb;
	buffer_info->skbuff_dma = dma_map_single(priv->dev, skb->data,
						 nopaged_len, DMA_TO_DEVICE);
	buffer_info->skbuff_dma_len = nopaged_len;

	csum_insertion = (skb->ip_summed == CHECKSUM_PARTIAL);

	if (is_tso) {
		/* send hdr */
		hdr_len = skb_transport_offset(skb) + tcp_hdrlen(skb);
		desc->tx.read.buffer1 = buffer_info->skbuff_dma;
		desc->tx.read.buffer1_length = hdr_len;
		desc->tx.read.first_desc = 1;
		desc->tx.read.tse = 1;
		desc->tx.read.payload_length = skb->len-hdr_len;
		desc->tx.read.tcp_hdr_length = (tcp_hdr(skb)->doff);
		if (hdr_len < nopaged_len) {
			desc->tx.read.buffer2 = buffer_info->skbuff_dma + hdr_len;
			desc->tx.read.buffer2_length = nopaged_len - hdr_len;
		}
		if (is_vlan)
			desc->tx.read.vlan_insertion = vtir_insert;
	} else {
		//pr_debug(">>> frame to be transmitted: ");
		//pr_info("no tso....hdr with data? %d %d \n", nopaged_len, skb->data_len );
		desc->tx.read.buffer1 = buffer_info->skbuff_dma;
		desc_prepare_tx_desc(desc, 1, nopaged_len, skb->len,
				csum_insertion, vlan_insertion);
	}

	if (desc != first)
		desc->tx.cntxt.own = 1;
	//dump = (struct tmp *) desc;
	//pr_info("desc[%d] %08x %08x %08x %08x\n", entry, dump->a, dump->b, dump->c, dump->d);
	//pr_info("is_gso:(%d)....skb_headlen: %d skb->data_len: %d nr_frags: %d \n", skb_is_gso(skb), nopaged_len, skb->data_len, nfrags);

	//print_pkt(skb->data, nopaged_len);

	/* send payload frags */
	for (i = 0; i < nr_frags; i++) {
		const skb_frag_t *frag = &skb_shinfo(skb)->frags[i];
		datalen =  skb_frag_size(frag);
		offset = 0;
		while (datalen) {
			entry = (++priv->cur_tx) % txsize;
			desc = priv->dma_tx + entry;
			buffer_info = priv->tx_buffer_info + entry;

			size = min(datalen, TX_MAX_BUF_SIZE);

			/* record skb info */
			buffer_info->skbuff = NULL;
			buffer_info->skbuff_dma = skb_frag_dma_map(priv->dev, frag, offset, size,
					DMA_TO_DEVICE);
			buffer_info->skbuff_dma_len = size;
			/* setup descriptor */
			desc->tx.read.buffer1 = buffer_info->skbuff_dma;
			desc_prepare_tx_desc(desc, 0, size, skb->len, csum_insertion, vlan_insertion);
			wmb();
			desc_set_tx_owner(desc);
			wmb();
			datalen -= size;
			offset += size;
		}
	}

	/* Interrupt on completition only for the latest segment */
	desc_close_tx_desc(desc);
	//pr_info("desc[%d] %08x %08x %08x %08x\n", entry, dump->a, dump->b, dump->c, dump->d);
	wmb();

	/* To avoid raise condition */
	desc_set_tx_owner(first);
	wmb();

	priv->cur_tx++;

	if (netif_msg_pktdata(priv)) {
		printk("%s: curr %d dirty=%d entry=%d, first=%p, nr_frags=%d\n",
		       __func__, (priv->cur_tx % txsize),
		       (priv->dirty_tx % txsize), entry, first, nr_frags);

		dweqos_display_ring((void *)priv->dma_tx, txsize);

		pr_debug(">>> frame to be transmitted: ");
		print_pkt(skb->data, skb->len);
	}

	if (unlikely(dweqos_tx_avail(priv) <= (MAX_SKB_FRAGS + 1))) {
		//if (netif_msg_hw(priv))
			pr_debug("%s: stop transmitted packets\n", __func__);
		netif_stop_queue(dev);
	}

	dev->stats.tx_bytes += skb->len;

	skb_tx_timestamp(skb);

	priv->hw->dma->enable_dma_transmission(priv->ioaddr);

	spin_unlock(&priv->tx_lock);

	return NETDEV_TX_OK;
}

static inline void stmmac_rx_refill(struct stmmac_priv *priv)
{
	unsigned int rxsize = priv->rx_desc_count;
	int bfsize = priv->dma_buf_sz;
	struct dma_rx_desc *p;
	struct dweqos_buffer *buffer_info;
	u32 value;

	for (; priv->cur_rx - priv->dirty_rx > 0; priv->dirty_rx++) {
		unsigned int entry = priv->dirty_rx % rxsize;

		p = priv->dma_rx + entry;
		buffer_info = priv->rx_buffer_info + entry;

		if (likely(buffer_info->skbuff == NULL)) {
			struct sk_buff *skb;

			skb =  netdev_alloc_skb_ip_align(priv->netdev, bfsize);

			if (unlikely(skb == NULL))
				break;

			buffer_info->skbuff = skb;
			buffer_info->skbuff_dma =
			    dma_map_single(priv->dev, skb->data, bfsize,
					   DMA_FROM_DEVICE);

			desc_reset_rx_desc(p);
			desc_set_rx_buf(p, buffer_info->skbuff_dma);

			if (netif_msg_rx_status(priv))
				pr_debug("\trefill entry #%d\n", entry);
		}
		wmb();
		desc_set_rx_owner(p);
		wmb();
	}

	value = readl(priv->ioaddr + DMA_CH0_RX_TAIL_ADDR);
	writel(value, priv->ioaddr + DMA_CH0_RX_TAIL_ADDR);
}

static int stmmac_rx(struct stmmac_priv *priv, int limit)
{
	unsigned int rxsize = priv->rx_desc_count;
	unsigned int entry = priv->cur_rx % rxsize;
	unsigned int next_entry;
	unsigned int count = 0;
	struct dma_rx_desc *p;
	struct dweqos_buffer *buffer_info;
	if (netif_msg_hw(priv)) {
		pr_debug(">>> stmmac_rx: descriptor ring:\n");
		dweqos_display_ring((void *)priv->dma_rx, rxsize);
	}

	while (count < limit) {
		int status;

		p = priv->dma_rx + entry;
		buffer_info = priv->rx_buffer_info + entry;

		if (desc_get_rx_owner(p))
			break;

		count++;

		next_entry = (++priv->cur_rx) % rxsize;

		prefetch(priv->dma_rx + next_entry);
		/* read the status of the incoming frame */
		status = (desc_get_rx_status(&priv->netdev->stats,
						    &priv->xstats, p));
		if (unlikely(status == discard_frame)) {
			priv->netdev->stats.rx_errors++;
			/*desc will be written back by device*/

			buffer_info->skbuff = NULL;
			dma_unmap_single(priv->dev,
					buffer_info->skbuff_dma,
					priv->dma_buf_sz,
					DMA_FROM_DEVICE);
		} else {
			struct sk_buff *skb;
			int frame_len;
			frame_len = desc_get_rx_frame_len(p,
					priv->plat->rx_coe);

			/* ACS is set; GMAC core strips PAD/FCS for IEEE 802.3
			 * Type frames (LLC/LLC-SNAP) */
			//if (unlikely(status != llc_snap))
			if (frame_len < 1536)
				frame_len -= ETH_FCS_LEN;

			if (netif_msg_rx_status(priv)) {
				printk("\tdesc: %p [entry %d] buff=0x%x\n",
						p, entry, buffer_info->skbuff_dma);
				if (frame_len > ETH_FRAME_LEN)
					printk("\tframe size %d, COE: %d\n",
							frame_len, status);
			}

			skb = buffer_info->skbuff;
			if (unlikely(!skb)) {
				pr_err("%s: Inconsistent Rx descriptor chain\n",
						priv->netdev->name);
				priv->netdev->stats.rx_dropped++;
				break;
			}
			prefetch(skb->data - NET_IP_ALIGN);
			buffer_info->skbuff = NULL;

			skb_put(skb, frame_len);
			dma_unmap_single(priv->dev,
					buffer_info->skbuff_dma,
					priv->dma_buf_sz, DMA_FROM_DEVICE);

			if (netif_msg_pktdata(priv)) {
				pr_info("frame received (%dbytes)\n", frame_len);
				print_pkt(skb->data, frame_len);
			}

			skb->protocol = eth_type_trans(skb, priv->netdev);

			if (unlikely(!priv->plat->rx_coe)) {
				skb_checksum_none_assert(skb);
			} else {
				skb->ip_summed = CHECKSUM_UNNECESSARY;
			}

			napi_gro_receive(&priv->napi, skb);

			priv->netdev->stats.rx_packets++;
			priv->netdev->stats.rx_bytes += frame_len;
		}
		entry = next_entry;
	}

	stmmac_rx_refill(priv);

	priv->xstats.rx_pkt_n += count;

	return count;
}

/**
 *  stmmac_poll - stmmac poll method (NAPI)
 *  @napi : pointer to the napi structure.
 *  @budget : maximum number of packets that the current CPU can receive from
 *	      all interfaces.
 *  Description :
 *   This function implements the the reception process.
 *   Also it runs the TX completion thread
 */
static int stmmac_poll(struct napi_struct *napi, int budget)
{
	struct stmmac_priv *priv = container_of(napi, struct stmmac_priv, napi);
	int work_done = 0;

	priv->xstats.poll_n++;
	dweqos_tx_clean(priv);

	work_done = stmmac_rx(priv, budget);
	if (work_done < budget) {
		napi_complete(napi);
		stmmac_enable_dma_irq(priv);
	}
	return work_done;
}

/**
 *  stmmac_tx_timeout
 *  @dev : Pointer to net device structure
 *  Description: this function is called when a packet transmission fails to
 *   complete within a reasonable time. The driver will mark the error in the
 *   netdev structure and arrange for the device to be reset to a sane state
 *   in order to transmit a new packet.
 */
static void stmmac_tx_timeout(struct net_device *dev)
{
	struct stmmac_priv *priv = netdev_priv(dev);

	/* Clear Tx resources and restart transmitting again */
	stmmac_tx_err(priv);
}

/* Configuration changes (passed on by ifconfig) */
static int stmmac_config(struct net_device *dev, struct ifmap *map)
{
	if (dev->flags & IFF_UP)	/* can't act on a running interface */
		return -EBUSY;

	/* Don't allow changing the I/O address */
	if (map->base_addr != dev->base_addr) {
		pr_warning("%s: can't change I/O address\n", dev->name);
		return -EOPNOTSUPP;
	}

	/* Don't allow changing the IRQ */
	if (map->irq != dev->irq) {
		pr_warning("%s: can't change IRQ number %d\n",
		       dev->name, dev->irq);
		return -EOPNOTSUPP;
	}

	/* ignore other fields */
	return 0;
}

/**
 *  stmmac_set_rx_mode - entry point for multicast addressing
 *  @dev : pointer to the device structure
 *  Description:
 *  This function is a driver entry point which gets called by the kernel
 *  whenever multicast addresses must be enabled/disabled.
 *  Return value:
 *  void.
 */
static void stmmac_set_rx_mode(struct net_device *dev)
{
	struct stmmac_priv *priv = netdev_priv(dev);

	spin_lock(&priv->lock);
	priv->hw->mac->set_filter(dev, priv->synopsys_id);
	spin_unlock(&priv->lock);
}

static netdev_features_t stmmac_fix_features(struct net_device *dev, netdev_features_t features)
{
	struct stmmac_priv *priv = netdev_priv(dev);

	if (priv->plat->rx_coe == STMMAC_RX_COE_NONE)
		features &= ~NETIF_F_RXCSUM;
	else if (priv->plat->rx_coe == STMMAC_RX_COE_TYPE1)
		features &= ~NETIF_F_IPV6_CSUM;
	if (!priv->plat->tx_coe)
		features &= ~NETIF_F_ALL_CSUM;

	/* Some GMAC devices have a bugged Jumbo frame support that
	 * needs to have the Tx COE disabled for oversized frames
	 * (due to limited buffer sizes). In this case we disable
	 * the TX csum insertionin the TDES and not use SF. */
	if (priv->plat->bugged_jumbo && (dev->mtu > ETH_DATA_LEN))
		features &= ~NETIF_F_ALL_CSUM;

	return features;
}

static irqreturn_t stmmac_interrupt(int irq, void *dev_id)
{
	struct net_device *ndev = (struct net_device *)dev_id;
	struct stmmac_priv *priv = netdev_priv(ndev);
	u32 intr = readl(priv->ioaddr + DMA_INTR_STATUS);

	//if (!netif_device_present(dev))
	//?	return IRQ_HANDLED;

	if (unlikely(!ndev)) {
		pr_err("%s: invalid dev pointer\n", __func__);
		return IRQ_NONE;
	}

	/* To handle GMAC own interrupts */
	if (intr & DMA_INTR_STATUS_MACIS) {
		priv->hw->mac->host_irq_status((void __iomem *) ndev->base_addr);
	}

	if (intr & DMA_INTR_STATUS_MTLIS) {
		stmmac_mtl_interrupt(priv);
	}

	/* To handle DMA interrupts */
	if (intr & DMA_INTR_STATUS_DC0IS) {
		stmmac_dma_interrupt(priv);
	}
	return IRQ_HANDLED;
}

#ifdef CONFIG_NET_POLL_CONTROLLER
/* Polling receive - used by NETCONSOLE and other diagnostic tools
 * to allow network I/O with interrupts disabled. */
static void stmmac_poll_controller(struct net_device *dev)
{
	disable_irq(dev->irq);
	stmmac_interrupt(dev->irq, dev);
	enable_irq(dev->irq);
}
#endif

/**
 *  stmmac_ioctl - Entry point for the Ioctl
 *  @dev: Device pointer.
 *  @rq: An IOCTL specefic structure, that can contain a pointer to
 *  a proprietary structure used to pass information to the driver.
 *  @cmd: IOCTL command
 *  Description:
 *  Currently there are no special functionality supported in IOCTL, just the
 *  phy_mii_ioctl(...) can be invoked.
 */
static int stmmac_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
	struct stmmac_priv *priv = netdev_priv(dev);
	int ret;

	if (!netif_running(dev))
		return -EINVAL;

	if (!priv->phydev)
		return -EINVAL;

	ret = phy_mii_ioctl(priv->phydev, rq, cmd);

	return ret;
}

#ifdef CONFIG_DWEQOS_DEBUG_FS
static struct dentry *stmmac_fs_dir;
static struct dentry *stmmac_rings_status;
static struct dentry *stmmac_mac_cap;

static int stmmac_sysfs_ring_read(struct seq_file *seq, void *v)
{
	struct tmp_s {
		unsigned int a;
		unsigned int b;
		unsigned int c;
		unsigned int d;
	};
	int i,j;
	struct net_device *dev = seq->private;
	struct stmmac_priv *priv = netdev_priv(dev);

	seq_printf(seq,"===HW feature===\n");
	seq_printf(seq,"reg[0x%08x] %08x\n", (u32)priv->ioaddr + (71 * 4), readl(priv->ioaddr + (71 * 4)));
	seq_printf(seq,"reg[0x%08x] %08x\n", (u32)priv->ioaddr + (72 * 4), readl(priv->ioaddr + (72 * 4)));
	seq_printf(seq,"reg[0x%08x] %08x\n", (u32)priv->ioaddr + (73 * 4), readl(priv->ioaddr + (73 * 4)));

	seq_printf(seq, "===MAC register===\n");
	for (i = 0; i < 50; i++) {
		seq_printf(seq,"reg[0x%08x] %08x\n", (u32)priv->ioaddr + (i * 4), readl(priv->ioaddr + (i * 4)));
	}
	seq_printf(seq, "===MTL register===\n");
	for (i = 0; i < 14; i++) {
		seq_printf(seq,"reg[0x%08x] %08x\n", (u32)priv->ioaddr +0xc00+ (i * 4), readl(priv->ioaddr +0xc00+ (i * 4)));
	}
	for (i = 64; i < 80; i++) {
		seq_printf(seq,"reg[0x%08x] %08x\n", (u32)priv->ioaddr +0xc00+ (i * 4), readl(priv->ioaddr +0xc00+ (i * 4)));
	}
	seq_printf(seq, "===DMA register===\n");
	for (i = 0; i < 5; i++) {
		seq_printf(seq,"reg[0x%08x] %08x\n", (u32)priv->ioaddr +0x1000+ (i * 4), readl(priv->ioaddr +0x1000+ (i * 4)));
	}

	seq_printf(seq, "===DMA Channel 0 register===\n");
	for (i = 0; i < 25; i++) {
		seq_printf(seq,"reg[0x%08x] %08x\n", (u32)priv->ioaddr +0x1100+ (i * 4), readl(priv->ioaddr +0x1100+ (i * 4)));
	}

	seq_printf(seq, "=======================\n");
	seq_printf(seq, " RX descriptor ring\n");
	seq_printf(seq, "=======================\n");

	for (i = 0; i < priv->rx_desc_count; i++) {
		struct tmp_s *x = (struct tmp_s *)(priv->dma_rx + i);
		seq_printf(seq, "%d [0x%08x] [0x%08x]: 0x%08x 0x%08x 0x%08x 0x%08x",
			   i, (u32)(priv->dma_rx + i),(unsigned int)(priv->dma_rx_phy+sizeof(struct dma_rx_desc)*i),
			   (u32)(x->a), (u32)(x->b),
			   (u32)(x->c), (u32)(x->d));
		seq_printf(seq, "\n");
	}

	seq_printf(seq, "\n");
	seq_printf(seq, "=======================\n");
	seq_printf(seq, "  TX descriptor ring\n");
	seq_printf(seq, "=======================\n");

	for (i = 0; i < priv->tx_desc_count; i++) {
		struct tmp_s *x = (struct tmp_s *)(priv->dma_tx + i);
		seq_printf(seq, "%d [0x%08x] [0x%08x]: 0x%08x 0x%08x 0x%08x 0x%08x, skblen %d",
			   i, (u32)(priv->dma_tx + i),(unsigned int)(priv->dma_tx_phy+sizeof(struct dma_tx_desc)*i),
			   (u32)(x->a), (u32)(x->b),
			   (u32)(x->c), (u32)(x->d), priv->tx_buffer_info[i].skbuff_dma_len);
		seq_printf(seq, "\n");

		if (priv->tx_buffer_info[i].skbuff) {
			struct sk_buff *skb = priv->tx_buffer_info[i].skbuff;
			int shiftlen =0;
			sys_print_pkt(seq, skb->data, skb_headlen(skb), shiftlen);
			shiftlen+=skb_headlen(skb);

			for (j = 0; j < skb_shinfo(skb)->nr_frags; j++) {
				const skb_frag_t *frag = &skb_shinfo(skb)->frags[j];
				int len =  skb_frag_size(frag);

				sys_print_pkt(seq, skb_frag_address(frag), len, shiftlen);
				shiftlen+=len;
			}


		}
	}

	return 0;
}

static int stmmac_sysfs_ring_open(struct inode *inode, struct file *file)
{
	return single_open(file, stmmac_sysfs_ring_read, inode->i_private);
}

static const struct file_operations stmmac_rings_status_fops = {
	.owner = THIS_MODULE,
	.open = stmmac_sysfs_ring_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int stmmac_sysfs_mac_cap_read(struct seq_file *seq, void *v)
{
	struct net_device *dev = seq->private;
	struct stmmac_priv *priv = netdev_priv(dev);

	if (!priv->hw_cap_support) {
		seq_printf(seq, "MAC HW features not supported\n");
		return 0;
	}

	seq_printf(seq, "==============================\n");
	seq_printf(seq, "\tMAC HW features\n");
	seq_printf(seq, "==============================\n");

	seq_printf(seq, "\t10/100 Mbps %s\n",
		   (priv->mac_cap.mbps_10_100) ? "Y" : "N");
	seq_printf(seq, "\t1000 Mbps %s\n",
		   (priv->mac_cap.mbps_1000) ? "Y" : "N");
	seq_printf(seq, "\tHalf duple %s\n",
		   (priv->mac_cap.half_duplex) ? "Y" : "N");
	seq_printf(seq, "\tPCS (TBI/SGMII/RTBI PHY interfatces): %s\n",
		   (priv->mac_cap.pcs) ? "Y" : "N");
	seq_printf(seq, "\tVLAN Hash Filter: %s\n",
		   (priv->mac_cap.vlan_hash_filter) ? "Y" : "N");
	seq_printf(seq, "\tSMA (MDIO) Interface: %s\n",
		   (priv->mac_cap.sma_mdio) ? "Y" : "N");
	seq_printf(seq, "\tPMT Remote wake up: %s\n",
		   (priv->mac_cap.pmt_remote_wake_up) ? "Y" : "N");
	seq_printf(seq, "\tPMT Magic Frame: %s\n",
		   (priv->mac_cap.pmt_magic_frame) ? "Y" : "N");
	seq_printf(seq, "\tRMON module: %s\n",
		   (priv->mac_cap.rmon) ? "Y" : "N");
	seq_printf(seq, "\tARP offload: %s\n",
		   (priv->mac_cap.arpoe) ? "Y" : "N");
	seq_printf(seq, "\tIEEE 1588-2008 v2 Time Stamp:%s\n",
		   (priv->mac_cap.time_stamp) ? "Y" : "N");
	seq_printf(seq, "\t802.3az - Energy-Efficient Ethernet (EEE) %s\n",
		   (priv->mac_cap.eee) ? "Y" : "N");
	seq_printf(seq, "\tChecksum Offload in TX: %s\n",
		   (priv->mac_cap.tx_coe) ? "Y" : "N");
	seq_printf(seq, "\tChecksum Offload in RX: %s\n",
		   (priv->mac_cap.rx_coe) ? "Y" : "N");
	seq_printf(seq, "\tMultiple MAC address registers 1-31: %s\n",
		   (priv->mac_cap.multi_addr) ? "Y" : "N");
	seq_printf(seq, "\tMultiple MAC address registers 32-63: %s\n",
		   (priv->mac_cap.multi_addr_32) ? "Y" : "N");
	seq_printf(seq, "\tMultiple MAC address registers 64-127: %s\n",
		   (priv->mac_cap.multi_addr_64) ? "Y" : "N");
	seq_printf(seq, "\tTimestamp System Time Source: ");
	switch (priv->mac_cap.time_stamp_src) {
		case 1:
			seq_printf(seq, "%s\n", "Internal");
			break;
		case 2:
			seq_printf(seq, "%s\n", "External");
			break;
		case 3:
			seq_printf(seq, "%s\n", "Both");
			break;
		default:
			seq_printf(seq, "%s\n", "Not support");
	}
	seq_printf(seq, "\tSource Address or VLAN Insertion: %s\n", (priv->mac_cap.sa_vlan_ins) ? "Y" : "N");

	/* HW Feature 1 */
	seq_printf(seq, "\tRXFIFO: %d\n",
		   1 << (priv->mac_cap.rxfifo_size + 7));
	seq_printf(seq, "\tTXFIFO: %d\n",
		   1 << (priv->mac_cap.txfifo_size + 7));
	seq_printf(seq, "\tIEEE 1588 High Word Register: %s\n", (priv->mac_cap.time_stamp_hiword) ? "Y" : "N");
	seq_printf(seq, "\tDCB features: %s\n", (priv->mac_cap.dcb) ? "Y" : "N");
	seq_printf(seq, "\tSplit Header Feature: %s\n", (priv->mac_cap.split_header) ? "Y" : "N");
	seq_printf(seq, "\tTCP Segmentation Offload: %s\n", (priv->mac_cap.tso) ? "Y" : "N");
	seq_printf(seq, "\tDMA Debug Registers: %s\n", (priv->mac_cap.dma_debug) ? "Y" : "N");
	seq_printf(seq, "\tAV features: %s\n", (priv->mac_cap.av) ? "Y" : "N");
	seq_printf(seq, "\tLow Power Mode: %s\n", (priv->mac_cap.low_power) ? "Y" : "N");
	seq_printf(seq, "\tHash Table Size: %d\n", (priv->mac_cap.hash_table_size) * 64);
	seq_printf(seq, "\tTotal number of L3 or L4 Filters: %d\n", (priv->mac_cap.l3l4_filter_num));

	/* HW Feature 2 */
	seq_printf(seq, "\tNumber of MTL Receive Queues: %d\n", (priv->mac_cap.number_rx_queue + 1));
	seq_printf(seq, "\tNumber of MTL Transmit Queues: %d\n", (priv->mac_cap.number_tx_queue + 1));
	seq_printf(seq, "\tNumber of DMA Receive Channels: %d\n", (priv->mac_cap.number_rx_channel + 1));
	seq_printf(seq, "\tNumber of DMA Transmit Channels: %d\n", (priv->mac_cap.number_tx_channel + 1));
	seq_printf(seq, "\tNumberof PPS Outputs: %d\n", priv->mac_cap.number_pps);
	seq_printf(seq, "\tNumber of Auxiliary Snapshot Inputs: %d\n", priv->mac_cap.number_aux_snapshot);
	return 0;
}

static int stmmac_sysfs_mac_cap_open(struct inode *inode, struct file *file)
{
	return single_open(file, stmmac_sysfs_mac_cap_read, inode->i_private);
}

static const struct file_operations stmmac_mac_cap_fops = {
	.owner = THIS_MODULE,
	.open = stmmac_sysfs_mac_cap_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int stmmac_init_fs(struct net_device *dev)
{
	/* Create debugfs entries */
	stmmac_fs_dir = debugfs_create_dir(STMMAC_RESOURCE_NAME, NULL);

	if (!stmmac_fs_dir || IS_ERR(stmmac_fs_dir)) {
		pr_err("ERROR %s, debugfs create directory failed\n",
		       STMMAC_RESOURCE_NAME);

		return -ENOMEM;
	}

	/* Entry to report DMA RX/TX rings */
	stmmac_rings_status = debugfs_create_file("descriptors_status",
					   S_IRUGO, stmmac_fs_dir, dev,
					   &stmmac_rings_status_fops);

	if (!stmmac_rings_status || IS_ERR(stmmac_rings_status)) {
		pr_info("ERROR creating stmmac ring debugfs file\n");
		debugfs_remove(stmmac_fs_dir);

		return -ENOMEM;
	}

	/* Entry to report the DMA HW features */
	stmmac_mac_cap = debugfs_create_file("mac_cap", S_IRUGO, stmmac_fs_dir,
					     dev, &stmmac_mac_cap_fops);

	if (!stmmac_mac_cap || IS_ERR(stmmac_mac_cap)) {
		pr_info("ERROR creating stmmac MMC debugfs file\n");
		debugfs_remove(stmmac_rings_status);
		debugfs_remove(stmmac_fs_dir);

		return -ENOMEM;
	}

	return 0;
}

static void stmmac_exit_fs(void)
{
	debugfs_remove(stmmac_rings_status);
	debugfs_remove(stmmac_mac_cap);
	debugfs_remove(stmmac_fs_dir);
}
#endif /* CONFIG_DWEQOS_DEBUG_FS */

static const struct net_device_ops stmmac_netdev_ops = {
	.ndo_open = stmmac_open,
	.ndo_start_xmit = dweqos_xmit_frame,
	.ndo_stop = stmmac_release,
	.ndo_change_mtu = eth_change_mtu,
	.ndo_fix_features = stmmac_fix_features,
	.ndo_set_rx_mode = stmmac_set_rx_mode,
	.ndo_tx_timeout = stmmac_tx_timeout,
	.ndo_do_ioctl = stmmac_ioctl,
	.ndo_set_config = stmmac_config,
#ifdef CONFIG_NET_POLL_CONTROLLER
	.ndo_poll_controller = stmmac_poll_controller,
#endif
	.ndo_set_mac_address = eth_mac_addr,
	.ndo_validate_addr = eth_validate_addr,
};

/**
 *  stmmac_hw_init - Init the MAC device
 *  @priv : pointer to the private device structure.
 *  Description: this function detects which MAC device
 *  (GMAC/MAC10-100) has to attached, checks the HW capability
 *  (if supported) and sets the driver's features (for example
 *  to use the ring or chaine mode or support the normal/enh
 *  descriptor structure).
 */
int stmmac_hw_init(struct stmmac_priv *priv)
{
	int ret = 0;
	struct mac_device_info *mac;

	/* Identify the MAC HW device */
	mac = dweqos_mac_setup(priv->ioaddr);

	if (!mac)
		return -ENOMEM;

	priv->hw = mac;

	/* Get and dump the chip ID */
	priv->synopsys_id = stmmac_get_synopsys_id(priv);

	priv->hw_cap_support = dweqos_get_hw_features(priv);
	if (priv->hw_cap_support) {
		pr_info(" DMA HW capability register supported");

		/* We can override some gmac/dma configuration fields: e.g.
		 * tso, tx_coe (e.g. that are passed through the
		 * platform) with the values from the HW capability
		 * register (if supported).
		 */
		priv->plat->pmt = priv->mac_cap.pmt_remote_wake_up;

		priv->plat->tx_coe = priv->mac_cap.tx_coe;
		priv->plat->rx_coe = priv->mac_cap.rx_coe;
	} else
		pr_info(" No HW DMA feature register supported");

	/* Enable the IPC (Checksum Offload) and check if the feature has been
	 * enabled during the core configuration. */
	ret = priv->hw->mac->rx_ipc(priv->ioaddr);
	if (!ret) {
		pr_warning(" RX IPC Checksum Offload not configured.\n");
		priv->plat->rx_coe = STMMAC_RX_COE_NONE;
	}

	if (priv->plat->rx_coe)
		pr_info(" RX Checksum Offload Engine supported\n");
	if (priv->plat->tx_coe)
		pr_info(" TX Checksum insertion supported\n");

	if (priv->plat->pmt) {
		pr_info(" Wake-Up On Lan supported\n");
		device_set_wakeup_capable(priv->dev, 1);
	}

	return ret;
}

/**
 * stmmac_probe - Initialization of the adapter .
 * @dev : device pointer
 * Description: The function initializes the network device structure for
 * the STMMAC driver. It also calls the low level routines
 * in order to init the HW (i.e. the DMA engine)
 */
int stmmac_probe(struct net_device *ndev)
{
	int ret = 0;
	struct stmmac_priv *priv = netdev_priv(ndev);

	/* Verify driver arguments */
	stmmac_verify_args();

	ether_setup(ndev);

	ndev->netdev_ops = &stmmac_netdev_ops;

	dweqos_set_ethtool_ops(ndev);

	ndev->hw_features = NETIF_F_SG | NETIF_F_IP_CSUM | NETIF_F_IPV6_CSUM |
			NETIF_F_RXCSUM | NETIF_F_TSO;
	ndev->features |= ndev->hw_features;
#ifdef STMMAC_VLAN_TAG_USED
	ndev->features |= NETIF_F_HW_VLAN_CTAG_TX | NETIF_F_HW_VLAN_CTAG_RX;
#endif
	ndev->priv_flags |= IFF_UNICAST_FLT;
	ndev->watchdog_timeo = msecs_to_jiffies(ETH_TX_TIMEOUT);
	priv->msg_enable = netif_msg_init(debug, default_msg_level);
	priv->pause = pause;

	if (flow_ctrl)
		priv->flow_ctrl = FLOW_AUTO;	/* RX/TX pause on */

	netif_napi_add(ndev, &priv->napi, stmmac_poll, 64);

	spin_lock_init(&priv->lock);
	spin_lock_init(&priv->tx_lock);

	ret = register_netdev(ndev);
	if (ret) {
		netif_napi_del(&priv->napi);
		pr_err("%s: ERROR %i registering the device\n", __func__, ret);
		return -ENODEV;
	}

	/* Setup clk_csr for MDC clock divider */
	dweqos_clk_csr_set(priv);

	return ret;
}
