/*
 *  drivers/net/ethernet/vatics/dweqos/dweqos_mdio.c
 *
 *  Copyright (C) 2007-2014  VATICS Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/mii.h>
#include <linux/phy.h>
#include <linux/slab.h>
#include <asm/io.h>

#include "stmmac.h"
#include "dweqos.h"

#define MII_BUSY        0x00000001
#define MII_WRITE       0x00000004
#define MII_READ        0x0000000c

#define MII_ADDRSHIFT   (21)
#define MII_REGSHIFT    (16)
#define MII_REGMSK      (0x1F << 16)
#define MII_ADDRMSK     (0x1F << 21)

static int dweqos_mdio_busy_wait(void __iomem *ioaddr, unsigned int mii_addr)
{
	unsigned long curr;
	unsigned long finish = jiffies + 3 * HZ;

	do {
		curr = jiffies;
		if (readl(ioaddr + mii_addr) & MII_BUSY)
			cpu_relax();
		else
			return 0;
	} while (!time_after_eq(curr, finish));

	return -EBUSY;
}

static int dweqos_mdio_read(struct mii_bus *bus, int phyaddr, int phyreg)
{
	struct net_device *ndev = bus->priv;
	struct stmmac_priv *priv = netdev_priv(ndev);

	int data;
	u32 value = ((phyaddr << MII_ADDRSHIFT) & MII_ADDRMSK) |
		       ((phyreg << MII_REGSHIFT) & MII_REGMSK) |
		       MII_READ;

	value |= MII_BUSY | ((priv->clk_csr & 0xF) << 8);

	if (dweqos_mdio_busy_wait(priv->ioaddr, GMAC_MII_ADDR))
		return -EBUSY;

	writel(value, priv->ioaddr + GMAC_MII_ADDR);

	if (dweqos_mdio_busy_wait(priv->ioaddr, GMAC_MII_ADDR))
		return -EBUSY;

	/* Read the data from the MII data register */
	data = (int)readl(priv->ioaddr + GMAC_MII_DATA);

	return data;
}

static int dweqos_mdio_write(struct mii_bus *bus, int phyaddr, int phyreg,
			     u16 phydata)
{
	struct net_device *ndev = bus->priv;
	struct stmmac_priv *priv = netdev_priv(ndev);

	u32 value = ((phyaddr << MII_ADDRSHIFT) & MII_ADDRMSK) |
		    ((phyreg << MII_REGSHIFT) & MII_REGMSK) |
		    MII_WRITE;

	value |= MII_BUSY | ((priv->clk_csr & 0xF) << 8);

	/* Wait until any existing MII operation is complete */
	if (dweqos_mdio_busy_wait(priv->ioaddr, GMAC_MII_ADDR))
		return -EBUSY;

	/* Set the MII address register to write */
	writel(phydata, priv->ioaddr + GMAC_MII_DATA);
	writel(value, priv->ioaddr + GMAC_MII_ADDR);

	/* Wait until any existing MII operation is complete */
	return dweqos_mdio_busy_wait(priv->ioaddr, GMAC_MII_ADDR);
}

static int dweqos_mdio_reset(struct mii_bus *bus)
{
	return 0;
}

static int dweqos_mdio_probe(struct net_device *ndev)
{
	struct stmmac_priv *priv = netdev_priv(ndev);
	struct phy_device *phydev = NULL;

	if (priv->plat->phy_addr == -1)
		phydev = phy_find_first(priv->mii_bus);
	else
		phydev = priv->mii_bus->phy_map[priv->plat->phy_addr];

	if (!phydev) {
		netdev_err(ndev, "No PHY found");
		return -ENXIO;
	}

	priv->phydev = phydev;

	return 0;
}

int dweqos_mdio_register(struct net_device *ndev)
{
	int err = 0;
	struct mii_bus *new_bus;
	struct stmmac_priv *priv = netdev_priv(ndev);
	struct stmmac_mdio_bus_data *mdio_bus_data = priv->plat->mdio_bus_data;

	if (!mdio_bus_data)
		return 0;

	priv->mii_bus = new_bus = mdiobus_alloc();
	if (priv->mii_bus == NULL)
		return -ENOMEM;

	new_bus->name = "dweqos_mdio";
	snprintf(new_bus->id, MII_BUS_ID_SIZE, "%s-%x",	new_bus->name,  priv->id);
	new_bus->priv = ndev;
	new_bus->read = &dweqos_mdio_read;
	new_bus->write = &dweqos_mdio_write;
	new_bus->reset = &dweqos_mdio_reset;
	new_bus->parent = priv->dev;
	new_bus->phy_mask = mdio_bus_data->phy_mask;
	new_bus->irq = mdio_bus_data->irqs;

	err = mdiobus_register(new_bus);
	if (err != 0) {
		netdev_err(ndev, "Cannot register as MDIO bus\n");
		goto bus_register_fail;
	}

	err = dweqos_mdio_probe(ndev);
	if (err) {
		netdev_err(ndev, "Failed to probe MDIO bus");
		goto err_unreg_mdio;
	}

	return 0;

err_unreg_mdio:
	mdiobus_unregister(priv->mii_bus);
bus_register_fail:
	mdiobus_free(priv->mii_bus);
	return err;
}

int dweqos_mdio_unregister(struct net_device *ndev)
{
	struct stmmac_priv *priv = netdev_priv(ndev);

	if (!priv->mii_bus)
		return 0;

	mdiobus_unregister(priv->mii_bus);
	priv->mii_bus->priv = NULL;
	mdiobus_free(priv->mii_bus);
	priv->mii_bus = NULL;
	priv->phydev = NULL;

	return 0;
}
