/*
 *  /drivers/net/ethernet/vatics/dweqos/dweqos_pesaro.c
 *
 * Copyright (C) 2013-2018  VATICS Inc.
 *
 * Author: ChangHsien Ho <vincent.ho@vatics.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <mach/platform.h>
#include <linux/phy.h>
#include "dweqos_platform.h"

/* -------------------------------------------------- */
static const char *phy_modes[] = {
	[PHY_INTERFACE_MODE_NA]         = "",
	[PHY_INTERFACE_MODE_MII]        = "mii",
	[PHY_INTERFACE_MODE_GMII]       = "gmii",
	[PHY_INTERFACE_MODE_SGMII]      = "sgmii",
	[PHY_INTERFACE_MODE_TBI]        = "tbi",
	[PHY_INTERFACE_MODE_RMII]       = "rmii",
	[PHY_INTERFACE_MODE_RGMII]      = "rgmii",
	[PHY_INTERFACE_MODE_RGMII_ID]   = "rgmii-id",
	[PHY_INTERFACE_MODE_RGMII_RXID] = "rgmii-rxid",
	[PHY_INTERFACE_MODE_RGMII_TXID] = "rgmii-txid",
	[PHY_INTERFACE_MODE_RTBI]       = "rtbi",
	[PHY_INTERFACE_MODE_SMII]       = "smii",
};

static inline char *cmdline_find_option(char *str)
{
	extern char *saved_command_line;
	return strstr(saved_command_line, str);
}

static phy_interface_t parse_cmdlin_phymod(void)
{
	phy_interface_t phy_mode = PHY_INTERFACE_MODE_GMII;
	char *cp;
	int i, rgmii = 0;

	/* Parse rgmii=0/1 or phy_mode = rgmii/gmii/mii/rmii */
	if ((cp = cmdline_find_option("rgmii="))) {
		cp+=strlen("rgmii=");
		if (get_option(&cp, &rgmii) && rgmii)
			phy_mode = PHY_INTERFACE_MODE_RGMII;
	}

	if ((cp = cmdline_find_option("phy_mode="))) {
		cp+=strlen("phy_mode=");
		for (i = 0; i < ARRAY_SIZE(phy_modes); i++)
			if (!strncasecmp(phy_modes[i], cp, strlen(phy_modes[i]))) {
				phy_mode = i;
				break;
			}
	}

	switch (phy_mode) {
		case PHY_INTERFACE_MODE_MII:
		case PHY_INTERFACE_MODE_RMII:
		case PHY_INTERFACE_MODE_RGMII:
			break;
		default:
			printk("PHY mode %s is Unsupported, force to RMII\n", cp);
			phy_mode = PHY_INTERFACE_MODE_RGMII;
	}
	pr_debug("PHY Mode: %s\n", phy_modes[phy_mode]);

	return phy_mode;
}

static struct stmmac_mdio_bus_data pesaro_dweqos_mdio_data = {
	.phy_mask = 0x1, /* mask addr 0, which is valid for most phy brodcast */
	.irqs = NULL,
	/*
	   .phy_reset = NULL
	 */
};

static void pesaro_dwmac_fix_mac_speed(void *bsp_priv, unsigned int speed)
{
	struct dweqos_platform_data *plat_dat = (struct dweqos_platform_data *) bsp_priv;
	void __iomem *sysc_base = ioremap(VPL_SYSC_MMR_BASE, SZ_4K);
	u32 reg = readl(sysc_base + 0x30);

	/*
	 *   [25:24] GMAC_EXT_CLK_FREQ_SEL:
	 *                     2'b00 125MHz; 2'b01 25MHz; 2'b10 2.5MHz; 2'b11 50MHz(RMII)
	 *   [21:20] GMAC_RMII_CLK_PHASE_SEL:
	 *                     2'b00 Normal; 2'b01 90; 2'b10 180; 2'b11 270
	 *   [18:16] GMAC_MODE_SEL:
	 *                     3'b000 GMII/MII; 3'b001 RGMII; 3'b100 RMII
	 *      [12] GMAC_TX_CLK_SRC:
	 *                     1'b0 External MAC<-PHY; 1'b1 Internal MAC->PHY;
	 *       [8] GMAC_RX_CLK_SRC:
	 *                     1'b0 External MAC<-PHY (default); 1'b1 Internal MAC->PHY (used in RMII)
	 *       [4] GMAC_REDUCED_MODE_EN: (TX_CLK PAD to PHY)
	 *	               1'b0: 180 degree, 1'b1: 90 degree
	 *   [01:00] GMAC_INT_CLK_FREQ_SEL:
	 *                     2'b00 125MHz; 2'b01 25MHz; 2'b10 2.5MHz;
	 */

	switch (plat_dat->interface) {
		case PHY_INTERFACE_MODE_MII:
			/* speed = 100 or 10, Tx Clk and Rx Clk are external */
			reg &= ~(1<<12 | 1<<8);
			/* reg = 0x1<<16 | 0x1<<12 | 0x0 << 8; */
			break;
		case PHY_INTERFACE_MODE_RGMII:
			switch (speed) {
				case 1000:
					reg = 0x0<<24 | 0x1<<16 | 0x1 <<12 | 0x0<<8 | 0x1 << 4 | 0x0<<0;
					break;
				case 100:
					reg = 0x1<<24 | 0x1<<16 | 0x1 <<12 | 0x0<<8 | 0x1 << 4 | 0x1<<0;
					break;
				case 10:
					reg = 0x2<<24 | 0x1<<16 | 0x1 <<12 | 0x0<<8 | 0x1 << 4 | 0x2<<0;
					break;
			}
			break;
		case PHY_INTERFACE_MODE_RMII:
			switch (speed) {
				case 100:
					reg = 0x3<<24 | 0x4<<16 | 0x1<<12 | 0x1<<8 | 0x1<<0 ;
					break;
				case 10:
					reg = 0x3<<24 | 0x4<<16 | 0x1<<12 | 0x1<<8 | 0x2<<0 ;
					break;
			}
			break;
		default:
			break;
	}
	writel(reg, sysc_base+0x30);
	iounmap(sysc_base);
}

static void pesaro_dwmac_bus_setup(void __iomem *ioaddr)
{
	pr_debug("[GMAC]: %s\n", __FUNCTION__);
}

static int pesaro_dwmac_init(struct platform_device *pdev)
{
	struct dweqos_platform_data *plat_dat = pdev->dev.platform_data;
	void __iomem *sysc_base = ioremap(VPL_SYSC_MMR_BASE, SZ_4K);
	u32 value;
	//u32 txclk_dir = 0, rxclk_dir = 1, mode_sel = 0, tx_ext_freq = 0, rx_freq = 0;

	/* ToDo: PAD/CLOCK enable/disable */
	pr_debug("[GMAC]: %s mac id %d\n", __FUNCTION__, pdev->id);

	/*
	 * Assign Interface. Overwrite by steps
	 * 1. Command line
	 * 2. Chip modle limitation.
	 */
	plat_dat->interface = parse_cmdlin_phymod();

	/* Chip modle limitation, None*/

	pr_info("GMAC: PHY mode is %s\n", phy_modes[plat_dat->interface]);

	/*
	 * To set GMAC work at correct interface mode.
	 * The following settings must be confirmed when reset MAC DMA Engine.
	 *
	 * 1. Set SYSC_GMAC_CTRL 0x30
	 *   [25:24] GMAC_EXT_CLK_FREQ_SEL:
	 *                     2'b00 125MHz; 2'b01 25MHz; 2'b10 2.5MHz; 2'b11 50MHz(RMII)
	 *   [21:20] GMAC_RMII_CLK_PHASE_SEL:
	 *                     2'b00 Normal; 2'b01 90; 2'b10 180; 2'b11 270
	 *   [18:16] GMAC_MODE_SEL:
	 *                     3'b000 GMII/MII; 3'b001 RGMII; 3'b100 RMII
	 *      [12] GMAC_TX_CLK_SRC:
	 *                     1'b0 Internal MAC->PHY; 1'b1 External MAC<-PHY
	 *       [8] GMAC_RX_CLK_SRC:
	 *                     1'b0 External MAC<-PHY (default); 1'b1 Internal MAC->PHY (used in RMII)
	 *       [4] GMAC_REDUCED_MODE_EN: (TX_CLK PAD to PHY)
	 *	                            1'b0: 180 degree, 1'b1: 90 degree
	 *   [01:00] GMAC_INT_CLK_FREQ_SEL:
	 *                     2'b00 125MHz; 2'b01 25MHz; 2'b10 2.5MHz;
	 *
	 *          | GMAC_RX_CLK_DIR | GMAC_TX_CLK_DIR | GMAC_MODE_SEL | GMAC_EXT_CLK_FREQ_SEL
	 * --------------------------------------------------------------------------------------
	 * MII      |    1'b1         |    1'b0         |  3'b000       |
	 * RGMII    |    1'b1         |    1'b0         |  3'b001       |
	 * RMII     |    1'b1         |    1'b0         |  3'b100       |	 2'b11 (50MHz for PHY)
	 *
	 * 2. Reset GMAC DMA in driver.
	 *
	 * After Reset DMA Engine, the Tx direction and Speed are free to change
	 * accroding to PHY's speed.
	 */

	switch (plat_dat->interface) {
		case PHY_INTERFACE_MODE_MII:
			value = 0x00000000;
			break;
		case PHY_INTERFACE_MODE_RGMII:
			value = 0x00011010;
			break;
		case PHY_INTERFACE_MODE_RMII:
			value = 0x03041100;
			break;
		default:
			value = 0x00000000;
	}
	writel(value, sysc_base+0x30);
	iounmap(sysc_base);

	plat_dat->bsp_priv = pdev->dev.platform_data;
	return 0;
};

static void pesaro_dwmac_exit(struct platform_device *pdev)
{
	pr_debug("%s\n", __FUNCTION__);
};

/*
 * tx_coe:
 *          For Pesaro, GMAC 3.42a without HW capability register.
 *          Accroding to databook, the IPC_FULL_OFFLOAD include Rx and Tx.
 *          Assigend manual tx_cot = 1
 *
 *          For Beethoven, GMAC 3.70 with HW capability register,
 *          The tx_coe can be check by driver.
 *
 *	    Note that, if tx_coe is avaliable in Hardware, MUST use Store-and-Forward Mode
 *
 * .bus_setup: call by driver when open device
 * .init:      call by driver probe
 * .exit:      call by driver exit
 * .pbl:       The driver can set x4(Pesaro) or x8 (Beethoven)
 *             The PBL will effect available space in FIFO.
 *             Frame Size < FIFO Depth - PBL - 3 FIFO Locations
 *             ex. 2048 (FIFO depth) /8(data bus width in byte) - 64 - 3
 *                = 189 beats = 1512 bytes
 *             Generally, net mtu is 1500, so we can't set PBL too large,ex. 32x4,
 *	       this will result not enought 1000 bytes threshold.
 *	       Reference to section 4.7 in databook v3.70
 *	       In TCP case, the maximum frame is 1460+IPHeader(20)+TCP Header(20)+eth Header(14)=1514 bytes.
 *	       Use pbl=16*4=64 , 1512 bytes threshold is also not enought.
 *	       Set to pbl=8*4=32, 1768 bytes
 */
static struct stmmac_dma_cfg pesaro_dweqos_dma_cfg = {
	.pbl = 4,
	.fixed_burst = 1,
	.mixed_burst = 0,
	.burst_len = 0,
};

/* of_data specifying platform hardware feature and callbacks */
struct dweqos_platform_data pesaro_dweqos_data = {
	.phy_addr = -1,
	.mdio_bus_data = &pesaro_dweqos_mdio_data,
	.dma_cfg = &pesaro_dweqos_dma_cfg,
	.tx_coe = 1,
	.rx_coe = STMMAC_RX_COE_TYPE2,
	.fix_mac_speed = pesaro_dwmac_fix_mac_speed,
	.bus_setup = pesaro_dwmac_bus_setup,
	.init = pesaro_dwmac_init,
	.exit = pesaro_dwmac_exit,
};
