/*
 *  drivers/net/ethernet/vatics/dweqos/dweqos_platform.c
 *
 * Copyright (C) 2013-2018  VATICS Inc.
 *
 * Author: ChangHsien Ho <vincent.ho@vatics.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/dma-mapping.h>
#include <linux/of_device.h>
#include <linux/etherdevice.h>
#include <linux/irqdomain.h>
#include "dweqos_platform.h"
#include "stmmac.h"

static const struct of_device_id dweqos_of_match[];

static struct dweqos_platform_data *dweqos_of_get_pdata(struct platform_device *pdev)
{
	const struct of_device_id *match;
	struct dweqos_platform_data *pdata = NULL;

	match = of_match_device(dweqos_of_match, &pdev->dev);
	if (match && match->data)
		pdata = (struct dweqos_platform_data *)match->data;

	/*
	 * struct device_node *np = pdev->dev.of_node;
	 * gpio = of_get_named_gpio_flags(np, "phy-reset-gpios", 0, NULL);
	 */

	return pdata;
}

static int dweqos_probe(struct platform_device *pdev)
{
	struct stmmac_priv *priv = NULL;
	struct net_device *netdev = NULL;
	struct device *dev = &pdev->dev;
	int ret = 0;
	struct resource *res;
	void __iomem *addr = NULL;
	struct dweqos_platform_data *pdata = NULL;
	int mac_virq;

	/* Get device I/O address */
	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	addr = devm_ioremap_resource(dev, res);
	if (IS_ERR(addr)) {
		dev_err(dev, "ioremap failed\n");
		return PTR_ERR(addr);
	}

	/* Get device IRQ number */
	mac_virq = platform_get_irq(pdev, 0);
	if (mac_virq < 0)
		return mac_virq;

	/* Initial platform dependance data */
	pdata = dweqos_of_get_pdata(pdev);
	if (!pdata) {
		dev_err(dev, "no platform data\n");
		return -ENODEV;
	}
	pdev->dev.platform_data = pdata;

	/* Set the DMA mask */
	ret = dma_set_mask(dev, DMA_BIT_MASK(32));
	if (ret == 0)
		dma_set_coherent_mask(dev, DMA_BIT_MASK(32));

	pdata->sysclk = devm_clk_get(dev, NULL);
	if (IS_ERR(pdata->sysclk)) {
		dev_err(dev, "devm_clk_get failed\n");
		return PTR_ERR(pdata->sysclk);
	}

	/* Custom initialisation (if needed)*/
	if (pdata->init) {
		ret = pdata->init(pdev);
		if (unlikely(ret))
			return ret;
	}

	/* initial network device and driver */
	netdev = alloc_etherdev(sizeof(struct stmmac_priv));
	if (!netdev) {
		dev_err(dev, "alloc_etherdev failed\n");
		return -ENOMEM;
	}
	SET_NETDEV_DEV(netdev, dev);
	netdev->base_addr = (unsigned int)addr;

	priv = netdev_priv(netdev);
	priv->netdev = netdev;
	priv->dev = dev;
	priv->id = pdev->id; //cahnge to pdev
	priv->plat = pdata;
	priv->ioaddr = addr;

	/* Init MAC and get the capabilities */
	stmmac_hw_init(priv);

	/* Network Device Registration */
	ret = stmmac_probe(netdev);
	if (ret < 0)
		goto out_free_netdev;

	pr_info("\t%s - (dev. name: %s - id: %d, IRQ #%d\n"
	       "\tIO base addr: 0x%p)\n", netdev->name, pdev->name,
	       pdev->id, netdev->irq, addr);

	platform_set_drvdata(pdev, netdev);

	/* MDIO bus Registration */
	ret = dweqos_mdio_register(netdev);
	if (ret < 0) {
		pr_debug("%s: MDIO bus (id: %d) registration failed",
			 __func__, pdev->id);
		goto out_unregister;
	}

	netdev->irq = mac_virq;

	/*
	 * On some platforms e.g. SPEAr the wake up irq differs from the mac irq
	 * The external wake up irq can be passed through the platform code
	 * named as "eth_wake_irq"
	 *
	 * In case the wake up interrupt is not passed from the platform
	 * so the driver will continue to use the mac irq (netdev->irq)
	 */
	priv->wol_irq = platform_get_irq_byname(pdev, "eth_wake_irq");
	if (priv->wol_irq == -ENXIO)
		priv->wol_irq = netdev->irq;

	pr_debug("STMMAC platform driver registration completed");

#ifdef CONFIG_DWEQOS_DEBUG_FS
	ret = stmmac_init_fs(netdev);
	if (ret < 0)
		pr_warning("\tFailed debugFS registration");
#endif

	return 0;

out_unregister:
	unregister_netdev(netdev);
out_free_netdev:
	free_netdev(netdev);
	platform_set_drvdata(pdev, NULL);

	return ret;
}

static int dweqos_remove(struct platform_device *pdev)
{
	struct net_device *netdev = platform_get_drvdata(pdev);
	struct stmmac_priv *priv = netdev_priv(netdev);

	pr_info("%s:\n\tremoving driver", __func__);

	priv->hw->dma->stop_rx(priv->ioaddr);
	priv->hw->dma->stop_tx(priv->ioaddr);

	stmmac_set_mac(priv->ioaddr, false);

	netif_carrier_off(netdev);

	dweqos_mdio_unregister(netdev);

	if (priv->plat->exit)
		priv->plat->exit(pdev);

	platform_set_drvdata(pdev, NULL);
	unregister_netdev(netdev);

#ifdef CONFIG_DWEQOS_DEBUG_FS
	stmmac_exit_fs();
#endif

	free_netdev(netdev);

	return 0;
}

static const struct of_device_id dweqos_of_match[] = {
	{ .compatible = "vatics,dweqos-v4.00a", .data = &pesaro_dweqos_data },
	{},
};
MODULE_DEVICE_TABLE(of, dweqos_of_match);

static struct platform_driver dweqos_driver = {
	.probe	= dweqos_probe,
	.remove	= dweqos_remove,
	.driver	= {
		.owner	= THIS_MODULE,
		.name	= "dweqos",
		.of_match_table = dweqos_of_match,
	}
};

module_platform_driver(dweqos_driver);
MODULE_DESCRIPTION("Designware Core Ethernet Quality-of-Service driver");
MODULE_AUTHOR("Vincent Ho <vincent.ho@vatics.com>");
MODULE_LICENSE("GPL");
