/*******************************************************************************
  This is the driver for the GMAC on-chip Ethernet controller for ST SoCs.
  DWC Ether MAC 10/100/1000 Universal version 3.41a  has been used for
  developing this code.

  This contains the functions to handle the dma.

  Copyright (C) 2007-2009  STMicroelectronics Ltd

  This program is free software; you can redistribute it and/or modify it
  under the terms and conditions of the GNU General Public License,
  version 2, as published by the Free Software Foundation.

  This program is distributed in the hope it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.

  The full GNU General Public License is included in this distribution in
  the file called "COPYING".

  Author: Giuseppe Cavallaro <peppe.cavallaro@st.com>
*******************************************************************************/

#include <asm/io.h>
#include "dweqos.h"
#include "dwmac1000.h"
#include "dwmac_dma.h"

static int dwmac1000_dma_init(void __iomem *ioaddr, int pbl, int fb,
			      int mb, int burst_len, u32 dma_tx, u32 dma_rx, u32 dma_tx_size, u32 dma_rx_size, u32 rx_bfsize)
{
	u32 value = readl(ioaddr + DMA_MODE);
	int limit;

	/* DMA SW reset */
	value |= DMA_MODE_SFT_RESET;
	writel(value, ioaddr + DMA_MODE);
	limit = 10;
	while (limit--) {
		if (!(readl(ioaddr + DMA_MODE) & DMA_MODE_SFT_RESET))
			break;
		mdelay(10);
	}
	if (limit < 0)
		return -EBUSY;

#ifdef CONFIG_STMMAC_DA
	value = DMA_BUS_MODE_DA;	/* Rx has priority over tx */
	writel(value, ioaddr + DMA_MODE);
#endif

	/* Set the Fixed burst mode */
	value = readl(ioaddr + DMA_SYSBUS_MODE);
	if (fb)
		value |= DMA_SYSBUS_MODE_UNDEF;

	/* Mixed Burst has no effect when fb is set */
	if (mb)
		value |= DMA_SYSBUS_MODE_MB;

	value |= DMA_SYSBUS_MODE_AAL;
	writel(value, ioaddr + DMA_SYSBUS_MODE);

	/*
	 * Set the DMA PBL (Programmable Burst Length) mode
	 * Before stmmac core 3.50 this mode bit was 4xPBL, and
	 * post 3.5 mode bit acts as 8*PBL.
	 * For core rev < 3.5, when the core is set for 4xPBL mode, the
	 * DMA transfers the data in 4, 8, 16, 32, 64 & 128 beats
	 * depending on pbl value.
	 * For core rev > 3.5, when the core is set for 8xPBL mode, the
	 * DMA transfers the data in 8, 16, 32, 64, 128 & 256 beats
	 * depending on pbl value.
	 */
	value = DMA_CH_CONTROL_PBLx8;
	writel(value, ioaddr + DMA_CH0_CONTROL);

	/* Set Tx Channel Control */
	value = (pbl << DMA_CH_TX_CONTROL_PBL_SHIFT) | DMA_CH_TX_CONTROL_TSE;//| DMA_CH_TX_CONTROL_OSP;
	writel(value, ioaddr + DMA_CH0_TX_CONTROL);

	/* Set Rx Channel Control */
	value = (pbl << DMA_CH_RX_CONTROL_PBL_SHIFT) | ((rx_bfsize&DMA_CH_RX_CONTROL_RBSZ_MASK) << DMA_CH_RX_CONTROL_RBSZ_SHIFT);
	writel(value, ioaddr + DMA_CH0_RX_CONTROL);

	/* Enable interrupts */
	writel(DMA_CH_INTR_DEFAULT_MASK, ioaddr + DMA_CH0_INTR_ENA);

	/* The base address of the RX/TX descriptor lists must be written into
	 * DMA CSR3 and CSR4, respectively. */
	writel(dma_tx, ioaddr + DMA_CH0_TX_BASE_ADDR);
	writel(dma_rx, ioaddr + DMA_CH0_RX_BASE_ADDR);
	writel(dma_tx_size - 1, ioaddr + DMA_CH0_TX_LEN_ADDR);
	writel(dma_rx_size - 1, ioaddr + DMA_CH0_RX_LEN_ADDR);

	writel(dma_tx + dma_tx_size * 16, ioaddr + DMA_CH0_TX_TAIL_ADDR);
	writel(dma_rx + dma_rx_size * 16, ioaddr + DMA_CH0_RX_TAIL_ADDR);
	return 0;
}

static void dwmac1000_dma_operation_mode(void __iomem *ioaddr, int txmode,
				    int rxmode)
{
	u32 txq = readl(ioaddr + MTL_TXQ0_OPERATION_MODE);
	u32 txc = readl(ioaddr + DMA_CH0_TX_CONTROL);
	u32 rxq = readl(ioaddr + MTL_RXQ0_OPERATION_MODE);

	if (txmode == SF_DMA_MODE) {
		CHIP_DBG(KERN_DEBUG "GMAC: enable TX store and forward mode\n");
		/* Transmit COE type 2 cannot be done in cut-through mode. */
		txq |= MTL_TXQ_OPERATION_MODE_TSF;
		/* Operating on second frame increase the performance
		 * especially when transmit store-and-forward is used.*/
		//txc |= DMA_CH_TX_CONTROL_OSP;
	} else {
		CHIP_DBG(KERN_DEBUG "GMAC: disabling TX store and forward mode"
			      " (threshold = %d)\n", txmode);
		txq &= ~MTL_TXQ_OPERATION_MODE_TSF;
		txq &= ~MTL_TXQ_OPERATION_MODE_TTC_MASK;
		/* Set the transmit threshold */
		if (txmode <= 32)
			txq |= MTL_TXQ_OPERATION_MODE_TTC_32;
		else if (txmode <= 64)
			txq |= MTL_TXQ_OPERATION_MODE_TTC_64;
		else if (txmode <= 128)
			txq |= MTL_TXQ_OPERATION_MODE_TTC_128;
		else if (txmode <= 192)
			txq |= MTL_TXQ_OPERATION_MODE_TTC_192;
		else
			txq |= MTL_TXQ_OPERATION_MODE_TTC_256;
	}

	if (rxmode == SF_DMA_MODE) {
		CHIP_DBG(KERN_DEBUG "GMAC: enable RX store and forward mode\n");
		rxq |= MTL_RXQ_OPERATION_MODE_RSF;
	} else {
		CHIP_DBG(KERN_DEBUG "GMAC: disabling RX store and forward mode"
			      " (threshold = %d)\n", rxmode);
		rxq &= ~MTL_RXQ_OPERATION_MODE_RSF;
		rxq &= ~MTL_RXQ_OPERATION_MODE_RTC_MASK;
		if (rxmode <= 32)
			rxq |= MTL_RXQ0_OPERATION_MODE_RTC_32;
		else if (rxmode <= 64)
			rxq |= MTL_RXQ0_OPERATION_MODE_RTC_64;
		else if (rxmode <= 96)
			rxq |= MTL_RXQ0_OPERATION_MODE_RTC_96;
		else
			rxq |= MTL_RXQ0_OPERATION_MODE_RTC_128;
	}

	writel(txq, ioaddr + MTL_TXQ0_OPERATION_MODE);
	writel(txc, ioaddr + DMA_CH0_TX_CONTROL);
	writel(rxq, ioaddr + MTL_RXQ0_OPERATION_MODE);

}

static void dwmac1000_dump_dma_regs(void __iomem *ioaddr)
{
	int i;
	pr_info(" DMA registers\n");
	for (i = 0; i < 22; i++) {
		if ((i < 9) || (i > 17)) {
			int offset = i * 4;
			pr_err("\t Reg No. %d (offset 0x%x): 0x%08x\n", i,
			       (DMA_MODE + offset),
			       readl(ioaddr + DMA_MODE + offset));
		}
	}
}

const struct stmmac_dma_ops dwmac1000_dma_ops = {
	.init = dwmac1000_dma_init,
	.dump_regs = dwmac1000_dump_dma_regs,
	.dma_mode = dwmac1000_dma_operation_mode,
	.enable_dma_transmission = dwmac_enable_dma_transmission,
	.enable_dma_irq = dwmac_enable_dma_irq,
	.disable_dma_irq = dwmac_disable_dma_irq,
	.start_tx = dwmac_dma_start_tx,
	.stop_tx = dwmac_dma_stop_tx,
	.start_rx = dwmac_dma_start_rx,
	.stop_rx = dwmac_dma_stop_rx,
	.dma_interrupt = dwmac_dma_interrupt,
	.mtl_interrupt = dwmac_mtl_interrupt,
};
