/*
 * Pinctrl data for the VATICS Pesaro pinmux
 *
 * Copyright (c) 2011-2014, VATICS CORPORATION.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 */

#include <linux/module.h>
#include <linux/of.h>
#include <linux/platform_device.h>
#include <linux/io.h>
#include <linux/pinctrl/pinctrl.h>
#include <linux/pinctrl/pinmux.h>
#include <linux/pinctrl/machine.h>
#include <linux/pinctrl/pinconf.h>
#include <linux/bitmap.h>
#include <linux/slab.h>

#include "core.h"
#include <mach/platform.h>
#include "pinctrl-pesaro.h"

DEFINE_SPINLOCK(pinmux_lock);
struct pesaro_group {
	const char *name;
	const unsigned int *pins;
	const unsigned num_pins;
	const unsigned int offset;
	const int bit;
	const int numbits;
};

struct pesaro_function {
	const char *name;
	const char * const *groups;
	unsigned ngroups;
};

struct pesaro_pinctrl_soc_data {
	unsigned ngpios;
	const struct pinctrl_pin_desc *pins;
	unsigned npins;
	const struct pesaro_function *functions;
	unsigned nfunctions;
	const struct pesaro_group *groups;
	unsigned ngroups;
};

struct pesaro_pmx {
	struct device *dev;
	struct pinctrl_dev *pctl;

	const struct pesaro_pinctrl_soc_data *soc;

	int nbanks;
	void __iomem **regs;
};

static struct pinctrl_gpio_range pesaro_pinctrl_gpio_range = {
	.name = "Pesaro-default-pin",
	.id = 0,
	.base = 0,
};

static const struct pinctrl_pin_desc pesaro_pins[] = {
	PINCTRL_PIN(GPIOC_0_IO_DATA_0, "GPIOC_0_IO_DATA_0"),
	PINCTRL_PIN(GPIOC_0_IO_DATA_1, "GPIOC_0_IO_DATA_1"),
	PINCTRL_PIN(GPIOC_0_IO_DATA_2, "GPIOC_0_IO_DATA_2"),
	PINCTRL_PIN(GPIOC_0_IO_DATA_3, "GPIOC_0_IO_DATA_3"),
	PINCTRL_PIN(GPIOC_0_IO_DATA_4, "GPIOC_0_IO_DATA_4"),
	PINCTRL_PIN(GPIOC_0_IO_DATA_5, "GPIOC_0_IO_DATA_5"),
	PINCTRL_PIN(GPIOC_0_IO_DATA_6, "GPIOC_0_IO_DATA_6"),
	PINCTRL_PIN(GPIOC_0_IO_DATA_7, "GPIOC_0_IO_DATA_7"),
	PINCTRL_PIN(GPIOC_0_IO_DATA_8, "GPIOC_0_IO_DATA_8"),
	PINCTRL_PIN(GPIOC_0_IO_DATA_9, "GPIOC_0_IO_DATA_9"),
	PINCTRL_PIN(GPIOC_0_IO_DATA_10, "GPIOC_0_IO_DATA_10"),
	PINCTRL_PIN(GPIOC_0_IO_DATA_11, "GPIOC_0_IO_DATA_11"),
	PINCTRL_PIN(GPIOC_0_IO_DATA_12, "GPIOC_0_IO_DATA_12"),
	PINCTRL_PIN(GPIOC_0_IO_DATA_13, "GPIOC_0_IO_DATA_13"),
	PINCTRL_PIN(GPIOC_0_IO_DATA_14, "GPIOC_0_IO_DATA_14"),
	PINCTRL_PIN(GPIOC_0_IO_DATA_15, "GPIOC_0_IO_DATA_15"),
	PINCTRL_PIN(SSIC_I_RXD, "SSIC_I_RXD"),
	PINCTRL_PIN(SSIC_O_BCLK, "SSIC_O_BCLK"),
	PINCTRL_PIN(SSIC_O_TXD, "SSIC_O_TXD"),
	PINCTRL_PIN(SSIC_O_nSEL_0, "SSIC_O_nSEL_0"),
	PINCTRL_PIN(SSIC_O_nSEL_1, "SSIC_O_nSEL_1"),
	PINCTRL_PIN(VIC_I_DEV_0_VSYNC, "VIC_I_DEV_0_VSYNC"),
	PINCTRL_PIN(VIC_I_DEV_0_HSYNC, "VIC_I_DEV_0_HSYNC"),
	PINCTRL_PIN(VIC_I_DEV_0_DATA_12, "VIC_I_DEV_0_DATA_12"),
	PINCTRL_PIN(VIC_I_DEV_0_DATA_13, "VIC_I_DEV_0_DATA_13"),
	PINCTRL_PIN(VIC_I_DEV_0_DATA_14, "VIC_I_DEV_0_DATA_14"),
	PINCTRL_PIN(VIC_I_DEV_0_DATA_15, "VIC_I_DEV_0_DATA_15"),
	PINCTRL_PIN(MSHC_0_I_CARD_nDETECT, "MSHC_0_I_CARD_nDETECT"),
	PINCTRL_PIN(MSHC_0_IO_DATA_0, "MSHC_0_IO_DATA_0"),
	PINCTRL_PIN(MSHC_0_IO_DATA_1, "MSHC_0_IO_DATA_1"),
	PINCTRL_PIN(MSHC_0_IO_DATA_2, "MSHC_0_IO_DATA_2"),
	PINCTRL_PIN(MSHC_0_IO_DATA_3, "MSHC_0_IO_DATA_3"),
	PINCTRL_PIN(MSHC_0_IO_CMD, "MSHC_0_IO_CMD"),
	PINCTRL_PIN(MSHC_0_O_TX_CLK, "MSHC_0_O_TX_CLK"),
	PINCTRL_PIN(MSHC_1_IO_DATA_0, "MSHC_1_IO_DATA_0"),
	PINCTRL_PIN(MSHC_1_IO_DATA_1, "MSHC_1_IO_DATA_1"),
	PINCTRL_PIN(MSHC_1_IO_DATA_2, "MSHC_1_IO_DATA_2"),
	PINCTRL_PIN(MSHC_1_IO_DATA_3, "MSHC_1_IO_DATA_3"),
	PINCTRL_PIN(MSHC_1_IO_CMD, "MSHC_1_IO_CMD"),
	PINCTRL_PIN(MSHC_1_O_TX_CLK, "MSHC_1_O_TX_CLK"),
	PINCTRL_PIN(I2SSC_I_RX_BCLK, "I2SSC_I_RX_BCLK"),
	PINCTRL_PIN(I2SSC_I_TX_BCLK, "I2SSC_I_TX_BCLK"),
	PINCTRL_PIN(I2SSC_I_RX_WS, "I2SSC_I_RX_WS"),
	PINCTRL_PIN(I2SSC_I_TX_WS, "I2SSC_I_TX_WS"),
	PINCTRL_PIN(I2SSC_I_RXD, "I2SSC_I_RXD"),
	PINCTRL_PIN(I2SSC_O_TXD, "I2SSC_O_TXD"),
	PINCTRL_PIN(I2SSC_O_MCLK, "I2SSC_O_MCLK"),
	PINCTRL_PIN(SYS_O_MON_CLK, "SYS_O_MON_CLK"),
	PINCTRL_PIN(IRDAC_I_SDA, "IRDAC_I_SDA"),
	PINCTRL_PIN(HDMITC_IO_DDCSCL, "HDMITC_IO_DDCSCL"),
	PINCTRL_PIN(HDMITC_IO_DDCSDA, "HDMITC_IO_DDCSDA"),
	PINCTRL_PIN(HDMITC_IO_CEC, "HDMITC_IO_CEC"),
	PINCTRL_PIN(VOC_I_REF_CLK, "VOC_I_REF_CLK"),
	PINCTRL_PIN(VOC_O_BLANK, "VOC_O_BLANK"),
	PINCTRL_PIN(VOC_O_VSYNC, "VOC_O_VSYNC"),
	PINCTRL_PIN(VOC_O_HSYNC, "VOC_O_HSYNC"),
	PINCTRL_PIN(VOC_O_DATA_0, "VOC_O_DATA_0"),
	PINCTRL_PIN(VOC_O_DATA_1, "VOC_O_DATA_1"),
	PINCTRL_PIN(VOC_O_DATA_2, "VOC_O_DATA_2"),
	PINCTRL_PIN(VOC_O_DATA_3, "VOC_O_DATA_3"),
	PINCTRL_PIN(VOC_O_DATA_4, "VOC_O_DATA_4"),
	PINCTRL_PIN(VOC_O_DATA_5, "VOC_O_DATA_5"),
	PINCTRL_PIN(VOC_O_DATA_6, "VOC_O_DATA_6"),
	PINCTRL_PIN(VOC_O_DATA_7, "VOC_O_DATA_7"),
	PINCTRL_PIN(VOC_O_DATA_8, "VOC_O_DATA_8"),
	PINCTRL_PIN(VOC_O_DATA_9, "VOC_O_DATA_9"),
	PINCTRL_PIN(VOC_O_DATA_10, "VOC_O_DATA_10"),
	PINCTRL_PIN(VOC_O_DATA_11, "VOC_O_DATA_11"),
	PINCTRL_PIN(VOC_O_DATA_12, "VOC_O_DATA_12"),
	PINCTRL_PIN(VOC_O_DATA_13, "VOC_O_DATA_13"),
	PINCTRL_PIN(VOC_O_DATA_14, "VOC_O_DATA_14"),
	PINCTRL_PIN(VOC_O_DATA_15, "VOC_O_DATA_15"),
	PINCTRL_PIN(VOC_O_CLK, "VOC_O_CLK"),
	PINCTRL_PIN(UARTC_3_I_SDA, "UARTC_3_I_SDA"),
	PINCTRL_PIN(UARTC_3_O_SDA, "UARTC_3_O_SDA"),
	PINCTRL_PIN(UARTC_2_I_SDA, "UARTC_2_I_SDA"),
	PINCTRL_PIN(UARTC_2_O_SDA, "UARTC_2_O_SDA"),
	PINCTRL_PIN(UARTC_1_I_SDA, "UARTC_1_I_SDA"),
	PINCTRL_PIN(UARTC_1_O_SDA, "UARTC_1_O_SDA"),
	PINCTRL_PIN(NFC_IO_DATA_0, "NFC_IO_DATA_0"),
	PINCTRL_PIN(NFC_IO_DATA_1, "NFC_IO_DATA_1"),
	PINCTRL_PIN(NFC_IO_DATA_2, "NFC_IO_DATA_2"),
	PINCTRL_PIN(NFC_IO_DATA_3, "NFC_IO_DATA_3"),
	PINCTRL_PIN(NFC_IO_DATA_4, "NFC_IO_DATA_4"),
	PINCTRL_PIN(NFC_IO_DATA_5, "NFC_IO_DATA_5"),
	PINCTRL_PIN(NFC_IO_DATA_6, "NFC_IO_DATA_6"),
	PINCTRL_PIN(NFC_IO_DATA_7, "NFC_IO_DATA_7"),
	PINCTRL_PIN(NFC_O_nWP, "NFC_O_nWP"),
	PINCTRL_PIN(NFC_O_nCE, "NFC_O_nCE"),
	PINCTRL_PIN(NFC_O_nRE, "NFC_O_nRE"),
	PINCTRL_PIN(NFC_O_nWE, "NFC_O_nWE"),
	PINCTRL_PIN(NFC_O_ALE, "NFC_O_ALE"),
	PINCTRL_PIN(NFC_O_CLE, "NFC_O_CLE"),
	PINCTRL_PIN(NFC_I_nRB, "NFC_I_nRB"),
	PINCTRL_PIN(I2CC_0_IO_SCL, "I2CC_0_IO_SCL"),
	PINCTRL_PIN(I2CC_0_IO_SDA, "I2CC_0_IO_SDA"),

	// non-gpio pins begin
	PINCTRL_PIN(UARTC_0_I_SDA, "UARTC_0_0_SDA"),
	PINCTRL_PIN(UARTC_0_O_SDA, "UARTC_0_O_SDA"),
	PINCTRL_PIN(ARM926U_PAD, "ARM926U_PAD"),
	PINCTRL_PIN(GMAC_PAD, "GMAC_PAD"),
	PINCTRL_PIN(VIC_REF_CLK, "VIC_REF_CLK"),
	PINCTRL_PIN(WDT_PAD, "WDT_PAD"),
	PINCTRL_PIN(VIC_DEFAULT, "VIC_DEFAULT"),
	PINCTRL_PIN(GMAC_REF_CLK, "GMAC_REF_CLK"),
	PINCTRL_PIN(GMAC_TX_CLK, "GMAC_TX_CLK"),
};

DECLARE_BITMAP(allpins, ARRAY_SIZE(pesaro_pins));

static const unsigned int i2c0_pos_0_pins[] = {
	I2CC_0_IO_SCL,
	I2CC_0_IO_SDA,
};
static const unsigned int i2c0_pos_1_pins[] = {
	GPIOC_0_IO_DATA_13,
	GPIOC_0_IO_DATA_14,
};
static const unsigned int i2c1_pins[] = {
	VIC_I_DEV_0_DATA_14,
	VIC_I_DEV_0_DATA_15,
};

static const unsigned int i2c2_pins[] = {
	MSHC_1_IO_DATA_0,
	MSHC_1_IO_DATA_1,
};

static const unsigned int uart0_pos_0_pins[] = {
	UARTC_0_I_SDA,
	UARTC_0_O_SDA,
};

static const unsigned int modem_pos_0_pins[] = {
	UARTC_3_I_SDA,
	UARTC_3_O_SDA,
	UARTC_2_I_SDA,
	UARTC_2_O_SDA,
	UARTC_1_I_SDA,
	UARTC_1_O_SDA,
	UARTC_0_I_SDA,
	UARTC_0_O_SDA,
};
static const unsigned int rs485_pos_0_pins[] = {
	UARTC_1_I_SDA,
	UARTC_1_O_SDA,
	UARTC_0_I_SDA,
	UARTC_0_O_SDA,
};
static const unsigned int uart0_pos_1_pins[] = {
	I2SSC_I_RX_BCLK,
	I2SSC_O_MCLK,
};
static const unsigned int modem_pos_1_pins[] = {
	I2SSC_I_RX_BCLK,
	I2SSC_O_MCLK,
	I2SSC_I_TX_BCLK,
	SYS_O_MON_CLK,
};
static const unsigned int uart1_pos_0_pins[] = {
	UARTC_1_I_SDA,
	UARTC_1_O_SDA,
};
static const unsigned int uart1_pos_1_pins[] = {
	HDMITC_IO_DDCSCL,
	HDMITC_IO_DDCSDA,
};
static const unsigned int uart2_pos_0_pins[] = {
	UARTC_2_I_SDA,
	UARTC_2_O_SDA,
};
static const unsigned int uart2_pos_1_pins[] = {
	I2SSC_I_RXD,
	I2SSC_O_TXD,
};
static const unsigned int uart3_pos_0_pins[] = {
	UARTC_3_I_SDA,
	UARTC_3_O_SDA,
};
static const unsigned int uart3_pos_1_pins[] = {
	NFC_O_nRE,
	NFC_I_nRB,
};

static const unsigned int arm926u_pins[] = {
	ARM926U_PAD,
};
static const unsigned int gmac_pins[] = {
	GMAC_PAD,
};
static const unsigned int gmac_ref_clk_pins[] = {
	GMAC_REF_CLK,
};
static const unsigned int gmac_tx_clk_pins[] = {
	GMAC_TX_CLK,
};
static const unsigned int hdmitc_pins[] = {
	HDMITC_IO_DDCSCL,
	HDMITC_IO_DDCSDA,
	HDMITC_IO_CEC,
};

static const unsigned int i2ssc_ext_slave_pins[] = {
	I2SSC_I_RX_BCLK,
	I2SSC_I_TX_BCLK,
	I2SSC_I_RX_WS,
	I2SSC_I_TX_WS,
	I2SSC_I_RXD,
	I2SSC_O_TXD,
	I2SSC_O_MCLK,
};

static const unsigned int i2ssc_ext_master_pins[] = {
	I2SSC_I_RX_BCLK,
	I2SSC_I_TX_BCLK,
	I2SSC_I_RX_WS,
	I2SSC_I_TX_WS,
	I2SSC_I_RXD,
	I2SSC_O_TXD,
	I2SSC_O_MCLK,
};

static const unsigned int irdac_pins[] = {
	IRDAC_I_SDA,
};

static const unsigned int monitor_pins[] = {
	SYS_O_MON_CLK,
};

static const unsigned int mshc0_pins[] = {
	MSHC_0_I_CARD_nDETECT,
	MSHC_0_IO_DATA_0,
	MSHC_0_IO_DATA_1,
	MSHC_0_IO_DATA_2,
	MSHC_0_IO_DATA_3,
	MSHC_0_IO_CMD,
	MSHC_0_O_TX_CLK,
};

static const unsigned int mshc0_pos_0_pins[] = {
	GPIOC_0_IO_DATA_15,
};
static const unsigned int mshc0_pos_1_pins[] = {
	HDMITC_IO_CEC,
};
static const unsigned int mshc1_pins[] = {
	MSHC_1_IO_DATA_0,
	MSHC_1_IO_DATA_1,
	MSHC_1_IO_DATA_2,
	MSHC_1_IO_DATA_3,
	MSHC_1_IO_CMD,
	MSHC_1_O_TX_CLK,
};

static const unsigned int nfc_pins[] = {
	NFC_IO_DATA_0,
	NFC_IO_DATA_1,
	NFC_IO_DATA_2,
	NFC_IO_DATA_3,
	NFC_IO_DATA_4,
	NFC_IO_DATA_5,
	NFC_IO_DATA_6,
	NFC_IO_DATA_7,
	NFC_O_nWP,
	NFC_O_nCE,
	NFC_O_nRE,
	NFC_O_nWE,
	NFC_O_ALE,
	NFC_O_CLE,
	NFC_I_nRB,
};

static const unsigned int ssic_pos_0_pins[] = {
	SSIC_I_RXD,
	SSIC_O_BCLK,
	SSIC_O_TXD,
	SSIC_O_nSEL_0,
	SSIC_O_nSEL_1,
};

static const unsigned int ssic_pos_1_pins[] = {
	VIC_I_DEV_0_DATA_12,
	VIC_I_DEV_0_DATA_13,
	VIC_I_DEV_0_DATA_14,
	VIC_I_DEV_0_DATA_15,
};

static const unsigned int vic_dev0_sec0_pins[] = {
	VIC_DEFAULT,
};

static const unsigned int vic_dev0_sec1_pins[] = {
	VIC_I_DEV_0_VSYNC,
	VIC_I_DEV_0_HSYNC,
};

static const unsigned int vic_dev0_sec2_pins[] = {
	VIC_I_DEV_0_DATA_12,
	VIC_I_DEV_0_DATA_13,
	VIC_I_DEV_0_DATA_14,
	VIC_I_DEV_0_DATA_15,
};
static const unsigned int vic_dev1_sec0_pins[] = {
	GPIOC_0_IO_DATA_0,
	GPIOC_0_IO_DATA_1,
	GPIOC_0_IO_DATA_2,
	GPIOC_0_IO_DATA_3,
	GPIOC_0_IO_DATA_4,
	GPIOC_0_IO_DATA_5,
	GPIOC_0_IO_DATA_6,
	GPIOC_0_IO_DATA_7,
	GPIOC_0_IO_DATA_8,
};
static const unsigned int vic_dev1_sec1_pins[] = {
	GPIOC_0_IO_DATA_13,
	GPIOC_0_IO_DATA_14,
};
static const unsigned int vic_dev1_sec2_pins[] = {
	GPIOC_0_IO_DATA_9,
	GPIOC_0_IO_DATA_10,
	GPIOC_0_IO_DATA_11,
	GPIOC_0_IO_DATA_12,

};
static const unsigned int vic_ref_clk_pins[] = {
	VIC_REF_CLK,
};

static const unsigned int voc_16bit_pins[] = {
	VOC_O_BLANK,
	VOC_O_VSYNC,
	VOC_O_HSYNC,
	VOC_O_DATA_0,
	VOC_O_DATA_1,
	VOC_O_DATA_2,
	VOC_O_DATA_3,
	VOC_O_DATA_4,
	VOC_O_DATA_5,
	VOC_O_DATA_6,
	VOC_O_DATA_7,
	VOC_O_DATA_8,
	VOC_O_DATA_9,
	VOC_O_DATA_10,
	VOC_O_DATA_11,
	VOC_O_DATA_12,
	VOC_O_DATA_13,
	VOC_O_DATA_14,
	VOC_O_DATA_15,
	VOC_O_CLK,
};
static const unsigned int voc_24bit_pins[] = {
	VOC_O_BLANK,
	VOC_O_VSYNC,
	VOC_O_HSYNC,
	VOC_O_DATA_0,
	VOC_O_DATA_1,
	VOC_O_DATA_2,
	VOC_O_DATA_3,
	VOC_O_DATA_4,
	VOC_O_DATA_5,
	VOC_O_DATA_6,
	VOC_O_DATA_7,
	VOC_O_DATA_8,
	VOC_O_DATA_9,
	VOC_O_DATA_10,
	VOC_O_DATA_11,
	VOC_O_DATA_12,
	VOC_O_DATA_13,
	VOC_O_DATA_14,
	VOC_O_DATA_15,
	VOC_O_CLK,
	UARTC_3_I_SDA,
	UARTC_3_O_SDA,
	UARTC_2_I_SDA,
	UARTC_2_O_SDA,
	UARTC_1_I_SDA,
	UARTC_1_O_SDA,
	UARTC_0_I_SDA,
	UARTC_0_O_SDA,
};

static const unsigned int voc_ref_clk_pins[] = {
	VOC_I_REF_CLK,
};

static const unsigned int wdt_pins[] = {
	WDT_PAD,
};

#define DECLARE_AGPO_PINS(n, p, pin)				\
static const unsigned int agpo##n##_pos_##p##_pins[] = {	\
	pin,							\
}

/* POS0 */
DECLARE_AGPO_PINS(0, 0, GPIOC_0_IO_DATA_8);
DECLARE_AGPO_PINS(1, 0, GPIOC_0_IO_DATA_9);
DECLARE_AGPO_PINS(2, 0, GPIOC_0_IO_DATA_10);
DECLARE_AGPO_PINS(3, 0, GPIOC_0_IO_DATA_11);
DECLARE_AGPO_PINS(4, 0, GPIOC_0_IO_DATA_12);
DECLARE_AGPO_PINS(5, 0, GPIOC_0_IO_DATA_13);
DECLARE_AGPO_PINS(6, 0, GPIOC_0_IO_DATA_14);
DECLARE_AGPO_PINS(7, 0, GPIOC_0_IO_DATA_15);

/* POS1 */
DECLARE_AGPO_PINS(0, 1, MSHC_0_IO_CMD);
DECLARE_AGPO_PINS(1, 1, MSHC_0_O_TX_CLK);
DECLARE_AGPO_PINS(2, 1, MSHC_1_IO_DATA_0);
DECLARE_AGPO_PINS(3, 1, MSHC_1_IO_DATA_1);
DECLARE_AGPO_PINS(4, 1, MSHC_1_IO_DATA_2);
DECLARE_AGPO_PINS(5, 1, MSHC_1_IO_DATA_3);
DECLARE_AGPO_PINS(6, 1, MSHC_1_IO_CMD);
DECLARE_AGPO_PINS(7, 1, MSHC_1_O_TX_CLK);

/* POS2 */
DECLARE_AGPO_PINS(0, 2, I2SSC_I_RX_BCLK);
DECLARE_AGPO_PINS(1, 2, I2SSC_I_TX_BCLK);
DECLARE_AGPO_PINS(2, 2, I2SSC_I_RX_WS);
DECLARE_AGPO_PINS(3, 2, I2SSC_I_TX_WS);
DECLARE_AGPO_PINS(4, 2, I2SSC_I_RXD);
DECLARE_AGPO_PINS(5, 2, I2SSC_O_TXD);
DECLARE_AGPO_PINS(6, 2, I2SSC_O_MCLK);
DECLARE_AGPO_PINS(7, 2, SYS_O_MON_CLK);

/* POS3 */
DECLARE_AGPO_PINS(0, 3, VOC_O_DATA_8);
DECLARE_AGPO_PINS(1, 3, VOC_O_DATA_9);
DECLARE_AGPO_PINS(2, 3, VOC_O_DATA_10);
DECLARE_AGPO_PINS(3, 3, VOC_O_DATA_11);
DECLARE_AGPO_PINS(4, 3, VOC_O_DATA_12);
DECLARE_AGPO_PINS(5, 3, VOC_O_DATA_13);
DECLARE_AGPO_PINS(6, 3, VOC_O_DATA_14);
DECLARE_AGPO_PINS(7, 3, VOC_O_DATA_15);

/* POS4 */
DECLARE_AGPO_PINS(0, 4, UARTC_3_I_SDA);
DECLARE_AGPO_PINS(1, 4, UARTC_3_O_SDA);
DECLARE_AGPO_PINS(2, 4, UARTC_2_I_SDA);
DECLARE_AGPO_PINS(3, 4, UARTC_2_O_SDA);
DECLARE_AGPO_PINS(4, 4, UARTC_1_I_SDA);
DECLARE_AGPO_PINS(5, 4, UARTC_1_O_SDA);
DECLARE_AGPO_PINS(6, 4, UARTC_0_I_SDA);
DECLARE_AGPO_PINS(7, 4, UARTC_0_O_SDA);

/* POS5 */
DECLARE_AGPO_PINS(0, 5, NFC_IO_DATA_0);
DECLARE_AGPO_PINS(1, 5, NFC_IO_DATA_1);
DECLARE_AGPO_PINS(2, 5, NFC_IO_DATA_2);
DECLARE_AGPO_PINS(3, 5, NFC_IO_DATA_3);
DECLARE_AGPO_PINS(4, 5, NFC_IO_DATA_4);
DECLARE_AGPO_PINS(5, 5, NFC_IO_DATA_5);
DECLARE_AGPO_PINS(6, 5, NFC_IO_DATA_6);
DECLARE_AGPO_PINS(7, 5, NFC_IO_DATA_7);

#define PESARO_GROUP(pg_name, _offset, _firstbit, _numbits)   \
        {                                                     \
	  .name = #pg_name,                                   \
	  .pins = pg_name##_pins,                             \
	  .num_pins = ARRAY_SIZE(pg_name##_pins),             \
	  .offset     = _offset,                              \
	  .bit        = _firstbit,                            \
	  .numbits    = _numbits                              \
	 }

#define PESARO_AGPO_GROUP(p)	\
	PESARO_GROUP(agpo0_pos_##p , ((p/4) ? VPL_PAD_1 : VPL_PAD_0), ((p%4)*8), 1),     \
	PESARO_GROUP(agpo1_pos_##p , ((p/4) ? VPL_PAD_1 : VPL_PAD_0), ((p%4)*8+1), 1),     \
	PESARO_GROUP(agpo2_pos_##p , ((p/4) ? VPL_PAD_1 : VPL_PAD_0), ((p%4)*8+2), 1),     \
	PESARO_GROUP(agpo3_pos_##p , ((p/4) ? VPL_PAD_1 : VPL_PAD_0), ((p%4)*8+3), 1),     \
	PESARO_GROUP(agpo4_pos_##p , ((p/4) ? VPL_PAD_1 : VPL_PAD_0), ((p%4)*8+4), 1),     \
	PESARO_GROUP(agpo5_pos_##p , ((p/4) ? VPL_PAD_1 : VPL_PAD_0), ((p%4)*8+5), 1),     \
	PESARO_GROUP(agpo6_pos_##p , ((p/4) ? VPL_PAD_1 : VPL_PAD_0), ((p%4)*8+6), 1),     \
	PESARO_GROUP(agpo7_pos_##p , ((p/4) ? VPL_PAD_1 : VPL_PAD_0), ((p%4)*8+7), 1)     \

static const struct pesaro_group pesaro_groups[] = {
	PESARO_GROUP(arm926u      , VPL_PAD_5, 0, 1),
	PESARO_GROUP(gmac         , VPL_PAD_5, 1, 1),
	PESARO_GROUP(gmac_ref_clk , VPL_PAD_5, 2, 1),
	PESARO_GROUP(gmac_tx_clk  , VPL_PAD_5, 3, 1),
	PESARO_GROUP(hdmitc       , VPL_PAD_5, 4, 1),
	PESARO_GROUP(i2c0_pos_0   , VPL_PAD_5, 5, 1),
	PESARO_GROUP(i2c0_pos_1   , VPL_PAD_5, 6, 1),
	PESARO_GROUP(i2c1         , VPL_PAD_5, 7, 1),
	PESARO_GROUP(i2c2         , VPL_PAD_5, 8, 1),
	PESARO_GROUP(i2ssc_ext_slave, VPL_PAD_5, 9, 3),
	PESARO_GROUP(i2ssc_ext_master, VPL_PAD_5,9, 4),
	PESARO_GROUP(irdac        , VPL_PAD_5,13, 1),
	PESARO_GROUP(monitor      , VPL_PAD_5,14, 1),
	PESARO_GROUP(mshc0        , VPL_PAD_5,15, 2),
	PESARO_GROUP(mshc0_pos_0  , VPL_PAD_5,17, 1),
	PESARO_GROUP(mshc0_pos_1  , VPL_PAD_5,18, 1),
	PESARO_GROUP(mshc1        , VPL_PAD_5,19, 1),
	PESARO_GROUP(nfc          , VPL_PAD_5,21, 1),
	PESARO_GROUP(ssic_pos_0   , VPL_PAD_5,22, 1),
	PESARO_GROUP(ssic_pos_1   , VPL_PAD_5,23, 1),
	PESARO_GROUP(uart0_pos_0  , VPL_PAD_5,24, 1),
	PESARO_GROUP(modem_pos_0  , VPL_PAD_5,24, 3),
	PESARO_GROUP(rs485_pos_0  , VPL_PAD_5,24, 2),	
	PESARO_GROUP(uart0_pos_1  , VPL_PAD_5,27, 1),
	PESARO_GROUP(modem_pos_1  , VPL_PAD_5,27, 2),
	PESARO_GROUP(uart1_pos_0  , VPL_PAD_5,29, 1),
	PESARO_GROUP(uart1_pos_1  , VPL_PAD_5,30, 1),
	PESARO_GROUP(uart2_pos_0  , VPL_PAD_5,31, 1),
	PESARO_GROUP(uart2_pos_1  , VPL_PAD_6, 0, 1),
	PESARO_GROUP(uart3_pos_0  , VPL_PAD_6, 1, 1),
	PESARO_GROUP(uart3_pos_1  , VPL_PAD_6, 2, 1),
	PESARO_GROUP(vic_dev0_sec0, VPL_PAD_6, 3, 1),
	PESARO_GROUP(vic_dev0_sec1, VPL_PAD_6, 4, 1),
	PESARO_GROUP(vic_dev0_sec2, VPL_PAD_6, 5, 1),
	PESARO_GROUP(vic_dev1_sec0, VPL_PAD_6, 6, 1),
	PESARO_GROUP(vic_dev1_sec1, VPL_PAD_6, 7, 1),
	PESARO_GROUP(vic_dev1_sec2, VPL_PAD_6, 8, 1),
	PESARO_GROUP(vic_ref_clk  , VPL_PAD_6, 9, 1),
	PESARO_GROUP(voc_16bit    , VPL_PAD_6,10, 1),
	PESARO_GROUP(voc_24bit    , VPL_PAD_6,10, 5),
	PESARO_GROUP(voc_ref_clk  , VPL_PAD_6,15, 1),
	PESARO_GROUP(wdt          , VPL_PAD_6,16, 1),
	PESARO_AGPO_GROUP(0),
	PESARO_AGPO_GROUP(1),
	PESARO_AGPO_GROUP(2),
	PESARO_AGPO_GROUP(3),
	PESARO_AGPO_GROUP(4),
	PESARO_AGPO_GROUP(5),
};

static const char * const i2c_groups[] = { "i2c0_pos_0", "i2c0_pos_1",
					    "i2c1", "i2c2"};

static const char * const uart_groups[] = { "uart0_pos_0", "uart0_pos_1",
                                             "rs485_pos_0",
					     "modem_pos_0", "modem_pos_1",
                                             "uart1_pos_0", "uart1_pos_1",
                                             "uart2_pos_0", "uart2_pos_1",
                                             "uart3_pos_0", "uart3_pos_1"};

static const char * const arm926u_groups[] = { "arm926u"};
static const char * const gmac_groups[] = { "gmac", "gmac_ref_clk", "gmac_tx_clk"};
static const char * const hdmitc_groups[] = { "hdmitc"};
static const char * const i2ssc_groups[] = { "i2ssc_ext_slave", "i2ssc_ext_master"};
static const char * const irdac_groups[] = { "irdac"};
static const char * const monitor_groups[] = { "monitor"};
static const char * const mshc_groups[] = { "mshc0", "mshc0_pos_0",
                                            "mshc0_pos_1", "mshc1"};
static const char * const nfc_groups[] = { "nfc"};
static const char * const ssic_groups[] = { "ssic_pos_0", "ssic_pos_1"};
static const char * const vic_groups[] = { "vic_dev0_sec0", "vic_dev0_sec1",
					   "vic_dev0_sec2",
					   "vic_dev1_sec0", "vic_dev1_sec1",
					   "vic_dev1_sec2",
					   "vic_ref_clk"};
static const char * const voc_groups[] = { "voc_16bit", "voc_24bit", "voc_ref_clk"};
static const char * const wdt_groups[] = { "wdt"};
static const char * const agpo0_groups[] = { "agpo0_pos_0", "agpo0_pos_1",
					     "agpo0_pos_2", "agpo0_pos_3",
					     "agpo0_pos_4", "agpo0_pos_5"};

static const char * const agpo1_groups[] = { "agpo1_pos_0", "agpo1_pos_1",
					     "agpo1_pos_2", "agpo1_pos_3",
					     "agpo1_pos_4", "agpo1_pos_5"};

static const char * const agpo2_groups[] = { "agpo2_pos_0", "agpo2_pos_1",
					     "agpo2_pos_2", "agpo2_pos_3",
					     "agpo2_pos_4", "agpo2_pos_5"};

static const char * const agpo3_groups[] = { "agpo3_pos_0", "agpo3_pos_1",
					     "agpo3_pos_2", "agpo3_pos_3",
					     "agpo3_pos_4", "agpo3_pos_5"};

static const char * const agpo4_groups[] = { "agpo4_pos_0", "agpo4_pos_1",
					     "agpo4_pos_2", "agpo4_pos_3",
					     "agpo4_pos_4", "agpo4_pos_5"};

static const char * const agpo5_groups[] = { "agpo5_pos_0", "agpo5_pos_1",
					     "agpo5_pos_2", "agpo5_pos_3",
					     "agpo5_pos_4", "agpo5_pos_5"};

static const char * const agpo6_groups[] = { "agpo6_pos_0", "agpo6_pos_1",
					     "agpo6_pos_2", "agpo6_pos_3",
					     "agpo6_pos_4", "agpo6_pos_5"};

static const char * const agpo7_groups[] = { "agpo7_pos_0", "agpo7_pos_1",
					     "agpo7_pos_2", "agpo7_pos_3",
					     "agpo7_pos_4", "agpo7_pos_5"};


#define FUNCTION(fname)                         \
{                                               \
	.name = #fname,                         \
	.groups = fname##_groups,               \
	.ngroups = ARRAY_SIZE(fname##_groups),  \
}

static const struct pesaro_function pesaro_functions[] = {
	FUNCTION(i2c),
	FUNCTION(uart),
	FUNCTION(arm926u),
	FUNCTION(gmac),
	FUNCTION(hdmitc),
	FUNCTION(i2ssc),
	FUNCTION(irdac),
	FUNCTION(monitor),
	FUNCTION(mshc),
	FUNCTION(nfc),
	FUNCTION(ssic),
	FUNCTION(vic),
	FUNCTION(voc),
	FUNCTION(wdt),
	FUNCTION(agpo0),
	FUNCTION(agpo1),
	FUNCTION(agpo2),
	FUNCTION(agpo3),
	FUNCTION(agpo4),
	FUNCTION(agpo5),
	FUNCTION(agpo6),
	FUNCTION(agpo7),
};

static int pesaro_get_groups_count(struct pinctrl_dev *pctldev)
{
	return ARRAY_SIZE(pesaro_groups);
}

static const char *pesaro_get_group_name(struct pinctrl_dev *pctldev,
		unsigned selector)
{
	return pesaro_groups[selector].name;
}

static int pesaro_get_group_pins(struct pinctrl_dev *pctldev, unsigned selector,
		const unsigned **pins,
		unsigned *num_pins)
{
	*pins = (unsigned *) pesaro_groups[selector].pins;
	*num_pins = pesaro_groups[selector].num_pins;
	return 0;
}
static void pesaro_pinctrl_dt_free_map(struct pinctrl_dev *pctldev,
				      struct pinctrl_map *map,
				      unsigned num_maps)
{
	int i;

	for (i = 0; i < num_maps; i++)
		if (map[i].type == PIN_MAP_TYPE_CONFIGS_GROUP)
			kfree(map[i].data.configs.configs);

	kfree(map);
}

static int add_map_mux(struct pinctrl_map **map, unsigned *reserved_maps,
		       unsigned *num_maps, const char *group,
		       const char *function)
{
	if (WARN_ON(*num_maps == *reserved_maps))
		return -ENOSPC;

	(*map)[*num_maps].type = PIN_MAP_TYPE_MUX_GROUP;
	(*map)[*num_maps].data.mux.group = group;
	(*map)[*num_maps].data.mux.function = function;
	(*num_maps)++;

	return 0;
}

static int reserve_map(struct device *dev, struct pinctrl_map **map,
		       unsigned *reserved_maps, unsigned *num_maps,
		       unsigned reserve)
{
	unsigned old_num = *reserved_maps;
	unsigned new_num = *num_maps + reserve;
	struct pinctrl_map *new_map;

	pr_debug("old_num:%d, new_num:%d\n", old_num, new_num);
	if (old_num >= new_num)
		return 0;

	new_map = krealloc(*map, sizeof(*new_map) * new_num, GFP_KERNEL);
	if (!new_map) {
		dev_err(dev, "krealloc(map) failed\n");
		return -ENOMEM;
	}

	memset(new_map + old_num, 0, (new_num - old_num) * sizeof(*new_map));

	*map = new_map;
	*reserved_maps = new_num;

	return 0;
}

__maybe_unused static int pesaro_default_gpio(struct device* dev,
				struct device_node *np)
{
	__maybe_unused int i;
	int len;
	const void* tmp;
	u32 gpios[VPL_GPIO_NR];

	tmp = of_get_property(np, "vpl,gpiopin", &len);
	if (tmp == NULL) {
		dev_err(dev, "could not parse property vpl,gpiopin\n");
		return -EINVAL;
	}
	len = len / sizeof(u32);
	of_property_read_u32_array(np, "vpl,gpiopin", gpios, len);
#ifdef DEBUG
	for (i = 0 ; i < len ; i++) {
		pr_debug("default gpio[%d]=%u\n", i, gpios[i]);
	}
#endif
	return 0;
}

static int pesaro_pinctrl_dt_subnode_to_map(struct device *dev,
					   struct device_node *np,
					   struct pinctrl_map **map,
					   unsigned *reserved_maps,
					   unsigned *num_maps)
{
	int ret;
	const char *function;
	const char *group;
	unsigned reserve = 0;

	ret = of_property_read_string(np, "vpl,function", &function);
	if (ret < 0) {
		/* EINVAL=missing, which is fine since it's optional */
		if (ret != -EINVAL)
			dev_err(dev,
				"could not parse property vpl,function\n");
		function = NULL;
	}

	ret = of_property_read_string(np, "vpl,group", &group);
	if (ret < 0) {
		/* EINVAL=missing, which is fine since it's optional */
		if (ret != -EINVAL)
			dev_err(dev,
					"could not parse property vpl,group\n");
		group = NULL;
	}
	if (function != NULL && group!= NULL)
		reserve++;

	ret = reserve_map(dev, map, reserved_maps, num_maps, reserve);
	if (ret < 0)
		return ret;

	ret = add_map_mux(map, reserved_maps, num_maps,
					  group, function);
	pr_debug("%s, function:%s, group:%s\n", __func__, function, group);

	ret = 0;

	return ret;
}

static int pesaro_pinctrl_dt_node_to_map(struct pinctrl_dev *pctldev,
					struct device_node *np_config,
					struct pinctrl_map **map,
					unsigned *num_maps)
{
	unsigned reserved_maps;
	struct device_node *np;
	int ret;

	reserved_maps = 0;
	*map = NULL;
	*num_maps = 0;

	for_each_child_of_node(np_config, np) {
		ret = pesaro_pinctrl_dt_subnode_to_map(pctldev->dev, np, map,
				                    &reserved_maps ,num_maps);
		if (ret < 0) {
			pesaro_pinctrl_dt_free_map(pctldev, *map, *num_maps);
			return ret;
		}
	}

	return 0;
}

static struct pinctrl_ops pesaro_pinctrl_ops = {
	.get_groups_count = pesaro_get_groups_count,
	.get_group_name = pesaro_get_group_name,
	.get_group_pins = pesaro_get_group_pins,
	.dt_node_to_map = pesaro_pinctrl_dt_node_to_map,
	.dt_free_map = pesaro_pinctrl_dt_free_map,
};

static int pesaro_pin_request(struct pinctrl_dev *pctldev, unsigned offset)
{
	const struct pinctrl_pin_desc *p = &pesaro_pins[offset];
	if (test_bit(offset, allpins)) {
		pr_err("request duplicate pin %d(%s) please check!!!\n", offset, p->name);
		
		/*XXX this is werid here, only booting linux enters the first
		 *    checking
		 */
		return -EPROBE_DEFER;
	};
	set_bit(offset, allpins);
	return 0;
}

static int pesaro_pin_free(struct pinctrl_dev *pctldev, unsigned offset)
{
	clear_bit(offset, allpins);
	return 0;
}

static int pesaro_gpio_request_enable(struct pinctrl_dev *pctldev,
					struct pinctrl_gpio_range *range, unsigned offset)
{
	int reg_off, bit_off;
	unsigned long val;
	reg_off = offset / 32;
	bit_off = offset % 32;
	pr_debug("%s, offset:%d, reg_off:%d, bit_off:%d\n",
			__func__, offset, reg_off, bit_off);

	if(pesaro_pin_request(pctldev, offset))
		return -EBUSY;
	// GPIO PAD BASE
	val = readl( GPIO_BASE + reg_off * 0x4);
	set_bit(bit_off, &val);
	writel(val, GPIO_BASE + reg_off * 0x4);

	return 0;
}

static void pesaro_gpio_disable_free(struct pinctrl_dev *pctldev,
					struct pinctrl_gpio_range *range, unsigned offset)
{
	int reg_off, bit_off;
	unsigned long val;
	reg_off = offset / 32;
	bit_off = offset % 32;
	pr_debug("%s, offset:%d, reg_off:%d, bit_off:%d\n",
			__func__, offset, reg_off, bit_off);

	pesaro_pin_free(pctldev, offset);
	// GPIO PAD BASE
	val = readl(GPIO_BASE + reg_off * 0x4);
	clear_bit(bit_off, &val);
	writel(val, GPIO_BASE + reg_off * 0x4);
	return;
}
int pesaro_get_functions_count(struct pinctrl_dev *pctldev)
{
	return ARRAY_SIZE(pesaro_functions);
}

const char *pesaro_get_fname(struct pinctrl_dev *pctldev, unsigned selector)
{
	return pesaro_functions[selector].name;
}

static int pesaro_get_groups(struct pinctrl_dev *pctldev, unsigned selector,
		const char * const **groups,
		unsigned * const num_groups)
{
	*groups = pesaro_functions[selector].groups;
	*num_groups = pesaro_functions[selector].ngroups;
	return 0;
}

unsigned long pad_reg[VPL_PAD_NUM] = {0};
static int isFirst = 1; // 1: for pinctrl_register, 0: for general usage
int pesaro_enable(struct pinctrl_dev *pctldev, unsigned selector,
		unsigned group)
{
	unsigned long flags;
	unsigned long val;
	int i;

	const struct pesaro_group *grp = &pesaro_groups[group];
	pr_debug("name:%s, reg:0x%x, bit:%d, bit_num:%d\n", grp->name,
		grp->offset, grp->bit, grp->numbits);
	spin_lock_irqsave(&pinmux_lock, flags);

	val = pad_reg[(grp->offset - VPL_PAD_0) /4];
	for (i = 0 ; i < grp->numbits ; i++) {
		set_bit(grp->bit + i, &val);
	}
	pad_reg[(grp->offset - VPL_PAD_0) /4] = val;	

	if (!isFirst)
		writel(val, SYSC_MMR_BASE + grp->offset);

	spin_unlock_irqrestore(&pinmux_lock, flags);
	return 0;
}

void pesaro_disable(struct pinctrl_dev *pctldev, unsigned selector,
		unsigned group)
{
	unsigned long flags;
	unsigned long val;
	int i;
	const struct pesaro_group *grp = &pesaro_groups[group];

	spin_lock_irqsave(&pinmux_lock, flags);

	val = readl(SYSC_MMR_BASE + grp->offset);
	for (i = 0 ; i < grp->numbits ; i++) {
		clear_bit(grp->bit + i, &val);
	}
	writel(val, SYSC_MMR_BASE + grp->offset);

	spin_unlock_irqrestore(&pinmux_lock, flags);
	return;
}

struct pinmux_ops pesaro_pinmux_ops = {
	.gpio_request_enable = pesaro_gpio_request_enable,
	.gpio_disable_free = pesaro_gpio_disable_free,
	.request = pesaro_pin_request,
	.free = pesaro_pin_free,
	.get_functions_count = pesaro_get_functions_count,
	.get_function_name = pesaro_get_fname,
	.get_function_groups = pesaro_get_groups,
	.enable = pesaro_enable,
	.disable = pesaro_disable,
};

static struct pinctrl_desc pesaro_pinctrl_desc = {
	.pctlops = &pesaro_pinctrl_ops,
	.pmxops = &pesaro_pinmux_ops,
	.owner = THIS_MODULE,
};

static const struct pesaro_pinctrl_soc_data pesaro_pinctrl = {
	.ngpios = ARRAY_SIZE(pesaro_pins),
	.pins = pesaro_pins,
	.npins = ARRAY_SIZE(pesaro_pins),
	.functions = pesaro_functions,
	.nfunctions = ARRAY_SIZE(pesaro_functions),
	.groups = pesaro_groups,
	.ngroups = ARRAY_SIZE(pesaro_groups),
};

__maybe_unused void reset_pad(void)
{
	int i;

	for( i = 0; i <= VPL_PAD_NUM ; i++) {
		writel(0x0, PAD_BASE + i * 0x4);
	}

	return;
}

void reverse_map(unsigned int addr_off, int bit)
{
	int i, j, pin;
	int max = ARRAY_SIZE(pesaro_groups);
	for(i = 0 ; i < max ; i++){
		if ( addr_off == pesaro_groups[i].offset &&
			bit == pesaro_groups[i].bit) {
			pr_debug("0x%x(%d) reverse found\n", addr_off, bit);

			for (j = 0 ; j < pesaro_groups[i].num_pins ; j++) {
				pin = *(pesaro_groups[i].pins+j);
				/* XXX we use this function to occupy the pins
				 *     and we can see update in debugfs, not only
				 *     in the allpins table
				 */
				pinctrl_request_gpio(pin);
				pr_debug("%s(%d) updated\n",
					pesaro_pins[pin].name, pesaro_pins[pin].number);
			}
			return;
		}
	}
	pr_debug("pad:0x%x,bit:%d not define, ignore it\n", addr_off, bit);
	return;
}

__maybe_unused static void pesaro_pin_bitmap_init(void)
{
	unsigned long val;
	int i, j;
	unsigned int offset = VPL_PAD_2;

	for( i = 0; offset <= VPL_PAD_6 ; offset = offset+0x4, i++) {
		val = readl(SYSC_MMR_BASE + offset);
		pr_debug("pad:0x%x = 0x%lx\n", offset, val);

		for(j = 0 ; j < 32 ; j++) {
			if(test_bit(j, &val)) {
				if (i < 3)
					/* XXX we use this function to occupy the pins
					 *     and we can see update in debugfs, not only
					 *     in the allpins table
					 */
					pinctrl_request_gpio(i*32+j);
				else {
					// non-gpio pad, we need to reverse map here
					reverse_map(offset, j);
				}
			}
		}
	}
}

static int pesaro_pinctrl_probe(struct platform_device *pdev)
{
	struct pesaro_pmx *pmx;
	u32 reg;
	int i;

	pmx = devm_kzalloc(&pdev->dev, sizeof(*pmx), GFP_KERNEL);
	if (!pmx) {
		dev_err(&pdev->dev, "Can't alloc pesaro_pmx\n");
		return -ENOMEM;
	}

	/*TODO: seperate soc information, if we have more projects in the future*/
	pmx->dev = &pdev->dev;
	pmx->soc = &pesaro_pinctrl;

	pesaro_pinctrl_gpio_range.npins = pmx->soc->ngpios;
	pesaro_pinctrl_desc.name = dev_name(&pdev->dev);
	pesaro_pinctrl_desc.pins = pmx->soc->pins;
	pesaro_pinctrl_desc.npins = pmx->soc->npins;


	pmx->pctl = pinctrl_register(&pesaro_pinctrl_desc, &pdev->dev, pmx);
	if (!pmx->pctl) {
		dev_err(&pdev->dev, "Couldn't register pinctrl driver\n");
		return -ENODEV;
	}
	platform_set_drvdata(pdev, pmx);
	pinctrl_add_gpio_range(pmx->pctl, &pesaro_pinctrl_gpio_range);
	//pesaro_pin_bitmap_init();

	for (i = 0; i < SYSC_PAD_NUM; i++) {
		reg = SYSC_PAD_0 + (i<<2);
		// keeps GPIO PAD settings untouched
		if ((reg < VPL_PAD_GPIO) || (reg >= (VPL_PAD_GPIO + (VPL_PAD_GPIO_NUM<<2)))) 
			writel(pad_reg[i], SYSC_MMR_BASE + reg);
	}
	isFirst = 0;
	
	dev_dbg(&pdev->dev, "Probed pesaro pinctrl driver\n");
	return 0;
}

int pesaro_pinctrl_remove(struct platform_device *pdev)
{
	struct pesaro_pmx *pmx = platform_get_drvdata(pdev);
	pinctrl_unregister(pmx->pctl);

	return 0;
}

static struct of_device_id pesaro_pinctrl_of_match[] = {
	{ .compatible = "vatics,pesaro-pinmux", },
	{ },
};

static struct platform_driver pesaro_pinctrl_driver = {
	.driver = {
		.name = "pesaro-pinctrl",
		.owner = THIS_MODULE,
		.of_match_table = pesaro_pinctrl_of_match,
	},
	.probe = pesaro_pinctrl_probe,
	.remove = pesaro_pinctrl_remove,
};

static int __init pesaro_pinctrl_init(void)
{
	return platform_driver_register(&pesaro_pinctrl_driver);
}
arch_initcall(pesaro_pinctrl_init);

static void __exit pesaro_pinctrl_exit(void)
{
	platform_driver_unregister(&pesaro_pinctrl_driver);
}
module_exit(pesaro_pinctrl_exit);

MODULE_AUTHOR("Paul Chen <paul.che@vatics.com.tw>");
MODULE_DESCRIPTION("VATICS pesaro pinctrl driver");
MODULE_LICENSE("GPL v2");
MODULE_DEVICE_TABLE(of, pesaro_pinctrl_of_match);
