/*
 *  drivers/rtc/rtc-vpl.c
 *
 *  Copyright (C) 2007-2014  VATICS Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/rtc.h>
#include <linux/platform_device.h>
#include <linux/io.h>

#define version_fourcc(a, b, c, d)\
	((__u32)(a) | ((__u32)(b) << 8) | ((__u32)(c) << 16) | ((__u32)(d) << 24))

#define VPL_RTC_VERSION version_fourcc(1, 0, 0, 0)

#define RTCC_VERSION	0x00
#define RTCC_SECOND	0x04
#define RTCC_STAT	0x08
#define RTCC_METADATA	0x0c
#define RTCC_CTRL	0x10

#define RTCC_STAT_SET_SEC_BUSY	0x1

struct vpl_rtc_info {
	void __iomem		*base;	/* physical */
	struct device		*dev;
	struct rtc_device	*rtc_dev;
};

static int vpl_rtc_wait_while_busy(struct device *dev)
{
	struct vpl_rtc_info *info = dev_get_drvdata(dev);

	int retries = 500;

	while (readl(info->base + RTCC_STAT) & RTCC_STAT_SET_SEC_BUSY) {
		if (!retries--)
			goto retry_failed;
		udelay(1);
	}

	return 0;

retry_failed:
	dev_err(dev, "write failed:retry count exceeded.\n");
	return -ETIMEDOUT;
}

static int vpl_rtc_read_time(struct device *dev, struct rtc_time *tm)
{
	struct vpl_rtc_info *info = dev_get_drvdata(dev);
	unsigned long sec;

	sec = readl(info->base + RTCC_SECOND);

	rtc_time_to_tm(sec, tm);

	dev_vdbg(dev, "time read as %lu. %d/%d/%d %d:%02u:%02u\n",
			sec,
			tm->tm_mon + 1,
			tm->tm_mday,
			tm->tm_year + 1900,
			tm->tm_hour,
			tm->tm_min,
			tm->tm_sec
		);

	return 0;
}

static int vpl_rtc_set_time(struct device *dev, struct rtc_time *tm)
{
	struct vpl_rtc_info *info = dev_get_drvdata(dev);
	unsigned long sec;
	int ret;

	ret = rtc_valid_tm(tm);
	if (ret)
		return ret;

	rtc_tm_to_time(tm, &sec);

	dev_vdbg(dev, "time set to %lu. %d/%d/%d %d:%02u:%02u\n",
		 sec,
		 tm->tm_mon+1,
		 tm->tm_mday,
		 tm->tm_year+1900,
		 tm->tm_hour,
		 tm->tm_min,
		 tm->tm_sec
	);

	ret = vpl_rtc_wait_while_busy(dev);
	if (!ret)
		writel(sec+2, info->base + RTCC_SECOND); // Pesaro RTCC takes 2 seconds to update time

	return ret;
}

static int vpl_rtc_proc(struct device *dev, struct seq_file *seq)
{
	struct vpl_rtc_info *info = dev_get_drvdata(dev);
 	int osc_ppm = 0;
	seq_printf(seq, "rtc_osc_ppm\t: %d\n",	osc_ppm);
	return 0;
}

static struct rtc_class_ops vpl_rtc_ops = {
	.read_time	= vpl_rtc_read_time,
	.set_time	= vpl_rtc_set_time,
	.proc		= vpl_rtc_proc,
};

static const struct of_device_id vpl_rtc_of_match[] = {
	{ .compatible = "vatics,vpl-rtc", },
	{}
};
MODULE_DEVICE_TABLE(of, vpl_rtc_of_match);

static int __init vpl_rtc_probe(struct platform_device *pdev)
{
	struct device *dev;
	struct vpl_rtc_info *info;
	struct resource *res;
	int ret;

	dev = &pdev->dev;

	info = devm_kzalloc(dev, sizeof(*info), GFP_KERNEL);
	if (!info) {
		dev_err(dev, "unable to alloacate driver data\n");
		return -ENOMEM;
	}

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		dev_err(dev, "no memory resource specified\n");
		return -ENOENT;
	}

	info->base = devm_ioremap_resource(dev, res);
	if (IS_ERR(info->base))
		return PTR_ERR(info->base);

	info->dev = &pdev->dev;
	platform_set_drvdata(pdev, info);

	info->rtc_dev = devm_rtc_device_register(&pdev->dev,
				dev_name(&pdev->dev), &vpl_rtc_ops,
				THIS_MODULE);

	if (IS_ERR(info->rtc_dev)) {
		ret = PTR_ERR(info->rtc_dev);
		dev_err(&pdev->dev, "Unable to register device (err=%d).\n",
			ret);
		return ret;
	}

	dev_info(dev, "VPL Real Time Clock Hardware Rev 0x%08x\n",
		readl(info->base + RTCC_VERSION));

	dev_info(dev, "Driver Version %d.%d.%d.%d\n",
		VPL_RTC_VERSION & 0xff,
		(VPL_RTC_VERSION >> 8) & 0xff,
		(VPL_RTC_VERSION >> 16) & 0xff,
		VPL_RTC_VERSION >> 24);

	return 0;
}

static int __exit vpl_rtc_remove(struct platform_device *pdev)
{
	struct vpl_rtc_info *info = platform_get_drvdata(pdev);
	rtc_device_unregister(info->rtc_dev);
	return 0;
}

static struct platform_driver vpl_rtc_driver = {
	.remove		= __exit_p(vpl_rtc_remove),
	.driver		= {
		.name   = "vpl_rtc",
		.owner  = THIS_MODULE,
		.of_match_table = vpl_rtc_of_match,
	},
};

module_platform_driver_probe(vpl_rtc_driver, vpl_rtc_probe);

MODULE_AUTHOR("Vincent Ho <vincent.ho@vatics.com>");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("VPL RTC driver");
