/*
 *  drivers/usb/gadget/vendor.c
 *
 * Copyright (C) 2016-2018  VATICS Inc.
 *
 * Author: Michael Tu <michael.tu@vatics.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/list.h>
#include <linux/mutex.h>
#include <linux/usb/ch9.h>
#include <linux/usb/gadget.h>
#include <linux/vmalloc.h>
#include <linux/wait.h>
#include <linux/kallsyms.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/utsname.h>
#include <linux/usb/composite.h>
#include <asm/unaligned.h>

static const struct usb_device_descriptor vndr_dev = {
	.bLength		= 0x12,
	.bDescriptorType	= 0x01,
	.bcdUSB			= cpu_to_le16(0x0200),
	.bDeviceClass		= 0x00,
	.bDeviceSubClass	= 0x00,
	.bDeviceProtocol	= 0x00,
	.bMaxPacketSize0	= 0x40,
	.idVendor		= 0x1234,
	.idProduct		= 0x1234,
	.bcdDevice		= 0x0100,
	.iManufacturer		= 0x00,
	.iProduct		= 0x02,
	.iSerialNumber		= 0x00,
	.bNumConfigurations	= 0x01,
};

static const struct usb_config_descriptor vndr_conf = {
	.bLength		= 0x09,
	.bDescriptorType	= 0x02,
	.wTotalLength		= 0x0020,
	.bNumInterfaces		= 0x01,
	.bConfigurationValue	= 0x01,
	.iConfiguration		= 0x00,
	.bmAttributes		= 0xC0,
	.bMaxPower		= 0x01,
};

static const struct usb_interface_descriptor vndr_infa = {
	.bLength		= 0x09,
	.bDescriptorType	= 0x04,
	.bInterfaceNumber	= 0x00,
	.bAlternateSetting	= 0x00,
	.bNumEndpoints		= 0x00,
	.bInterfaceClass	= 0xFF,
	.bInterfaceSubClass	= 0x01,
	.bInterfaceProtocol	= 0x00,
	.iInterface		= 0x00,
};

static const struct usb_endpoint_descriptor vndr_epin = {
	.bLength		= 0x07,
	.bDescriptorType	= 0x05,
	.bEndpointAddress   = 0x81,
	.bmAttributes       = 0x02,
	.wMaxPacketSize	= 0x0200,
	.bInterval          = 0x00,
};

static const struct usb_endpoint_descriptor vndr_epout = {
	.bLength		= 0x07,
	.bDescriptorType	= 0x05,
	.bEndpointAddress   = 0x01,
	.bmAttributes       = 0x02,
	.wMaxPacketSize	= 0x0200,
	.bInterval          = 0x00,
};

static void
vndr_ep0_complete(struct usb_ep *ep, struct usb_request *req)
{
	if (req->status || req->actual != req->length)
		printk(KERN_INFO "setup complete --> %d, %d/%d\n",
				req->status, req->actual, req->length);
}

#define DeviceInRequest \
	((USB_DIR_IN | USB_TYPE_STANDARD | USB_RECIP_DEVICE) << 8)
#define DeviceOutRequest \
	((USB_DIR_OUT | USB_TYPE_STANDARD | USB_RECIP_DEVICE) << 8)
#define InterfaceInRequest \
	((USB_DIR_IN | USB_TYPE_STANDARD | USB_RECIP_INTERFACE) << 8)
#define InterfaceOutRequest \
	((USB_DIR_OUT | USB_TYPE_STANDARD | USB_RECIP_INTERFACE) << 8)

static int
vndr_setup_standard(struct usb_gadget *gadget,
	const struct usb_ctrlrequest *ctrl)
{
	struct usb_composite_dev *cdev = get_gadget_data(gadget);
	struct usb_request *req = cdev->req;
	u16 wIndex = le16_to_cpu(ctrl->wIndex);
	u16 wValue = le16_to_cpu(ctrl->wValue);
	int value = -EOPNOTSUPP;

	switch ((ctrl->bRequestType << 8) | ctrl->bRequest) {
	case DeviceInRequest | USB_REQ_GET_DESCRIPTOR:
		switch (wValue >> 8) {
		case USB_DT_DEVICE:
			memcpy(req->buf, &vndr_dev, 18);
			value = 18;
			break;
		case USB_DT_CONFIG:
			memcpy(req->buf, &vndr_conf, 9);
			memcpy(req->buf+9, &vndr_infa, 9);
			memcpy(req->buf+18, &vndr_epin, 7);
			memcpy(req->buf+25, &vndr_epout, 7);
			value = 32;
			break;
		case USB_DT_STRING:
			if((wValue & 0xFF) == 0x02){
				memcpy(req->buf, "\x0A\x03\x76\x00\x6E\x00\x64\x00\x72\x00", 10);//vndr
				value = 10;
			}
			else{
				memcpy(req->buf, "\x04\x03\x09\x04", 4);
				value = 4;
			}
			break;
		default:
			printk(KERN_INFO "Unsupported descriptor type %u\n",
				wIndex >> 8);
			break;
		}
		break;
	case DeviceOutRequest | USB_REQ_SET_CONFIGURATION:
		printk(KERN_INFO "set cfg\n");
		value = 1;
		break;
	default:
		printk(KERN_INFO "Unsupported request bRequestType 0x%02x "
			"bRequest 0x%02x wValue 0x%04x wIndex 0x%04x\n",
			ctrl->bRequestType, ctrl->bRequest, wValue, wIndex);
		break;
	}
	return value;
}

static int
vndr_setup(struct usb_gadget *gadget, const struct usb_ctrlrequest *ctrl)
{
	struct usb_composite_dev *cdev = get_gadget_data(gadget);
	struct usb_request *req = cdev->req;
	u16 wLength = le16_to_cpu(ctrl->wLength);
	int value = -EOPNOTSUPP;

	switch (ctrl->bRequestType & USB_TYPE_MASK) {
	case USB_TYPE_STANDARD:
		value = vndr_setup_standard(gadget, ctrl);
		break;
	default:
		printk(KERN_INFO "Unsupported request type 0x%02x\n",
			ctrl->bRequestType);
		break;
	}

	if (value >= 0) {
		req->length = min((int)wLength, value);
		req->zero = value < wLength;
		value = usb_ep_queue(gadget->ep0, req, GFP_ATOMIC);
	}

	return (value == -EAGAIN) ? 0 : value;
}

void vndr_disconnect(struct usb_gadget *gadget)
{}

void vndr_suspend(struct usb_gadget *gadget)
{}

static ssize_t usbvndr_cfg_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	return 0;
}

static DEVICE_ATTR(config, 0444, usbvndr_cfg_show, NULL);

/* --------------------------------------------------------------------------
 * USB probe and disconnect
 */
static void
vndr_unbind(struct usb_gadget *gadget)
{
	struct usb_composite_dev *dev = get_gadget_data(gadget);

	printk(KERN_INFO "vndr_unbind: gadget %s\n", gadget->name);
	device_remove_file(&gadget->dev, &dev_attr_config);
	set_gadget_data(gadget, NULL);
	kfree(dev);
}

static int
vndr_bind(struct usb_gadget *gadget,
		struct usb_gadget_driver *gdriver)
{
	struct usb_composite_dev *dev;
	int ret;

	printk(KERN_INFO "vndr_bind: gadget %s\n", gadget->name);
	dev = kmalloc(sizeof(*dev), GFP_KERNEL);
	if (dev == NULL)
		return -ENOMEM;

	dev->gadget = gadget;
	set_gadget_data(gadget, dev);

	/* Preallocate control endpoint request. */
	dev->req = usb_ep_alloc_request(gadget->ep0, GFP_KERNEL);
	if (dev->req == NULL){
		ret = -ENOMEM;
		goto error;
	}

	/* prevent bad params*/
	dev->req->complete = vndr_ep0_complete;
	dev->req->buf = kmalloc(USB_COMP_EP0_BUFSIZ, GFP_KERNEL);
	if (!dev->req->buf){
		ret = -ENOMEM;
		goto error;
	}

	/* Create the sysfs configuration file. */
	ret = device_create_file(&gadget->dev, &dev_attr_config);
	if(ret)
		goto error;

	return 0;

error:
	kfree(dev->req->buf);
	vndr_unbind(gadget);
	return ret;
}
/* --------------------------------------------------------------------------
 * Driver
 */

static struct usb_gadget_driver vndr_driver = {
	.function	= "USB VNDR Class",
	.bind		= vndr_bind,
	.unbind		= vndr_unbind,
	.setup		= vndr_setup,
	.disconnect	= vndr_disconnect,
	.suspend		= vndr_suspend,
	.driver		= {
		.name		= "vndr",
		.owner		= THIS_MODULE,
	},
};

static int __init vndr_init (void)
{
	return usb_gadget_probe_driver(&vndr_driver);
}

static void __exit vndr_cleanup (void)
{
	usb_gadget_unregister_driver(&vndr_driver);
}

module_init(vndr_init);
module_exit(vndr_cleanup);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Michael.Tu");
MODULE_DESCRIPTION("VATICS VENDOR driver");;
MODULE_ALIAS("usb:vndr");

