/*
 *	webcam.c -- USB webcam gadget driver
 *
 *	Copyright (C) 2009-2010
 *	    Laurent Pinchart (laurent.pinchart@ideasonboard.com)
 *
 *  VATICS added UAC supports, but UAC has not been verified yet.
 *  OTUS added compiler options at webcam.h header file, to allow
 *  kernel-mode driver and use-mode application sharing the same information.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 */

#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/usb/video.h>

#include "f_uvc.h"
#include "webcam.h"

/*
 * Kbuild is not very cooperative with respect to linking separately
 * compiled library objects into one module.  So for now we won't use
 * separate compilation ... ensuring init/exit sections work to shrink
 * the runtime footprint, and giving us at least some parts of what
 * a "gcc --combine ... part1.c part2.c part3.c ... " build would.
 */
#include "uvc_queue.c"
#include "uvc_video.c"
#include "uvc_v4l2.c"
#include "f_uvc.c"

#if WEBCAM_UAC_AUDIO_ENABLE	> 0

#include "f_uac.c"
#include "uac_alsa.c"
#include "uac_audio.c"

static int webcam_audio = 1;
module_param_named(audio, webcam_audio, int, S_IRUGO);
MODULE_PARM_DESC(audio, "Audio support");

#endif

USB_GADGET_COMPOSITE_OPTIONS();
/* --------------------------------------------------------------------------
 * Device descriptor
 */

#define WEBCAM_VENDOR_ID		0x1d6b	/* Linux Foundation */
#define WEBCAM_PRODUCT_ID		0x0102	/* Webcam A/V gadget */
#define WEBCAM_DEVICE_BCD		0x0010	/* 0.10 */

#if WEBCAM_UAC_AUDIO_ENABLE	> 0
static char webcam_vendor_label[] = "Linux Foundation";
static char webcam_product_label[] = "Webcam A/V gadget";
static char webcam_config_label[] = "Audio/Video";
#else
static char webcam_vendor_label[] = "Linux Foundation";
static char webcam_product_label[] = "Webcam gadget";
static char webcam_config_label[] = "Video";
#endif

/* string IDs are assigned dynamically */

#define STRING_DESCRIPTION_IDX		USB_GADGET_FIRST_AVAIL_IDX

static struct usb_string webcam_strings[] = {
	[USB_GADGET_MANUFACTURER_IDX].s = webcam_vendor_label,
	[USB_GADGET_PRODUCT_IDX].s = webcam_product_label,
	[USB_GADGET_SERIAL_IDX].s = "",
	[STRING_DESCRIPTION_IDX].s = webcam_config_label,
	{  }
};

static struct usb_gadget_strings webcam_stringtab = {
	.language = 0x0409,	/* en-us */
	.strings = webcam_strings,
};

static struct usb_gadget_strings *webcam_device_strings[] = {
	&webcam_stringtab,
	NULL,
};

static struct usb_device_descriptor webcam_device_descriptor = {
	.bLength		= USB_DT_DEVICE_SIZE,
	.bDescriptorType	= USB_DT_DEVICE,
	.bcdUSB			= cpu_to_le16(0x0200),
	.bDeviceClass		= USB_CLASS_MISC,
	.bDeviceSubClass	= 0x02,
	.bDeviceProtocol	= 0x01,
	.bMaxPacketSize0	= 0, /* dynamic */
	.idVendor		= cpu_to_le16(WEBCAM_VENDOR_ID),
	.idProduct		= cpu_to_le16(WEBCAM_PRODUCT_ID),
	.bcdDevice		= cpu_to_le16(WEBCAM_DEVICE_BCD),
	.iManufacturer		= 0, /* dynamic */
	.iProduct		= 0, /* dynamic */
	.iSerialNumber		= 0, /* dynamic */
	.bNumConfigurations	= 0, /* dynamic */
};

DECLARE_UVC_HEADER_DESCRIPTOR(1);

static const struct UVC_HEADER_DESCRIPTOR(1) uvc_control_header = {
	.bLength		= UVC_DT_HEADER_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VC_HEADER,
	.bcdUVC			= cpu_to_le16(0x0100),
	.wTotalLength		= 0, /* dynamic */
	.dwClockFrequency	= cpu_to_le32(48000000),
	.bInCollection		= 0, /* dynamic */
	.baInterfaceNr[0]	= 0, /* dynamic */
};

static const struct uvc_camera_terminal_descriptor uvc_camera_terminal = {
	.bLength		= UVC_DT_CAMERA_TERMINAL_SIZE(3),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VC_INPUT_TERMINAL,
	.bTerminalID		= WEBCAM_CAMERA_TERMINAL_ID,
	.wTerminalType		= cpu_to_le16(UVC_ITT_CAMERA),
	.bAssocTerminal		= 0,	// associate terminal
	.iTerminal		= 0,
	.wObjectiveFocalLengthMin	= cpu_to_le16(0),
	.wObjectiveFocalLengthMax	= cpu_to_le16(0),
	.wOcularFocalLength		= cpu_to_le16(0),
	.bControlSize		= 3,
	.bmControls[0]		= 0,//2,
	.bmControls[1]		= 0,
	.bmControls[2]		= 0,
};

static const struct uvc_processing_unit_descriptor uvc_processing = {
	.bLength		= UVC_DT_PROCESSING_UNIT_SIZE(2),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VC_PROCESSING_UNIT,
	.bUnitID		= WEBCAM_PROCESS_UNIT_ID,	// processing unit
	.bSourceID		= WEBCAM_CAMERA_TERMINAL_ID,	// source is input terminal
	.wMaxMultiplier		= cpu_to_le16(16*1024),
	.bControlSize		= 2,
	.bmControls[0]		= 0,//1,
	.bmControls[1]		= 0,
	.iProcessing		= 0,
};

#if WEBCAM_EXTENSION_UNIT_ID > 0
DECLARE_UVC_EXTENSION_UNIT_DESCRIPTOR(1, 2);

static const struct UVC_EXTENSION_UNIT_DESCRIPTOR(1, 2) uvc_extension_unit = {
	.bLength            = UVC_DT_EXTENSION_UNIT_SIZE(1, 2),
	.bDescriptorType    = USB_DT_CS_INTERFACE,
	.bDescriptorSubType = UVC_VC_EXTENSION_UNIT,
	.bUnitID            = WEBCAM_EXTENSION_UNIT_ID,
	.guidExtensionCode  =
		{0x55, 0x41, 0x14, 0x23, 0xea, 0xa6, 0x44, 0x2a,
		 0xa5, 0xed, 0xbb, 0x2e, 0xc8, 0xa8, 0xa8, 0x8b},
	.bNumControls       = 10,
	.bNrInPins          = 1,
	.baSourceID[0]      = WEBCAM_PROCESS_UNIT_ID,	// source is process unit
	.bControlSize       = 2,
	.bmControls[0]      = 0x07,
	.bmControls[1]      = 0x7f,
	.iExtension         = 0,
};
#endif

static const struct uvc_output_terminal_descriptor uvc_output_terminal = {
	.bLength		= UVC_DT_OUTPUT_TERMINAL_SIZE,
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VC_OUTPUT_TERMINAL,
	.bTerminalID		= WEBCAM_USB_TERMINAL_ID,	// output terminal
	.wTerminalType		= cpu_to_le16(UVC_TT_STREAMING),
	.bAssocTerminal		= 0,	// associate terminal
#if WEBCAM_EXTENSION_UNIT_ID > 0
	.bSourceID		= WEBCAM_EXTENSION_UNIT_ID, // source is extension unit
#else
	.bSourceID		= WEBCAM_PROCESS_UNIT_ID,	// source is process unit
#endif
	.iTerminal		= 0,
};

#if WEBCAM_FORMAT_NUMBERS == 3
DECLARE_UVC_INPUT_HEADER_DESCRIPTOR(1, 3);

static const struct UVC_INPUT_HEADER_DESCRIPTOR(1, 3) uvc_input_header = {
	.bLength		= UVC_DT_INPUT_HEADER_SIZE(1, 3),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_INPUT_HEADER,
	.bNumFormats		= 3, // three of I420, MJPEG, H264
	.wTotalLength		= 0, /* dynamic */
	.bEndpointAddress	= 0, /* dynamic */
	.bmInfo			= 0,
	.bTerminalLink		= WEBCAM_USB_TERMINAL_ID,	// link to output terminal
	.bStillCaptureMethod	= 0,
	.bTriggerSupport	= 0,
	.bTriggerUsage		= 0,
	.bControlSize		= 1,
	.bmaControls[0][0]	= 0,
	.bmaControls[1][0]	= 0,
	.bmaControls[2][0]	= 0,
};
#elif WEBCAM_FORMAT_NUMBERS == 2
DECLARE_UVC_INPUT_HEADER_DESCRIPTOR(1, 2);

static const struct UVC_INPUT_HEADER_DESCRIPTOR(1, 2) uvc_input_header = {
	.bLength		= UVC_DT_INPUT_HEADER_SIZE(1, 2),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_INPUT_HEADER,
	.bNumFormats		= 2, // two of I420, MJPEG, H264
	.wTotalLength		= 0, /* dynamic */
	.bEndpointAddress	= 0, /* dynamic */
	.bmInfo			= 0,
	.bTerminalLink		= WEBCAM_USB_TERMINAL_ID,	// link to output terminal
	.bStillCaptureMethod	= 0,
	.bTriggerSupport	= 0,
	.bTriggerUsage		= 0,
	.bControlSize		= 1,
	.bmaControls[0][0]	= 0,
	.bmaControls[1][0]	= 0,
};
#elif WEBCAM_FORMAT_NUMBERS == 1
DECLARE_UVC_INPUT_HEADER_DESCRIPTOR(1, 1);

static const struct UVC_INPUT_HEADER_DESCRIPTOR(1, 1) uvc_input_header = {
	.bLength		= UVC_DT_INPUT_HEADER_SIZE(1, 1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_INPUT_HEADER,
	.bNumFormats		= 1, // one of I420, MJPEG, H264
	.wTotalLength		= 0, /* dynamic */
	.bEndpointAddress	= 0, /* dynamic */
	.bmInfo			= 0,
	.bTerminalLink		= WEBCAM_USB_TERMINAL_ID,	// link to output terminal
	.bStillCaptureMethod	= 0,
	.bTriggerSupport	= 0,
	.bTriggerUsage		= 0,
	.bControlSize		= 1,
	.bmaControls[0][0]	= 0,
};
#endif


#define FPS2INTERVAL(fps)   (10000000 / fps)

#if WEBCAM_H264_FORMAT > 0
static const struct uvc_format_frame_based uvc_format_h264 = {
	.bLength              = UVC_DT_FORMAT_FRAME_BASED_SIZE,
	.bDescriptorType      = USB_DT_CS_INTERFACE,
	.bDescriptorSubType   = UVC_VS_FORMAT_FRAME_BASED,
	.bFormatIndex         = uvc_format_index_h264,	// format index,
	.bNumFrameDescriptors = h264_frame_index_max-1,	// 480p, 720p, 1080p
	.guidFormat           =
		{ 'H',  '2',  '6',  '4', 0x00, 0x00, 0x10, 0x00,
		 0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71},
	.bBitsPerPixel        = 16,
	.bDefaultFrameIndex   = 1,	// default frame index
	.bAspectRatioX        = 0,
	.bAspectRatioY        = 0,
	.bmInterfaceFlags     = 0,
	.bCopyProtect         = 0,
	.bVariableSize        = 1,
};

DECLARE_UVC_FRAME_FRAME_BASED(1);
DECLARE_UVC_FRAME_FRAME_BASED(3);

#if WEBCAM_H264_640x480  > 0
static const struct UVC_FRAME_FRAME_BASED(3) uvc_frame_h264_480p = {
	.bLength                = UVC_DT_FRAME_FRAME_BASED_SIZE(3),
	.bDescriptorType        = USB_DT_CS_INTERFACE,
	.bDescriptorSubType     = UVC_VS_FRAME_FRAME_BASED,
	.bFrameIndex            = h264_frame_index_640x480,	// frame index
	.bmCapabilities         = 0,
	.wWidth                 = cpu_to_le16(640),
	.wHeight                = cpu_to_le16(480),
	.dwMinBitRate           = cpu_to_le32(640*480*12*WEBCAM_H264_640x480_FPS_2/16),
	.dwMaxBitRate           = cpu_to_le32(640*480*12*WEBCAM_H264_640x480_FPS_0/16),
	.dwDefaultFrameInterval = cpu_to_le32(FPS2INTERVAL(WEBCAM_H264_640x480_FPS_0)),
	.bFrameIntervalType     = 3,
	.dwBytesPerLine         = 0,
	.dwFrameInterval[0]     = cpu_to_le32(FPS2INTERVAL(WEBCAM_H264_640x480_FPS_0)),
	.dwFrameInterval[1]     = cpu_to_le32(FPS2INTERVAL(WEBCAM_H264_640x480_FPS_1)),
	.dwFrameInterval[2]     = cpu_to_le32(FPS2INTERVAL(WEBCAM_H264_640x480_FPS_2)),
};
#endif

#if WEBCAM_H264_1280x720  > 0
static const struct UVC_FRAME_FRAME_BASED(3) uvc_frame_h264_720p = {
	.bLength                = UVC_DT_FRAME_FRAME_BASED_SIZE(3),
	.bDescriptorType        = USB_DT_CS_INTERFACE,
	.bDescriptorSubType     = UVC_VS_FRAME_FRAME_BASED,
	.bFrameIndex            = h264_frame_index_1280x720,	// frame index
	.bmCapabilities         = 0,
	.wWidth                 = cpu_to_le16(1280),
	.wHeight                = cpu_to_le16(720),
	.dwMinBitRate           = cpu_to_le32(1280*720*12*WEBCAM_H264_1280x720_FPS_2/16),
	.dwMaxBitRate           = cpu_to_le32(1280*720*12*WEBCAM_H264_1280x720_FPS_0/16),
	.dwDefaultFrameInterval = cpu_to_le32(FPS2INTERVAL(WEBCAM_H264_1280x720_FPS_0)),
	.bFrameIntervalType     = 3,
	.dwBytesPerLine         = 0,
	.dwFrameInterval[0]     = cpu_to_le32(FPS2INTERVAL(WEBCAM_H264_1280x720_FPS_0)),
	.dwFrameInterval[1]     = cpu_to_le32(FPS2INTERVAL(WEBCAM_H264_1280x720_FPS_1)),
	.dwFrameInterval[2]     = cpu_to_le32(FPS2INTERVAL(WEBCAM_H264_1280x720_FPS_2)),
};
#endif

#if WEBCAM_H264_1920x1080  > 0
static const struct UVC_FRAME_FRAME_BASED(3) uvc_frame_h264_1080p = {
	.bLength                = UVC_DT_FRAME_FRAME_BASED_SIZE(3),
	.bDescriptorType        = USB_DT_CS_INTERFACE,
	.bDescriptorSubType     = UVC_VS_FRAME_FRAME_BASED,
	.bFrameIndex            = h264_frame_index_1920x1080,	// frame index
	.bmCapabilities         = 0,
	.wWidth                 = cpu_to_le16(1920),
	.wHeight                = cpu_to_le16(1080),
	.dwMinBitRate           = cpu_to_le32(1920*1080*12*WEBCAM_H264_1920x1080_FPS_2/16),
	.dwMaxBitRate           = cpu_to_le32(1920*1080*12*WEBCAM_H264_1920x1080_FPS_0/16),
	.dwDefaultFrameInterval = cpu_to_le32(FPS2INTERVAL(WEBCAM_H264_1920x1080_FPS_0)),
	.bFrameIntervalType     = 3,
	.dwBytesPerLine         = 0,
	.dwFrameInterval[0]     = cpu_to_le32(FPS2INTERVAL(WEBCAM_H264_1920x1080_FPS_0)),
	.dwFrameInterval[1]     = cpu_to_le32(FPS2INTERVAL(WEBCAM_H264_1920x1080_FPS_1)),
	.dwFrameInterval[2]     = cpu_to_le32(FPS2INTERVAL(WEBCAM_H264_1920x1080_FPS_2)),
};
#endif
#endif

#if WEBCAM_YUV420_FORMAT > 0
static const struct uvc_format_uncompressed uvc_format_yuv = {
	.bLength				= UVC_DT_FORMAT_UNCOMPRESSED_SIZE,
	.bDescriptorType		= USB_DT_CS_INTERFACE,
	.bDescriptorSubType		= UVC_VS_FORMAT_UNCOMPRESSED,
	.bFormatIndex			= uvc_format_index_yuv420,	// format index
	.bNumFrameDescriptors	= yuv420_frame_index_max-1,	// 480p, 720p, 1080p
	.guidFormat		=
		{ 'I',  '4',  '2',  '0', 0x00, 0x00, 0x10, 0x00,
		 0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71},
	.bBitsPerPixel			= 12,
	.bDefaultFrameIndex		= 1,
	.bAspectRatioX			= 0,
	.bAspectRatioY			= 0,
	.bmInterfaceFlags		= 0,
	.bCopyProtect			= 0,
};

DECLARE_UVC_FRAME_UNCOMPRESSED(1);
DECLARE_UVC_FRAME_UNCOMPRESSED(3);

#if WEBCAM_YUV420_640x480 > 0
static const struct UVC_FRAME_UNCOMPRESSED(3) uvc_frame_yuv_480p = {
	.bLength				= UVC_DT_FRAME_UNCOMPRESSED_SIZE(3),
	.bDescriptorType		= USB_DT_CS_INTERFACE,
	.bDescriptorSubType		= UVC_VS_FRAME_UNCOMPRESSED,
	.bFrameIndex			= yuv420_frame_index_640x480,	// frame index
	.bmCapabilities			= 0,
	.wWidth					= cpu_to_le16(640),
	.wHeight				= cpu_to_le16(480),
	.dwMinBitRate			= cpu_to_le32(640*480*12*WEBCAM_YUV420_640x480_FPS_2),
	.dwMaxBitRate			= cpu_to_le32(640*480*12*WEBCAM_YUV420_640x480_FPS_0),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(640*480*3/2),
	.dwDefaultFrameInterval	= cpu_to_le32(FPS2INTERVAL(WEBCAM_YUV420_640x480_FPS_0)),
	.bFrameIntervalType		= 3,
	.dwFrameInterval[0]		= cpu_to_le32(FPS2INTERVAL(WEBCAM_YUV420_640x480_FPS_0)),
	.dwFrameInterval[1]		= cpu_to_le32(FPS2INTERVAL(WEBCAM_YUV420_640x480_FPS_1)),
	.dwFrameInterval[2]		= cpu_to_le32(FPS2INTERVAL(WEBCAM_YUV420_640x480_FPS_2)),
};
#endif

#if WEBCAM_YUV420_1280x720 > 0
static const struct UVC_FRAME_UNCOMPRESSED(3) uvc_frame_yuv_720p = {
	.bLength				= UVC_DT_FRAME_UNCOMPRESSED_SIZE(3),
	.bDescriptorType		= USB_DT_CS_INTERFACE,
	.bDescriptorSubType		= UVC_VS_FRAME_UNCOMPRESSED,
	.bFrameIndex			= yuv420_frame_index_1280x720,	// frame index
	.bmCapabilities			= 0,
	.wWidth					= cpu_to_le16(1280),
	.wHeight				= cpu_to_le16(720),
	.dwMinBitRate			= cpu_to_le32(1280*720*12*WEBCAM_YUV420_1280x720_FPS_2),
	.dwMaxBitRate			= cpu_to_le32(1280*720*12*WEBCAM_YUV420_1280x720_FPS_0),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(1280*720*3/2),
	.dwDefaultFrameInterval	= cpu_to_le32(FPS2INTERVAL(WEBCAM_YUV420_1280x720_FPS_0)),
	.bFrameIntervalType		= 3,
	.dwFrameInterval[0]		= cpu_to_le32(FPS2INTERVAL(WEBCAM_YUV420_1280x720_FPS_0)),
	.dwFrameInterval[1]		= cpu_to_le32(FPS2INTERVAL(WEBCAM_YUV420_1280x720_FPS_1)),
	.dwFrameInterval[2]		= cpu_to_le32(FPS2INTERVAL(WEBCAM_YUV420_1280x720_FPS_2)),
};
#endif

#if WEBCAM_YUV420_1920x1080 > 0
static const struct UVC_FRAME_UNCOMPRESSED(3) uvc_frame_yuv_1080p = {
	.bLength				= UVC_DT_FRAME_UNCOMPRESSED_SIZE(3),
	.bDescriptorType		= USB_DT_CS_INTERFACE,
	.bDescriptorSubType		= UVC_VS_FRAME_UNCOMPRESSED,
	.bFrameIndex			= yuv420_frame_index_1920x1080,	// frame index
	.bmCapabilities			= 0,
	.wWidth					= cpu_to_le16(1920),
	.wHeight				= cpu_to_le16(1080),
	.dwMinBitRate			= cpu_to_le32(1920*1080*12*WEBCAM_YUV420_1920x1080_FPS_2),
	.dwMaxBitRate			= cpu_to_le32(1920*1080*12*WEBCAM_YUV420_1920x1080_FPS_0),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(1920*1080*3/2),
	.dwDefaultFrameInterval	= cpu_to_le32(FPS2INTERVAL(WEBCAM_YUV420_1920x1080_FPS_0)),
	.bFrameIntervalType		= 3,
	.dwFrameInterval[0]		= cpu_to_le32(FPS2INTERVAL(WEBCAM_YUV420_1920x1080_FPS_0)),
	.dwFrameInterval[1]		= cpu_to_le32(FPS2INTERVAL(WEBCAM_YUV420_1920x1080_FPS_1)),
	.dwFrameInterval[2]		= cpu_to_le32(FPS2INTERVAL(WEBCAM_YUV420_1920x1080_FPS_2)),
};
#endif
#endif

#if WEBCAM_MPJEG_FORMAT > 0
static const struct uvc_format_mjpeg uvc_format_mjpg = {
	.bLength				= UVC_DT_FORMAT_MJPEG_SIZE,
	.bDescriptorType		= USB_DT_CS_INTERFACE,
	.bDescriptorSubType		= UVC_VS_FORMAT_MJPEG,
	.bFormatIndex			= uvc_format_index_mjpeg,	// format index
	.bNumFrameDescriptors	= mjpeg_frame_index_max-1,	// 480p, 720p, 1080p
	.bmFlags				= 0,
	.bDefaultFrameIndex		= 1,
	.bAspectRatioX			= 0,
	.bAspectRatioY			= 0,
	.bmInterfaceFlags		= 0,
	.bCopyProtect			= 0,
};

DECLARE_UVC_FRAME_MJPEG(1);
DECLARE_UVC_FRAME_MJPEG(3);

#if WEBCAM_MJPEG_640x480 > 0
static const struct UVC_FRAME_MJPEG(3) uvc_frame_mjpg_480p = {
	.bLength				= UVC_DT_FRAME_MJPEG_SIZE(3),
	.bDescriptorType		= USB_DT_CS_INTERFACE,
	.bDescriptorSubType		= UVC_VS_FRAME_MJPEG,
	.bFrameIndex			= mjpeg_frame_index_640x480,	// frame index
	.bmCapabilities			= 0,
	.wWidth					= cpu_to_le16(640),
	.wHeight				= cpu_to_le16(480),
	.dwMinBitRate			= cpu_to_le32(640*480*12*WEBCAM_MJPEG_640x480_FPS_2/8),
	.dwMaxBitRate			= cpu_to_le32(640*480*12*WEBCAM_MJPEG_640x480_FPS_0/8),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(640*480*3/2/8),
	.dwDefaultFrameInterval	= cpu_to_le32(FPS2INTERVAL(WEBCAM_MJPEG_640x480_FPS_0)),
	.bFrameIntervalType		= 3,
	.dwFrameInterval[0]		= cpu_to_le32(FPS2INTERVAL(WEBCAM_MJPEG_640x480_FPS_0)),
	.dwFrameInterval[1]		= cpu_to_le32(FPS2INTERVAL(WEBCAM_MJPEG_640x480_FPS_1)),
	.dwFrameInterval[2]		= cpu_to_le32(FPS2INTERVAL(WEBCAM_MJPEG_640x480_FPS_2)),
};
#endif

#if WEBCAM_MJPEG_1280x720 > 0
static const struct UVC_FRAME_MJPEG(3) uvc_frame_mjpg_720p = {
	.bLength				= UVC_DT_FRAME_MJPEG_SIZE(3),
	.bDescriptorType		= USB_DT_CS_INTERFACE,
	.bDescriptorSubType		= UVC_VS_FRAME_MJPEG,
	.bFrameIndex			= mjpeg_frame_index_1280x720,	// frame index
	.bmCapabilities			= 0,
	.wWidth					= cpu_to_le16(1280),
	.wHeight				= cpu_to_le16(720),
	.dwMinBitRate			= cpu_to_le32(1280*720*12*WEBCAM_MJPEG_1280x720_FPS_2/8),
	.dwMaxBitRate			= cpu_to_le32(1280*720*12*WEBCAM_MJPEG_1280x720_FPS_0/8),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(1280*720*3/2/8),
	.dwDefaultFrameInterval	= cpu_to_le32(FPS2INTERVAL(WEBCAM_MJPEG_1280x720_FPS_0)),
	.bFrameIntervalType		= 3,
	.dwFrameInterval[0]		= cpu_to_le32(FPS2INTERVAL(WEBCAM_MJPEG_1280x720_FPS_0)),
	.dwFrameInterval[1]		= cpu_to_le32(FPS2INTERVAL(WEBCAM_MJPEG_1280x720_FPS_1)),
	.dwFrameInterval[2]		= cpu_to_le32(FPS2INTERVAL(WEBCAM_MJPEG_1280x720_FPS_2)),
};
#endif

#if WEBCAM_MJPEG_1920x1080 > 0
static const struct UVC_FRAME_MJPEG(3) uvc_frame_mjpg_1080p = {
	.bLength				= UVC_DT_FRAME_MJPEG_SIZE(3),
	.bDescriptorType		= USB_DT_CS_INTERFACE,
	.bDescriptorSubType		= UVC_VS_FRAME_MJPEG,
	.bFrameIndex			= mjpeg_frame_index_1920x1080,	// frame index
	.bmCapabilities			= 0,
	.wWidth					= cpu_to_le16(1920),
	.wHeight				= cpu_to_le16(1080),
	.dwMinBitRate			= cpu_to_le32(1920*1080*12*WEBCAM_MJPEG_1920x1080_FPS_2/8),
	.dwMaxBitRate			= cpu_to_le32(1920*1080*12*WEBCAM_MJPEG_1920x1080_FPS_0/8),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(1920*1080*3/2/8),
	.dwDefaultFrameInterval	= cpu_to_le32(FPS2INTERVAL(WEBCAM_MJPEG_1920x1080_FPS_0)),
	.bFrameIntervalType		= 3,
	.dwFrameInterval[0]		= cpu_to_le32(FPS2INTERVAL(WEBCAM_MJPEG_1920x1080_FPS_0)),
	.dwFrameInterval[1]		= cpu_to_le32(FPS2INTERVAL(WEBCAM_MJPEG_1920x1080_FPS_1)),
	.dwFrameInterval[2]		= cpu_to_le32(FPS2INTERVAL(WEBCAM_MJPEG_1920x1080_FPS_2)),
};
#endif
#endif

static const struct uvc_color_matching_descriptor uvc_color_matching = {
	.bLength		= UVC_DT_COLOR_MATCHING_SIZE,
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_COLORFORMAT,
	.bColorPrimaries	= 1,
	.bTransferCharacteristics	= 1,
	.bMatrixCoefficients	= 4,
};

static const struct uvc_descriptor_header * const uvc_fs_control_cls[] = {
	(const struct uvc_descriptor_header *) &uvc_control_header,
	(const struct uvc_descriptor_header *) &uvc_camera_terminal,
	(const struct uvc_descriptor_header *) &uvc_processing,
#if WEBCAM_EXTENSION_UNIT_ID > 0
	(const struct uvc_descriptor_header *) &uvc_extension_unit,
#endif
	(const struct uvc_descriptor_header *) &uvc_output_terminal,
	NULL,
};

static const struct uvc_descriptor_header * const uvc_ss_control_cls[] = {
	(const struct uvc_descriptor_header *) &uvc_control_header,
	(const struct uvc_descriptor_header *) &uvc_camera_terminal,
	(const struct uvc_descriptor_header *) &uvc_processing,
#if WEBCAM_EXTENSION_UNIT_ID > 0
	(const struct uvc_descriptor_header *) &uvc_extension_unit,
#endif
	(const struct uvc_descriptor_header *) &uvc_output_terminal,
	NULL,
};

static const struct uvc_descriptor_header * const uvc_fs_streaming_cls[] = {
	(const struct uvc_descriptor_header *) &uvc_input_header,
#if WEBCAM_H264_FORMAT > 0
	(const struct uvc_descriptor_header *) &uvc_format_h264,
#if WEBCAM_H264_640x480 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_h264_480p,
#endif
#if WEBCAM_H264_1280x720 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_h264_720p,
#endif
#if WEBCAM_H264_1920x1080 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_h264_1080p,
#endif
#endif
#if WEBCAM_MPJEG_FORMAT > 0
	(const struct uvc_descriptor_header *) &uvc_format_mjpg,
#if WEBCAM_MJPEG_640x480 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_mjpg_480p,
#endif
#if WEBCAM_MJPEG_1280x720 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_mjpg_720p,
#endif
#if WEBCAM_MJPEG_1920x1080 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_mjpg_1080p,
#endif
#endif
#if WEBCAM_YUV420_FORMAT > 0
	(const struct uvc_descriptor_header *) &uvc_format_yuv,
#if WEBCAM_YUV420_640x480 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_yuv_480p,
#endif
#if WEBCAM_YUV420_1280x720 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_yuv_720p,
#endif
#if WEBCAM_YUV420_1920x1080 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_yuv_1080p,
#endif
#endif
	(const struct uvc_descriptor_header *) &uvc_color_matching,
	NULL,
};

static const struct uvc_descriptor_header * const uvc_hs_streaming_cls[] = {
	(const struct uvc_descriptor_header *) &uvc_input_header,
#if WEBCAM_H264_FORMAT > 0
	(const struct uvc_descriptor_header *) &uvc_format_h264,
#if WEBCAM_H264_640x480 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_h264_480p,
#endif
#if WEBCAM_H264_1280x720 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_h264_720p,
#endif
#if WEBCAM_H264_1920x1080 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_h264_1080p,
#endif
#endif
#if WEBCAM_MPJEG_FORMAT > 0
	(const struct uvc_descriptor_header *) &uvc_format_mjpg,
#if WEBCAM_MJPEG_640x480 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_mjpg_480p,
#endif
#if WEBCAM_MJPEG_1280x720 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_mjpg_720p,
#endif
#if WEBCAM_MJPEG_1920x1080 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_mjpg_1080p,
#endif
#endif
#if WEBCAM_YUV420_FORMAT > 0
	(const struct uvc_descriptor_header *) &uvc_format_yuv,
#if WEBCAM_YUV420_640x480 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_yuv_480p,
#endif
#if WEBCAM_YUV420_1280x720 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_yuv_720p,
#endif
#if WEBCAM_YUV420_1920x1080 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_yuv_1080p,
#endif
#endif
	(const struct uvc_descriptor_header *) &uvc_color_matching,
	NULL,
};

static const struct uvc_descriptor_header * const uvc_ss_streaming_cls[] = {
	(const struct uvc_descriptor_header *) &uvc_input_header,
#if WEBCAM_H264_FORMAT > 0
	(const struct uvc_descriptor_header *) &uvc_format_h264,
#if WEBCAM_H264_640x480 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_h264_480p,
#endif
#if WEBCAM_H264_1280x720 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_h264_720p,
#endif
#if WEBCAM_H264_1920x1080 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_h264_1080p,
#endif
#endif
#if WEBCAM_MPJEG_FORMAT > 0
	(const struct uvc_descriptor_header *) &uvc_format_mjpg,
#if WEBCAM_MJPEG_640x480 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_mjpg_480p,
#endif
#if WEBCAM_MJPEG_1280x720 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_mjpg_720p,
#endif
#if WEBCAM_MJPEG_1920x1080 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_mjpg_1080p,
#endif
#endif
#if WEBCAM_YUV420_FORMAT > 0
	(const struct uvc_descriptor_header *) &uvc_format_yuv,
#if WEBCAM_YUV420_640x480 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_yuv_480p,
#endif
#if WEBCAM_YUV420_1280x720 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_yuv_720p,
#endif
#if WEBCAM_YUV420_1920x1080 > 0
	(const struct uvc_descriptor_header *) &uvc_frame_yuv_1080p,
#endif
#endif
	(const struct uvc_descriptor_header *) &uvc_color_matching,
	NULL,
};

/* --------------------------------------------------------------------------
 * USB configuration
 */

static int __init
webcam_config_bind(struct usb_configuration *c)
{
	int ret;

	// bind UVC
	ret = uvc_bind_config(c, uvc_fs_control_cls, uvc_ss_control_cls,
		uvc_fs_streaming_cls, uvc_hs_streaming_cls,
		uvc_ss_streaming_cls);

	if (ret < 0)
		return ret;

#if WEBCAM_UAC_AUDIO_ENABLE	> 0
	// bind UAC
	if (webcam_audio) {
		ret = audio_bind_config(c);
		if (ret < 0)
			return ret;
	}
#endif

	return 0;
}

static struct usb_configuration webcam_config_driver = {
	.label			= webcam_config_label,
	.bConfigurationValue	= 1,
	.iConfiguration		= 0, /* dynamic */
	.bmAttributes		= USB_CONFIG_ATT_SELFPOWER,
	.MaxPower		= CONFIG_USB_GADGET_VBUS_DRAW,
};

static int /* __init_or_exit */
webcam_unbind(struct usb_composite_dev *cdev)
{
	return 0;
}

static int __init
webcam_bind(struct usb_composite_dev *cdev)
{
	int ret;

	/* Allocate string descriptor numbers ... note that string contents
	 * can be overridden by the composite_dev glue.
	 */
	ret = usb_string_ids_tab(cdev, webcam_strings);
	if (ret < 0)
		goto error;
	// .iManufacturer, .iProduct, .iConfiguration
	webcam_device_descriptor.iManufacturer =
		webcam_strings[USB_GADGET_MANUFACTURER_IDX].id;
	webcam_device_descriptor.iProduct =
		webcam_strings[USB_GADGET_PRODUCT_IDX].id;
	webcam_config_driver.iConfiguration =
		webcam_strings[STRING_DESCRIPTION_IDX].id;

	/* Register our configuration. */
	if ((ret = usb_add_config(cdev, &webcam_config_driver,
					webcam_config_bind)) < 0)
		goto error;

	usb_composite_overwrite_options(cdev, &coverwrite);
#if WEBCAM_UAC_AUDIO_ENABLE	> 0
	INFO(cdev, "Webcam Audio/Video Gadget\n");
#else
	INFO(cdev, "Webcam Video Gadget\n");
#endif
	return 0;

error:
	webcam_unbind(cdev);
	return ret;
}

/* --------------------------------------------------------------------------
 * sysfs
 */

struct class *webcam_class;

/* --------------------------------------------------------------------------
 * Driver
 */

static __refdata struct usb_composite_driver webcam_driver = {
	.name		= "g_webcam",
	.dev		= &webcam_device_descriptor,
	.strings	= webcam_device_strings,
	.max_speed	= USB_SPEED_SUPER,
	.bind		= webcam_bind,
	.unbind		= webcam_unbind,
};

static int __init
webcam_init(void)
{
	int ret;

	webcam_class = class_create(THIS_MODULE, "webcam");
	if (IS_ERR(webcam_class))
	{
		ret = PTR_ERR(webcam_class);
		goto err;
	}

	ret = usb_composite_probe(&webcam_driver);
	if (ret == 0)
		return ret;

	class_destroy(webcam_class);
err:
	return ret;
}

static void __exit
webcam_cleanup(void)
{
	usb_composite_unregister(&webcam_driver);
	class_destroy(webcam_class);
}

module_init(webcam_init);
module_exit(webcam_cleanup);

#if WEBCAM_UAC_AUDIO_ENABLE	> 0
MODULE_AUTHOR("Laurent Pinchart");
MODULE_DESCRIPTION("Webcam Audio/Video Gadget");
MODULE_LICENSE("GPL");
MODULE_VERSION("0.1.0");
#else
MODULE_AUTHOR("Laurent Pinchart");
MODULE_DESCRIPTION("Webcam Video Gadget");
MODULE_LICENSE("GPL");
MODULE_VERSION("0.1.0");
#endif

