#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/err.h>
#include <linux/kdev_t.h>
#include <linux/string.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/irqdomain.h>
#include <asm/io.h>
#include <linux/sched.h>
#include <mach/hardware.h>
#include <mach/maps.h>
#include <mach/irqs.h>
#include <linux/irqchip/pesaro.h>
#include <linux/firmware.h>
#include <linux/dma-mapping.h>

#include <linux/device.h>
#include <linux/cdev.h>
#include <linux/fs.h>
#include <asm/uaccess.h>
#include <asm/cacheflush.h>

#include "n9.h"

#define IVB_BASE 			0x1500000
#define SYSC_BASE 			IO_ADDRESS(VPL_SYSC_MMR_BASE)
#define SYSC_RST_CTRL 			SYSC_BASE + 0x24
#define SYSC_N903U_CTRL 		SYSC_BASE + 0x68
#define SYSC_CLK_EN_CTRL_1 		SYSC_BASE + 0x98

#define N903U_RST_off 9
#define N903U_IVB_off 12

static DEFINE_MUTEX(sysfs_lock);
static DEFINE_MUTEX(ch_lock);
static DEFINE_MUTEX(pwm_lock);
static wait_queue_head_t buf_wq;

static unsigned long g_cmdAddr = OSDE_SHARE_ADDR;
static void *cmdAddr;
struct n9_dev {
	int start;
	char* resetAddr;
	int bufAddr;
	int width;
	int height;
	ipc_channel_t *ch[N9_CHANNEL_NUM];
	dma_addr_t dma_handle[N9_CHANNEL_NUM];
	dma_addr_t n9_status[N9_CHANNEL_NUM];
	dma_addr_t n2a_isr_handle;
	dma_addr_t a2n_isr_handle;
	int current_ch;
};

static struct n9_dev *gDev;
static int global_n9_start = 0;

int n9_start(void)
{
	return global_n9_start;
}
EXPORT_SYMBOL(n9_start);

static ssize_t n9_start_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	return sprintf(buf, "%d\n", gDev->start);
}
static ssize_t n9_start_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size)
{
	int val;

	mutex_lock(&sysfs_lock);

	gDev->start = 1;
	val = readl(SYSC_RST_CTRL);
	printk("%s, before SYSC_RST_CTRL:0x%x\n", __func__, val);
	val |= 1 << N903U_RST_off;
	printk("%s, SYSC_RST_CTRL:0x%x\n", __func__, val);
	writel(val, SYSC_RST_CTRL);
	mdelay(20);
	global_n9_start = 1;

	mutex_unlock(&sysfs_lock);

	return size;

}
static /*const*/ DEVICE_ATTR(start, 0644,
		n9_start_show, n9_start_store);

static ssize_t n9_resetAddr_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	int val;
	val = readl(SYSC_RST_CTRL);
	printk("SYSC_RST_CTRL:0x%x\n", val);
	val = readl(SYSC_N903U_CTRL);
	printk("SYSC_N903U_CTRL:0x%x\n", val);
	return sprintf(buf, "0x%x\n", (unsigned int)gDev->resetAddr);
}
static ssize_t n9_resetAddr_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size)
{
	/* this is just pwm example */
#if 0
	int status;
	ipc_message_t *data = kzalloc(sizeof(ipc_message_t), GFP_KERNEL);
	data->pwm.pwm_start = 3;

	data->pwm.pwm_count[0] = 100;
	data->pwm.pwm_count[1] = 200;
	// share info address
	data->pwm.pwm_gap = 100000;
	data->pwm.pwm_gpioc = 1;

	data->pwm.pwm_gpio_num[0] = 2;
	data->pwm.pwm_gpio_pin[0][0] = 8;
	data->pwm.pwm_gpio_pin[0][1] = 10;
	data->pwm.pwm_status_num[0] = 4;

	data->pwm.pwm_gpio_num[1] = 4;
	data->pwm.pwm_gpio_pin[1][0] = 14;
	data->pwm.pwm_gpio_pin[1][1] = 9;
	data->pwm.pwm_gpio_pin[1][2] = 11;
	data->pwm.pwm_gpio_pin[1][3] = 13;
	data->pwm.pwm_status_num[1] = 4;

	data->pwm.pwm_status[0][0][0]  = 1;
	data->pwm.pwm_status[0][1][0]  = 0;

	data->pwm.pwm_status[0][0][1]  = 0;
	data->pwm.pwm_status[0][1][1]  = 1;

	data->pwm.pwm_status[0][0][2]  = 1;
	data->pwm.pwm_status[0][1][2]  = 0;

	data->pwm.pwm_status[0][0][3]  = 0;
	data->pwm.pwm_status[0][1][3]  = 1;

	data->pwm.pwm_status[1][0][0]  = 1;
	data->pwm.pwm_status[1][1][0]  = 0;
	data->pwm.pwm_status[1][2][0]  = 1;
	data->pwm.pwm_status[1][3][0]  = 0;

	data->pwm.pwm_status[1][0][1]  = 0;
	data->pwm.pwm_status[1][1][1]  = 1;
	data->pwm.pwm_status[1][2][1]  = 0;
	data->pwm.pwm_status[1][3][1]  = 1;

	data->pwm.pwm_status[1][0][2]  = 1;
	data->pwm.pwm_status[1][1][2]  = 0;
	data->pwm.pwm_status[1][2][2]  = 1;
	data->pwm.pwm_status[1][3][2]  = 0;

	data->pwm.pwm_status[1][0][3]  = 0;
	data->pwm.pwm_status[1][1][3]  = 1;
	data->pwm.pwm_status[1][2][3]  = 0;
	data->pwm.pwm_status[1][3][3]  = 1;

	n9_pwm_init();
	n9_pwm_send(data);

	mdelay(100);
	status = n9_pwm_status();
	printk("0x%x\n", status);
	mdelay(2000);
	status = n9_pwm_status();
	printk("0x%x\n", status);
	kfree(data);
#endif
	return size;
}

static /*const*/ DEVICE_ATTR(resetAddr, 0644,
		n9_resetAddr_show, n9_resetAddr_store);

static void intr_to_n9(void)
{
	int val;
	ipc_channel_t *ich = gDev->ch[ATN_ISR_CHANNEL];
	int ret;
	int count = 0;
	val = readl(IO_ADDRESS(VPL_INTC_MMR_BASE + INTC_SET_LO));
	val |= 0x1 << ARM926U_IRQ_NUM;
	writel(val, IO_ADDRESS(VPL_INTC_MMR_BASE + INTC_SET_LO));
	/*XXX is it good to wait here? */
	do {
		dma_sync_single_for_cpu(NULL, gDev->a2n_isr_handle,
				N9_CHANNEL_SIZE - 32, DMA_FROM_DEVICE);
		ret = ich->done;
		dma_sync_single_for_device(NULL, gDev->a2n_isr_handle,
				N9_CHANNEL_SIZE - 32, DMA_FROM_DEVICE);
		count++;
		if (LOOP_TIMES == count) {
			printk("%s, n9 status error\n", __func__);
			break;
		}

	} while (ret != OWNER_ARM);
}

inline ipc_channel_t* get_channel(void)
{	
	return gDev->ch[gDev->current_ch];
}

void wait_for_prevcmd(void)
{
	ipc_channel_t *ich = gDev->ch[ATN_ISR_CHANNEL];
	int ret;
	int count = 0;
	do {
		dma_sync_single_for_cpu(NULL, gDev->a2n_isr_handle,
				N9_CHANNEL_SIZE - 32, DMA_FROM_DEVICE);
		ret = ich->done;
		dma_sync_single_for_device(NULL, gDev->a2n_isr_handle,
				N9_CHANNEL_SIZE - 32, DMA_FROM_DEVICE);
		count++;
		if (LOOP_TIMES == count) {
			printk("%s, n9 status error\n", __func__);
			break;
		}
	} while (ret != OWNER_ARM);
}

static void postcmd_new(ipc_message_t* cmd)
{
	ipc_channel_t *cch = gDev->ch[gDev->current_ch];
	ipc_channel_t *ich = gDev->ch[ATN_ISR_CHANNEL];

	/* prepare interrupt channel num */
	wait_for_prevcmd();
	dma_sync_single_for_cpu(NULL, gDev->dma_handle[ATN_ISR_CHANNEL],
			N9_CHANNEL_SIZE - 32, DMA_BIDIRECTIONAL);
	ich->ch_type = gDev->current_ch;
	ich->done = OWNER_N9;
	dma_sync_single_for_device(NULL, gDev->dma_handle[ATN_ISR_CHANNEL],
			N9_CHANNEL_SIZE - 32, DMA_BIDIRECTIONAL);

	dma_sync_single_for_cpu(NULL, gDev->dma_handle[gDev->current_ch],
			N9_CHANNEL_SIZE - 32, DMA_BIDIRECTIONAL);

	cch->arm[TX_STATUS] = 1;
	cch->command = cmd->head;
	cch->done = 0;
	switch (cmd->head) {
		case AUDIO_AEC:
			memcpy(&cch->data,&cmd->aec, sizeof(aec_t));
			break;
		case MBQP_COMPUTE:
			memcpy(&cch->data,&cmd->mbqp, sizeof(mbqp_t));
			break;
		case WBLEND:
			memcpy(&cch->data,&cmd->wblend, sizeof(weighted_blending_msg_t));
			break;
#if DELTA_FLAG
		case DELTA_INTERLACE:
			memcpy(&cch->data,&cmd->delta, sizeof(delta_t));			
			break;
#endif
	};
	dma_sync_single_for_device(NULL, gDev->dma_handle[gDev->current_ch],
			N9_CHANNEL_SIZE - 32, DMA_BIDIRECTIONAL);
}

static dma_addr_t pwm_dma;
void n9_pwm_init(void)
{
	pwm_dma = dma_map_single(NULL, gDev->ch[PWM_STATUS_CHANNEL],
		N9_CHANNEL_SIZE, DMA_FROM_DEVICE);
	dma_sync_single_for_device(NULL, pwm_dma, N9_CHANNEL_SIZE, DMA_FROM_DEVICE);
	return;
}

int n9_pwm_status(void)
{
	int ret;
	ipc_channel_t *cch = gDev->ch[PWM_STATUS_CHANNEL];

	mutex_lock(&pwm_lock);

	dma_sync_single_for_cpu(NULL, pwm_dma, N9_CHANNEL_SIZE, DMA_FROM_DEVICE);
	ret = cch->command; 
	dma_sync_single_for_device(NULL, pwm_dma, N9_CHANNEL_SIZE, DMA_FROM_DEVICE);

	mutex_unlock(&pwm_lock);
	return ret;
}

void n9_pwm_send(ipc_message_t* cmd)
{
	// hard code for channel 0, share data
	ipc_channel_t *cch = gDev->ch[PWM_INFO_CHANNEL];

	mutex_lock(&pwm_lock);

	dma_sync_single_for_cpu(NULL, gDev->dma_handle[PWM_INFO_CHANNEL],
			N9_CHANNEL_SIZE - 32, DMA_BIDIRECTIONAL);
	memcpy(&cch->data,&cmd->pwm, sizeof(pwm_t));
	dma_sync_single_for_device(NULL, gDev->dma_handle[PWM_INFO_CHANNEL],
			N9_CHANNEL_SIZE - 32, DMA_BIDIRECTIONAL);
	intr_to_n9();

	mutex_unlock(&pwm_lock);
	return;
}

static ssize_t n9_intr_show(struct device *dev,
               struct device_attribute *attr, char *buf)
{
       int val;
       val = readl(IO_ADDRESS(VPL_INTC_MMR_BASE + INTC_HOSTU_0_STAT_LO));
       printk("HOST_0_STATUS_LO:0x%x\n", val);
       val = readl(IO_ADDRESS(VPL_INTC_MMR_BASE + INTC_HOSTU_0_MASK_LO));
       printk("HOST_0_MASK_LO:0x%x\n", val);
       val = readl(IO_ADDRESS(VPL_INTC_MMR_BASE + INTC_TRIGGER_MODE_LO));
       printk("HOST_0_MODE_LO:0x%x\n", val);
       return sprintf(buf, "0x%x\n", (unsigned int)gDev->resetAddr);
}

static ssize_t n9_intr_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size)
{
	intr_to_n9();	
	return size;
}

static /*const*/ DEVICE_ATTR(intr, 0644,
		n9_intr_show, n9_intr_store);

static int n9_open(struct inode * inode, struct file * filp)
{
       return 0;
}

static int n9_release (struct inode *inode, struct file *filp)
{
       return 0;
}

static long n9_ioctl (struct file *file, unsigned int cmd, 
		unsigned long arg)
{
	int retval = 0;
	ipc_message_t msg;
	ipc_channel_t *cch;
	if (copy_from_user(&msg, (ipc_message_t  __user *)arg, sizeof(msg))) {
		retval = -EFAULT;
		goto done;
	}
		
	mutex_lock(&ch_lock);
	switch(msg.head){
		case N9_MBQP_COMPUTE:
			//printk("cmd:n9_mbqp_compute\n");
			gDev->current_ch = 1;
			cch = get_channel();
			msg.head = MBQP_COMPUTE;
			postcmd_new(&msg);
//			printk("t1s\n");
			intr_to_n9();	
			mutex_unlock(&ch_lock);
			wait_event_interruptible(buf_wq, cch->done);
			break;
		case N9_AUDIO_AEC:
			//printk("cmd:N9_AUDIO_AEC\n");
			gDev->current_ch = 0;
			cch = get_channel();
			msg.head = AUDIO_AEC;
			postcmd_new(&msg);
//			printk("t0s\n");
			intr_to_n9();
			mutex_unlock(&ch_lock);
			wait_event_interruptible(buf_wq, cch->done);
			break;
		case N9_WBLEND:
			//printk("cmd:N9_WBLEND\n");
			gDev->current_ch = 5;
			cch = get_channel();
			msg.head = WBLEND;
			postcmd_new(&msg);
//			printk("t5s\n");
			intr_to_n9();
			mutex_unlock(&ch_lock);
			wait_event_interruptible(buf_wq, cch->done);
			break;
#if DELTA_FLAG
		case N9_DELTA_INTERLACE:
			//printk("cmd:N9_DELTA_INTERLACE\n");
			gDev->current_ch = 1;
			cch = get_channel();
			msg.head = DELTA_INTERLACE;
			postcmd_new(&msg);
//			printk("t1s\n");
			intr_to_n9();
			mutex_unlock(&ch_lock);
			wait_event_interruptible(buf_wq, cch->done);
			break;
#endif
		default:
			retval = -ENOTTY;
			goto done;
	}
done:
	return retval;
}

static struct cdev *cdev = NULL;
static struct file_operations n9_fops = {
	open: 		n9_open,
	unlocked_ioctl: n9_ioctl,
	release: 	n9_release,
};

static irqreturn_t n9_osde_isr(int irq, void *dev_id)
{
	ipc_channel_t *cch;
	ipc_channel_t *ich = gDev->ch[NTA_ISR_CHANNEL];
	int result;
	int i;
	int count = 0;
	/*XXX better to move the checking code to work queue? */
	do {
		dma_sync_single_for_cpu(NULL, gDev->n2a_isr_handle,
				N9_CHANNEL_SIZE - 32, DMA_FROM_DEVICE);
		result = ich->done;
		dma_sync_single_for_device(NULL, gDev->n2a_isr_handle,
				N9_CHANNEL_SIZE - 32, DMA_FROM_DEVICE);
		count++;
		if (LOOP_TIMES == count) {
			printk("%s, n9 status error\n", __func__);
			return IRQ_HANDLED;
		}
	} while (result != OWNER_ARM);

	dma_sync_single_for_cpu(NULL, gDev->n2a_isr_handle,
			N9_CHANNEL_SIZE - 32, DMA_FROM_DEVICE);
	i = ich->ch_type;
	dma_sync_single_for_device(NULL, gDev->n2a_isr_handle,
			N9_CHANNEL_SIZE - 32, DMA_FROM_DEVICE);

	cch = gDev->ch[i];

	dma_sync_single_for_cpu(NULL, gDev->dma_handle[i],
			N9_CHANNEL_SIZE - 32, DMA_BIDIRECTIONAL);
	dma_sync_single_for_cpu(NULL, gDev->n9_status[i],
			32, DMA_FROM_DEVICE);

	/* this means task complete interrupt */
	if (cch->arm[TX_STATUS] && cch->n9[TX_STATUS]) {
		cch->arm[TX_STATUS] = 0;
		cch->done = 1;
	}
	dma_sync_single_for_device(NULL, gDev->dma_handle[i],
			N9_CHANNEL_SIZE - 32, DMA_BIDIRECTIONAL);
	dma_sync_single_for_device(NULL, gDev->n9_status[i],
			32, DMA_FROM_DEVICE);

	wake_up_interruptible(&buf_wq);

	dma_sync_single_for_cpu(NULL, gDev->dma_handle[NTA_ISR_CHANNEL],
			N9_CHANNEL_SIZE - 32, DMA_BIDIRECTIONAL);
	ich->done = OWNER_N9;
	dma_sync_single_for_device(NULL, gDev->dma_handle[NTA_ISR_CHANNEL],
			N9_CHANNEL_SIZE - 32, DMA_BIDIRECTIONAL);
	return IRQ_HANDLED;
}

static int firmware_update(void)
{
	const char *fw_name = "RDEMO.bin";
	const struct firmware *fw_p;
	struct device dev;
	void* fw_base;
	fw_base = phys_to_virt(IVB_BASE);

	if (request_firmware(&fw_p, fw_name, &dev)) {
		pr_info("%s - %s not found\n", __func__, fw_name);
		return -ENODEV;
	}
	/*XXX do we need to flush here?*/
	memcpy(fw_base, fw_p->data, fw_p->size);
	pr_info("n9 firmware_size:%d\n", fw_p->size);
	return 0;
}

void n9_ipc_channel_init(void)
{
	int i;
	unsigned long addr;
	for (i = 0 ; i < N9_CHANNEL_NUM ; i++) {
		addr = N9_IPC_CHANNEL_BASE_ADDR + i * N9_CHANNEL_SIZE;
		gDev->ch[i] = (ipc_channel_t*)phys_to_virt(addr);
		gDev->dma_handle[i] = dma_map_single(NULL, phys_to_virt(addr) + 32,
				N9_CHANNEL_SIZE - 32, DMA_BIDIRECTIONAL);

		dma_sync_single_for_cpu(NULL, gDev->dma_handle[i],
				N9_CHANNEL_SIZE - 32, DMA_BIDIRECTIONAL);

		memset(gDev->ch[i], 0, sizeof(ipc_channel_t));


		gDev->n9_status[i] = dma_map_single(NULL, phys_to_virt(addr),
				32, DMA_FROM_DEVICE);
		dma_sync_single_for_device(NULL, gDev->n9_status[i],
				32, DMA_FROM_DEVICE);
		if (NTA_ISR_CHANNEL == i) {
			gDev->n2a_isr_handle = dma_map_single(NULL, phys_to_virt(addr) + 32,
					N9_CHANNEL_SIZE - 32, DMA_FROM_DEVICE);
			dma_sync_single_for_device(NULL, gDev->n2a_isr_handle,
					N9_CHANNEL_SIZE - 32, DMA_FROM_DEVICE);
		} else if (ATN_ISR_CHANNEL == i) {
			gDev->a2n_isr_handle = dma_map_single(NULL, phys_to_virt(addr) + 32,
					N9_CHANNEL_SIZE - 32, DMA_FROM_DEVICE);
			gDev->ch[i]->done = OWNER_ARM;
			dma_sync_single_for_device(NULL, gDev->dma_handle[i],
					N9_CHANNEL_SIZE - 32, DMA_BIDIRECTIONAL);
			dma_sync_single_for_device(NULL, gDev->a2n_isr_handle,
					N9_CHANNEL_SIZE - 32, DMA_FROM_DEVICE);
		}
#if 0
		printk("%d, dma_handle:0x%x\n", i, gDev->dma_handle[i]);
		printk("%d, n9_status:0x%x\n", i, gDev->n9_status[i]);
		printk("%d, n2a_isr_handle:0x%x\n", i, gDev->n2a_isr_handle);
		printk("%d, a2n_isr_handle:0x%x\n", i, gDev->a2n_isr_handle);
#endif
	}
	return;
}

static int n9_osde_probe(struct platform_device *pdev)
{
	struct n9_dev   *n9_dev;
	struct class    *sysfs_class;
	struct device   *dev;
	int val, err, ret, major, virq;
	dev_t dev_no;

	/* check clock is enable */
	val = readl(SYSC_CLK_EN_CTRL_1);
	if ((val & 0x1<<1) == 0) {
		pr_err("NO N9 clock!\n");
		return -EINVAL;
	}

	ret = firmware_update();
	if (ret) {
		return ret;
	}

	n9_dev = kzalloc(sizeof(struct n9_dev), GFP_KERNEL);
	gDev = n9_dev;
	gDev->start = 0;
	gDev->resetAddr = (char*)IVB_BASE; 

	// set SYSC N9 reset/IVB base
	val = readl(SYSC_N903U_CTRL);
	val |= (IVB_BASE >> 20) << N903U_IVB_off;
	writel( val, SYSC_N903U_CTRL);

	virq = irq_create_mapping(NULL, N903U_IRQ_NUM);
	err = request_irq(virq, n9_osde_isr, IRQF_DISABLED | IRQF_TRIGGER_RISING,
			"n9_osde", pdev);
	if (err) {
		dev_err(&pdev->dev, "failed to allocate irq.\n");
	}

	// unmask and set edge trigger
#if 0
	val = readl(IO_ADDRESS(VPL_INTC_MMR_BASE + INTC_TRIGGER_MODE_LO));
	val |= 0x1 << ARM926U_INTC_NUM;
	writel(val, IO_ADDRESS(VPL_INTC_MMR_BASE + INTC_TRIGGER_MODE_LO));

	val = readl(IO_ADDRESS(VPL_INTC_MMR_BASE + INTC_HOSTU_0_MASK_LO));
	val |= 0x1 << ARM926U_INTC_NUM;
	writel(val, IO_ADDRESS(VPL_INTC_MMR_BASE + INTC_HOSTU_0_MASK_LO));
#endif

	init_waitqueue_head(&buf_wq);

	cdev = cdev_alloc();
	if (!cdev)
		return -ENOMEM;
	cdev->ops = &n9_fops;
	cdev->owner = THIS_MODULE;

	ret = alloc_chrdev_region(&dev_no, 0 , 1, "n9");
	if (ret < 0) {
		printk(KERN_WARNING "%s, allocate major number failed!\n", __func__);
		return -ENODEV;
	}
	major = MAJOR(dev_no);
        ret = cdev_add(cdev, dev_no, 1);
	if (ret < 0)
		return -ENOMEM;

	// create n9 sysfs
	sysfs_class = class_create(THIS_MODULE, "n9_osde");
	if (!sysfs_class) {
		dev_err(&pdev->dev, "couldn't create class\n");
		return -ENODEV;
	}
	dev = device_create(sysfs_class, NULL, MKDEV(major, 0),
							NULL, "n9");
	
	if (dev) {
		device_create_file(dev, &dev_attr_intr);
		device_create_file(dev, &dev_attr_start);
		device_create_file(dev, &dev_attr_resetAddr);
	}

	n9_ipc_channel_init();

	cmdAddr = phys_to_virt(g_cmdAddr);
	return 0;
}

static int n9_osde_remove(struct platform_device *pdev)
{
	return 0;
}

static struct platform_driver n9_osde_driver = {
	.probe		= n9_osde_probe,
	.remove		= n9_osde_remove,
	.driver		= {
		.name	= "n9_osde",
	},
};
static struct platform_device *n9_device;
static int __init n9_osde_init(void)
{
	n9_device = platform_device_alloc("n9_osde", 0);
	platform_device_add(n9_device);
	return platform_driver_register(&n9_osde_driver);
}

static void __exit n9_osde_exit(void)
{
        cdev_del(cdev);
	platform_driver_unregister(&n9_osde_driver);
}

module_init(n9_osde_init);
module_exit(n9_osde_exit);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("N9 OSDE driver");
MODULE_AUTHOR("Paul Chen <paul.chen@vatics.com.tw>");

