/*
 * Watchdog Driver for VPL watchdog timer.
 *
 * Copyright (C) 2013-2018  VATICS Inc.
 *
 * Author: ChangHsien Ho <vincent.ho@vatics.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/types.h>
#include <linux/watchdog.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/clk.h>
#include <linux/err.h>
#include <linux/platform_device.h>
#include <linux/io.h>

#include "vpl_wdt.h"

static bool nowayout = WATCHDOG_NOWAYOUT;
module_param(nowayout, bool, 0);
MODULE_PARM_DESC(nowayout, "Watchdog cannot be stopped once started (default="
			__MODULE_STRING(WATCHDOG_NOWAYOUT) ")");

static unsigned int timeout = WDTC_DEFAULT_TIMEOUT;
module_param(timeout, uint, 0);
MODULE_PARM_DESC(timeout, "Watchdog timeout in seconds (default = 60s)");

static bool testmode = 0;
module_param(testmode, bool, 0);
MODULE_PARM_DESC(testmode, "Watchdog test mode (1 = no reboot) (default = 0)");

struct vpl_wdt_drvdata {
	void __iomem    *base;	/* physical */
	struct device   *dev;
	int		irq;
	u32		clk_freq;
	struct clk	*clk;
};

static void vpl_wdt_set_timer(struct vpl_wdt_drvdata *vpl_wdev, unsigned int timeout)
{
	void __iomem *base = vpl_wdev->base;
	writel(timeout * vpl_wdev->clk_freq, base + WDTC_RELOAD_VALUE);
}

static void vpl_wdt_reload(struct vpl_wdt_drvdata *vpl_wdev)
{
	void __iomem *base = vpl_wdev->base;
	writel(WDTC_RELOAD_PASSWD, base + WDTC_RELOAD_CTRL);
}

static void vpl_wdt_enable(struct vpl_wdt_drvdata *vpl_wdev)
{
	void __iomem *base = vpl_wdev->base;
	u32 ctrl;
	ctrl = readl(base + WDTC_CTRL);
	if (testmode) {
		writel(0, base + WDTC_RST_LEN);
		ctrl |= WDTC_CTRL_ACK_EN;
	} else
		writel(WDTC_RESET_LEN, base + WDTC_RST_LEN);

	writel(ctrl | WDTC_CTRL_OP_EN, base + WDTC_CTRL);
}

static void vpl_wdt_disable(struct vpl_wdt_drvdata *vpl_wdev)
{
	void __iomem *base = vpl_wdev->base;
	u32 ctrl;
	ctrl = readl(base + WDTC_CTRL);
	ctrl &= ~WDTC_CTRL_OP_EN;
	writel(ctrl, base + WDTC_CTRL);
}

static int vpl_wdt_start(struct watchdog_device *wdd)
{
	struct vpl_wdt_drvdata *vpl_wdev = watchdog_get_drvdata(wdd);

	vpl_wdt_set_timer(vpl_wdev, wdd->timeout);
	vpl_wdt_reload(vpl_wdev);
	vpl_wdt_enable(vpl_wdev);

	return 0;
}

static int vpl_wdt_stop(struct watchdog_device *wdd)
{
	struct vpl_wdt_drvdata *vpl_wdev = watchdog_get_drvdata(wdd);

	vpl_wdt_disable(vpl_wdev);

	return 0;
}

static int vpl_wdt_ping(struct watchdog_device *wdd)
{
	struct vpl_wdt_drvdata *vpl_wdev = watchdog_get_drvdata(wdd);

	vpl_wdt_reload(vpl_wdev);

	return 0;
}

static int vpl_wdt_set_timeout(struct watchdog_device *wdd, unsigned int timeout)
{
	struct vpl_wdt_drvdata *vpl_wdev = watchdog_get_drvdata(wdd);

	vpl_wdt_set_timer(vpl_wdev, timeout);
	vpl_wdt_reload(vpl_wdev);
	wdd->timeout = timeout;

	return 0;
}

static irqreturn_t vpl_wdt_irq(int irqno, void *param)
{
	struct platform_device *pdev = param;
	struct watchdog_device *wdd = platform_get_drvdata(pdev);
	dev_info(&pdev->dev, "watchdog timer expired (irq)\n");

	vpl_wdt_ping(wdd);
	return IRQ_HANDLED;
}

static const struct watchdog_info vpl_wdt_info = {
	.options = WDIOF_SETTIMEOUT | WDIOF_KEEPALIVEPING,
	.firmware_version = VPL_WDT_VERSION,
	.identity = "VPL Watchdog",
};

static const struct watchdog_ops vpl_wdt_ops = {
	.owner      = THIS_MODULE,
	.start      = vpl_wdt_start,
	.stop       = vpl_wdt_stop,
	.ping       = vpl_wdt_ping,
	.set_timeout	= vpl_wdt_set_timeout,
};

static int vpl_wdt_probe(struct platform_device *pdev)
{
	struct device *dev;
	struct watchdog_device *wdd;
	struct resource *res;
	int irq;
	struct vpl_wdt_drvdata *vpl_wdev;
	int ret;

	dev = &pdev->dev;

	wdd = devm_kzalloc(dev, sizeof(*wdd), GFP_KERNEL);
	if (!wdd) {
		dev_err(dev, "unable to alloacate watchdog device\n");
		return -ENOMEM;
	}

	vpl_wdev = devm_kzalloc(dev, sizeof(*vpl_wdev), GFP_KERNEL);
	if (!vpl_wdev) {
		dev_err(dev, "unable to alloacate driver data\n");
		return -ENOMEM;
	}

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		dev_err(dev, "no memory resource specified\n");
		return -ENOENT;
	}

	vpl_wdev->base = devm_ioremap_resource(dev, res);
	if (IS_ERR(vpl_wdev->base))
		return PTR_ERR(vpl_wdev->base);

	vpl_wdev->clk = devm_clk_get(dev, NULL);
	if (IS_ERR(vpl_wdev->clk))
		return PTR_ERR(vpl_wdev->clk);

	vpl_wdev->clk_freq = clk_get_rate(vpl_wdev->clk);
	if (!vpl_wdev->clk_freq)
		return -EINVAL;

	irq = platform_get_irq(pdev, 0);
	if (irq < 0) {
		dev_err(dev, "no irq resource specified\n");
		return irq;
	}

	ret = devm_request_irq(dev, irq, vpl_wdt_irq, 0,
			"vpl_wdt", pdev);
	if (ret) {
		dev_err(dev, "fail to install irq (%d)\n", ret);
		return ret;
	}

	vpl_wdev->irq = irq;
	vpl_wdev->dev = &pdev->dev;

	wdd->info		= &vpl_wdt_info;
	wdd->ops		= &vpl_wdt_ops;
	wdd->timeout		= WDTC_DEFAULT_TIMEOUT,
	wdd->min_timeout	= 1;
	wdd->max_timeout	= (UINT_MAX / vpl_wdev->clk_freq);

	watchdog_init_timeout(wdd, timeout, &pdev->dev);
	watchdog_set_drvdata(wdd, vpl_wdev);
	watchdog_set_nowayout(wdd, nowayout);

	platform_set_drvdata(pdev, wdd);

	ret = watchdog_register_device(wdd);
	if (ret) {
		dev_err(dev, "watchdog_register_device() failed: %d\n",	ret);
		return ret;
	}

	dev_info(dev, "VPL Watchdog Timer Hardware Rev 0x%08x\n",
		readl(vpl_wdev->base + WDTC_VERSION));

	dev_info(dev, "Driver Version %d.%d.%d.%d: initial timeout %d sec\n",
		wdd->info->firmware_version & 0xff,
		(wdd->info->firmware_version >> 8) & 0xff,
		(wdd->info->firmware_version >> 16) & 0xff,
		wdd->info->firmware_version >> 24,
		wdd->timeout);

	return 0;
}

static int vpl_wdt_remove(struct platform_device *pdev)
{
	struct watchdog_device *wdd = platform_get_drvdata(pdev);

	watchdog_unregister_device(wdd);

	return 0;
}

static const struct of_device_id vpl_wdt_of_match[] = {
	{ .compatible = "vatics,vpl-wdt" },
	{},
};
MODULE_DEVICE_TABLE(of, vpl_wdt_of_match);

static struct platform_driver vpl_wdt_driver = {
	.probe		= vpl_wdt_probe,
	.remove		= vpl_wdt_remove,
	.driver		= {
		.owner  = THIS_MODULE,
		.name   = "vpl_wdt",
		.of_match_table = vpl_wdt_of_match,
	},
};

module_platform_driver(vpl_wdt_driver);
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("VPL Watchdog Timer driver");
