/*
 *  agpo_consumer.c - Sample AGPO consumer driver
 *
 * Copyright (C) 2013-2018  VATICS Inc.
 *
 * Author: ChangHsien Ho <vincent.ho@vatics.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/err.h>
#include <linux/vpl_agpo.h>

/*
 * This sample driver will show how to configur a AGPO port with
 * cfg1 pattern, repeat 5 times.
 * Interrupt will be trigger after finish cfg1, then config to cfg2 pattern
 * repeat 20 times.
 */

static struct agpo_desc *agpo = NULL;
static int virq;
static int port_num = 2;

static struct agpo_pattern_cfg cfg1 = {
	.pattern = {0x000000AA, 0x0, 0x0, 0x0},
	.period = 20000000,
	.bits_length = 8,
	.repeat = 5,
	.interval = 12,
};

static struct agpo_pattern_cfg cfg2 = {
	.pattern = {0x0000000F, 0x0, 0x0, 0x0},
	.period = 2000000,
	.bits_length = 5,
	.repeat = 20,
	.interval = 5,
};

static irqreturn_t agpo_isr(int irq, void *data)
{
	/* coinfgur to cfg2 */
	agpo_config(agpo, &cfg2);
	/* kick enable */
	agpo_enable(agpo);
	return IRQ_HANDLED;
}

int __init agpo_consumer_module_init(void)
{
	int ret;

	/* reques port_num agpo */
	agpo = agpo_request(port_num, NULL);
	if (IS_ERR(agpo))
		printk("request fail %ld\n", PTR_ERR(agpo));

	/* set to open source type */
	ret = agpo_set_type(agpo, AGPO_PORT_TYPE_OPEN_SOURCE, 0);
	if (ret)
		printk("set type fail %d\n", ret);

	/* configure to cfg1 */
	ret = agpo_config(agpo, &cfg1);
	if (ret)
		printk("config fail %d\n", ret);

	/* get virq number and register a ISR */
	virq = agpo_to_irq(agpo);
	ret = request_irq(virq, agpo_isr, 0, "agpo-test", NULL);

	/* enable agpo */
	ret = agpo_enable(agpo);
	if (ret)
		printk("enable fail %d\n", ret);

	return 0;
}

void __exit agpo_consumer_module_exit(void)
{
	free_irq(virq, NULL);
	agpo_disable(agpo);
	agpo_free(agpo);
}

module_init(agpo_consumer_module_init);
module_exit(agpo_consumer_module_exit);

MODULE_LICENSE("GPL");
